/*
#       lidsadm.c --- The Linux Intrusion Detection System Configuration Tool 
#       (C) Huagang Xie 1999-2001 All rights reserved.
#       EMail complaints to xie@gnuchina.org
#
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 2 of the License, or
#       (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#
*/

/* ------------------------------------------------------------------------- */

/* Includes */

#ifdef HAVE_CONFIG_H
#include <../config.h>
#endif

/*#define _GNU_SOURCE*/
#include <stdio.h>
#include <limits.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <linux/capability.h>
#include <linux/kdev_t.h>
#include <string.h>
#include <getopt.h>

#include "lidstools.h"
#include "lids_capflag.h"

#include <linux/lidsif.h>

/*#define DEBUG*/

#define LIDS_STR2(x) #x
#define LIDS_STR(X) LIDS_STR2(X)

#ifdef DEBUG
#define LIDS_DBG(msg...)  printf("LIDS." __FUNCTION__ ".l" LIDS_STR(__LINE__) ": " ##msg)
#else
#define LIDS_DBG(msg...)
#endif



/* ------------------------------------------------------------------------- */


/*  globals */

char lids_type_desc[6][9] = {"DENY","READONLY","APPEND","WRITE","IGNORE","GRANT\0"};
int  lids_type_val[6] = { LIDS_DENY,
			  LIDS_DENY|LIDS_READONLY,
			  LIDS_DENY|LIDS_READONLY|LIDS_APPEND,
			  LIDS_DENY|LIDS_READONLY|LIDS_APPEND|LIDS_WRITE,
			  LIDS_DENY|LIDS_READONLY|LIDS_APPEND|LIDS_WRITE,
			  LIDS_CAP
};

char lids_special_target_val[2] = {0,1};

lids_t lids[1024];
int	last;
int 	bindport[LIDS_PORT_ITEM][2]; 


/* ------------------------------------------------------------------------- */


void exit_error (int status, char *msg)
{
	fprintf (stderr, "lidsconf: %s\n", msg);
	if(status == 3 )
		perror("reason:");
	printf("\n");
	exit (status);
}

void exit_version ()
{
	printf ("lidsconf version " VERSION " for the LIDS project\n");
	exit(1);
}

void exit_normal ()
{
	printf ("lidsconf version " VERSION " for the LIDS project\n"
		"Use 'lidsconf -h' for help\n");
	exit(1);
}

void exit_help ()
{
	entry_t *entry;

	
	printf ("lidsconf version " VERSION " for the LIDS project\n"
		"       Huagang Xie<xie@gnuchina.org>\n"
		"       Philippe Biondi <philippe.biondi@webmotion.net>\n\n"
		"Usage: lidsconf -A [-s subject] -o object [-d] [-t from-to] [-i level] -j ACTION\n"
		"       lidsconf -D [-s file] [-o file] \n"
		"       lidsconf -Z\n"
		"       lidsconf -U\n"
		"       lidsconf -L [-e]\n"
		"       lidsconf -P\n"
		"       lidsconf -v\n"
		"       lidsconf -h\n\n"
		"Commands:\n"
		"       -A,--add	To add an entry\n"
		"       -D,--delete	To delete an entry\n"
		"       -Z,--zero	To delete all entries \n"
		"       -U,--update	To update dev/inode numbers\n"
		"       -L,--list	To list all entries \n"
		"       -P,--passwd	To encrypt a password with RipeMD-160\n"
		"       -v,--version	To show the version\n"
 		"       -h,--help	To list this help \n\n"
		"subject: -s,--subject subj\n"
		"       can be any program, must be a file\n" 
		"object: -o,--object [obj]\n"
		"       can be a file, directory or special device (e.g. MEM, HD, NET, IO,\n"
                "                                                        HIDDEN, KILL)\n"
		"ACTION: -j,--jump\n"
		"       DENY     deny access\n"
		"       READONLY read only\n" 
		"       APPEND   append only\n"
	        "       WRITE    writable\n"
		"       GRANT    grant capability to subject\n"
		"       IGNORE   ignore any permissions set on this object\n"
		
		"OPTION:\n"	
		"      -d,--domain	The object is an EXEC Domain\n"
		"      -i,--inheritance Inheritance level\n"
		"      -t,--time	Time dependency\n"
		"      -e,--extended	Extended list\n"
);

	printf("\nAvailable capabilities:\n");
	for_each_entry(cap_list,entry)
		printf("%20s %s\n",entry->name,entry->desc);
	


	exit(1);
}




void lids_del_all()
{
	FILE *fd;
	
	if((fd = fopen(LIDS_CONF_FILE,"w"))== NULL ) {
		exit_error(3,"cannot open "LIDS_CONF_FILE);
	}
	fwrite(LIDS_CONF_COMMENT,1,strlen(LIDS_CONF_COMMENT),fd);
	fclose(fd);
}

/*
 *	lids_get_str get the ino,dev,filename from the buff.
 *	the format of buffer is "inode:dev:filename"
 */
int lids_get_str(char *buffer,unsigned long * ino,int *dev,char *filename)
{
	char    *p,*q;
	char 	ino_str[1024];
	char 	ino_dev[1024];

	memset(ino_str,'\0',1024);
	memset(ino_dev,'\0',1024);

	p = q = NULL;
	p = strchr(buffer,':');	
	if ( p == NULL ) return -1; 
	*p = '\0';
	strncpy(ino_str,buffer,p-buffer);
	p++;
	q = strchr(p+1,':');
	if ( q == NULL ) return -1; 
	strncpy(ino_dev,p,q-p);

	strcpy(filename,q+1);
	*ino = atol(ino_str);
	*dev = atoi(ino_dev);
	return 0;
}
/*
 * check_format check if the given buf is the valid string for current version
 */
int check_format(char *buf)
{
	int i,j;
	char *p=NULL;

	i=0;
	p = strchr(buf,':');
	while(p!=NULL) {
		p = strchr(p+1,':');
		i++;
	}
	if(i!=8) return -1;
	return 0;
}
/*
 *  lids read conf read /etc/lids.conf into a data structure.
 */ 
void lids_read_conf()
{
	FILE 	*fd;
	int	i=0,type;
	char 	buffer[1024];
	char    *p,*q;
		
	if((fd = fopen(LIDS_CONF_FILE,"r"))== NULL ) {
		exit_error(3,"cannot open "LIDS_CONF_FILE);
	}
	while(fgets(buffer,1024,fd) && i < 1024) {
		if ( buffer[0] == '#' ) continue;
		if ( buffer[strlen(buffer)-1] == '\n' )
			buffer[strlen(buffer)-1] = '\0' ;
		if(check_format(buffer) < 0) {
			printf("syntax error in %s\n", LIDS_CONF_FILE);
			exit(-1);
		}
		p = strchr(buffer,':');	
		if ( p == NULL ) continue; 
		q = strchr(p+1,':');
		if ( q == NULL ) continue; 
		p = strchr(q+1,':');	
		if ( p == NULL || *(p+1) == ':' ) continue; 
		*p = '\0';
		/* get the permission type */
		q = strchr(p+1,':');
		if(q==NULL) continue;
		
		*q='\0';
		lids[i].type = atoi(p+1);

		/* get the inherit level */ 	
		p = strchr(q+1,':');
		if ( p == NULL || *(p+1) == ':' ) continue;
		*p = '\0';
		lids[i].inherit = atoi(q+1);

		/* get the time like :19283-89283 */
		q = rindex(p+1,':');
		*q = '\0';

		if( lids_get_time(q+1,lids[i].time)<0 ) 
			continue;
		
		if (lids_get_str(buffer,&lids[i].s_ino,&lids[i].s_dev,lids[i].s_file) <0 ) continue;
		
		if (lids_get_str(p+1,&lids[i].o_ino,&lids[i].o_dev,lids[i].o_file) <0 ) continue;
		if ((lids[i].type==LIDS_CAP) && (lids[i].o_dev==getentrybyname(cap_list,"CAP_NET_BIND_SERVICE")->val))
			str2data(p+1,lids[i].port,1,LIDS_PORT_ITEM);

#ifdef DEBUG
		printf("inode= %d, dev= %d, file= %s, type= %d, inherit= %d, o_inode= %ld, o_dev= %d, o_file= %s, from:%ld, to_%ld\n",lids[i].s_ino,
                                                                                                                    lids[i].s_dev,
                                                                                                                    lids[i].s_file,
                                                                                                                    lids[i].type,
                                                                                                                    lids[i].inherit,
                                                                                                                    lids[i].o_ino,
                                                                                                                    lids[i].o_dev,
                                                                                                                    lids[i].o_file,
														    lids[i].time[0][0],
														    lids[i].time[1][0]);
#endif
		i++;
	}
	last = i;

	fclose(fd);
}

void lids_write_file()
{
	int i=0;
	FILE *fd;
	
	if((fd = fopen(LIDS_CONF_FILE,"w"))== NULL ) {
		exit_error(3,"cannot open "LIDS_CONF_FILE);
	}
	fwrite(LIDS_CONF_COMMENT,1,strlen(LIDS_CONF_COMMENT),fd);
	for(i=0;i<last;i++) {
		if(lids[i].type != LIDS_DELETE_FLAG ) {
			if(lids[i].type == LIDS_CAP && 
			  lids[i].o_dev==getentrybyname(cap_list,"CAP_NET_BIND_SERVICE")->val)

				fprintf(fd,"%d:%d:%s:%d:%d:%s:%d:%s:%s\n",
					lids[i].s_ino, lids[i].s_dev,
                                        lids[i].s_file, lids[i].type,
                                        lids[i].inherit,
                                        disp_multi_data(lids[i].port,1),
					lids[i].o_dev,
                                        lids[i].o_file, 
					disp_multi_data(lids[i].time,0),0);
			else 
				fprintf(fd,"%d:%d:%s:%d:%d:%d:%d:%s:%s\n",
					lids[i].s_ino, lids[i].s_dev,
                                        lids[i].s_file, lids[i].type,
                                        lids[i].inherit,
                                        lids[i].o_ino, lids[i].o_dev,
                                        lids[i].o_file, 
					disp_multi_data(lids[i].time,0),0);
		}
	}

	fclose(fd);
}
/*
 *	get the str desc number from type number
 */ 
int get_type_str(int type)
{
	int k,j;
	
	k = -1;
	for(j=0;j<6;j++) {
		if(type == lids_type_val[j]) {
			k=j;
			break;
		}
	}
	if(k == -1 ) exit_error(2,"type mismatch in "LIDS_CONF_FILE);
	return k;
}

void lids_list_file(int type)
{
	int i,j,k;
	static char anyfile[] = "Any file";
	char *src,*obj;
	
	lids_read_conf();
#ifdef DEBUG
	printf("last = %d \n",last);
#endif
	if ( type == LIDS_EXTEND) 
		printf("Subj ino,dev   Obj ino,dev  type  ");
	printf("                Subject   ACCESS(inherit)        time        Object\n");
                                                                           
	printf("-----------------------------------------------------\n");
	for(i=0;i<last;i++) {
		src=anyfile;
		obj=anyfile;

		if(*lids[i].s_file)
			src=lids[i].s_file;

		if(*lids[i].o_file)
			obj=lids[i].o_file;

		if( type == LIDS_EXTEND) 
			printf("%6d,%6d %6d,%6d %6d",lids[i].s_ino,lids[i].s_dev,lids[i].o_ino,lids[i].o_dev,lids[i].type);


		printf("%23s  %8s%s:%3d  %s  %20s %s\n",
		       src,
		       lids_type_desc[get_type_str(abs(lids[i].type))],
		       lids[i].type < 0 ? "" : "(domain)",
		       lids[i].inherit,
		       time2str(lids[i].time),
		       obj,
		       strncmp("CAP_NET_BIND_SERVICE",obj,20) ? "" : disp_multi_data(lids[i].port,1) );		
	}
	printf("\n\n");
}
/*
 *	lids_search_file()
 *
 *	lids search the file by given file and type.
 *
 */
int lids_search_file(char *s_file,char *o_file) 
{
	int i ,flag,number=0;

	for ( i = 0; i<last ; i++ ) {
		flag = 0;
		if (*s_file != '\0') {
		 	flag = strcmp(lids[i].s_file , s_file);
		}
		if (*o_file != '\0')
		 	flag |= strcmp(lids[i].o_file , o_file);
		if(!flag) {
			number++;
			lids[i].type = LIDS_DELETE_FLAG;
		}
	}
        printf("delete %d items\n",number);
	return number;
}

/*
 *	lids_search_inode()
 *
 *	lids search the file by given inode , dev  and type.
 *
 */
int lids_search_inode(struct stat s_st,struct stat o_st) 
{
	int i;

	for ( i = 0; i<last ; i++ ) {
		if ( lids[i].s_ino == s_st.st_ino && 
		     lids[i].s_dev == s_st.st_dev &&
		     lids[i].o_ino == o_st.st_ino && 
		     lids[i].o_dev == o_st.st_dev 
		     ) {   
			return lids[i].type;
		}
	}	
	return LIDS_DELETE_FLAG;
}
void lids_del_file(char *subject_file,char *object_file)
{
	int i;
	char canonical_s_path[PATH_MAX];
	char canonical_o_path[PATH_MAX];

	lids_read_conf();

	i = lids_search_file(subject_file,object_file);
		
	if(!i) {
		realpath(subject_file,canonical_s_path);
		realpath(object_file,canonical_o_path);
		i = lids_search_file(canonical_s_path,canonical_o_path);
		if (!i) 
			exit_error(2,"the file does not exit in "LIDS_CONF_FILE);
	}

	lids_write_file();
}
/*
 *	get special type from typestr, when lidsadm -A -s ..-o .. -t -j ..
 */ 

int get_object_type(char *filename)
{
	int i;
	char o_file[1024];
	char *p=NULL;
		
	strcpy(o_file,filename);
	strcat(o_file,"/");

	while( p != o_file) { 
		p = rindex(o_file,'/');
		if(!p)
			break;
		*p = '\0';
		for(i=0; i<last;i++) {
			if(*(lids[i].s_file)=='\0' && strcmp(o_file,lids[i].o_file) == 0 ) {
				return lids[i].type;

			}
		}
	}
	return -1;
	
}
/*
 *	lids_add_file 
 */ 
void lids_add_file(char *subject_file, char *object_file,int type,int inherit,int domain,time_t timescale[][2])
{
	struct stat s_st,o_st,z_st;
	char canonical_s_path[PATH_MAX];
	char canonical_o_path[PATH_MAX];
	int  default_rule=-1;
	int  sys_cap = 0;	/* 1 for system cap, 0 for individual cap */
	entry_t *entry;

	lids_read_conf();

        if ( last >= 1024 ) {
                exit_error(2, "cannot exceed 1024 entries.");
        }
	
#ifdef DEBUG	
	printf("sub=[%s],type=%d\n",subject_file,type);
#endif
	if(*subject_file == '\0' ) {
		sys_cap = 1;
	}


	if(!sys_cap) {	
		realpath(subject_file,canonical_s_path);
		if ( stat(canonical_s_path,&s_st) ) {
			exit_error(2, " cannot find the subject file");
		}
	}
	else {
		s_st.st_ino = s_st.st_dev = 0;
	}
	if(type != LIDS_CAP) {
		realpath(object_file,canonical_o_path);
		if(stat(canonical_o_path,&o_st) ) {
			exit_error(2, " cannot find the object file");
		}
	}
	else {
		o_st.st_ino = -1;	/* special type's ino */
		if (entry=getentrybyname(cap_list,object_file)) {
			o_st.st_dev=entry->val;
		} else {
			exit_error(2, "special type must be one of CAP_XXX_XXX. use lidsadm -h for details.");
		}
		/*
		if(o_st.st_dev == getentrybyname(cap_list,"CAP_NET_BIND_SERVICE")) {
			o_st.st_ino = bindport;
		}
		*/
		strcpy(canonical_o_path,object_file);
	}
	/* check the subject */
	if(!sys_cap) {
		if((default_rule = get_object_type(canonical_s_path)) == -1)
			exit_error(2,"the subject file is not protected");
		if(default_rule > LIDS_READONLY )
			exit_error(2,"the subject files must be protected by READONLY or DENY"); 
	}
	/* check the object */
	if(type != LIDS_CAP) {
		if( (default_rule = get_object_type(canonical_o_path)) == -1) {
			if(!sys_cap)
				exit_error(2,"you must define the default rules for object files");
		}
	}
	
	if(domain) type = -type;

	if (!(lids_search_inode(s_st,o_st) != type ||
		lids_search_inode(s_st,o_st) == LIDS_DELETE_FLAG))
			exit_error(2,"a file with the same inode and dev is already referenced.");

	lids[last].s_ino= s_st.st_ino;
	lids[last].s_dev = s_st.st_dev;
	lids[last].o_ino= o_st.st_ino;
	lids[last].o_dev = o_st.st_dev;
#ifdef DEBUG	
	printf("canonical_o_file=%s,o_ino =%d\n",canonical_o_path,o_st.st_ino);
#endif
	lids[last].type = type;
	lids[last].inherit = inherit;

	strcpy(lids[last].s_file,canonical_s_path);
	strcpy(lids[last].o_file,canonical_o_path);

	memcpy(lids[last].time, timescale, LIDS_TIME_ITEM*2*sizeof(int *));
	memcpy(lids[last].port, bindport, LIDS_PORT_ITEM*2*sizeof(int *));
		
	last++;
	lids_write_file();
}

void lids_update()
{
	struct stat s_st,o_st;
	int i;
	lids_read_conf();
	

	for (i=0; i<last; i++) {
		/*
		 *	check subject files
		 */ 
		if (*lids[i].s_file != '\0' ) {
			if(stat(lids[i].s_file,&s_st)) {
				printf("%s doesn't exist anymore. not removed.\n",lids[i].s_file);
			}
			else {
				if ((s_st.st_ino != lids[i].s_ino) || (s_st.st_dev != lids[i].s_dev) ) {
				printf("subject file %s was (%u:%u inode %lu) instead of (%u:%u %lu). corrected.\n",
				       lids[i].s_file,
				       MAJOR(lids[i].s_dev),MINOR(lids[i].s_dev),lids[i].s_ino,
				       MAJOR((unsigned long)s_st.st_dev),MINOR((unsigned long)s_st.st_dev),s_st.st_ino);
				lids[i].s_ino = s_st.st_ino;
				lids[i].s_dev = s_st.st_dev;
				}
			}	
		}
		/*
		 *	check object files
		 */ 
		if (*lids[i].o_file != '\0' && stat(lids[i].o_file,&o_st)) {
			if( (getentrybyname(cap_list,lids[i].o_file) < 0) && lids[i].type != LIDS_CAP) {
				printf("%s doesn't exist anymore. not removed.\n",lids[i].o_file);
			}
		}
		else {
			if (o_st.st_ino != lids[i].o_ino || o_st.st_dev != lids[i].o_dev) {
				printf("object file %s was (%u:%u inode %lu) instead of (%u:%u %lu). corrected.\n",
				       lids[i].o_file,
				       MAJOR(lids[i].o_dev),MINOR(lids[i].o_dev),lids[i].o_ino,
				       MAJOR((unsigned long)o_st.st_dev),MINOR((unsigned long)o_st.st_dev),o_st.st_ino);

				lids[i].o_ino = o_st.st_ino;
				lids[i].o_dev = o_st.st_dev;
			}
		}
	}
	lids_write_file();
}

void lids_make_rmd160_passwd()
{
	char passwd[64*2];
	int times=0;
	int ok;
	FILE *fd;
	
	memset(passwd,'\0',64*2);
	while(times < 3) {
		times++;	
		if ((ok=read_rmd160_passwd(passwd,1,2))==0)  
			break;
	}
	
	if (ok==0) {
		fd = fopen(LIDS_PW_FILE,"w");
		if (fd == NULL) return;	
		fwrite(passwd,1,strlen(passwd),fd);
		fclose(fd);
		printf("wrote password to %s\n", LIDS_PW_FILE); 
	}
	
}
	

static char shortopts[] = "UADZLPs:o::j:i:deht:v";
static struct option longopts[] = {
	{ "update", 0, 0, 'U' },
	{ "add", 0, 0, 'A' },
	{ "delete", 0, 0, 'D' },
	{ "zero", 0, 0, 'Z' },
	{ "list", 0, 0, 'L' },
	{ "passwd", 0, 0, 'P' },
	{ "subject", 1, 0, 's' },
	{ "object", 2, 0, 'o' },
	{ "jump", 1, 0, 'j' },
	{ "inheritance", 1, 0, 'i' },
	{ "domain", 0, 0, 'd' },
	{ "extended", 0, 0, 'e' },
	{ "time", 1, 0, 't' },
	{ "help", 0, 0, 'h' },
	{ "version", 0, 0, 'v' },
	{ 0, 0, 0, 0 }
};


main(int argc,char **argv)
{
	int	command = LIDS_NONE;
	int 	type=LIDS_NONE;
	char 	subject_file[1024],object_file[1024];
	char 	del_file[1024];
	char 	sw_flag[16];
	char 	lids_time[176];
	int	c,i;
	int     index=0;
	int	cap_type=0; /* program capability type */
	int	inherit=0;/* if the acl is inheritable*/
	int	domain = 0;/* here define the DOMAIN */
	time_t	timescale[LIDS_TIME_ITEM][2];


	setentry(cap_list);
	memset(subject_file,'\0',1024);
	memset(object_file,'\0',1024);
	memset(timescale,0,sizeof(timescale));
	if(getuid()!=0) {
		exit_error(2, "you must be root to run this program");
	}
	while ((c = getopt_long (argc, argv, shortopts, longopts, &index)) != -1) {
        	switch (c)
			{
		        case 'U':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_UPDATE;
				break;
			case 'A':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_ADD;
				break;
			case 'D':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_DELETE;
				break;
			case 'L':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_LIST;
				break;
			case 'e':
				if (command != LIDS_LIST)
					exit_error (2, "error commands specified");
				type = LIDS_EXTEND;
				break;
			case 'Z':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_DELETE_ALL;
				break;
			case 'o':	/* for object */
				if (command == LIDS_NONE)
					exit_error (2, "no comands specified");
			
				strcpy(object_file,argv[optind++]);
				if(!strcmp(object_file,"CAP_NET_BIND_SERVICE")) {
					if (optind != argc) {
						 
					if(str2data(argv[optind],bindport,1,LIDS_PORT_ITEM)<0)
						exit_error(2,"The binding port is invalid");	
					optind++;
					}
					else 
						bindport[0][0]=bindport[0][1]=0;
				}
				if(strchr(object_file,':'))
					exit_error (2, "filename can not contain :");
				break;
			case 's':	/* program capability subject*/
				if (command == LIDS_NONE)
					exit_error (2, "no comands specified");
				strcpy(subject_file,optarg);
				if(strchr(object_file,':'))
					exit_error (2, "filename must not contain :");
				break;
			case 't':	/* time scale restrition*/
				if (command == LIDS_NONE)
					exit_error (2, "no comands specified");
				if(strlen(optarg)>176) 
					exit_error(2, "invalid time syntax");
				memset(lids_time,'\0',176);
				strncpy(lids_time,optarg,176);
				/* use '-1" to indicate the default value*/
				memset(timescale,'\0',LIDS_TIME_ITEM*2*sizeof(time_t *));
				/* FIXME */

				if(str2data(lids_time,timescale,0,LIDS_TIME_ITEM)<0)
					exit_error(2, "invalid time format");
				break;
			case 'i':
				if (command == LIDS_NONE)
					exit_error (2, "no comands specified");
				inherit= atoi(optarg);
				break;
			case 'd':
				if(command == LIDS_NONE)
					exit_error (2, "no comands specified");
				domain=1;
				break;
			case 'j':
				if (command == LIDS_NONE)
					exit_error (2, "no comands specified");

				if ( type != LIDS_NONE )
					exit_error (2, "multiple types specified");
				type = -1;
				
				for(i=0;i<6;i++) {
					if( !strcmp(optarg,lids_type_desc[i])) {
						type=lids_type_val[i];
					}
				}
				if(type == -1)
					exit_error(2,"cap type must be READONLY, WRITE, APPEND, DENY or GRANT");
				break;
			case 'P':
				if (command != LIDS_NONE)
					exit_error (2, "multiple commands specified");
				command = LIDS_MK_PASSWD;
				break;
			case 'v':
				exit_version();
				break;
			case 'h':
				exit_help();
			default:
			}
	}
	
	if (optind < argc)
        	exit_error (2, "unknown arguments found on commandline");
	if ( (command == LIDS_NONE) || (argc < 2) )
		exit_normal();
	if ( (command == LIDS_ADD) && (type == LIDS_NONE) ) 
		exit_normal();
	if ( (subject_file[0] != 0)  && (object_file[0] == 0) && (command == LIDS_ADD) ) 
		exit_error(2,"defined subject must be related to an object");

	switch(command) {
	case LIDS_ADD:
		printf("ADD\n");
	       	lids_add_file(subject_file,object_file,type,inherit,domain,timescale);		
		break;
	case LIDS_DELETE:
		printf("DELETE\n");
		lids_del_file(subject_file,object_file);
		break;
	case LIDS_LIST:
		printf("LIST\n");
		lids_list_file(type);
		break;
	case LIDS_DELETE_ALL:
		printf("DELETE_ALL\n");
		lids_del_all();
		break;
	case LIDS_MK_PASSWD:
		printf("MAKE PASSWD\n");
		lids_make_rmd160_passwd();
		break;
	case LIDS_UPDATE:
		printf("UPDATE\n");
		lids_update();
		break;
	}
	exit(0);
}
