/*
 * Copyright (C) 2005-2008 by CERN/IT/GD/SC
 * All rights reserved
 */

#ifndef lint
static char sccsid[] = "@(#)$RCSfile: dpm-drain.c,v $ $Revision: 1.22 $ $Date: 2009/12/14 10:07:40 $ CERN IT-GD/SC Jean-Philippe Baud";
#endif /* not lint */

/*      dpm-drain - drain a component of the Light Weight Disk Pool Manager */

#include <grp.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/types.h>
#include <signal.h>
#include <unistd.h>
#include <errno.h>
#include <stdarg.h>
#include "Cgetopt.h"
#include "dpm_api.h"
#include "dpns_api.h"
#include "serrno.h"
#include "u64subr.h"
#include "Cthread_api.h"
#include "Cpool_api.h"
int help_flag = 0;
volatile sig_atomic_t sig_int_received = 0;

struct dpm_drain_global_s;

struct dpm_drain_thread_info_s {
	struct Cns_filereplica lp;
	u_signed64 filesize;
	int rc;
	int good;
	int hardrc;
	int idx;
	struct dpm_drain_global_s *global;
	char logbuf[2048];
	size_t loglen;
};

struct dpm_drain_global_s {
	struct dpm_drain_thread_info_s *thip;
	int nbgids;
	gid_t *gid_list;	
	u_signed64 min_size;

	int condvar;

	int rc;
	int nbthreads;
	int pass_good;
	int pass_rc;

	/* access only by main thread */
	u_signed64 current_size;

	/* can have concurrent access between threads, require lock */
	enum states { FINISHED, NOTFINISHED } *thip_state;
	int nbfree;
};

thread_logit(struct dpm_drain_thread_info_s *thip, FILE *stream, char *msg, ...)
{
	va_list args;
	time_t current_time;
	int has_newline = 0;
	struct dpm_drain_thread_info_s lthip;
	int Tid = 0;
	struct tm tres;

	va_start (args, msg);

	if (strlen (msg) && msg[strlen (msg)-1]=='\n')
		has_newline = 1;

	if (!thip) {
		memset (&lthip, '\0', sizeof(lthip));
		thip = &lthip;
	}

	if (thip->loglen == 0) {
		current_time = time (0);
		Cglobals_getTid (&Tid);
		if (!strftime (thip->logbuf, sizeof(thip->logbuf), "%a %b %d %H:%M:%S %Y",
		    localtime_r (&current_time, &tres)))
			*thip->logbuf = '\0';
		if (Tid<0)
			Csnprintf (thip->logbuf + strlen (thip->logbuf),
			    sizeof(thip->logbuf) - strlen (thip->logbuf), " %5d: ", getpid ());
		else
			Csnprintf (thip->logbuf + strlen (thip->logbuf),
			    sizeof(thip->logbuf) - strlen (thip->logbuf), " %5d,%d: ", getpid (), Tid);
		thip->loglen = strlen (thip->logbuf);
	}

	Cvsnprintf (thip->logbuf+thip->loglen, sizeof(thip->logbuf) - thip->loglen, msg, args);
	thip->loglen = strlen (thip->logbuf);

	if (has_newline) {
		fprintf (stdout, "%s", thip->logbuf);
		fflush (stdout);
		thip->logbuf[0] = '\0';
		thip->loglen = 0;
	}

	va_end (args);
	return (0);
}

void collect_finished_and_reinit(s)
struct dpm_drain_global_s *s;
{
	int i;
	for (i=0;i < s->nbthreads;++i) {
		if (s->thip_state[i] == FINISHED) {
			if (s->min_size != 0 && s->thip[i].good)
				s->current_size += s->thip[i].filesize;
			if (s->thip[i].hardrc)
				s->rc = s->thip[i].hardrc;
			if (s->thip[i].good)
				s->pass_good++;
			if (s->thip[i].rc)
				s->pass_rc = s->thip[i].rc;
			memset (&s->thip[i], '\0', sizeof(struct dpm_drain_thread_info_s));
			s->thip[i].global = s;
			s->thip[i].idx = i;
			s->thip_state[i] = NOTFINISHED;
		}
	}
}

void sig_handler(signum)
int signum;
{
	const char notice[34] = {"Interrupt received, finishing up.\n"};
     	const char warn[50] = {"Interrupt received. One more to exit immediately.\n"};

	switch (++sig_int_received) {
		case 1:
			write (STDERR_FILENO, notice, sizeof (notice));
			break;
		case 2:
			write (STDERR_FILENO, warn, sizeof (warn));
			break;
		case 3:
			_exit (USERR);
			break;
	}
}

void *handle_one_file(arg)
void *arg;
{
	struct dpm_drain_thread_info_s *thip = (struct dpm_drain_thread_info_s *)arg;
	struct Cns_filestatg statbuf;
	struct Cns_filereplica *lp = &thip->lp;
	struct dpm_drain_global_s *s = thip->global;
	time_t current_time;
	char path[CA_MAXPATHLEN+1];
	char u64buf[21];
	int nbentries;
	struct dpns_filereplicax *rep_entries = NULL;
	int i;
	int j;
	char buf[256];
	struct tm tres;
	struct dpm_space_metadata *spacemd = NULL;
	int replicate;
	time_t f_lifetime;
	char pfn[CA_MAXSFNLEN+1];
	char *p;

	if (s->nbgids != 0 || s->min_size != 0) {
		if (Cns_statr (lp->sfn, &statbuf) < 0) {
			thread_logit (thip, stderr, "Cns_statr %s: %s\n", lp->sfn, sstrerror (serrno));
			thip->rc = 1;
			goto finish;
		}
		thip->filesize = statbuf.filesize;
	}
	if (s->nbgids != 0) {
		for (i=0; i < s->nbgids; i++) {
			if (statbuf.gid == s->gid_list[i])
				break;
		}
		if (i >= s->nbgids)
			goto finish;
	}
	if (lp->status != '-') {	/* file is being populated/deleted */
		if (lp->status == 'P') {
			thread_logit (thip, stdout, "The file %s is in the process of being uploaded, ignoring\n", lp->sfn);
		} else {
			thread_logit (thip, stdout, "The file %s is recorded as being in the process of being deleted, ignoring it during drain\n", lp->sfn);
		}
		if (s->min_size == 0)
			thip->rc = 1;
		goto finish;
	}
	current_time = time (0);
	if (lp->ptime > current_time) {	/* file is pinned */
		if (!strftime (buf, sizeof(buf), "%a %b %d %H:%M:%S %Y",
		    localtime_r (&lp->ptime, &tres)))
			*buf = '\0';
		thread_logit (thip, stdout, "%s pinned until %s\n", lp->sfn, buf);
		if (s->min_size == 0)
			thip->rc = 1;
		goto finish;
	}

	if (dpns_getpath (NULL, lp->fileid, path) < 0) {
		thread_logit (thip, stderr, 
			"dpns_getpath: %s (%s): %s\n", lp->sfn,
			u64tostr (lp->fileid, u64buf, 0), sstrerror (serrno));
		thip->rc = 1;
		goto finish;
	}

	if (dpns_getreplicax (path, NULL, NULL, &nbentries, &rep_entries) < 0) {
		thread_logit (thip, stderr, "dpns_getreplicax: %s: %s\n", path, sstrerror (serrno));
		thip->rc = 1;
		goto finish;
	}
	for (i=0; i < nbentries; i++) {
		if (!strcmp (lp->sfn, rep_entries[i].sfn))
			break;
	}
	if (i >= nbentries) {
		thread_logit (thip, stderr, "could not find replica of %s with pfn %s\n", path, lp->sfn);
		thip->rc = 1;
		goto finish;
	}
	thread_logit (thip, stdout, "File:\t\t%s\n", path);
	thread_logit (thip, stdout, "pfn:\t\t%s (of %d)\n", lp->sfn, nbentries);
	thread_logit (thip, stdout, "replica type:\t%s\n", (rep_entries[i].r_type == 'P') ? "primary" : "secondary");
	switch(lp->f_type) {
		case 'V':
			strcpy (buf,"volatile");
			break;
		case 'D':
			strcpy (buf,"durable");
			break;
		case 'P':
			strcpy (buf,"permanent");
			break;
		default:
			sprintf (buf,"'%c'",lp->f_type);
			break;
	}
	thread_logit (thip, stdout, "file type:\t%s", buf);
	if (lp->f_type == 'P')
		thread_logit (thip, stdout, " (does not expire)\n");
	else {
		if (!strftime (buf, sizeof(buf), "%a %b %d %H:%M:%S %Y",
		    localtime_r (&rep_entries[i].ltime, &tres)))
			*buf = '\0';
		thread_logit (thip, stdout, " (%s on %s)\n",
		    (rep_entries[i].ltime <= current_time) ? "expired" : "expires", buf);
	}
	if (s->min_size != 0)
		thread_logit (thip, stdout, "size:\t\t%s\n", u64tostru (statbuf.filesize, u64buf, 0));
	if (rep_entries[i].setname == NULL || *rep_entries[i].setname == '\0')
		thread_logit (thip, stdout, "space:\t\tnot in any space\n");
	else {
		p = rep_entries[i].setname;
		thread_logit (thip, stdout, "space:\t\t%s", p);
		j = 0;
		if (dpm_getspacemd (1, &p, &j, &spacemd) < 0) {
			if (serrno == EINVAL) {
				thread_logit (thip, stdout, " (invalid space)\n");
				*p = '\0';
			} else
				thread_logit (thip, stdout, "\n");
		} else if (j == 1 && spacemd)
			thread_logit (thip, stdout, " (%s)\n", spacemd[0].u_token);
		else
			thread_logit (thip, stdout, "\n");
		free (spacemd);
		spacemd = NULL;
	}
	replicate = 1;
	if (lp->f_type == 'V' && rep_entries[i].ltime <= current_time)
		replicate = 0;
	if (replicate) {
		thread_logit (thip, stdout, "replicating...\n");
		f_lifetime = rep_entries[i].ltime;
		if (dpm_replicatex (lp->sfn, lp->f_type, rep_entries[i].setname, f_lifetime, pfn) < 0) {
			thread_logit (thip, stdout, "failed\n");
			thread_logit (thip, stderr, "dpm_replicatex %s: %s\n", lp->sfn, sstrerror (serrno));
			thip->rc = 1;
			goto finish;
		}
		if (lp->f_type != 'P') {
			free (rep_entries);
			rep_entries = NULL;
			if (dpns_getreplicax (path, NULL, NULL, &nbentries, &rep_entries) < 0) {
				thread_logit (thip, stdout, "failed\n");
				thread_logit (thip, stderr, "dpns_getreplicax: %s: %s\n",
				    path, sstrerror (serrno));
				if (dpm_delreplica (pfn) < 0)
					thip->hardrc = 1;
				thip->rc = 1;
				goto finish;
			}
			for (i=0; i < nbentries; i++) {
				if (!strcmp (pfn, rep_entries[i].sfn))
					break;
			}
			if (i>=nbentries) {
				thread_logit (thip, stdout, "failed\n");
				thread_logit (thip, stderr, "could not find new replica\n");
				if (dpm_delreplica (pfn) < 0)
					thip->hardrc = 1;
				thip->rc = 1;
				goto finish;
			}
			if (rep_entries[i].ltime < f_lifetime) {
				thread_logit (thip, stdout, "failed\n");
				thread_logit (thip, stderr, "could not replicate to a new file with sufficient lifetime\n");
				if (dpm_delreplica (pfn) < 0)
					thip->hardrc = 1;
				thip->rc = 1;
				goto finish;
			}
		}
	}
	thread_logit (thip, stdout, "deleting %s\n",lp->sfn);
	if (dpm_delreplica (lp->sfn) < 0) {
		thread_logit (thip, stderr, "dpm_delreplica %s: %s\n", lp->sfn, sstrerror (serrno));
		thip->rc = 1;
		thip->hardrc = 1;
		goto finish;
	}
	thip->good = 1;

finish:
	free (rep_entries);
	rep_entries = NULL;
	if (Cthread_mutex_lock (&s->condvar) < 0) {
		fprintf (stderr, "Cthread_mutex_lock", sstrerror (serrno));
		exit (SYERR);
	}
	s->thip_state[thip->idx] = FINISHED;
	s->nbfree++;
	if (Cthread_cond_signal (&s->condvar) < 0) {
		fprintf (stderr, "Cthread_cond_signal", sstrerror (serrno));
		exit (SYERR);
	}
	if (Cthread_mutex_unlock (&s->condvar) < 0) {
		fprintf (stderr, "Cthread_mutex_unlock", sstrerror (serrno));
		exit (SYERR);
	}
	return (NULL);
}

int main(argc, argv)
int argc;
char **argv;
{
	int c;
	char *dp;
	struct dpm_fs *dpm_fs;
	struct dpm_pool *dpm_pools;
	int errflg = 0;
	int flags;
	int found_fs;
	char *fs = NULL;
	struct fs_list_s {
		char server[CA_MAXHOSTNAMELEN+1];
		char fs[80];
	} *fs_list = NULL;
	int fs_status;
#ifndef VIRTUAL_ID
	struct group *gr;
#endif
	char groupname[256];
	int i = 0;
	int ipool;
	int j = 0;
	Cns_list list;
	static struct Coptions longopts[] = {
		{"fs", REQUIRED_ARGUMENT, 0, OPT_FS},
		{"gid",REQUIRED_ARGUMENT,0,OPT_POOL_GID},
		{"group",REQUIRED_ARGUMENT,0,OPT_POOL_GROUP},
		{"help", NO_ARGUMENT, &help_flag, 1},
		{"size", REQUIRED_ARGUMENT, 0, OPT_DRAIN_SZ},
		{"poolname", REQUIRED_ARGUMENT, 0, OPT_POOL_NAME},
		{"server", REQUIRED_ARGUMENT, 0, OPT_FS_SERVER},
		{"threads", REQUIRED_ARGUMENT, 0, OPT_NBTHREADS},
		{0, 0, 0, 0}
	};
	struct Cns_filereplica *lp;
	int nbfs;	
	int nbpools;	
	char *p = NULL;
	char *poolname = NULL;
	struct dpm_drain_global_s s;
	int save_serrno;
	struct sigaction sigact;
	char *server = NULL;
	int target_fs_status;
	int thread_index;
	char u64buf[21];
	int test_tid[20];

	(void) dpm_copyfile_activate ();

	memset(&s, '\0', sizeof(struct dpm_drain_global_s));
	s.nbthreads = 1;
	Copterr = 1;
	Coptind = 1;
	while ((c = Cgetopt_long (argc, argv, "", longopts, NULL)) != EOF) {
		switch (c) {
		case OPT_FS:
			if (strlen (Coptarg) > 79) {
				fprintf (stderr,
					"filesystem name too long: %s\n", Coptarg);
				errflg++;
			} else
				fs = Coptarg;
			break;
		case OPT_FS_SERVER:
			if (strlen (Coptarg) > CA_MAXHOSTNAMELEN) {
				fprintf (stderr,
					"server name too long: %s\n", Coptarg);
				errflg++;
			} else
				server = Coptarg;
			break;
		case OPT_DRAIN_SZ:
			p = Coptarg;
			while (*p >= '0' && *p <= '9') p++;
			if (! (*p == '\0' || ((*p == 'k' || *p == 'M' ||
			    *p == 'G' || *p == 'T' || *p == 'P') && *(p+1) == '\0'))) {
				fprintf (stderr,
				    "invalid minimum amount to drain %s\n", Coptarg);
				errflg++;
			} else
				s.min_size = strutou64 (Coptarg);
			break;
		case OPT_NBTHREADS:
			p = Coptarg;
			while (*p >= '0' && *p <= '9') p++;
			if (*p != '\0') {
				fprintf (stderr,
				    "invalid number of threads %s\n", Coptarg);
				errflg++;
			} else {
				s.nbthreads = atoi (Coptarg);
				if (s.nbthreads <= 0) {
					fprintf (stderr,
					    "invalid number of threads %s\n", Coptarg);
					errflg++;
				}
			}
			break;
		case OPT_POOL_NAME:
			if (strlen (Coptarg) > CA_MAXPOOLNAMELEN) {
				fprintf (stderr,
					"pool name too long: %s\n", Coptarg);
				errflg++;
			} else
				poolname = Coptarg;
			break;
		case OPT_POOL_GID:
			p = Coptarg;
			//check that the user didn't provide already a list of gid
			if (s.nbgids == 0) i = 0;
			while (*p) {
				if (*p == ',') s.nbgids++;
				p++;
			}
			s.nbgids++;
			if ((s.gid_list = (gid_t *) realloc (s.gid_list, s.nbgids * sizeof(gid_t))) == NULL) {
				fprintf (stderr, "Could not allocate memory for gids\n");
				exit (USERR);
			}
			p = strtok (Coptarg, ",");
			while (p) {
				if ((s.gid_list[i] = strtol (p, &dp, 10)) < 0 || *dp != '\0') {
					fprintf (stderr, "Invalid gid %s \n",p);
					errflg++;
				} else {
						
#ifdef VIRTUAL_ID
					if (s.gid_list[i] > 0 && Cns_getgrpbygid (s.gid_list[i], groupname) < 0) {
#else
					if (s.gid_list[i] > 0 && ! getgrgid (s.gid_list[i])) {
#endif
						fprintf (stderr, "Invalid gid %s \n", p);
						errflg++;
					}
				}
				i++;
				if ((p = strtok (NULL, ",")))
					*(p - 1) = ',';
			}
			break;
		case OPT_POOL_GROUP:
			/* if it contains ALL, it is like a normal dpm-drain without filtering */
			if (strstr (Coptarg, "ALL") != NULL)
				break;
			/* check that the user didn't provide already a list of gid */
			if (s.nbgids == 0) i = 0;
			p = Coptarg;
		  	while (*p) {
				if (*p == ',') s.nbgids++;
				p++;
			}
			s.nbgids++;
			if ((s.gid_list = (gid_t *) realloc (s.gid_list, s.nbgids * sizeof(gid_t))) == NULL) {
				fprintf (stderr, "Could not allocate memory for gids\n");
				exit (USERR);
			}
			p = strtok (Coptarg, ",");
			while (p) {
#ifdef VIRTUAL_ID
				if (strcmp (p, "root") == 0)
					s.gid_list[i] = 0;
				else if (Cns_getgrpbynam (p, &s.gid_list[i]) < 0) {
#else
				if ((gr = getgrnam (p)))
					s.gid_list[i] = gr->gr_gid;
				else {
#endif
					fprintf (stderr, "Invalid group : %s\n", p);
					errflg++;
				}
				i++;
				if ((p = strtok (NULL, ",")))
					*(p - 1) = ',';
			}
			break;
		case '?':
			errflg++;
			break;
		default:
			break;
		}
	}
	if (Coptind < argc || (poolname == NULL && server == NULL && fs == NULL))
		errflg++;
	if (fs && server == NULL)
		errflg++;
	if (fs && poolname)
		errflg++;
	if (errflg || help_flag) {
		fprintf (stderr, "usage:\n%s\n%s\n%s\n%s",
		    "dpm-drain --poolname pool_name [--server fs_server] [--gid gid(s)]\n"
		    "\t[--group group_name(s)] [--size amount_to_drain]\n",
		    "dpm-drain --server fs_server [--gid gid(s)] [--group group_name(s)]\n"
		    "\t[--size amount_to_drain]\n",
		    "dpm-drain --server fs_server --fs fs_name [--gid gid(s)]\n"
		    "\t[--group group_name(s)] [--size amount_to_drain]\n",
		    "dpm-drain --help\n");
		exit (help_flag ? 0 : USERR);
	}

	/* set status to FS_RDONLY unless a specific server & fs is specified along with a limit */
	target_fs_status = ((s.nbgids == 0 && s.min_size == 0) || !(server && fs)) ? FS_RDONLY : -1;

	if (dpm_getpools (&nbpools, &dpm_pools) < 0) {
		fprintf (stderr, "dpm_getpools: %s\n", sstrerror (serrno));
		exit (USERR);
	}

	for (i = 0; i < nbpools; i++) {
		if (poolname && strcmp ((dpm_pools + i)->poolname, poolname))
			continue;
		if (dpm_getpoolfs ((dpm_pools + i)->poolname, &nbfs, &dpm_fs) < 0) {
			fprintf (stderr, 
				"dpm_getpoolfs %s: %s\n", (dpm_pools + i)->poolname, 
				sstrerror (serrno));
			exit (USERR);
		}
		for (j = 0; j < nbfs; j++) {
			if (server && strcmp ((dpm_fs + j)->server, server))
				continue;
			if (fs && strcmp (dpm_fs[j].fs, fs))
				continue;
			fs_status = target_fs_status;
			if (dpm_fs[j].status & FS_DISABLED)
				fs_status = -1;
			if (dpm_modifyfs ((dpm_fs + j)->server, (dpm_fs + j)->fs, fs_status) < 0) {
				fprintf (stderr, 
					"dpm_modifyfs %s %s: %s\n", (dpm_fs + j)->server, 
					(dpm_fs + j)->fs, sstrerror (serrno));
				exit (USERR);
			}
			p = realloc (fs_list, sizeof(struct fs_list_s)*(found_fs+1));
			fs_list = (struct fs_list_s *)p;
			strcpy (fs_list[found_fs].server,(dpm_fs + j)->server);
			strcpy (fs_list[found_fs].fs,(dpm_fs + j)->fs);
			found_fs++;
		}
	}
	if (!found_fs) {
		fprintf (stderr, "No file systems matching specification\n");
		exit (USERR);
	}

	/* install a signal handler for SIGINT */

	memset(&sigact, '\0', sizeof(sigact));
	sigemptyset(&sigact.sa_mask);
	sigact.sa_handler = &sig_handler;
#ifdef SA_RESTART
	sigact.sa_flags |= SA_RESTART;
#endif
	sigaction(SIGINT, &sigact, (struct sigaction *)NULL);

	s.thip = (struct dpm_drain_thread_info_s *)
	    calloc (s.nbthreads, sizeof(struct dpm_drain_thread_info_s));

	s.thip_state = (enum states *)
	    calloc (s.nbthreads, sizeof(enum states));

	s.nbfree = s.nbthreads;

	/* moves all the newly created thread specific states to NOTFINISHED */
	collect_finished_and_reinit(&s);

	ipool = Cpool_create (s.nbthreads, NULL);
	if (ipool < 0) {
		fprintf (stderr, "Could not create thread pool\n");
		exit (SYERR);
	}

	if (Cthread_mutex_lock (&s.condvar) < 0) {
		fprintf (stderr, "Cthread_mutex_lock", sstrerror (serrno));
		exit (SYERR);
	}

	thread_logit (NULL, stdout, "Starting to drain with %d threads\n", s.nbthreads);

	while (1) {
		flags = CNS_LIST_BEGIN;
		s.pass_good = 0;
		s.pass_rc = 0;
		while (!sig_int_received && (serrno=0, lp = Cns_listreplicax (poolname, server, fs, flags, &list)) != NULL) {
			if (flags != CNS_LIST_CONTINUE)
				flags = CNS_LIST_CONTINUE;
			for (i=0; i < found_fs; i++) {
				if (!strcmp (fs_list[i].server, lp->host) &&
				    !strcmp (fs_list[i].fs, lp->fs))
					break;
			}
			if (i >= found_fs)
				continue;
			while (!s.nbfree) {
				if (Cthread_cond_wait (&s.condvar) < 0) {
					fprintf (stderr, "Error in Cthread_cond_wait: %s\n", sstrerror (serrno));
					exit (SYERR);
				}
			}
			collect_finished_and_reinit(&s);

			if (s.min_size != 0 && s.current_size >= s.min_size)
				break;

			if (sig_int_received)
				break;

			if ((thread_index = Cpool_next_index (ipool)) < 0) {
				fprintf (stderr, "Error in Cpool_next_index: %s\n", sstrerror (serrno));
				exit (SYERR);
			}
			memcpy (&s.thip[thread_index].lp, lp, sizeof(struct Cns_filereplica));
			if (Cpool_assign (ipool, &handle_one_file, &s.thip[thread_index], -1) < 0) {
				fprintf (stderr, "Error in Cpool_assign: %s\n", sstrerror (serrno));
				exit (SYERR);
			}
			s.nbfree--;
		}
		save_serrno = serrno;
		(void) Cns_listreplicax (poolname, server, fs, CNS_LIST_END, &list);
		while (s.nbfree<s.nbthreads) {
			if (Cthread_cond_wait (&s.condvar) < 0) {
				fprintf (stderr, "Error in Cthread_cond_wait: %s\n", sstrerror (serrno));
				exit (SYERR);
			}
		}
		collect_finished_and_reinit(&s);

		if (sig_int_received || lp != NULL || s.rc || save_serrno != SETIMEDOUT || s.pass_good == 0) {
			if (s.pass_rc)
				s.rc = 1;
			if (!sig_int_received && lp == NULL && save_serrno)
				fprintf (stderr, "Cns_listreplicax: %s\n", sstrerror (save_serrno));
			break;
		}
	}
	if (Cthread_mutex_unlock (&s.condvar) < 0)
		fprintf (stderr, "Cthread_mutex_unlock", sstrerror (serrno));
	if (sig_int_received) {
		thread_logit (NULL, stderr, "Finishing after user requested that the drain stop\n");
		s.rc = 1;
	} else if (s.rc)
		thread_logit (NULL, stderr, "There were some errors which prevented dpm-drain from completing fully\n");
	else
		thread_logit (NULL, stderr, "Finished draining\n");
	if (s.min_size != 0) {
		thread_logit (NULL, stdout, "number of bytes drained %s\n", u64tostru (s.current_size, u64buf, 0));
		if (s.current_size < s.min_size)
			s.rc = 1;
	}
	return (s.rc);
}
