/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*- */
/*
 * Linux Desktop Testing Project http://ldtp.freedesktop.org
 *
 * Author:
 *    Nagappan A <nagappan@gmail.com>
 *    S Thanikachalam <sthanikachalam@novell.com>
 *    Premkumar J <jpremkumar@novell.com>
 *
 * Copyright 2004 - 2006 Novell, Inc.
 * Copyright 2007 - 2008 Nagappan Alagappan
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110, USA.
 */

#include "ldtp.h"
#include "ldtp-gui.h"
#include "ldtp-utils.h"
#include "ldtp-error.h"
#include "ldtp-logger.h"
#include "ldtp-appmap.h"
#include "ldtp-command.h"
#include "ldtp-gui-comp.h"

extern gboolean ldtp_debug;
static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

static gchar*
read_delimiter (int fd, gchar delimit)
{
	int len = 128;
	int size = 0;
	gchar ch;
	gchar *data = NULL;

	while (read (fd, &ch, 1) > 0) {
		if (!data)
			data = (gchar *) malloc (sizeof (gchar)*len);
		if (size%len == 0)
			data = (gchar *) realloc (data, sizeof (gchar)*len+size+1);
		data[size++] = ch;
		if (data[0] == '\n' || ch == delimit || (size && data[size - 1] == ']')) {
			if (data[0] != '\n' && read (fd, &ch, 1) > 0) {
				/*
				 * Don't do any operation
				 */
			}
			if (size%len == 0)
				data = (gchar *) realloc (data, sizeof (gchar)+size+1);
			data[size] = '\0';
			return data;
		}
	}
	return NULL;
}

static void
add_child_attributes (gchar *cur_entry, GHashTable *cur_context_table)
{
	gchar *token = NULL;
	int offset;
	gchar *component = NULL;
	    
	token = strtok (strdup (cur_entry), "=");
	if (token)
		component = strdup (token);
	    
	offset = strlen (component) + 1;
	if (cur_entry[offset] == '{') {
		gchar *end = strstr (cur_entry, "}");
		if (end) {
			gchar *key, *value;
			gchar *all_attributes;
			GHashTable *hash_attributes;
			int len = (end - cur_entry) - (offset+1);
	 
			all_attributes = (gchar *) malloc (sizeof (gchar) * len + 1);
			strncpy (all_attributes, (cur_entry + offset + 1), len);
			all_attributes[len] = '\0';

			/*
			  Create new attribute hash table
			*/
			hash_attributes = g_hash_table_new (&g_str_hash, &g_str_equal);
			key = strtok (strdup (all_attributes), "=");
			while (key) {
				gchar *hash_key;
				gchar *hash_value;
				hash_key = g_strdup (key);
				value = strtok (NULL, ",");
				hash_value = g_strdup (value);
				if (hash_key && hash_value)
					g_print ("%s: %s\n", hash_key, hash_value);
				if (hash_attributes && hash_key && hash_value)
					g_hash_table_insert (hash_attributes, hash_key, hash_value);
				key = strtok (NULL, "=");
			}
			if (component && cur_context_table && hash_attributes)
				g_hash_table_insert (cur_context_table, component, hash_attributes);
		}
	}
}

void
print_attributes (gchar *key, gchar *value, gchar *userdata)
{
	if (!key || !value)
		return;
	g_print ("\t|-------%s = %s\n", key, value);
}

void
print_component (gchar *key, GHashTable *component, gchar *userdata)
{
	if (!key || !component)
		return;
	g_print ("|-------%s\n", key);
	g_hash_table_foreach (component, (GHFunc)&print_attributes, userdata);
}

void
print_context (gchar *key, GHashTable *context, gchar *userdata)
{
	if (!key || !context || !ldtp_debug)
		return;
	g_print (":%s:\n", key);
	g_hash_table_foreach (context, (GHFunc)&print_component, userdata);
}

/*
  Initialize application map
*/
GHashTable*
appmap_init (gchar *gui_map_filename, FILE *log_fp)
{
	int fd;
	//int quit_flag = 0;
	FILE *fp;
	gchar *cur_entry = NULL;
	GHashTable *appmap;
	GHashTable *cur_context_table = NULL;

	appmap = g_hash_table_new (&g_str_hash, &g_str_equal);
    
	fp = fopen (gui_map_filename, "r");
	if (fp == NULL) {
		g_print ("Unable to open appmap %s file\n", gui_map_filename);
		log_msg (LDTP_LOG_CAUSE, "Unable to open appmap file", log_fp);
		return NULL;
	}
	fd = fileno (fp);

	cur_entry = read_delimiter (fd, '}');
	while (1) {
		int cur_entry_len;
		gchar *context = NULL;
		if (!cur_entry)
			break;
		cur_entry_len = strlen (cur_entry);
		if (cur_entry [0] == '[' && cur_entry [cur_entry_len - 1] == ']') {
			cur_context_table = g_hash_table_new (&g_str_hash, &g_str_equal);
			context = (gchar *) g_new0 (gchar, sizeof (gchar) * (cur_entry_len - 1));

			/* Copy everything except the leading '[' and trailing ']' */
			memcpy (context, cur_entry+1, cur_entry_len-2);
			if (appmap && context && cur_context_table) 
				g_hash_table_insert (appmap, context, cur_context_table);
		}
		else if (cur_entry[0] == '\n') {
			cur_entry = read_delimiter (fd, '}');
			continue;
		}
		else
			add_child_attributes (cur_entry, cur_context_table);
		cur_entry = read_delimiter (fd, '}');
	}
	if (ldtp_debug)
		g_hash_table_foreach (appmap, (GHFunc)&print_context, NULL);
	return appmap;
}

void
destroy_hashtable (gpointer data)
{
	if (data) {
		g_hash_table_destroy (data);
		data = NULL;
	}
}

void
destroy_string (gpointer data)
{
	if (data) {
		g_free (data);
		data = NULL;
	}
}

static gboolean
is_remaped (gpointer key, gpointer value, gpointer user_data)
{
	if (key == NULL || value == NULL)
		return FALSE;
	g_print ("%s - %s ******* DEBUG\n", (gchar *) key, (gchar *) value);
	if (g_utf8_collate (key, "remap") == 0)
		return TRUE;
	return FALSE;
}

gboolean
remove_remapped_entry (gpointer key, gpointer value, gpointer user_data)
{
	if (key == NULL || value == NULL)
		return TRUE;
	g_hash_table_foreach_remove (value, is_remaped, key);
	return TRUE;
}

/*
gboolean
remove_context_entries (gpointer key, gpointer value, gpointer context)
{
	GHashTable *component = NULL;

	if (context == NULL || key == NULL)
		return TRUE;
	component = g_hash_table_lookup (context, (gchar *)key);
	if (component) {
		g_hash_table_destroy (component);
		g_hash_table_remove (context, component);
		component = NULL;
	}
	return TRUE;
}
*/

static gboolean
remove_appmap_entries (gpointer key, gpointer value, gpointer user_data)
{
	GHashTable *context = NULL;
	GHashTable *appmap = user_data;

	if (!appmap || !key)
		return TRUE;
	context = g_hash_table_lookup (appmap, (gchar *)key);
	if (context) {
		//g_hash_table_foreach_remove (context, (GHRFunc)&remove_context_entries, context);
		g_print ("Context: %s %s\n", (gchar *)key, context);
		g_hash_table_destroy (context);
		context = NULL;
	}
	return TRUE;
}

/*
  Free application map resource
*/
void
ldtp_appmap_free (GHashTable *appmap)
{
	if (!appmap)
		return;

	pthread_mutex_lock (&mutex);
	//g_hash_table_foreach_remove (appmap, (GHRFunc)&remove_appmap_entries, appmap);
	g_hash_table_destroy (appmap);
	pthread_mutex_unlock (&mutex);
	appmap = NULL;
}

static gboolean
search_key_based (gpointer key, gpointer value, gpointer user_data)
{
	OBJInfo *obj = (OBJInfo *) user_data;
	if (!key || !obj || !obj->key) {
		return FALSE;
	}
	if (g_pattern_match_simple (obj->key, key)) {
		if (ldtp_debug && obj->key && key) {
			g_print ("Key: %s %s\n",
				 (gchar *) key,
				 obj->key);
		}
		if (obj->parent_name && value) {
			gchar *parent = get_property (value, "parent", NULL);
			if (parent && g_pattern_match_simple (obj->parent_name, parent))
				return TRUE;
		} else
			return TRUE;
	}
	return FALSE;
}

static void
search_window_key_based (gpointer key, gpointer value, gpointer user_data)
{
	if (search_key_based (key, value, user_data)) {
		OBJInfo *obj;
		obj = (OBJInfo *) user_data;
		g_hash_table_insert (obj->ht_data_list, g_strdup ((gchar *) key), value);
	}
}

static gboolean
search_obj_based_on_index (gpointer key, gpointer value, gpointer user_data)
{
	gchar *obj_index = NULL;
	OBJInfo *obj = (OBJInfo *) user_data;
	if (!value || !obj || !obj->key) {
		return FALSE;
	}
	obj_index = get_property (value, "obj_index", NULL);
	if (obj_index && g_pattern_match_simple (obj->key, obj_index)) {
		if (obj->parent_name) {
			gchar *parent = get_property (value, "parent", NULL);
			if (parent && g_pattern_match_simple (obj->parent_name, parent))
				return TRUE;
		} else
			return TRUE;
	}
	return FALSE;
}

static gboolean
search_key_glob_based (gpointer key, gpointer value, gpointer user_data)
{
	gchar *tmp;
	OBJInfo *obj;
	if (!key || !user_data) {
		return FALSE;
	}
	/*
	  FIXME: Use get_object_info function to get the the type of object
	  if type and key matches then return true
	*/
	obj = (OBJInfo *) user_data;
	if (ldtp_debug && obj->key && key) {
		g_print ("glob - key: %d - %d - %d - %s %s %ld %ld\n",
			 obj->obj_is_window,
			 g_pattern_match_simple ("frm*", obj->key),
			 g_pattern_match_simple ("dlg*", obj->key),
			 obj->key, (gchar *) key, g_utf8_strlen (key, -1),
			 g_utf8_strlen (obj->key, -1));
	}
	if (obj->obj_is_window) {
		if (obj->pattern) {
			if (g_pattern_match_string (obj->pattern, key)) {
				if (obj->parent_name && value) {
					gchar *parent = get_property (value, "parent", NULL);
					if (parent && g_pattern_match_simple (obj->parent_name, parent))
						return TRUE;
				} else
					return TRUE;
			}
		}
		tmp = g_strdup_printf ("frm%s", (gchar *)key);
		if (tmp && g_pattern_match_simple (tmp, key)) {
			g_free (tmp);
			tmp = NULL;
			if (obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return TRUE;
		}
		if (tmp)
			g_free (tmp);
		tmp = g_strdup_printf ("dlg%s", obj->key);
		if (tmp && g_pattern_match_simple (tmp, key)) {
			g_free (tmp);
			tmp = NULL;
			if (obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return TRUE;
		}
		if (tmp)
			g_free (tmp);
		tmp = g_strdup_printf ("frm%s*", obj->key);
		if (tmp && g_pattern_match_simple (tmp, key)) {
			g_free (tmp);
			tmp = NULL;
			if (obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return TRUE;
		}
		if (tmp)
			g_free (tmp);
		tmp = g_strdup_printf ("dlg%s*", obj->key);
		if (tmp && g_pattern_match_simple (tmp, key)) {
			g_free (tmp);
			tmp = NULL;
			if (obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return TRUE;
		}
		if (tmp)
			g_free (tmp);
		return FALSE;
	}
	if (obj->pattern && !obj->obj_is_window) {
		gboolean flag = g_pattern_match_string (obj->pattern, key);
		if (flag && obj->parent_name && value) {
			gchar *parent = get_property (value, "parent", NULL);
			if (parent && g_pattern_match_simple (obj->parent_name, parent))
				return TRUE;
		} else
			return flag;
	}
	return FALSE;
}

static void
search_window_key_glob_based (gpointer key, gpointer value, gpointer user_data)
{
	if (search_key_glob_based (key, value, user_data)) {
		OBJInfo *obj;
		obj = (OBJInfo *) user_data;
		g_hash_table_insert (obj->ht_data_list, g_strdup ((gchar *) key), value);
	}
}

static gboolean
search_label_based (gpointer key, gpointer value, gpointer user_data)
{
	OBJInfo *obj;
	gchar *tmp   = NULL;
	gchar *class = NULL;
	gchar *label = NULL;
	gchar *label_by = NULL;
	gchar *tmp_data = NULL;
	gboolean flag = FALSE;
	GPatternSpec *pattern = NULL;

	if (!value || !user_data) {
		return FALSE;
	}
	if (key)
		g_print ("Key: %s\n", (gchar *) key);
	obj = (OBJInfo *) user_data;
	g_hash_table_lookup_extended (value, "class", NULL, (gpointer) &class);
	g_hash_table_lookup_extended (value, "label", NULL, (gpointer) &label);

	tmp_data = obj->key;
	if (class)
		pattern = g_pattern_spec_new (class);
	if (pattern && (g_pattern_match_string (pattern, "frame") ||
	    g_pattern_match_string (pattern, "dialog") ||
	    g_pattern_match_string (pattern, "alert") ||
	    g_pattern_match_string (pattern, "file_chooser") ||
	    g_pattern_match_string (pattern, "font_chooser"))) {
		flag = TRUE;
	}
	if (pattern)
		g_pattern_spec_free (pattern);
	/*
	  FIXME: Use get_object_info function to get the the type of object
	  if type and key matches then return true
	*/
	if (label && tmp_data) {
		if (ldtp_debug && (gchar *)key && label && tmp_data)
			g_print ("Label: %s - %s - %s\n", (gchar *)key, (gchar *)tmp_data, label);
		if (!obj->obj_is_window || flag)
			if (g_utf8_collate (label, tmp_data) == 0) {
				if (obj->parent_name && value) {
					gchar *parent = get_property (value, "parent", NULL);
					if (parent && g_pattern_match_simple (obj->parent_name, parent))
						return TRUE;
				} else
					return TRUE;
			}
		// Search for mnemonics, if available then remove just _
		if (g_utf8_strchr (label, -1, '_') == NULL)
			return FALSE;
		tmp = escape_character (label, '_');
		if (tmp && tmp_data)
			if (g_utf8_collate (tmp, tmp_data) == 0) {
				g_free (tmp);
				tmp = NULL;
				if (obj->parent_name && value) {
					gchar *parent = get_property (value, "parent", NULL);
					if (parent)
						g_print ("Parent: %s - %s\n", parent, obj->parent_name);
					if (parent && g_pattern_match_simple (obj->parent_name, parent))
						return TRUE;
				} else
					return TRUE;
			}
		if (tmp)
			g_free (tmp);
		return FALSE;
	}
	g_hash_table_lookup_extended (value, "label_by", NULL, (gpointer) &label_by);
	if (label_by && tmp_data) {
		if ((gchar *)key && label_by && tmp_data)
			g_print ("LabelBy: %s - %s - %s\n", (gchar *)key, (gchar *)tmp_data, label_by);
		if (!obj->obj_is_window || flag)
			if (g_utf8_collate (label_by, tmp_data) == 0) {
				if (obj->parent_name && value) {
					gchar *parent = get_property (value, "parent", NULL);
					if (parent && g_pattern_match_simple (obj->parent_name, parent))
						return TRUE;
				} else
					return TRUE;
			}
		// Search for mnemonics, if available then remove just _
		if (g_utf8_strchr (label_by, -1, '_') == NULL)
			return FALSE;
		tmp = escape_character (label, '_');
		if (tmp && tmp_data)
			if (g_utf8_collate (tmp, tmp_data) == 0) {
				g_free (tmp);
				tmp = NULL;
				if (obj->parent_name && value) {
					gchar *parent = get_property (value, "parent", NULL);
					if (parent && g_pattern_match_simple (obj->parent_name, parent))
						return TRUE;
				} else
					return TRUE;
			}
		if (tmp)
			g_free (tmp);
		return FALSE;
	}
	return FALSE;
}

static void
search_window_label_based (gpointer key, gpointer value, gpointer user_data)
{
	if (search_label_based (key, value, user_data)) {
		OBJInfo *obj;
		obj = (OBJInfo *) user_data;
		g_hash_table_insert (obj->ht_data_list, g_strdup ((gchar *) key), value);
	}
}

static gboolean
search_label_glob_based (gpointer key, gpointer value, gpointer user_data)
{
	OBJInfo *obj;
	gchar *class = NULL;
	gchar *tmp_label = NULL;
	gchar *tmp_label_by = NULL;
	GPatternSpec *pattern = NULL;
	gboolean flag = FALSE;

	if (!value || !user_data)
		return FALSE;
	/*
	  FIXME: Remove under score from label or label_by and do a pattern match too
	*/

	obj = (OBJInfo *) user_data;
	if (g_hash_table_lookup_extended (value, "class", NULL, (gpointer) &class))
		pattern = g_pattern_spec_new (class);
	if (pattern && (g_pattern_match_string (pattern, "frame") ||
	    g_pattern_match_string (pattern, "dialog") ||
	    g_pattern_match_string (pattern, "alert") ||
	    g_pattern_match_string (pattern, "file_chooser") ||
	    g_pattern_match_string (pattern, "font_chooser"))) {
		flag = TRUE;
	}
	if (pattern)
		g_pattern_spec_free (pattern);
	if (g_hash_table_lookup_extended (value, "label", NULL, (gpointer) &tmp_label)) {
		if (key && value)
			g_print ("Label glob: %s - %s\n", (gchar *)key, (gchar *)value);
		if (obj->pattern && (!obj->obj_is_window || flag == TRUE)) {
			gboolean tmp_flag = g_pattern_match_string (obj->pattern, tmp_label);
			if (tmp_flag && obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return tmp_flag;
		}
	}

	if (g_hash_table_lookup_extended (value, "label_by", NULL, (gpointer) &tmp_label_by)) {
		g_print ("Label_By glob: %s\n", tmp_label_by);
		if (obj->pattern && (!obj->obj_is_window || flag)) {
			gboolean tmp_flag = g_pattern_match_string (obj->pattern, tmp_label_by);
			if (tmp_flag && obj->parent_name && value) {
				gchar *parent = get_property (value, "parent", NULL);
				if (parent && g_pattern_match_simple (obj->parent_name, parent))
					return TRUE;
			} else
				return tmp_flag;
		}
	}
	return FALSE;
}

static void
search_window_label_glob_based (gpointer key, gpointer value, gpointer user_data)
{
	if (search_label_glob_based (key, value, user_data)) {
		OBJInfo *obj;
		obj = (OBJInfo *) user_data;
		g_hash_table_insert (obj->ht_data_list, g_strdup ((gchar *) key), value);
	}
}

/*
static gboolean 
search_label_name (gpointer key, gpointer value, gpointer user_data)
{
	gchar *child_index = NULL;
	UnknLabelProperty *tmp_data = (UnknLabelProperty *) user_data;
	if (!value && !tmp_data && !key) {
		return FALSE;
	}
	if (g_hash_table_lookup_extended (value, "child_index", NULL, (gpointer) &child_index)) {
		if ((gchar *)key && tmp_data->str_child_index && tmp_data->parent_name)
			g_print ("Key - Child index - Parent: %s - %s - %s\n",
				 (gchar *)key, tmp_data->str_child_index,
				 tmp_data->parent_name);
		if (child_index && tmp_data->str_child_index &&
		    g_utf8_collate (child_index, tmp_data->str_child_index) == 0) {
			gchar *parent_name = NULL;
			g_hash_table_lookup_extended (value, "parent", NULL, (gpointer) &parent_name);
			g_print ("Key - Parent: %s - %s\n",
				 (gchar *)key,
				 parent_name);
			if (parent_name && tmp_data->parent_name &&
			    g_utf8_collate (parent_name, tmp_data->parent_name) == 0) {
				tmp_data->obj_name = (gchar *) key;
				return TRUE;
			}
		}
 	}
	return FALSE;
}
*/

static gboolean
search_obj_after_stripping_space (gpointer key, gpointer value, gpointer user_data)
{
	gchar *tmp;
	OBJInfo *obj = (OBJInfo *) user_data;
	if (!key || !user_data || !obj->key)
		return FALSE;
	if (g_utf8_strchr (obj->key, -1, ' ') == NULL)
		return FALSE;
	tmp = escape_character (obj->key, ' ');
	if (!tmp)
		return FALSE;
	g_print ("DEBUG: %s - %s - %s\n", tmp, (gchar *)obj->key, (gchar *)key);
	if (g_utf8_collate (key, tmp) == 0 || g_utf8_collate (key, obj->key) == 0) {
		g_free (tmp);
		return TRUE;
	}
	else {
		gchar *glob_str = NULL;
		/*
		    FIXME: Use get_object_info function to get the the type of object
		      and based on that use the convention, instead of directly using *
		*/
		glob_str = g_strdup_printf ("*%s", tmp);
		g_print ("Glob str: %s - %s - %s\n", glob_str, (gchar *)key, tmp);
		if (g_pattern_match_simple (glob_str, key) ||
		    g_pattern_match_simple (key, tmp)) {
			g_print ("search_obj_after_stripping_space - matched\n");
			g_free (tmp);
			g_free (glob_str);
			return TRUE;
		}
		g_free (glob_str);
		g_free (tmp);
	}
	return FALSE;
}

static void
search_window_after_stripping_space (gpointer key, gpointer value, gpointer user_data)
{
	OBJInfo *obj;
	if (!key || !user_data || !value)
		return;
	obj = (OBJInfo *) user_data;
	if (search_obj_after_stripping_space (key, value, obj)) {
		if (obj && obj->ht_data_list && key && value)
			g_hash_table_insert (obj->ht_data_list, g_strdup ((gchar *) key), value);
	}
}

GHashTable*
get_all_matching_def (GHashTable *ht, gchar *context, gboolean obj_is_window, FILE *log_fp)
{
	GHashTable *ht_data_list = NULL;
	LDTPErrorCode err;

	OBJInfo obj;
	GPatternSpec *pattern = NULL;
	if (ht == NULL) {
		err = LDTP_ERROR_APPMAP_NOT_INITIALIZED;
		g_print ("%s - %d - %s\n", __FILE__, __LINE__, ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	if (context == NULL) {
		err = LDTP_ERROR_ARGUMENT_NULL;
		g_print ("%s\n", ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	ht_data_list = g_hash_table_new_full (&g_str_hash, &g_str_equal,
						 destroy_string, NULL);
	obj.key = context;
	obj.parent_name = NULL;
	obj.obj_is_window = obj_is_window;
	g_hash_table_foreach (ht, search_window_key_based, &obj);

	if (!g_hash_table_size (ht_data_list)) {
		/*
		  Search key based on glob expression
		*/
		g_print ("Search key based on glob expression: %s %s %d\n", context, __FILE__, __LINE__);
		if (context)
			pattern = g_pattern_spec_new ((const gchar *)context);

		obj.obj_is_window = obj_is_window;
		obj.pattern = pattern;
		obj.key = context;
		obj.ht_data_list = ht_data_list;
		g_hash_table_foreach (ht, search_window_key_glob_based, &obj);
		if (pattern)
			g_pattern_spec_free (pattern);
	}
	if (!g_hash_table_size (ht_data_list)) {
		/*
		  Search label based
		*/
		g_print ("Search label based: %s\n", context);

		obj.pattern = NULL;
		obj.key = context;
		obj.parent_name = NULL;
		obj.obj_is_window = obj_is_window;
		g_hash_table_foreach (ht, search_window_label_based, &obj);
	}
	if (!g_hash_table_size (ht_data_list)) {
		/*
		  Search label based on glob expression
		*/
		OBJInfo obj;
		GPatternSpec *pattern = NULL;

		g_print ("Search label based on glob expression: %s\n", context);
		if (context)
			pattern = g_pattern_spec_new (context);

		obj.pattern = pattern;
		obj.key = context;
		obj.parent_name = NULL;
		obj.obj_is_window = obj_is_window;
		g_hash_table_foreach (ht, search_window_label_glob_based, &obj);
		if (pattern)
			g_pattern_spec_free (pattern);
	}
	if (!g_hash_table_size (ht_data_list)) {
		/*
		  Search key after stripping space
		*/
		g_print ("Search key after stripping space: %s\n", context);
		obj.key = context;
		obj.parent_name = NULL;
		obj.obj_is_window = obj_is_window;
		g_hash_table_foreach (ht, search_window_after_stripping_space, &obj);
	}
	return ht_data_list;
}

gboolean
match_object_def (gchar *key, gchar *value, gboolean obj_is_window)
{
	OBJInfo obj;
	gchar msg [256];
	GPatternSpec *pattern = NULL;
	/*
	  Search key based on glob expression
	*/
	g_print ("Search key based on glob expression: %s %s %d\n", key, __FILE__, __LINE__);
	if (key)
		pattern = g_pattern_spec_new (/*(const ggchar *)*/key);

	obj.obj_is_window = obj_is_window;
	obj.pattern = pattern;
	obj.key = key;
	if (search_key_glob_based (value, NULL, &obj)) {
		if (pattern)
			g_pattern_spec_free (pattern);
		return TRUE;
	}
	if (pattern)
		g_pattern_spec_free (pattern);

	g_print ("Search label based: %s - %d\n", value, obj_is_window);

	obj.obj_is_window = obj_is_window;
	obj.pattern = NULL;
	obj.key = key;
	if (search_label_based (NULL, value, &obj))
		return TRUE;

	g_print ("Search label based on glob expression\n");
	if (key)
		pattern = g_pattern_spec_new (key);

	obj.obj_is_window = obj_is_window;
	obj.pattern = pattern;
	obj.key = key;
	if (search_label_glob_based (NULL, value, &obj)) {
		if (pattern)
			g_pattern_spec_free (pattern);
		return TRUE;
	}
	if (pattern)
		g_pattern_spec_free (pattern);
	/*
	  Search key after stripping space
	*/
	g_print ("Search key after stripping space\n");
	if (search_obj_after_stripping_space (key, NULL, value))
		return TRUE;

	g_sprintf (msg, "Object definition %s not match", key);
	g_print ("%s\n", msg);
	return FALSE;
}

/*
  Get window definition
*/
GHashTable*
get_object_def (GHashTable *ht, gchar *context, gchar *parent, FILE *log_fp, gboolean obj_is_window)
{
	OBJInfo obj;
	gchar msg [256];
	LDTPErrorCode err;
	GPatternSpec *pattern  = NULL;
	GHashTable *ht_context = NULL;

	if (ht == NULL) {
		err = LDTP_ERROR_APPMAP_NOT_INITIALIZED;
		g_print ("%s - %d - %s\n", __FILE__, __LINE__, ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	if (context == NULL) {
		err = LDTP_ERROR_ARGUMENT_NULL;
		g_print ("%s\n", ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	/*
	  Search given context in hash table
	*/
	if (g_utf8_strchr (context, -1, '#')) {
		/*
		  Search key based on object index
		  ex: btn#1, mnu#1
		*/
		g_print ("Search based on index: %s\n", context);
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_obj_based_on_index, &obj);
	}
	if (!ht_context) {
		g_print ("Search based on key: %s\n", context);
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_key_based, &obj);
	}
	if (!ht_context) {
		/*
		  Search key based on glob expression
		*/
		g_print ("Search key based on glob expression: %s - %d %s %d\n", context, obj_is_window, __FILE__, __LINE__);
		if (context)
			pattern = g_pattern_spec_new ((const gchar *)context);

		obj.pattern = pattern;
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_key_glob_based, &obj);
		if (pattern)
			g_pattern_spec_free (pattern);
	}
	if (!ht_context) {
		/*
		  Search label based
		*/
		OBJInfo obj;

		g_print ("Search label based: %s - %d\n", context, obj_is_window);

		obj.pattern = NULL;
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_label_based, &obj);
	}
	if (!ht_context) {
		/*
		  Search label based on glob expression
		*/
		OBJInfo obj;
		GPatternSpec *pattern = NULL;

		g_print ("Search label based on glob expression: %s\n", context);
		if (context)
			pattern = g_pattern_spec_new (context);

		obj.pattern = pattern;
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_label_glob_based, &obj);
		if (pattern)
			g_pattern_spec_free (pattern);
	}
	if (!ht_context) {
		/*
		  Search key after stripping space
		*/
		g_print ("Search key after stripping space: %s\n", context);
		obj.key = context;
		obj.parent_name = parent;
		obj.obj_is_window = obj_is_window;
		ht_context = g_hash_table_find (ht, search_obj_after_stripping_space, &obj);
	}
	if (!ht_context) {
		g_sprintf (msg, "Object definition %s not found in appmap", context);
		g_print ("%s\n", msg);
		log_msg (LDTP_LOG_DEBUG, msg, log_fp);
	}
	return ht_context;
}

static gboolean
find_property (gpointer key, gpointer value, gpointer user_data)
{
	if (!key || !value || !user_data)
		return FALSE;
	if (ldtp_debug) {
		g_print ("User data: %s\n", (gchar *) user_data);
		//g_print ("Value: %s\n", (gchar *) value);
		g_print ("Key: %s\n", (gchar *) key);
	}

	if (g_utf8_collate (key, user_data) == 0)
		return TRUE;
	return FALSE;
}

gchar*
get_property (GHashTable *ht, gchar *property, FILE *log_fp)
{
	gchar *value = NULL;
	gchar msg [256];
	LDTPErrorCode err;

	if (ht == NULL) {
		err = LDTP_ERROR_APPMAP_NOT_INITIALIZED;
		g_print ("%s - %d - %s\n", __FILE__, __LINE__, ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	if (property == NULL) {
		err = LDTP_ERROR_ARGUMENT_NULL;
		g_print ("%s\n", ldtp_error_get_message (err));
		log_msg (LDTP_LOG_CAUSE, ldtp_error_get_message (err), log_fp);
		return NULL;
	}
	value = (gchar *) g_hash_table_find (ht, find_property, (gpointer) property);
	if (!value) {
		g_sprintf (msg, "Property %s not found in appmap", property);
		g_print ("%s\n", msg);
		log_msg (LDTP_LOG_DEBUG, msg, log_fp);
	}
	else
		g_print ("Property: %s - Value: %s\n", property, value);
	return value;
}

gboolean
search_title_based (gpointer key, gpointer value, gpointer user_data)
{
	gchar *tmp_key = NULL;
	if (!key && !user_data)
		return FALSE;
	if (g_utf8_collate (key, user_data) == 0)
		return TRUE;
	if (g_pattern_match_simple (key, user_data))
		return TRUE;
	tmp_key = g_strdup_printf ("*%s", (gchar *)key);
	if (tmp_key && g_pattern_match_simple (tmp_key, user_data)) {
		g_free (tmp_key);
		return TRUE;
	}
	if (tmp_key)
		g_free (tmp_key);
	return FALSE;
}
