<?php
/*
$Id: sambaSamAccount.inc,v 1.158 2009/11/26 10:59:00 gruberroland Exp $

  This code is part of LDAP Account Manager (http://www.ldap-account-manager.org/)
  Copyright (C) 2003 - 2006  Tilo Lutz
                2005 - 2009  Roland Gruber

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/**
* Manages Samba 3 accounts for users and hosts.
*
* @package modules
*
* @author Tilo Lutz
* @author Roland Gruber
* @author Michael Duergner
*/

/** terminals server options */
include_once('sambaSamAccount/sambaMungedDial.inc');

/**
* Manages the object class "sambaSamAccount" for users and hosts.
*
* @package modules
*/
class sambaSamAccount extends baseModule implements passwordService {

	// Variables
	/** use no password? */
	private $nopwd;
	/** password does not expire? */
	private $noexpire;
	/** account deactivated? */
	private $deactivated;
	/** array of well known rids */
	private $rids;
	/** HEX to binary conversion table */
	private $hex2bitstring = array('0' => '0000', '1' => '0001', '2' => '0010', '3' => '0011', '4' => '0100',
		'5' => '0101', '6' => '0110', '7' => '0111', '8' => '1000', '9' => '1001', 'A' => '1010',
		'B' => '1011', 'C' => '1100', 'D' => '1101', 'E' => '1110', 'F' => '1111');
	/** specifies if the password should be expired */
	private $expirePassword = false;


	/**
	* Creates a new sambaSamAccount object.
	*
	* @param string $scope account type (user, group, host)
	*/
	function __construct($scope) {
		// List of well known rids
		$this->rids = array(
			_('Domain admins') => 512, _('Domain users') => 513, _('Domain guests') => 514,
			_('Domain computers') => 515, _('Domain controllers') => 516, _('Domain certificate admins') => 517,
			_('Domain schema admins') => 518, _('Domain enterprise admins') => 519, _('Domain policy admins') => 520);
		// call parent constructor
		parent::__construct($scope);
		$this->autoAddObjectClasses = false;
	}

	/** this functin fills the error message array with messages
	**/
	function load_Messages() {
		// error messages for input checks
		$this->messages['homePath'][0] = array('ERROR', _('Home path'), _('Home path is invalid.'));
		$this->messages['homePath'][1] = array('INFO', _('Home path'), _('Inserted user or group name in home path.'));
		$this->messages['homePath'][2] = array('ERROR', _('Account %s:') . ' sambaSamAccount_homePath', _('Home path is invalid.'));
		$this->messages['profilePath'][0] = array('ERROR', _('Profile path'), _('Profile path is invalid!'));
		$this->messages['profilePath'][1] = array('INFO', _('Profile path'), _('Inserted user or group name in profile path.'));
		$this->messages['profilePath'][2] = array('ERROR', _('Account %s:') . ' sambaSamAccount_profilePath', _('Profile path is invalid!'));
		$this->messages['logonScript'][0] = array('ERROR', _('Logon script'), _('Logon script is invalid!'));
		$this->messages['logonScript'][1] = array('INFO', _('Logon script'), _('Inserted user or group name in logon script.'));
		$this->messages['logonScript'][2] = array('ERROR', _('Account %s:') . ' sambaSamAccount_logonScript', _('Logon script is invalid!'));
		$this->messages['workstations'][0] = array('ERROR', _('Samba workstations'), _('Please enter a comma separated list of host names!'));
		$this->messages['workstations'][1] = array('ERROR', _('Account %s:') . ' sambaSamAccount_workstations', _('Please enter a comma separated list of host names!'));
		$this->messages['sambaLMPassword'][2] = array('ERROR', _('Account %s:') . ' sambaSamAccount_password', _('Password contains invalid characters. Valid characters are: a-z, A-Z, 0-9 and #*,.;:_-+!%&/|?{[()]}=@$ !'));
		$this->messages['rid'][2] = array('ERROR', _('Account %s:') . ' sambaSamAccount_rid', _('Please enter a RID number or the name of a special account!'));
		$this->messages['rid'][3] = array('ERROR', _('Account %s:') . ' sambaSamAccount_rid', _('This is not a valid RID number!'));
		$this->messages['displayName'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_displayName', _('Please enter a valid display name!'));
		$this->messages['displayName'][1] = array('ERROR', _('Display name'), _('Please enter a valid display name!'));
		$this->messages['pwdUnix'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_pwdUnix', _('This value can only be \"true\" or \"false\"!'));
		$this->messages['noPassword'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_noPassword', _('This value can only be \"true\" or \"false\"!'));
		$this->messages['noExpire'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_noExpire', _('This value can only be \"true\" or \"false\"!'));
		$this->messages['deactivated'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_deactivated', _('This value can only be \"true\" or \"false\"!'));
		$this->messages['pwdCanChange'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_pwdCanChange', _('Please enter a valid date in format DD-MM-YYYY.'));
		$this->messages['pwdMustChange'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_pwdMustChange', _('Please enter a valid date in format DD-MM-YYYY.'));
		$this->messages['expireDate'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_expireDate', _('Please enter a valid date in format DD-MM-YYYY.'));
		$this->messages['homeDrive'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_homeDrive', _('Please enter a valid drive letter.'));
		$this->messages['domain'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_domain', _('LAM was unable to find a domain with this name!'));
		$this->messages['logonHours'][0] = array('ERROR', _('Logon hours'), _('The format of the logon hours field is invalid!'));
		$this->messages['logonHours'][1] = array('ERROR', _('Account %s:') . ' sambaSamAccount_logonHours', _('The format of the logon hours field is invalid!'));
		$this->messages['group'][0] = array('ERROR', _('Account %s:') . ' sambaSamAccount_group', _('Please enter a valid group name!'));
		$this->messages['profileCanMustChange'][0] = array('ERROR', _('The value for the Samba 3 field "User can/must change password" needs to be a number.'));
	}

	/**
	* Returns meta data that is interpreted by parent class
	*
	* @return array array with meta data
	* 
	* @see baseModule::get_metaData()
	*/
	function get_metaData() {
		$return = array();
		// icon
		$return['icon'] = 'samba.png';
		// manages user and host accounts
		$return["account_types"] = array("user", "host");
		// alias name
		$return["alias"] = _('Samba 3');
		// RDN attribute
		$return["RDN"] = array("sambaSID" => "low");
		// module dependencies
		$return['dependencies'] = array('depends' => array('posixAccount'), 'conflicts' => array());
		// LDAP filter
		$return["ldap_filter"] = array('or' => "(objectClass=sambaSamAccount)");
		// managed object classes
		$return['objectClasses'] = array('sambaSamAccount');
		// managed attributes
		$return['attributes'] = array('uid', 'sambaSID', 'sambaLMPassword', 'sambaNTPassword', 'sambaPwdLastSet',
			'sambaLogonTime', 'sambaLogoffTime', 'sambaKickoffTime', 'sambaPwdCanChange', 'sambaPwdMustChange', 'sambaAcctFlags',
			'sambaPwdLastSet', 'displayName', 'sambaHomePath', 'sambaHomeDrive', 'sambaLogonScript', 'sambaProfilePath',
			'sambaUserWorkstations', 'sambaPrimaryGroupSID', 'sambaDomainName', 'sambaLogonHours', 'sambaMungedDial');
		// PHP extensions
		$return['extensions'] = array('hash', 'iconv');
		// profile checks
		$return['profile_checks']['sambaSamAccount_smbhome'] = array(
			'type' => 'ext_preg',
			'regex' => 'UNC',
			'error_message' => $this->messages['homePath'][0]);
		$return['profile_checks']['sambaSamAccount_profilePath'] = array(
			'type' => 'ext_preg',
			'regex' => 'UNC',
			'error_message' => $this->messages['profilePath'][0]);
		$return['profile_checks']['sambaSamAccount_logonScript'] = array(
			'type' => 'ext_preg',
			'regex' => 'logonscript',
			'error_message' => $this->messages['logonScript'][0]);
		$return['profile_checks']['sambaSamAccount_userWorkstations'] = array(
			'type' => 'ext_preg',
			'regex' => 'unixhost',
			'error_message' => $this->messages['workstations'][0]);
		$return['profile_checks']['sambaSamAccount_logonHours'] = array(
			'type' => 'ext_preg',
			'regex' => 'sambaLogonHours',
			'error_message' => $this->messages['logonHours'][0]);
		$return['profile_checks']['sambaSamAccount_pwdCanChange'] = array(
			'type' => 'ext_preg',
			'regex' => 'digit',
			'error_message' => $this->messages['profileCanMustChange'][0]);
		$return['profile_checks']['sambaSamAccount_pwdMustChange'] = array(
			'type' => 'ext_preg',
			'regex' => 'digit',
			'error_message' => $this->messages['profileCanMustChange'][0]);
		// profile mappings
		$return['profile_mappings'] = array(
			'sambaSamAccount_sambaDomainName' => 'sambaDomainName',
		);
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
			$return['profile_mappings']['sambaSamAccount_smbhome'] = 'sambaHomePath';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
			$return['profile_mappings']['sambaSamAccount_profilePath'] = 'sambaProfilePath';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
			$return['profile_mappings']['sambaSamAccount_logonScript'] = 'sambaLogonScript';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideWorkstations')) {
			$return['profile_mappings']['sambaSamAccount_userWorkstations'] = 'sambaUserWorkstations';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonHours')) {
			$return['profile_mappings']['sambaSamAccount_logonHours'] = 'sambaLogonHours';
		}
		// available PDF fields
		$return['PDF_fields'] = array(
			'displayName', 'sambaDomainName', 'sambaPrimaryGroupSID'
		);
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
			$return['PDF_fields'][] = 'sambaHomeDrive';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
			$return['PDF_fields'][] = 'sambaHomePath';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
			$return['PDF_fields'][] = 'sambaProfilePath';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
			$return['PDF_fields'][] = 'sambaLogonScript';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideWorkstations')) {
			$return['PDF_fields'][] = 'sambaUserWorkstations';
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideTerminalServer')) {
			$return['PDF_fields'][] = 'tsAllowLogin';
			$return['PDF_fields'][] = 'tsHomeDir';
			$return['PDF_fields'][] = 'tsHomeDrive';
			$return['PDF_fields'][] = 'tsProfilePath';
			$return['PDF_fields'][] = 'tsInherit';
			$return['PDF_fields'][] = 'tsInitialProgram';
			$return['PDF_fields'][] = 'tsWorkDirectory';
			$return['PDF_fields'][] = 'tsConnectionLimit';
			$return['PDF_fields'][] = 'tsDisconnectionLimit';
			$return['PDF_fields'][] = 'tsIdleLimit';
			$return['PDF_fields'][] = 'tsConnectDrives';
			$return['PDF_fields'][] = 'tsConnectPrinters';
			$return['PDF_fields'][] = 'tsClientPrinterDefault';
			$return['PDF_fields'][] = 'tsShadowing';
			$return['PDF_fields'][] = 'tsBrokenConn';
			$return['PDF_fields'][] = 'tsReconnect';
		}
		$return['selfServiceFieldSettings'] = array(
			'syncNTPassword' => _('Sync Samba NT password with Unix password'),
			'syncLMPassword' => _('Sync Samba LM password with Unix password'),
			'syncSambaPwdLastSet' => _('Update attribute "sambaPwdLastSet" on password change'),
			'syncSambaPwdMustChange' => _('Update attribute "sambaPwdMustChange" on password change'),
			'syncSambaPwdCanChange' => _('Update attribute "sambaPwdCanChange" on password change')
		);
		// help Entries
		$return['help'] = array (
			"displayName" => array(
				"Headline" => _("Display name"),
				"Text" => _("This is the account's full name on Windows systems.")),
			"password" => array(
				"Headline" => _("Samba password"),
				"Text" => _("This is the account's Windows password.")),
			"resetPassword" => array(
				"Headline" => _("Reset password"),
				"Text" => _("This will reset the host's password to a default value.")),
			"pwdUnix" => array(
				"Headline" => _("Use Unix password"),
				"Text" => _("If checked Unix password will also be used as Samba password.")),
			"pwdUnixUpload" => array(
				"Headline" => _("Use Unix password"),
				"Text" => _("If set to \"true\" Unix password will also be used as Samba password.")),
			"noPassword" => array(
				"Headline" => _("Use no password"),
				"Text" => _("If checked no password will be used.")),
			"noPasswordUpload" => array(
				"Headline" => _("Use no password"),
				"Text" => _("If set to \"true\" no password will be used.")),
			"noExpire" => array(
				"Headline" => _("Password does not expire"),
				"Text" => _("If checked password does not expire. (Setting X-Flag)")),
			"noExpireUpload" => array(
				"Headline" => _("Password does not expire"),
				"Text" => _("If set to \"true\" password does not expire. (Setting X-Flag)")),
			"deactivated" => array(
				"Headline" => _("Account is deactivated"),
				"Text" => _("If checked then the account will be deactivated. (Setting D-Flag)")),
			"locked" => array(
				"Headline" => _("Account is locked"),
				"Text" => _("If checked then the account will be locked (setting L-Flag). You usually want to use this setting to unlock user accounts which were locked because of failed login attempts.")),
			"deactivatedUpload" => array(
				"Headline" => _("Account is deactivated"),
				"Text" => _("If set to \"true\" account will be deactivated. (Setting D-Flag)")),
			"passwordIsExpired" => array(
				"Headline" => _("Password change at next login"),
				"Text" => _("If you set this option then the user has to change his password at the next login.")),
			"pwdCanChange" => array(
				"Headline" => _("User can change password"),
				"Text" => _("Date after the user is able to change his password. Format: DD-MM-YYYY")),
			"pwdMustChange" => array ("Headline" => _("User must change password"),
				"Text" => _("Date after the user must change his password. Format: DD-MM-YYYY")),
			"homeDrive" => array(
				"Headline" => _("Home drive"),
				"Text" => _("The home directory will be connected under this drive letter.")),
			"homePath" => array(
				"Headline" => _("Home path"),
				"Text" => _('UNC-path (\\\\server\\share) of homedirectory. $user and $group are replaced with user and group name.'). ' '. _("Can be left empty.")),
			"profilePath" => array(
				"Headline" => _("Profile path"),
				"Text" => _('Path of the user profile. Can be a local absolute path or a UNC-path (\\\\server\\share). $user and $group are replaced with user and group name.'). ' '. _("Can be left empty.")),
			"scriptPath" => array(
				"Headline" => _("Logon script"),
				"Text" => _('File name and path relative to netlogon-share which should be executed on logon. $user and $group are replaced with user and group name.'). ' '. _("Can be left empty.")),
			"userWorkstations" => array(
				"Headline" => _("Samba workstations"),
				"Text" => _("List of Samba workstations the user is allowed to login. Empty means every workstation.")),
			"workstations" => array(
				"Headline" => _("Samba workstations"),
				"Text" => _("Comma separated list of Samba workstations the user is allowed to login. Empty means every workstation."). ' '. _("Can be left empty.")),
			"group" => array(
				"Headline" => _("Windows primary group"),
				"Text" => _("This is the user's primary Windows group.")),
			"groupUpload" => array(
				"Headline" => _("Windows primary group SID"),
				"Text" => _("This is the SID of the user's primary Windows group.")),
			"specialUser" => array(
				"Headline" => _("Special user"),
				"Text" => _("This allows you to define this account as a special user like administrator or guest.")),
			"ridUpload" => array(
				"Headline" => _("Samba RID"),
				"Text" => _("This is the relative ID number for your Windows account. You can either enter a number or one of these special accounts: ") .
				implode(", ", array_keys($this->rids)) . "<br><br>" . _("If you leave this empty LAM will use: uidNumber*2 + sambaAlgorithmicRidBase.")),
			"ridUploadHost" => array(
				"Headline" => _("Samba RID"),
				"Text" => _("This is the relative ID number for your host account. If you leave this empty LAM will use: uidNumber*2 + sambaAlgorithmicRidBase.")),
			"domain" => array(
				"Headline" => _("Domain"),
				"Text" => _("Windows domain name of account.")),
			"logonHours" => array(
				"Headline" => _("Logon hours"),
				"Text" => _("This option defines the allowed logon hours for this account.")),
			"logonHoursUpload" => array(
				"Headline" => _("Logon hours"),
				"Text" => _("This option defines the allowed logon hours for this account. The format is the same as for the LDAP attribute. The 24*7 hours are represented as 168 bit which are saved as 21 hex (21*8 = 168) values. The first bit represents Sunday 0:00 - 0:59 in GMT.")),
			'expireDate' => array (
				"Headline" => _("Account expiration date"),
				"Text" => _("This is the date when the account will expire. Format: DD-MM-YYYY")),
			'timeZone' => array (
				"Headline" => _("Time zone"),
				"Text" => _("This is the time zone of your Samba server. LAM needs this information to display the logon hours correctly.")),
			'tsAllowLogin' => array (
				"Headline" => _("Allow terminal server login"),
				"Text" => _("Activate this checkbox to allow this user to use the terminal service.")),
			'tsHomeDir' => array (
				"Headline" => _("Home directory"),
				"Text" => _("This is the path to the user's home directory.")),
			'tsProfilePath' => array (
				"Headline" => _("Profile path"),
				"Text" => _("Path of the user profile.")),
			'tsInherit' => array (
				"Headline" => _("Inherit client startup configuration"),
				"Text" => _("Activate this checkbox to inherit the initial program and working directory from the client machine.")),
			'tsInitialProgram' => array (
				"Headline" => _("Initial program"),
				"Text" => _("This program is run after the login.")),
			'tsWorkDirectory' => array (
				"Headline" => _("Working directory"),
				"Text" => _("Working directory of initial program.")),
			'tsTimeLimit' => array (
				"Headline" => _("Time limit"),
				"Text" => _("Please enter the time limit in minutes. 0 means unlimited.")),
			'tsConnectDrives' => array (
				"Headline" => _("Connect client drives"),
				"Text" => _("Activate this checkbox to connect drives from the client machine.")),
			'tsConnectPrinters' => array (
				"Headline" => _("Connect client printers"),
				"Text" => _("Activate this checkbox to connect printers from the client machine.")),
			'tsClientPrinterDefault' => array (
				"Headline" => _("Client printer is default"),
				"Text" => _("Activate this checkbox to set the client's printer as default printer.")),
			'tsShadowing' => array (
				"Headline" => _("Shadowing"),
				"Text" => _("Here you can specify the shadowing mode.")),
			'tsBrokenConn' => array (
				"Headline" => _("On broken or timed out connection"),
				"Text" => _("This specifies what to do when the client connection is broken.")),
			'tsReconnect' => array (
				"Headline" => _("Reconnect if disconnected"),
				"Text" => _("This specifies the reconnect policy.")),
			'terminalServer' => array (
				"Headline" => _("Terminal server options"),
				"Text" => _("Here you can change the settings for the terminal server access.")),
			'profilePwdCanMustChange' => array (
				"Headline" => _("User can/must change password"),
				"Text" => _("This is the number of seconds after when the user may or has to change his password.")),
			'lmHash' => array (
				"Headline" => _("Disable LM hashes"),
				"Text" => _("Windows password hashes are saved by default as NT and LM hashes. LM hashes are insecure and only needed for old versions of Windows. You should disable them unless you really need them.")),
			'hiddenOptions' => array(
				"Headline" => _("Hidden options"),
				"Text" => _("The selected options will not be managed inside LAM. You can use this to reduce the number of displayed input fields."))
			);
		// upload dependencies
		$return['upload_preDepends'] = array('posixAccount', 'inetOrgPerson');
		// upload options
		if ($this->get_scope() == "user") {
			$return['upload_columns'] = array(
				array(
					'name' => 'sambaSamAccount_domain',
					'description' => _('Domain'),
					'required' => true,
					'help' => 'domain',
					'example' => _('mydomain')
				),
				array(
					'name' => 'sambaSamAccount_displayName',
					'description' => _('Display name'),
					'help' => 'displayName',
					'example' => _('Steve Miller')
				),
				array(
					'name' => 'sambaSamAccount_password',
					'description' => _('Password'),
					'help' => 'password',
					'example' => _('secret')
				),
				array(
					'name' => 'sambaSamAccount_pwdUnix',
					'description' => _('Use Unix password'),
					'help' => 'pwdUnixUpload',
					'default' => 'true',
					'values' => 'true, false',
					'example' => 'true'
				),
				array(
					'name' => 'sambaSamAccount_noPassword',
					'description' => _('Use no password'),
					'help' => 'noPasswordUpload',
					'default' => 'false',
					'values' => 'true, false',
					'example' => 'false'
				),
				array(
					'name' => 'sambaSamAccount_noExpire',
					'description' => _('Password does not expire'),
					'help' => 'noExpireUpload',
					'default' => 'true',
					'values' => 'true, false',
					'example' => 'true'
				),
				array(
					'name' => 'sambaSamAccount_deactivated',
					'description' => _('Account is deactivated'),
					'help' => 'deactivatedUpload',
					'default' => 'false',
					'values' => 'true, false',
					'example' => 'false'
				),
				array(
					'name' => 'sambaSamAccount_pwdCanChange',
					'description' => _('User can change password'),
					'help' => 'pwdCanChange',
					'default' => '31-12-2030',
					'example' => '15-11-2006'
				),
				array(
					'name' => 'sambaSamAccount_pwdMustChange',
					'description' => _('User must change password'),
					'help' => 'pwdMustChange',
					'default' => '31-12-2030',
					'example' => '15-10-2006'
				),
				array(
					'name' => 'sambaSamAccount_expireDate',
					'description' => _('Account expiration date'),
					'help' => 'expireDate',
					'default' => '31-12-2030',
					'example' => '15-10-2006'
				),
				array(
					'name' => 'sambaSamAccount_group',
					'description' => _('Windows group'),
					'help' => 'groupUpload',
					'example' => _('mygroup'),
					'default' => 'Domain Users'
				),
				array(
					'name' => 'sambaSamAccount_rid',
					'description' => _('Samba RID'),
					'help' => 'ridUpload',
					'example' => '1235',
					'default' => '&lt;uidNumber&gt;*2 + &lt;sambaAlgorithmicRidBase&gt;'
				),
			);
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_homeDrive',
					'description' => _('Home drive'),
					'help' => 'homeDrive',
					'example' => 'k:'
				);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_homePath',
					'description' => _('Home path'),
					'help' => 'homePath',
					'example' => _('\\\\server\\homes\\smiller')
				);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_profilePath',
					'description' => _('Profile path'),
					'help' => 'profilePath',
					'example' => _('\\\\server\\profiles\\smiller')
				);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_logonScript',
					'description' => _('Logon script'),
					'help' => 'scriptPath',
					'example' => 'logon.bat'
				);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideWorkstations')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_workstations',
					'description' => _('Samba workstations'),
					'help' => 'workstations',
					'example' => 'PC01,PC02,PC03'
				);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonHours')) {
				$return['upload_columns'][] = array(
					'name' => 'sambaSamAccount_logonHours',
					'description' => _('Logon hours'),
					'help' => 'logonHoursUpload',
					'example' => 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF'
				);
			}
		}
		elseif ($this->get_scope() == "host") {
			$return['upload_columns'] = array(
				array(
					'name' => 'sambaSamAccount_domain',
					'description' => _('Domain'),
					'required' => true,
					'help' => 'domain',
					'example' => _('mydomain')
				),
				array(
					'name' => 'sambaSamAccount_rid',
					'description' => _('Samba RID'),
					'help' => 'ridUploadHost',
					'example' => '1235',
					'default' => '&lt;uidNumber&gt;*2 + &lt;sambaAlgorithmicRidBase&gt;'
				)
			);
		}
		// configuration options
		$return['config_options']['user'] = array(
		array(
			array('kind' => 'text', 'text' => '<b>' . _("Time zone") . ': &nbsp;</b>'),
			array('kind' => 'select', 'name' => 'sambaSamAccount_timeZone', 'size' => '1',
				'options' => array(
					array('-12', "GMT-12: " . _("Eniwetok, Kwajalein")),
					array('-11', "GMT-11: " . _("Midway Island, Samoa")),
					array('-10', "GMT-10: " . _("Hawaii")),
					array('-9', "GMT-09: " . _("Alaska")),
					array('-8', "GMT-08: " . _("Pacific Time (US & Canada)")),
					array('-7', "GMT-07: " . _("Mountain Time (US & Canada)")),
					array('-6', "GMT-06: " . _("Central Time (US & Canada), Mexico City")),
					array('-5', "GMT-05: " . _("Eastern Time (US & Canada), Bogota")),
					array('-4', "GMT-04: " . _("Atlantic time (Canada), Caracas")),
					array('-3', "GMT-03: " . _("Brazil, Buenos Aires")),
					array('-2', "GMT-02: " . _("Mid-Atlantic")),
					array('-1', "GMT-01: " . _("Azores, Cape Verde Islands")),
					array('0', 'GMT: ' . _("Western Europe Time, London, Lisbon")),
					array('1', "GMT+01: " . _("Central Europe Time, Paris, Berlin")),
					array('2', "GMT+02: " . _("Eastern Europe Time, South Africa")),
					array('3', "GMT+03: " . _("Baghdad, Riyadh, Moscow")),
					array('4', "GMT+04: " . _("Abu Dhabi, Muscat, Baku")),
					array('5', "GMT+05: " . _("Islamabad, Karachi")),
					array('6', "GMT+06: " . _("Almaty, Dhaka, Colombo")),
					array('7', "GMT+07: " . _("Bangkok, Hanoi, Jakarta")),
					array('8', "GMT+08: " . _("Beijing, Perth, Singapore")),
					array('9', "GMT+09: " . _("Tokyo, Seoul, Osaka, Yakutsk")),
					array('10', "GMT+10: " . _("East Australian Standard, Guam")),
					array('11', "GMT+11: " . _("Magadan, Solomon Islands")),
					array('12', "GMT+12: " . _("Auckland, Fiji, Kamchatka")),
					),
					'options_selected' => array('0'),
					'descriptiveOptions' => true),
			array('kind' => 'help', 'value' => 'timeZone')),
		array(
			array('kind' => 'text', 'text' => '<b>' . _("Disable LM hashes") . ': &nbsp;</b>'),
			array('kind' => 'select', 'name' => 'sambaSamAccount_lmHash', 'size' => '1',
				'options' => array(
					array('yes', _('yes')),
					array('no', _('no'))
					),
				'options_selected' => array('yes'),
				'descriptiveOptions' => true),
			array('kind' => 'help', 'value' => 'lmHash')
		),
		array(
			array('kind' => 'text', 'text' => '<b>' . _('Hidden options') . ":</b> ", 'td' => array('valign' => 'top')),
			array('kind' => 'table', 'value' => array(
				array(
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideHomeDrive', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Home drive')),
					array('kind' => 'text', 'text' => '&nbsp;'),
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideHomePath', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Home path')),
					array('kind' => 'text', 'text' => '&nbsp;'),
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideProfilePath', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Profile path')),
				),
				array(
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideLogonScript', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Logon script')),
					array('kind' => 'text', 'text' => '&nbsp;'),
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideWorkstations', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Samba workstations')),
					array('kind' => 'text', 'text' => '&nbsp;'),
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideLogonHours', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Logon hours')),
				),
				array(
					array('kind' => 'input', 'name' => 'sambaSamAccount_hideTerminalServer', 'type' => 'checkbox'),
					array('kind' => 'text', 'text' => _('Terminal server options')),
				),
			)),
			array('kind' => 'text', 'text' => '&nbsp;'),
			array('kind' => 'help', 'value' => 'hiddenOptions'),
		)
		);
		return $return;
	}

	/**
	* Initializes the module after it became part of an accountContainer
	*
	* @param string $base the name of the accountContainer object ($_SESSION[$base])
	*/
	function init($base) {
		// call parent init
		parent::init($base);
		$this->noexpire = true;
		$this->nopwd = false;
		$this->deactivated = false;
		$sambaDomains = search_domains();
		if (sizeof($sambaDomains) == 0) {
			StatusMessage("ERROR", _('No Samba 3 domains found in LDAP! Please create one first.'), '');
			return;
		}
	}

	/**
	* This function is used to check if this module page can be displayed.
	* It returns false if a module depends on data from other modules which was not yet entered.
	*
	* @return boolean true, if page can be displayed
	*/
	function module_ready() {
		$attrs = $this->getAccountContainer()->getAccountModule('posixAccount')->getAttributes();
		if ($attrs['gidNumber'][0]=='') return false;
		if ($attrs['uidNumber'][0]=='') return false;
		if ($attrs['uid'][0]=='') return false;
		return true;
	}
	
	/**
	* This function is used to check if all settings for this module have been made.
	* 
	* @see baseModule::module_complete
	*
	* @return boolean true, if settings are complete
	*/
	public function module_complete() {
		if (!in_array('sambaSamAccount', $this->attributes['objectClass'])) {
			return true;
		}
		if ($this->get_scope() == "host") {
			$attrs = $this->getAccountContainer()->getAccountModule('posixAccount')->getAttributes();
			if (substr($attrs['uid'][0], -1, 1) != '$') {
				return false;			
			}
		}
		return true;
	}


	/**
	* This function loads the LDAP attributes for this module.
	*
	* @param array $attributes attribute list
	*/
	function load_attributes($attr) {
		parent::load_attributes($attr);
		if (isset($this->attributes['sambaAcctFlags'][0])) {
			if (strpos($this->attributes['sambaAcctFlags'][0], "D")) $this->deactivated = true;
			else $this->deactivated = false;
			if (strpos($this->attributes['sambaAcctFlags'][0], "N")) $this->nopwd = true;
			else $this->nopwd = false;
			if (strpos($this->attributes['sambaAcctFlags'][0], "X")) $this->noexpire = true;
			else $this->noexpire = false;
		}
		if (isset($this->attributes['sambaPwdLastSet'][0]) && ($this->attributes['sambaPwdLastSet'][0] === '0')) {
			$this->expirePassword = true;
		}
	}

	/**
	* Returns a list of modifications which have to be made to the LDAP account.
	*
	* @return array list of modifications
	* <br>This function returns an array with 3 entries:
	* <br>array( DN1 ('add' => array($attr), 'remove' => array($attr), 'modify' => array($attr)), DN2 .... )
	* <br>DN is the DN to change. It may be possible to change several DNs (e.g. create a new user and add him to some groups via attribute memberUid)
	* <br>"add" are attributes which have to be added to LDAP entry
	* <br>"remove" are attributes which have to be removed from LDAP entry
	* <br>"modify" are attributes which have to been modified in LDAP entry
	*/
	function save_attributes() {
		if (!in_array('sambaSamAccount', $this->attributes['objectClass'])) {
			return array();
		}
		if ($this->expirePassword === true) {
			$this->attributes['sambaPwdLastSet'][0] = '0';
		}
		elseif ((isset($this->attributes['sambaPwdLastSet'][0])) && ($this->attributes['sambaPwdLastSet'][0] == '0')) {
			$this->attributes['sambaPwdLastSet'][0] = time();
		}
		$return = $this->getAccountContainer()->save_module_attributes($this->attributes, $this->orig);
		return $return;
	}

	/**
	* Processes user input of the primary module page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_attributes() {
		if (!in_array('sambaSamAccount', $this->attributes['objectClass'])) {
			return array();
		}
		// delete LM hash if needed
		if (!isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) || ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'yes')) {
			if (isset($this->attributes['sambaLMPassword'])) {
				unset($this->attributes['sambaLMPassword']);
			}
		}
		$errors = array();
		$sambaDomains = search_domains();
		if (sizeof($sambaDomains) == 0) {
			return array(array(array("ERROR", _('No Samba 3 domains found in LDAP! Please create one first.'))));
		}
		$attrs = $this->getAccountContainer()->getAccountModule('posixAccount')->getAttributes();
		// Save attributes
		$this->attributes['sambaDomainName'][0] = $_POST['sambaDomainName'];
		// Get Domain SID from name
		for ($i=0; $i<count($sambaDomains); $i++ ) {
			if ($this->attributes['sambaDomainName'][0] == $sambaDomains[$i]->name) {
				$SID = $sambaDomains[$i]->SID;
				$RIDbase = $sambaDomains[$i]->RIDbase;
				break;
			}
		}
		$flag = "[";
		if (isset($_POST['sambaAcctFlagsD'])) {
			$flag .= "D";
			$this->deactivated = true;
		}
		else {
			$this->deactivated = false;
		}
		if (isset($_POST['sambaAcctFlagsX'])) {
			$flag .= "X";
			$this->noexpire = true;
		}
		else {
			$this->noexpire = false;
		}
		if (isset($_POST['sambaAcctFlagsN'])) {
			$flag .= "N";
			$this->nopwd = true;
		}
		else {
			$this->nopwd = false;
		}
		if (isset($_POST['sambaAcctFlagsS'])) $flag .= "S";
		if (isset($_POST['sambaAcctFlagsH'])) $flag .= "H";
		if (isset($_POST['sambaAcctFlagsW'])) $flag .= "W";
		if (isset($_POST['sambaAcctFlagsU'])) $flag .= "U";
		if (isset($_POST['sambaAcctFlagsL'])) $flag .= "L";
		// Expand string to fixed length
		$flag = str_pad($flag, 12);
		// End character
		$flag = $flag. "]";
		$this->attributes['sambaAcctFlags'][0] = $flag;

		// display name
		$this->attributes['displayName'][0] = $_POST['displayName'];
		if (!($this->attributes['displayName'][0] == '') && !(get_preg($this->attributes['displayName'][0], 'realname'))) {
			$errors[] = $this->messages['displayName'][1];
		}

		// host attributes
		if ($this->get_scope()=='host') {
			$this->attributes['sambaPrimaryGroupSID'][0] = $SID."-".$this->rids[_('Domain computers')];
			if ($_POST['ResetSambaPassword'] || !$this->attributes['sambaNTPassword'][0]) {
				$hostname = $attrs['uid'][0];
				$hostname = substr($hostname, 0, strlen($hostname) - 1);
				if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
					$this->attributes['sambaLMPassword'][0] = lmPassword($hostname);
				}
				$this->attributes['sambaNTPassword'][0] = ntPassword($hostname);
				$this->attributes['sambaPwdLastSet'][0] = time();
			}
		}
		// user attributes
		$unixGroupName = $this->getGroupName($attrs['gidNumber'][0]);
		if ($this->get_scope()=='user') {
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
				$this->attributes['sambaHomePath'][0] = $_POST['sambaHomePath'];
				$this->attributes['sambaHomePath'][0] = str_replace('$user', $attrs['uid'][0], $this->attributes['sambaHomePath'][0]);
				$this->attributes['sambaHomePath'][0] = str_replace('$group', $unixGroupName, $this->attributes['sambaHomePath'][0]);
				if ($this->attributes['sambaHomePath'][0] != $_POST['sambaHomePath']) $errors[] = $this->messages['homePath'][1];
				if ( (!$this->attributes['sambaHomePath'][0]=='') && (!get_preg($this->attributes['sambaHomePath'][0], 'UNC'))) {
					$errors[] = $this->messages['homePath'][0];
				}
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
				if ($_POST['sambaHomeDrive'] == "-") $this->attributes['sambaHomeDrive'][0] = '';
				else $this->attributes['sambaHomeDrive'][0] = $_POST['sambaHomeDrive'];
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
				$this->attributes['sambaLogonScript'][0] = $_POST['sambaLogonScript'];
				$this->attributes['sambaLogonScript'][0] = str_replace('$user', $attrs['uid'][0], $this->attributes['sambaLogonScript'][0]);
				$this->attributes['sambaLogonScript'][0] = str_replace('$group', $unixGroupName, $this->attributes['sambaLogonScript'][0]);
				if ($this->attributes['sambaLogonScript'][0] != $_POST['sambaLogonScript']) $errors[] = $this->messages['logonScript'][1];
				if ( (!$this->attributes['sambaLogonScript'][0]=='') && (!get_preg($this->attributes['sambaLogonScript'][0], 'logonscript'))) {
					$errors[] = $this->messages['logonScript'][0];
				}
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
				$this->attributes['sambaProfilePath'][0] = $_POST['sambaProfilePath'];
				$this->attributes['sambaProfilePath'][0] = str_replace('$user', $attrs['uid'][0], $this->attributes['sambaProfilePath'][0]);
				$this->attributes['sambaProfilePath'][0] = str_replace('$group', $unixGroupName, $this->attributes['sambaProfilePath'][0]);
				if ($this->attributes['sambaProfilePath'][0] != $_POST['sambaProfilePath']) $errors[] = $this->messages['profilePath'][1];
				if (!($this->attributes['sambaProfilePath'][0] == '') &&
					!(get_preg($this->attributes['sambaProfilePath'][0], 'UNC') xor get_preg($this->attributes['sambaProfilePath'][0], 'homeDirectory'))) {
						$errors[] = $this->messages['profilePath'][0];
				}
			}
			$rids = array_keys($this->rids);
			$wrid = false;
			for ($i=0; $i<count($rids); $i++) {
				if ($_POST['sambaPrimaryGroupSID'] == $rids[$i]) {
					$wrid = true;
					// Get Domain SID
					$this->attributes['sambaPrimaryGroupSID'][0] = $SID."-".$this->rids[$rids[$i]];
					}
				}
			if (!$wrid) {
				$gidnumber = $attrs['gidNumber'][0];
				$groups = $_SESSION['cache']->get_cache(array('gidNumber', 'sambaSID'), 'sambaGroupMapping', 'group');
				$groupKeys = array_keys($groups);
				for ($i = 0; $i < sizeof($groupKeys); $i++) {
					if ($groups[$groupKeys[$i]]['gidNumber'][0] == $gidnumber) {
						if (isset($groups[$groupKeys[$i]]['sambaSID'][0])) {
							$this->attributes['sambaPrimaryGroupSID'][0] = $groups[$groupKeys[$i]]['sambaSID'][0];
						}
						break;
					}
				}
			}

			$specialRids = array_flip($this->rids);
			// set special RID if selected
			if (in_array($_POST['sambaSID'], $specialRids)) {
				$this->attributes['sambaSID'][0] = $SID . '-' . $this->rids[$_POST['sambaSID']];
			}
			// standard RID
			else if ($_POST['sambaSID'] == "-") {
				$rid = substr($this->attributes['sambaSID'][0], strrpos($this->attributes['sambaSID'][0], '-') + 1, strlen($this->attributes['sambaSID'][0]));
				// change only if not yet set, previously set to special SID or domain changed
				if (!isset($this->attributes['sambaSID'][0])
					|| in_array($rid, $this->rids)
					|| (strpos($this->attributes['sambaSID'][0], $SID) === false)) {
					$this->attributes['sambaSID'][0] = $SID."-". (($attrs['uidNumber'][0]*2)+$RIDbase);
				}
			}
		}
		else {  // host
			if (!$this->attributes['sambaSID'][0]) {
				$this->attributes['sambaSID'][0] = $SID."-". (($attrs['uidNumber'][0]*2)+$RIDbase);
			}
		}
		if (isset($_POST['passwordIsExpired'])) {
			$this->expirePassword = true;
		}
		else {
			$this->expirePassword = false;
		}
		return $errors;
	}

	/**
	* Processes user input of the primary module page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_sambaUserWorkstations() {
		// Load attributes
		if ($this->get_scope()=='user') {
			if (isset($_POST['availableSambaUserWorkstations']) && isset($_POST['sambaUserWorkstations_add'])) { // Add workstations to list
				$temp = str_replace(' ', '', $this->attributes['sambaUserWorkstations'][0]);
				$workstations = explode (',', $temp);
				for ($i=0; $i<count($workstations); $i++) {
					if ($workstations[$i]=='') unset($workstations[$i]);
				}
				$workstations = array_values($workstations);
				// Add new // Add workstations
				$workstations = array_merge($workstations, $_POST['availableSambaUserWorkstations']);
				// remove doubles
				$workstations = array_flip($workstations);
				array_unique($workstations);
				$workstations = array_flip($workstations);
				// sort workstations
				sort($workstations);
				// Recreate workstation string
				$this->attributes['sambaUserWorkstations'][0] = $workstations[0];
				for ($i=1; $i<count($workstations); $i++) {
					$this->attributes['sambaUserWorkstations'][0] = $this->attributes['sambaUserWorkstations'][0] . "," . $workstations[$i];
					}
				}
			elseif (isset($_POST['sambaUserWorkstations']) && isset($_POST['sambaUserWorkstations_remove'])) { // remove // Add workstations from list
				// Put all workstations in array
				$temp = str_replace(' ', '', $this->attributes['sambaUserWorkstations'][0]);
				$workstations = explode (',', $temp);
				for ($i=0; $i<count($workstations); $i++) {
					if ($workstations[$i]=='') unset($workstations[$i]);
				}
				$workstations = array_values($workstations);
				// Remove unwanted workstations from array
				$workstations = array_delete($_POST['sambaUserWorkstations'], $workstations);
				// Recreate workstation string
				$this->attributes['sambaUserWorkstations'][0] = $workstations[0];
				for ($i=1; $i<count($workstations); $i++) {
					$this->attributes['sambaUserWorkstations'][0] = $this->attributes['sambaUserWorkstations'][0] . "," . $workstations[$i];
				}
			}
		}
		return array();
	}

	/**
	* Processes user input of the logon hours page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_logonHours() {
		if (isset($_POST['form_subpage_sambaSamAccount_attributes_abort'])) return array();
		// set new logon hours
		$logonHours = '';
		for ($i = 0; $i < 24*7; $i++) {
			$logonHours .= isset($_POST['lh_' . $i]) ? '1' : '0';
		}
		// reconstruct HEX string
		$bitstring2hex = array_flip($this->hex2bitstring);
		$logonHoursNew = '';
		for ($i = 0; $i < 21; $i++) {
			$part = strrev(substr($logonHours, $i * 8, 8));
			$byte[hi] = substr($part,0,4);
			$byte[low] = substr($part,4,4);
			$hex = $bitstring2hex[$byte[hi]].$bitstring2hex[$byte[low]];
			$logonHoursNew = $logonHoursNew . $hex;
		}
		$this->attributes['sambaLogonHours'][0] = $logonHoursNew;
		return array();
	}

	/**
	* Processes user input of the time selection page.
	*
	* @return array list of info/error messages
	*/
	function process_time() {
		$return = array();
		// find button name
		$buttonName = '';
		$postKeys = array_keys($_POST);
		for ($i = 0; $i < sizeof($postKeys); $i++) {
			if (strpos($postKeys[$i], 'form_subpage_sambaSamAccount_attributes_') !== false) {
				$buttonName = $postKeys[$i];
			}
		}
		if (($buttonName == '') || (strpos($buttonName, '_back') !== false)) return array();
		// get attribute name
		$attr = '';
		if (strpos($buttonName, 'sambaPwdCanChange') !== false) {
			$attr = 'sambaPwdCanChange';
		}
		elseif (strpos($buttonName, 'sambaPwdMustChange') !== false) {
			$attr = 'sambaPwdMustChange';
		}
		elseif (strpos($buttonName, 'sambaKickoffTime') !== false) {
			$attr = 'sambaKickoffTime';
		}
		if ($attr == '') return array();
		// determine action
		if (strpos($buttonName, '_change') !== false) {
			// set new time
			$this->attributes[$attr][0] = gmmktime(0, 0, 0, intval($_POST['expire_mon']), intval($_POST['expire_day']),
				intval($_POST['expire_yea']));
		}
		elseif (strpos($buttonName, '_del') !== false) {
			// remove attribute value
			unset($this->attributes[$attr]);
		}
		return $return;
	}

	/**
	* Processes user input of the terminal server page.
	* It checks if all input values are correct and updates the associated LDAP attributes.
	*
	* @return array list of info/error messages
	*/
	function process_terminalServer() {
		if (isset($_POST['form_subpage_sambaSamAccount_attributes_abort'])) return array();
		$mDial = new sambaMungedDial();
		if (isset($this->attributes['sambaMungedDial'][0])) {
			$mDial->load($this->attributes['sambaMungedDial'][0]);
		}
		$mDial->setTsLogin(!isset($_POST['tsAllowLogin']));	
		$mDial->ctx['CtxWFHomeDir'] = $_POST['tsHomeDir'];
		$mDial->ctx['CtxWFHomeDirDrive'] = $_POST['tsHomeDrive'];
		$mDial->ctx['CtxWFProfilePath'] = $_POST['tsProfilePath'];
		$mDial->setInheritMode(isset($_POST['tsInherit']));
		$mDial->ctx['CtxInitialProgram'] = $_POST['tsInitialProgram'];
		$mDial->ctx['CtxWorkDirectory'] = $_POST['tsWorkDirectory'];
		$mDial->ctx['CtxMaxConnectionTime'] = $_POST['tsConnectionLimit'];
		$mDial->ctx['CtxMaxDisconnectionTime'] = $_POST['tsDisconnectionLimit'];
		$mDial->ctx['CtxMaxIdleTime'] = $_POST['tsIdleLimit'];
		$mDial->setConnectClientDrives(isset($_POST['tsConnectDrives']));
		$mDial->setConnectClientPrinters(isset($_POST['tsConnectPrinters']));
		$mDial->setDefaultPrinter(isset($_POST['tsClientPrinterDefault']));
		$mDial->setShadow(true, $_POST['tsShadowing']);
		$mDial->setBrokenConn($_POST['tsBrokenConn']);
		$mDial->setReConn($_POST['tsReconnect']);
		$this->attributes['sambaMungedDial'][0] = $mDial->getMunged();
		return array();
	}

	/**
	 * Returns the HTML meta data for the main account page.
	 * 
	 * @return array HTML meta data
	 */
	function display_html_attributes() {
		if (isset($_POST['form_subpage_sambaSamAccount_attributes_addObjectClass'])) {
			$this->attributes['objectClass'][] = 'sambaSamAccount';
		}
		$return = array();
		if (in_array('sambaSamAccount', $this->attributes['objectClass'])) {
			if ($this->get_scope() == "host") {
				$attrs = $this->getAccountContainer()->getAccountModule('posixAccount')->getAttributes();
				if (substr($attrs['uid'][0], -1, 1) != '$') {
					StatusMessage("ERROR", _('Host name must end with $!'), _('Please check your settings on the Unix page!'));
				}
			}
			// Get Domain SID from user SID
			$sambaDomains = search_domains();
			if (sizeof($sambaDomains) == 0) {
				StatusMessage("ERROR", _('No Samba 3 domains found in LDAP! Please create one first.'), '');
				return array();
			}
			if (isset($this->attributes['sambaSID'][0]) && $this->attributes['sambaSID'][0] != '') {
				$domainSID = substr($this->attributes['sambaSID'][0], 0, strrpos($this->attributes['sambaSID'][0], "-"));
			}
			$sel_domain = array();
			for ($i=0; $i<count($sambaDomains); $i++ ) {
				$sambaDomainNames[] = $sambaDomains[$i]->name;
				if (isset($domainSID)) {
					if ($domainSID == $sambaDomains[$i]->SID) {
						$SID = $domainSID;
						$sel_domain = array($sambaDomains[$i]->name);
					}
				}
				elseif (isset($this->attributes['sambaDomainName'][0]) && ($this->attributes['sambaDomainName'][0]!='')) {
					if ($this->attributes['sambaDomainName'][0] == $sambaDomains[$i]->name) {
						$SID = $sambaDomains[$i]->SID;
						$sel_domain = array($sambaDomains[$i]->name);
					}
				}
			}
	
			$displayName = '';
			if (isset($this->attributes['displayName'][0])) $displayName = $this->attributes['displayName'][0];
			$return[] = array(
				array('kind' => 'text', 'text' => _('Display name')),
				array('kind' => 'input', 'type' => 'text', 'name' => 'displayName', 'size' => '20', 'value' => $displayName),
				array('kind' => 'help', 'value' => 'displayName'));
			if ($this->get_scope()=='user') {
				$return[] = array(
						array('kind' => 'input', 'name' => 'sambaAcctFlagsU', 'type' => 'hidden', 'value' => 'true'));
				$return[] = array(
					array('kind' => 'text', 'text' => _('Use no password') ),
					array('kind' => 'input', 'name' => 'sambaAcctFlagsN', 'type' => 'checkbox', 'checked' => $this->nopwd),
					array('kind' => 'help', 'value' => 'noPassword'));
				$return[] = array(
					array('kind' => 'text', 'text' => _('Password does not expire') ),
					array('kind' => 'input', 'name' => 'sambaAcctFlagsX', 'type' => 'checkbox', 'checked' => $this->noexpire),
					array('kind' => 'help', 'value' => 'noExpire'));
				$return[] = array(
					array('kind' => 'text', 'text' => _('Account is deactivated') ),
					array('kind' => 'input', 'name' => 'sambaAcctFlagsD', 'type' => 'checkbox', 'checked' => $this->deactivated),
					array('kind' => 'help', 'value' => 'deactivated'));
				$locked = false;
				if (isset($this->attributes['sambaAcctFlags'][0]) && (strpos($this->attributes['sambaAcctFlags'][0], "L") !== false)) {
					$locked = true;
				}
				$return[] = array(
					array('kind' => 'text', 'text' => _('Account is locked') ),
					array('kind' => 'input', 'name' => 'sambaAcctFlagsL', 'type' => 'checkbox', 'checked' => $locked),
					array('kind' => 'help', 'value' => 'locked'));
	
				$return[] = array(
					array('kind' => 'text', 'text' => _('Password change at next login') ),
					array('kind' => 'input', 'name' => 'passwordIsExpired', 'type' => 'checkbox', 'checked' => $this->expirePassword),
					array('kind' => 'help', 'value' => 'passwordIsExpired'));

				$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;-&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
				if (isset($this->attributes['sambaPwdCanChange'][0])) {
					if ($this->attributes['sambaPwdCanChange'][0] > 2147483648) {
						$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;∞&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
					}
					else {
						$date = getdate($this->attributes['sambaPwdCanChange'][0]);
						$dateValue = $date['mday'] . "." . $date['mon'] . "." . $date['year'];
					}
				}
				$return[] = array(
					array('kind' => 'text', 'text' => _('User can change password')),
					array('kind' => 'table', 'value' => array(array(
						array('kind' => 'text', 'text' => $dateValue),
						array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_time_sambaPwdCanChange', 'type' => 'submit', 'value' => _('Change'))
					))),
					array('kind' => 'help', 'value' => 'pwdCanChange' ));
				$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;-&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
				if (isset($this->attributes['sambaPwdMustChange'][0])) {
					if ($this->attributes['sambaPwdMustChange'][0] > 2147483648) {
						$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;∞&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
					}
					else {
						$date = getdate($this->attributes['sambaPwdMustChange'][0]);
						$dateValue = $date['mday'] . "." . $date['mon'] . "." . $date['year'];
					}
				}
				$return[] = array(
					array('kind' => 'text', 'text' => _('User must change password')),
					array('kind' => 'table', 'value' => array(array(
						array('kind' => 'text', 'text' => $dateValue),
						array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_time_sambaPwdMustChange', 'type' => 'submit', 'value' => _('Change'))
					))),
					array('kind' => 'help', 'value' => 'pwdMustChange' ));
				$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;-&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
				if (isset($this->attributes['sambaKickoffTime'][0])) {
					if ($this->attributes['sambaKickoffTime'][0] > 2147483648) {
						$dateValue = "&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;∞&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;";
					}
					else {
						$date = getdate($this->attributes['sambaKickoffTime'][0]);
						$dateValue = $date['mday'] . "." . $date['mon'] . "." . $date['year'];
					}
				}
				$return[] = array(
					array('kind' => 'text', 'text' => _('Account expiration date')),
					array('kind' => 'table', 'value' => array(array(
						array('kind' => 'text', 'text' => $dateValue),
						array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_time_sambaKickoffTime', 'type' => 'submit', 'value' => _('Change'))
					))),
					array('kind' => 'help', 'value' => 'expireDate' ));
	
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
					$drives = array('-');
					for ($i=90; $i>67; $i--) $drives[] = chr($i).':';
					if (isset($this->attributes['sambaHomeDrive'][0])) {
						$selected = array ($this->attributes['sambaHomeDrive'][0]);
					}
					else $selected = array('-');
					$return[] = array(
						array('kind' => 'text', 'text' => _('Home drive') ),
						array('kind' => 'select', 'name' => 'sambaHomeDrive', 'options' => $drives, 'options_selected' => $selected),
						array('kind' => 'help', 'value' => 'homeDrive' ));
				}
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
					$sambaHomePath = '';
					if (isset($this->attributes['sambaHomePath'][0])) {
						$sambaHomePath = $this->attributes['sambaHomePath'][0];
					}
					$return[] = array(
						array('kind' => 'text', 'text' => _('Home path') ),
						array('kind' => 'input', 'type' => 'text', 'name' => 'sambaHomePath', 'size' => '20', 'maxlength' => '255', 'value' => $sambaHomePath),
						array('kind' => 'help', 'value' => 'homePath' ));
				}
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
					$sambaProfilePath = '';
					if (isset($this->attributes['sambaProfilePath'][0])) {
						$sambaProfilePath = $this->attributes['sambaProfilePath'][0];
					}
					$return[] = array(
						array('kind' => 'text', 'text' => _('Profile path') ),
						array('kind' => 'input', 'type' => 'text', 'name' => 'sambaProfilePath', 'size' => '20', 'maxlength' => '255', 'value' => $sambaProfilePath),
						array('kind' => 'help', 'value' => 'profilePath' ));
				}
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
					$sambaLogonScript = '';
					if (isset($this->attributes['sambaLogonScript'][0])) {
						$sambaLogonScript = $this->attributes['sambaLogonScript'][0];
					}
					$return[] = array(
						array('kind' => 'text', 'text' => _('Logon script') ),
						array('kind' => 'input', 'type' => 'text', 'name' => 'sambaLogonScript', 'size' => '20', 'maxlength' => '255', 'value' => $sambaLogonScript),
						array('kind' => 'help', 'value' => 'scriptPath' ));
				}
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideWorkstations')) {
					$return[] = array(
						array('kind' => 'text', 'text' => _('Samba workstations') ),
						array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_sambaSamAccount_sambaUserWorkstations_open', 'value' => _('Edit workstations')),
						array('kind' => 'help', 'value' => 'userWorkstations' ));
				}
				$names = array_keys($this->rids);
				$wrid=false;
				$options = array();
				$selected = array();
				for ($i=0; $i<count($names); $i++) {
					if (isset($this->attributes['sambaPrimaryGroupSID'][0]) && ($this->attributes['sambaPrimaryGroupSID'][0] == $SID . "-" . $this->rids[$names[$i]])) {
						$selected[] = $names[$i];
						$wrid=true;
					}
					else $options[] = $names[$i];
				}
				$attrs = $this->getAccountContainer()->getAccountModule('posixAccount')->getAttributes();
				if ($wrid) $options[] = $this->getGroupName($attrs['gidNumber'][0]);
					else $selected[] = $this->getGroupName($attrs['gidNumber'][0]);
				$return[] = array(
					array('kind' => 'text', 'text' => _('Windows group')),
					array('kind' => 'select', 'name' => 'sambaPrimaryGroupSID', 'options' => $options, 'options_selected' => $selected),
					array('kind' => 'help', 'value' => 'group'));
				// display if group SID should be mapped to a well known SID
				$options = array_keys($this->rids);
				$options[] = '-';
				$selected = array();
				if (isset($this->attributes['sambaSID'][0]) && ($this->attributes['sambaSID'][0] != '')) {
					$rid = substr($this->attributes['sambaSID'][0], strrpos($this->attributes['sambaSID'][0], '-') + 1, strlen($this->attributes['sambaSID'][0]));
					$specialRids = array_flip($this->rids);
					if (in_array($rid, $this->rids)) {
						$selected = array($specialRids[$rid]);
					}
					else {
						$selected = array('-');
					}
				}
				else $selected[] = "-";
				$return[] = array(
					array('kind' => 'text', 'text' => _('Special user') ),
					array('kind' => 'select', 'name' => 'sambaSID', 'options' => $options, 'options_selected' => $selected),
					array('kind' => 'help', 'value' => 'specialUser' ));
				}
			$return[] = array(
				array('kind' => 'text', 'text' => _('Domain') . "*" ),
				array('kind' => 'select', 'name' => 'sambaDomainName', 'options' => $sambaDomainNames, 'options_selected' => $sel_domain),
				array('kind' => 'help', 'value' => 'domain' ));
			// logon hours and terminal server options
			if ($this->get_scope()=='user') {
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonHours')) {
					$return[] = array(
						array('kind' => 'text', 'text' => _('Logon hours')),
						array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_sambaSamAccount_logonHours_open', 'value' => _('Edit')),
						array('kind' => 'help', 'value' => 'logonHours'));
				}
				if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideTerminalServer')) {
					$return[] = array(
						array('kind' => 'text', 'text' => _('Terminal server options') ),
						array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_sambaSamAccount_terminalServer_open', 'value' => _('Edit')),
						array('kind' => 'help', 'value' => 'terminalServer'));
				}
			}
			// reset host password
			if ($this->get_scope()=='host') {
				$return[] = array(array ( 'kind' => 'input', 'name' => 'sambaAcctFlagsW', 'type' => 'hidden', 'value' => 'true' ));
				$return[] = array(
					array('kind' => 'text', 'text' => _('Reset password') ),
					array('kind' => 'input', 'type' => 'submit', 'name' => 'ResetSambaPassword', 'value' => _('Reset')),
					array('kind' => 'help', 'value' => 'resetPassword' ));
				}
		}
		else {
			$return[] = array(
				array('kind' => 'text', 'text' => '&nbsp;'),
				array('kind' => 'input', 'type' => 'submit', 'name' => 'form_subpage_sambaSamAccount_attributes_addObjectClass', 'value' => _('Add Samba 3 account'))
			);
		}
		return $return;
	}

	/**
	* This function will create the HTML page to edit the allowed workstations.
	*
	* @return array meta HTML code
	*/
	function display_html_sambaUserWorkstations() {
		if ($this->get_scope()=='user') {
			// Get list of all hosts.
			$result = $_SESSION['cache']->get_cache('uid', 'sambaSamAccount', 'host');
			if (is_array($result)) {
				foreach ($result as $host) $availableUserWorkstations[] = str_replace("$", '', $host[0]);
				sort($availableUserWorkstations, SORT_STRING);
				$result = str_replace(' ', '', $this->attributes['sambaUserWorkstations'][0]);
				$userWorkstations = explode (',', $result);
				$availableUserWorkstations = array_delete($userWorkstations, $availableUserWorkstations);
				}

			$return[] = array( 0 => array ( 'kind' => 'fieldset', 'legend' => _("Allowed workstations"), 'value' =>
				array ( 0 => array ( 0 => array ('kind' => 'fieldset', 'td' => array ('valign' => 'top'), 'legend' => _("Allowed workstations"), 'value' =>
					array ( 0 => array ( 0 => array ( 'kind' => 'select', 'name' => 'sambaUserWorkstations', 'size' => '15', 'multiple' => true, 'options' => $userWorkstations)))),
				1 => array ( 'kind' => 'table', 'value' => array ( 0 => array ( 0 => array ( 'kind' => 'input', 'type' => 'submit', 'name' => 'sambaUserWorkstations_add',
					'value' => '<=')), 1 => array ( 0 => array ( 'kind' => 'input', 'type' => 'submit', 'name' => 'sambaUserWorkstations_remove', 'value' => '=>' )),
					2 => array ( 0 => array ( 'kind' => 'help', 'value' => 'sambaUserWorkstations' )))),
				2 => array ('kind' => 'fieldset', 'td' => array ('valign' => 'top'), 'legend' => _("Available workstations"), 'value' =>
					array ( 0 => array ( 0 => array ( 'kind' => 'select', 'name' => 'availableSambaUserWorkstations', 'size' => '15', 'multiple' => true, 'options' => $availableUserWorkstations))))
					))));

			$return[] = array ( 0 => array ( 'kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_back', 'type' => 'submit', 'value' => _('Back') ),
				1 => array ( 'kind' => 'text'),
				2 => array ('kind' => 'text'));
			}
		return $return;
	}

	/**
	* This function will create the HTML page to edit logon hours.
	*
	* @return array meta HTML code
	*/
	function display_html_logonHours() {
		$timeZone = 0;
		if (isset($this->moduleSettings['sambaSamAccount_timeZone']) && is_array($this->moduleSettings['sambaSamAccount_timeZone'])) {
			$timeZone = $this->moduleSettings['sambaSamAccount_timeZone'][0];
		}
		$days = array(1 => _('Monday'), 2 => _('Tuesday'), 3 => _('Wednesday'), 4 => _('Thursday'),
			5 => _('Friday'), 6 => _('Saturday'), 0 => _('Sunday'));
		if (!$this->attributes['sambaLogonHours'][0]) {
			$this->attributes['sambaLogonHours'][0] = 'FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF';
		}
		// convert existing logonHours string to bit array
		$logonHours = $this->attributes['sambaLogonHours'][0];
		$temp = array();
		for ($i = 0; $i < strlen($logonHours); $i++) {
			$temp[] = $this->hex2bitstring[$logonHours[$i]];
		}
		$logonHoursRev = implode('', $temp);
		// reverse bits low to high (1 is 0:00 sunday, 2 is 1:00 sunday, etc)
		$logonHours = "";
		for ($i = 0; $i < 21; $i++) {
			$logonHours .= strrev(substr($logonHoursRev, $i*8, 8));
		}
		$hour = array();
		for ($i = 0; $i < 24*7; $i++) {
			$hour[$i] = substr($logonHours, $i, 1);
		}
		// display input
		for ($i = 0; $i < 24*7; $i++) {
			$hr = $i + $timeZone;
			if ($hr < 0) {
				$hr = $hr + 24*7;
			}
			elseif ($hr >= 24*7) {
				$hr = $hr - 24*7;
			}
			if ($i % 7 == 0) {
				$return[0][floor($i / 24)+1] = array('kind' => 'text',
					'text' => '<b>' . $days[floor($i / 24)] . '</b>',
					'td' => array('width' => "11%", 'align' => 'center'));
			}
			$return[$i % 24 + 1][floor($i/24) + 1] = array('kind' => 'input',
				'name' => 'lh_' . $hr,
				'type' => 'checkbox',
				'checked' => $hour[$hr] ? true : false,
				'td' => array('align' => 'center'));
		}
		$return[0][0] = array('kind' => 'text', 'text' => '<b>' . _('Time') . '</b>', 'td' => array('width' => "12.5%"));
		for ($h = 0; $h < 24; $h++) {
			$return[1 + $h][0] = array('kind' => 'text', 'text' => "$h:00 - $h:59");
		}
		$return[] = array(0 => array('kind' => 'text', 'text' => '&nbsp;', 'td' => array('colspan' => 8)));
		$return[] = array(
			0 => array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_submit', 'type' => 'submit', 'value' => _('Ok'), 'td' => array('align' => 'right')),
			1 => array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_abort', 'type' => 'submit', 'value' => _('Cancel'), 'td' => array('align' => 'left'))
		);
		return $return;
	}

	/**
	* This function will create the meta HTML code to show a page to change time values.
	*
	* @return array meta HTML code
	*/
	function display_html_time() {
		$return = array();
		// determine attribute
		if (isset($_POST['form_subpage_sambaSamAccount_time_sambaPwdCanChange'])) {
			$attr = 'sambaPwdCanChange';
			$text = _('User can change password');
			$help = "pwdCanChange";
		}
		elseif (isset($_POST['form_subpage_sambaSamAccount_time_sambaPwdMustChange'])) {
			$attr = 'sambaPwdMustChange';
			$text = _('User must change password');
			$help = "pwdMustChange";
		}
		if (isset($_POST['form_subpage_sambaSamAccount_time_sambaKickoffTime'])) {
			$attr = 'sambaKickoffTime';
			$text = _('Account expiration date');
			$help = "expireDate";
		}
		$time = 0;
		if (isset($this->attributes[$attr][0])) {
			$time = $this->attributes[$attr][0];
		}
		$date = getdate($time);
		for ( $i=1; $i<=31; $i++ ) $mday[] = $i;
		for ( $i=1; $i<=12; $i++ ) $mon[] = $i;
		for ( $i=2003; $i<=2030; $i++ ) $year[] = $i;
		$return[] = array(
			array('kind' => 'text', 'text' => $text),
			array('kind' => 'table', 'value' => array(array(
				array('kind' => 'select', 'name' => 'expire_day', 'options' => $mday, 'options_selected' => $date['mday']),
				array('kind' => 'select', 'name' => 'expire_mon', 'options' => $mon, 'options_selected' => $date['mon']),
				array('kind' => 'select', 'name' => 'expire_yea', 'options' => $year, 'options_selected' => $date['year'])))),
			array('kind' => 'help', 'value' => $help));
		$buttons = array();
		$buttons[] = array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_change' . $attr, 'type' => 'submit', 'value' => _('Change'));
		if (isset($this->attributes[$attr][0])) {
			$buttons[] = array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_del' . $attr, 'type' => 'submit', 'value' => _('Remove'));
		}
		$buttons[] = array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_back' . $attr, 'type' => 'submit', 'value' => _('Cancel'));
		$return[] = array(
			array('kind' => 'table', 'td' => array('colspan' => 3), 'value' => array($buttons))
		);
		return $return;	
	}

	/**
	* This function will create the HTML page to edit the terminal server options.
	*
	* @return array meta HTML code
	*/
	function display_html_terminalServer() {
		$return = array();
		$mDial = new sambaMungedDial();
		if (isset($this->attributes['sambaMungedDial'][0])) {
			$mDial->load($this->attributes['sambaMungedDial'][0]);
		}
		// terminal server login
		$return[] = array(
			array('kind' => 'text', 'text' => _('Allow terminal server login')),
			array('kind' => 'input', 'name' => 'tsAllowLogin', 'type' => 'checkbox', 'checked' => $mDial->getTsLogin()),
			array('kind' => 'help', 'value' => 'tsAllowLogin'));
		// home directory
		$return[] = array(
			array('kind' => 'text', 'text' => _('Home directory')),
			array('kind' => 'input', 'name' => 'tsHomeDir', 'type' => 'text', 'value' => $mDial->ctx['CtxWFHomeDir']),
			array('kind' => 'help', 'value' => 'tsHomeDir'));
		// home drive
		$drives = array();
		for ($i=90; $i>67; $i--) $drives[] = chr($i).':';
		$selTsDrive = array();
		if (isset($mDial->ctx['CtxWFHomeDirDrive'])) {
			$selTsDrive = array($mDial->ctx['CtxWFHomeDirDrive']);
		}
		$return[] = array(
			array('kind' => 'text', 'text' => _('Home drive')),
			array('kind' => 'select', 'size' => 1, 'name' => 'tsHomeDrive', 'options' => $drives, 'options_selected' => $selTsDrive),
			array('kind' => 'help', 'value' => 'homeDrive'));
		// profile path
		$return[] = array(
			array('kind' => 'text', 'text' => _('Profile path')),
			array('kind' => 'input', 'name' => 'tsProfilePath', 'type' => 'text', 'value' => $mDial->ctx['CtxWFProfilePath']),
			array('kind' => 'help', 'value' => 'tsProfilePath'));
		// use startup program and working dir from client
		$return[] = array(
			array('kind' => 'text', 'text' => _('Inherit client startup configuration')),
			array('kind' => 'input', 'name' => 'tsInherit', 'type' => 'checkbox', 'checked' => $mDial->getInheritMode()),
			array('kind' => 'help', 'value' => 'tsInherit'));
		// startup program
		$return[] = array(
			array('kind' => 'text', 'text' => _('Initial program')),
			array('kind' => 'input', 'name' => 'tsInitialProgram', 'type' => 'text', 'value' => $mDial->ctx['CtxInitialProgram']),
			array('kind' => 'help', 'value' => 'tsInitialProgram'));
		// working dir
		$return[] = array(
			array('kind' => 'text', 'text' => _('Working directory')),
			array('kind' => 'input', 'name' => 'tsWorkDirectory', 'type' => 'text', 'value' => $mDial->ctx['CtxWorkDirectory']),
			array('kind' => 'help', 'value' => 'tsWorkDirectory'));
		// connection time limit
		$return[] = array(
			array('kind' => 'text', 'text' => _('Connection time limit')),
			array('kind' => 'input', 'name' => 'tsConnectionLimit', 'type' => 'text', 'value' => $mDial->ctx['CtxMaxConnectionTime']),
			array('kind' => 'help', 'value' => 'tsTimeLimit'));
		// disconnection time limit
		$return[] = array(
			array('kind' => 'text', 'text' => _('Disconnection time limit')),
			array('kind' => 'input', 'name' => 'tsDisconnectionLimit', 'type' => 'text', 'value' => $mDial->ctx['CtxMaxDisconnectionTime']),
			array('kind' => 'help', 'value' => 'tsTimeLimit'));
		// idle time limit
		$return[] = array(
			array('kind' => 'text', 'text' => _('Idle time limit')),
			array('kind' => 'input', 'name' => 'tsIdleLimit', 'type' => 'text', 'value' => $mDial->ctx['CtxMaxIdleTime']),
			array('kind' => 'help', 'value' => 'tsTimeLimit'));
		// connect client drives
		$return[] = array(
			array('kind' => 'text', 'text' => _('Connect client drives')),
			array('kind' => 'input', 'name' => 'tsConnectDrives', 'type' => 'checkbox', 'checked' => $mDial->getConnectClientDrives()),
			array('kind' => 'help', 'value' => 'tsConnectDrives'));
		// connect client printers
		$return[] = array(
			array('kind' => 'text', 'text' => _('Connect client printers')),
			array('kind' => 'input', 'name' => 'tsConnectPrinters', 'type' => 'checkbox', 'checked' => $mDial->getConnectClientPrinters()),
			array('kind' => 'help', 'value' => 'tsConnectPrinters'));
		// client printer is default
		$return[] = array(
			array('kind' => 'text', 'text' => _('Client printer is default')),
			array('kind' => 'input', 'name' => 'tsClientPrinterDefault', 'type' => 'checkbox', 'checked' => $mDial->getDefaultPrinter()),
			array('kind' => 'help', 'value' => 'tsClientPrinterDefault'));
		// shadowing
		$shadowOptions = array(
			array("0", _("disabled")),
			array("1", _("input on, notify on")),
			array("2", _("input on, notify off")),
			array("3", _("input off, notify on")),
			array("4", _("input off, notify off")));
		$selShadow = array($mDial->getShadow());
		$return[] = array(
			array('kind' => 'text', 'text' => _('Shadowing')),
			array('kind' => 'select', 'size' => 1, 'name' => 'tsShadowing', 'descriptiveOptions' => true, 'options' => $shadowOptions, 'options_selected' => $selShadow),
			array('kind' => 'help', 'value' => 'tsShadowing'));
		// broken connection
		$brokenConnOptions = array(
			array("0", _("disconnect")),
			array("1", _("reset")));
		$selbrokenConn = array($mDial->getBrokenConn());
		$return[] = array(
			array('kind' => 'text', 'text' => _('On broken or timed out connection')),
			array('kind' => 'select', 'size' => 1, 'name' => 'tsBrokenConn', 'descriptiveOptions' => true, 'options' => $brokenConnOptions, 'options_selected' => $selbrokenConn),
			array('kind' => 'help', 'value' => 'tsBrokenConn'));
		// reconnect
		$reconnectOptions = array(
			array("0", _("from any client")),
			array("1", _("from previous client only")));
		$selReconnect = array($mDial->getReConn());
		$return[] = array(
			array('kind' => 'text', 'text' => _('Reconnect if disconnected')),
			array('kind' => 'select', 'size' => 1, 'name' => 'tsReconnect', 'descriptiveOptions' => true, 'options' => $reconnectOptions, 'options_selected' => $selReconnect),
			array('kind' => 'help', 'value' => 'tsReconnect'));
		// buttons
		$return[] = array(array('kind' => 'text', 'text' => '&nbsp;'));
		$return[] = array(
			array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_submit', 'type' => 'submit', 'value' => _('Ok'), 'td' => array('align' => 'right')),
			array('kind' => 'input', 'name' => 'form_subpage_sambaSamAccount_attributes_abort', 'type' => 'submit', 'value' => _('Cancel'), 'td' => array('align' => 'left'))
		);
		return $return;	
	}
	
	/**
	* Returns a list of elements for the account profiles.
	*
	* @return profile elements
	*/
	function get_profileOptions() {
		$return = array();
		if ($this->get_scope() == 'user') {
			// lists for expiration date
			$day = array(); $mon = array(); $year = array();
			for ( $i=1; $i<=31; $i++ ) $day[] = $i;
			for ( $i=1; $i<=12; $i++ ) $mon[] = $i;
			for ( $i=2003; $i<=2030; $i++ ) $year[] = $i;
			// use no password at all
			$return[] = array(
				array('kind' => 'text', 'text' => _('Use no password') . ': '),
				array('kind' => 'input', 'name' => 'sambaSamAccount_sambaAcctFlagsN', 'type' => 'checkbox', 'checked' => false),
				array('kind' => 'help', 'value' => 'noPassword')
				);
			// account deactivation
			$return[] = array(
				array('kind' => 'text', 'text' => _('Account is deactivated') . ': '),
				array('kind' => 'input', 'name' => 'sambaSamAccount_sambaAcctFlagsD', 'type' => 'checkbox', 'checked' => false),
				array('kind' => 'help', 'value' => 'deactivated')
				);
			// password never expires
			$return[] = array(
				array('kind' => 'text', 'text' => _('Password does not expire') . ': '),
				array('kind' => 'input', 'name' => 'sambaSamAccount_sambaAcctFlagsX', 'type' => 'checkbox', 'checked' => true),
				array('kind' => 'help', 'value' => 'noExpire')
				);
			// expiration date
			$return[] = array(
				array('kind' => 'text', 'text' => _('Account expiration date')),
				array('kind' => 'table', 'value' => array(
					array (
						array('kind' => 'select', 'name' => 'sambaSamAccount_expire_day', 'options' => $day, 'options_selected' => array('1')),
						array('kind' => 'select', 'name' => 'sambaSamAccount_expire_mon', 'options' => $mon, 'options_selected' => array('1')),
						array('kind' => 'select', 'name' => 'sambaSamAccount_expire_yea', 'options' => $year, 'options_selected' => array('2030'))
					)
				)),
				array('kind' => 'help', 'value' => 'expireDate'));
			// user can change password
			$return[] = array(
				array('kind' => 'text', 'text' => _('User can change password')),
				array('kind' => 'input', 'name' => 'sambaSamAccount_pwdCanChange', 'type' => 'text', 'size' => '20', 'maxlength' => '10', 'value' => ""),
				array('kind' => 'help', 'value' => 'profilePwdCanMustChange'));
			// user must change password
			$return[] = array(
				array('kind' => 'text', 'text' => _('User must change password')),
				array('kind' => 'input', 'name' => 'sambaSamAccount_pwdMustChange', 'type' => 'text', 'size' => '20', 'maxlength' => '10', 'value' => ""),
				array('kind' => 'help', 'value' => 'profilePwdCanMustChange'));
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
				// letter of home drive
				$drives = array('-');
				for ($i = 90; $i > 67; $i--) $drives[] = chr($i) . ':';
				$return[] = array(
					array('kind' => 'text', 'text' => _('Home drive') . ': '),
					array('kind' => 'select', 'name' => 'sambaSamAccount_sambaHomeDrive', 'options' => $drives, 'options_selected' => array('-')),
					array('kind' => 'help', 'value' => 'homeDrive')
					);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomePath')) {
				// path to home directory
				$return[] = array(
					array('kind' => 'text', 'text' => _('Home path') . ': '),
					array('kind' => 'input', 'type' => 'text', 'name' => 'sambaSamAccount_smbhome', 'size' => '20', 'maxlength' => '255', 'value' => ''),
					array('kind' => 'help', 'value' => 'homePath')
					);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideProfilePath')) {
				// profile path
				$return[] = array(
					array('kind' => 'text', 'text' => _('Profile path') . ': '),
					array('kind' => 'input', 'type' => 'text', 'name' => 'sambaSamAccount_profilePath', 'size' => '20', 'maxlength' => '255', 'value' => ''),
					array('kind' => 'help', 'value' => 'profilePath')
					);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonScript')) {
				// logon script
				$return[] = array(
					array('kind' => 'text', 'text' => _('Logon script') . ': '),
					array('kind' => 'input', 'type' => 'text', 'name' => 'sambaSamAccount_logonScript', 'size' => '20', 'maxlength' => '255', 'value' => ''),
					array('kind' => 'help', 'value' => 'scriptPath')
					);
			}
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideWorkstations')) {
				// allowed workstations
				$return[] = array(
					array('kind' => 'text', 'text' => _('Samba workstations') . ': '),
					array('kind' => 'input', 'type' => 'text', 'name' => 'sambaSamAccount_userWorkstations', 'value' => ''),
					array('kind' => 'help', 'value' => 'workstations')
					);
			}
			// domains
			$sambaDomains = search_domains();
			$sambaDomainNames = array();
			for ($i = 0; $i < count($sambaDomains); $i++) {
				$sambaDomainNames[] = $sambaDomains[$i]->name;
			}
			$return[] = array(
				array('kind' => 'text', 'text' => _('Domain') . ': '),
				array('kind' => 'select', 'name' => 'sambaSamAccount_sambaDomainName', 'options' => $sambaDomainNames, 'options_selected' => array()),
				array('kind' => 'help', 'value' => 'domain')
				);
			// Windows group
			$groups = array();
			foreach ($this->rids as $key => $value) {
				$groups[] = array($value, $key);
			}
			$groups[] = array("-", "-");
			$return[] = array(
				array('kind' => 'text', 'text' => _('Windows group') . ': '),
				array('kind' => 'select', 'name' => 'sambaSamAccount_group', 'options' => $groups, 'options_selected' => array('513'), 'descriptiveOptions' => true),
				array('kind' => 'help', 'value' => 'group')
				);
			if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideLogonHours')) {
				// logon hours
				$return[] = array(
					array('kind' => 'text', 'text' => _('Logon hours') . ': '),
					array('kind' => 'input', 'type' => 'text', 'name' => 'sambaSamAccount_logonHours', 'value' => ''),
					array('kind' => 'help', 'value' => 'logonHoursUpload')
					);
			}
		}
		elseif ($this->get_scope() == 'host') {
			// domains
			$sambaDomains = search_domains();
			$sambaDomainNames = array();
			for ($i = 0; $i < count($sambaDomains); $i++) {
				$sambaDomainNames[] = $sambaDomains[$i]->name;
			}
			$return[] = array(
				array('kind' => 'text', 'text' => _('Domain') . ': '),
				array('kind' => 'select', 'name' => 'sambaSamAccount_sambaDomainName', 'options' => $sambaDomainNames, 'options_selected' => array()),
				array('kind' => 'help', 'value' => 'domain')
				);
		}
		return $return;
	}

	/**
	* Loads the values of an account profile into internal variables.
	*
	* @param array $profile hash array with profile values (identifier => value)
	*/
	function load_profile($profile) {
		// profile mappings in meta data
		parent::load_profile($profile);
		// special profile options
		// use no password
		if ($profile['sambaSamAccount_sambaAcctFlagsN'][0] == "true") {
			$this->nopwd = true;
		}
		elseif ($profile['sambaSamAccount_sambaAcctFlagsN'][0] == "false") {
			$this->nopwd = false;
		}
		// password expiration
		if ($profile['sambaSamAccount_sambaAcctFlagsX'][0] == "true") {
			$this->noexpire = true;
		}
		elseif ($profile['sambaSamAccount_sambaAcctFlagsX'][0] == "false") {
			$this->noexpire = false;
		}
		// use no password
		if ($profile['sambaSamAccount_sambaAcctFlagsD'][0] == "true") {
			$this->deactivated = true;
		}
		elseif ($profile['sambaSamAccount_sambaAcctFlagsD'][0] == "false") {
			$this->deactivated = false;
		}
		if (!$this->isBooleanConfigOptionSet('sambaSamAccount_hideHomeDrive')) {
			// home drive
			if ($profile['sambaSamAccount_sambaHomeDrive'][0] == "-") {
				$this->attributes['sambaHomeDrive'][0] = '';
			}
			elseif ($profile['sambaSamAccount_sambaHomeDrive'][0]) {
				$this->attributes['sambaHomeDrive'][0] = $profile['sambaSamAccount_sambaHomeDrive'][0];
			}
		}
		// expiration date
		if (isset($profile['sambaSamAccount_expire_day'][0]) && ($profile['sambaSamAccount_expire_day'][0] != "")) {
			$date = mktime(0, 0, 0, intval($profile['sambaSamAccount_expire_mon'][0]),
				intval($profile['sambaSamAccount_expire_day'][0]), intval($profile['sambaSamAccount_expire_yea'][0]));
			$this->attributes['sambaKickoffTime'][0] = $date;
		}
		// domain -> change SID
		if (isset($this->attributes['sambaSID'][0])) {
			if (isset($profile['sambaSamAccount_sambaDomainName'][0]) && ($profile['sambaSamAccount_sambaDomainName'][0] != "")) {
				$domains = search_domains();
				$domSID = '';
				// find domain SID
				for ($i = 0; $i < sizeof($domains); $i++) {
					if ($domains[$i]->name == $profile['sambaSamAccount_sambaDomainName'][0]) {
						$domSID = $domains[$i]->SID;
						break;
					}
				}
				// replace domain part of SID
				if ($domSID != '') {
					$SID = $this->attributes['sambaSID'][0];
					$rid = substr($SID, strrpos($SID, '-') + 1);
					$SID = $domSID . '-' . $rid;
					$this->attributes['sambaSID'][0] = $SID;
				}
			}
		}
		// primary group
		if (isset($profile['sambaSamAccount_sambaDomainName'][0])) {
			$domains = search_domains();
			$domSID = '';
			// find domain SID
			for ($i = 0; $i < sizeof($domains); $i++) {
				if ($domains[$i]->name == $profile['sambaSamAccount_sambaDomainName'][0]) {
					$domSID = $domains[$i]->SID;
					break;
				}
			}
			if ($domSID != '') {
				// set primary group if selected
				if (isset($profile['sambaSamAccount_group'][0]) && ($profile['sambaSamAccount_group'][0] != "-")) {
					$this->attributes['sambaPrimaryGroupSID'][0] = $domSID . "-" . $profile['sambaSamAccount_group'][0];
				}
			}
		}
		// user can change password
		if (isset($profile['sambaSamAccount_pwdCanChange'][0]) && ($profile['sambaSamAccount_pwdCanChange'][0] != '')) {
			$this->attributes['sambaPwdCanChange'][0] = time() + $profile['sambaSamAccount_pwdCanChange'][0];
		}
		// user must change password
		if (isset($profile['sambaSamAccount_pwdMustChange'][0]) && ($profile['sambaSamAccount_pwdMustChange'][0] != '')) {
			$this->attributes['sambaPwdMustChange'][0] = time() + $profile['sambaSamAccount_pwdMustChange'][0];
		}
	}

	/**
	* Returns the PDF entries for this module.
	*
	* @return array list of possible PDF entries
	*/
	function get_pdfEntries() {
		$return = array();
		if (isset($this->attributes['displayName'][0])) {
			$return['sambaSamAccount_displayName'] = array('<block><key>' . _('Display name') . '</key><value>' . $this->attributes['displayName'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaHomeDrive'][0])) {
			$return['sambaSamAccount_sambaHomePath'] = array('<block><key>' . _('Home path') . '</key><value>' . $this->attributes['sambaHomeDrive'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaHomePath'][0])) {
			$return['sambaSamAccount_sambaHomeDrive'] = array('<block><key>' . _('Home drive') . '</key><value>' . $this->attributes['sambaHomePath'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaLogonScript'][0])) {
			$return['sambaSamAccount_sambaLogonScript'] = array('<block><key>' . _('Logon script') . '</key><value>' . $this->attributes['sambaLogonScript'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaProfilePath'][0])) {
			$return['sambaSamAccount_sambaProfilePath'] = array('<block><key>' . _('Profile path') . '</key><value>' . $this->attributes['sambaProfilePath'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaUserWorkstations'][0])) {
			$return['sambaSamAccount_sambaUserWorkstations'] = array('<block><key>' . _('Samba workstations') . '</key><value>' . $this->attributes['sambaUserWorkstations'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaDomainName'][0])) {
			$return['sambaSamAccount_sambaDomainName'] = array('<block><key>' . _('Domain') . '</key><value>' . $this->attributes['sambaDomainName'][0] . '</value></block>');
		}
		if (isset($this->attributes['sambaPrimaryGroupSID'][0])) {
			$return['sambaSamAccount_sambaPrimaryGroupSID'] = array('<block><key>' . _('Windows group') . '</key><value>' . $this->attributes['sambaPrimaryGroupSID'][0] . '</value></block>');
		}
		// terminal server options
		if (isset($this->attributes['sambaMungedDial'][0])) {
			$mDial = new sambaMungedDial();
			$mDial->load($this->attributes['sambaMungedDial'][0]);
			$tsAllowLogin = _('yes');
			if (!$mDial->getTsLogin()) {
				$tsAllowLogin = _('no');
			}
			$return['sambaSamAccount_tsAllowLogin'] = array('<block><key>' . _('Allow terminal server login') . '</key><value>' . $tsAllowLogin . '</value></block>');
			$return['sambaSamAccount_tsHomeDir'] = array('<block><key>' . _('Home directory') . '</key><value>' . $mDial->ctx['CtxWFHomeDir'] . '</value></block>');
			$return['sambaSamAccount_tsHomeDrive'] = array('<block><key>' . _('Home drive') . '</key><value>' . $mDial->ctx['CtxWFHomeDirDrive'] . '</value></block>');
			$return['sambaSamAccount_tsProfilePath'] = array('<block><key>' . _('Profile path') . '</key><value>' . $mDial->ctx['CtxWFProfilePath'] . '</value></block>');
			$tsInherit = _('yes');
			if (!$mDial->getInheritMode()) {
				$tsInherit = _('no');
			}
			$return['sambaSamAccount_tsInherit'] = array('<block><key>' . _('Inherit client startup configuration') . '</key><value>' . $tsInherit . '</value></block>');
			$return['sambaSamAccount_tsInitialProgram'] = array('<block><key>' . _('Initial program') . '</key><value>' . $mDial->ctx['CtxInitialProgram'] . '</value></block>');
			$return['sambaSamAccount_tsWorkDirectory'] = array('<block><key>' . _('Working directory') . '</key><value>' . $mDial->ctx['CtxWorkDirectory'] . '</value></block>');
			$return['sambaSamAccount_tsConnectionLimit'] = array('<block><key>' . _('Connection time limit') . '</key><value>' . $mDial->ctx['CtxMaxConnectionTime'] . '</value></block>');
			$return['sambaSamAccount_tsDisconnectionLimit'] = array('<block><key>' . _('Disconnection time limit') . '</key><value>' . $mDial->ctx['CtxMaxDisconnectionTime'] . '</value></block>');
			$return['sambaSamAccount_tsIdleLimit'] = array('<block><key>' . _('Idle time limit') . '</key><value>' . $mDial->ctx['CtxMaxIdleTime'] . '</value></block>');
			$tsConnectDrives = _('yes');
			if (!$mDial->getConnectClientDrives()) {
				$tsConnectDrives = _('no');
			}
			$return['sambaSamAccount_tsConnectDrives'] = array('<block><key>' . _('Connect client drives') . '</key><value>' . $tsConnectDrives . '</value></block>');
			$tsConnectPrinters = _('yes');
			if (!$mDial->getConnectClientPrinters()) {
				$tsConnectPrinters = _('no');
			}
			$return['sambaSamAccount_tsConnectPrinters'] = array('<block><key>' . _('Connect client printers') . '</key><value>' . $tsConnectPrinters . '</value></block>');
			$tsClientPrinterDefault = _('yes');
			if (!$mDial->getDefaultPrinter()) {
				$tsClientPrinterDefault = _('no');
			}
			$return['sambaSamAccount_tsClientPrinterDefault'] = array('<block><key>' . _('Client printer is default') . '</key><value>' . $tsClientPrinterDefault . '</value></block>');
			$shadowOptions = array(
				'0' => _("disabled"),
				'1' => _("input on, notify on"),
				'2' => _("input on, notify off"),
				'3' => _("input off, notify on"),
				'4' => _("input off, notify off"));
			$tsShadowing = '';
			if (($mDial->getShadow() != null) && is_numeric($mDial->getShadow())) {
				$tsShadowing = $shadowOptions[$mDial->getShadow()];
			}
			$return['sambaSamAccount_tsShadowing'] = array('<block><key>' . _('Shadowing') . '</key><value>' . $tsShadowing . '</value></block>');
			$brokenConnOptions = array(
				'0' => _("disconnect"),
				'1' => _("reset"));
			$tsBrokenConn = '';
			if (($mDial->getBrokenConn() != null) && is_numeric($mDial->getBrokenConn())) {
				$tsBrokenConn = $brokenConnOptions[$mDial->getBrokenConn()];
			}
			$return['sambaSamAccount_tsBrokenConn'] = array('<block><key>' . _('On broken or timed out connection') . '</key><value>' . $tsBrokenConn . '</value></block>');
			$reconnectOptions = array(
				'0' => _("from any client"),
				'1' => _("from previous client only"));
			$tsReconnect = '';
			if (($mDial->getReConn() != null) && is_numeric($mDial->getReConn())) {
				$tsReconnect = $reconnectOptions[$mDial->getReConn()];
			}
			$return['sambaSamAccount_tsReconnect'] = array('<block><key>' . _('Reconnect if disconnected') . '</key><value>' . $tsReconnect . '</value></block>');
		}
		return $return;
	}

	/**
	* In this function the LDAP account is built up.
	*
	* @param array $rawAccounts list of hash arrays (name => value) from user input
	* @param array $partialAccounts list of hash arrays (name => value) which are later added to LDAP
	* @param array $ids list of IDs for column position (e.g. "posixAccount_uid" => 5)
	* @return array list of error messages if any
	*/
	function build_uploadAccounts($rawAccounts, $ids, &$partialAccounts) {
		$errors = array();
		// get list of Samba 3 domains
		$domains = search_domains();
		// get list of Unix groups and their sambaSID + gidNumber
		$groups = $_SESSION['cache']->get_cache(array('cn', 'sambaSID', 'gidNumber'), 'posixGroup', 'group');
		$groups_k = array_keys($groups);
		$groups_cn = array();
		for ($i = 0; $i < sizeof($groups_k); $i++) {
			if (isset($groups[$groups_k[$i]]['sambaSID'][0])) {
				$groups_cn[$groups[$groups_k[$i]]['cn'][0]]['SID'] = $groups[$groups_k[$i]]['sambaSID'][0];
			}
			if (isset($groups[$groups_k[$i]]['gidNumber'][0])) {
				$groups_cn[$groups[$groups_k[$i]]['cn'][0]]['gid'] = $groups[$groups_k[$i]]['gidNumber'][0];
			}
		}
		if ($this->get_scope() == 'user') {
			for ($i = 0; $i < sizeof($rawAccounts); $i++) {
				if (!in_array("sambaSamAccount", $partialAccounts[$i]['objectClass'])) $partialAccounts[$i]['objectClass'][] = "sambaSamAccount";
				// displayName
				if ($rawAccounts[$i][$ids['sambaSamAccount_displayName']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_displayName']], 'realname')) {
						$partialAccounts[$i]['displayName'] = $rawAccounts[$i][$ids['sambaSamAccount_displayName']];
					}
					else {
						$errMsg = $this->messages['displayName'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// password
				$partialAccounts[$i]['sambaPwdLastSet'] = time();
				if (!get_preg($rawAccounts[$i][$ids['sambaSamAccount_password']], 'password')) {
					$errMsg = $this->messages['sambaLMPassword'][2];
					array_push($errMsg, array($i));
					$errors[] = $errMsg;
				}
				// use Unix password
				if ($rawAccounts[$i][$ids['sambaSamAccount_pwdUnix']] == "") {  // default: use Unix
					if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
						$partialAccounts[$i]['sambaLMPassword'] = lmPassword($rawAccounts[$i][$ids['posixAccount_password']]);
					}
					$partialAccounts[$i]['sambaNTPassword'] = ntPassword($rawAccounts[$i][$ids['posixAccount_password']]);
				}
				elseif (in_array($rawAccounts[$i][$ids['sambaSamAccount_pwdUnix']], array('true', 'false'))) {
					if ($rawAccounts[$i][$ids['sambaSamAccount_pwdUnix']] == 'true') {  // use Unix
						if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
							$partialAccounts[$i]['sambaLMPassword'] = lmPassword($rawAccounts[$i][$ids['posixAccount_password']]);
						}
						$partialAccounts[$i]['sambaNTPassword'] = ntPassword($rawAccounts[$i][$ids['posixAccount_password']]);
					}
					else {  // use given password
						if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
							$partialAccounts[$i]['sambaLMPassword'] = lmPassword($rawAccounts[$i][$ids['sambaSamAccount_password']]);
						}
						$partialAccounts[$i]['sambaNTPassword'] = ntPassword($rawAccounts[$i][$ids['sambaSamAccount_password']]);
					}
				}
				else {
					$errMsg = $this->messages['pwdUnix'][0];
					array_push($errMsg, array($i));
					$errors[] = $errMsg;
				}
				// use no password
				if ($rawAccounts[$i][$ids['sambaSamAccount_noPassword']] != "") {
					if (in_array($rawAccounts[$i][$ids['sambaSamAccount_noPassword']], array('true', 'false'))) {
						if ($rawAccounts[$i][$ids['sambaSamAccount_noPassword']] == 'true') {
							$partialAccounts[$i]['sambaLMPassword'] = 'NO PASSWORD*****';
							$partialAccounts[$i]['sambaNTPassword'] = 'NO PASSWORD*****';
						}
					}
					else {
						$errMsg = $this->messages['noPassword'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// account flags
				$flag_expire = false;
				$flag_deactivated = false;
				// password does not expire
				if ($rawAccounts[$i][$ids['sambaSamAccount_noExpire']] != "") {
					if (in_array($rawAccounts[$i][$ids['sambaSamAccount_noExpire']], array('true', 'false'))) {
						if ($rawAccounts[$i][$ids['sambaSamAccount_noExpire']] == 'false') {
							$flag_expire = true;
						}
					}
					else {
						$errMsg = $this->messages['noExpire'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// account is deactivated
				if ($rawAccounts[$i][$ids['sambaSamAccount_deactivated']] != "") {
					if (in_array($rawAccounts[$i][$ids['sambaSamAccount_deactivated']], array('true', 'false'))) {
						if ($rawAccounts[$i][$ids['sambaSamAccount_deactivated']] == 'true') {
							$flag_deactivated = true;
						}
					}
					else {
						$errMsg = $this->messages['deactivated'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// set flags
				$flags = "[";
				if ($flag_deactivated) $flags = $flags . "D";
				if (!$flag_expire) $flags = $flags . "X";
				$flags = $flags . "U";
				// Expand string to fixed length
				$flags = str_pad($flags, 12);
				// End character
				$flags = $flags . "]";
				$partialAccounts[$i]['sambaAcctFlags'] = $flags;
				// passsword can be changed
				if ($rawAccounts[$i][$ids['sambaSamAccount_pwdCanChange']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_pwdCanChange']], 'date')) {
						$parts = explode("-", $rawAccounts[$i][$ids['sambaSamAccount_pwdCanChange']]);
						$time = mktime(0, 0, 0, intval($parts[1]), intval($parts[0]), intval($parts[2]));
						$partialAccounts[$i]['sambaPwdCanChange'] = $time;
					}
					else {
						$errMsg = $this->messages['pwdCanChange'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// passsword must be changed
				if ($rawAccounts[$i][$ids['sambaSamAccount_pwdMustChange']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_pwdMustChange']], 'date')) {
						$parts = explode("-", $rawAccounts[$i][$ids['sambaSamAccount_pwdMustChange']]);
						$time = mktime(0, 0, 0, intval($parts[1]), intval($parts[0]), intval($parts[2]));
						$partialAccounts[$i]['sambaPwdMustChange'] = $time;
					}
					else {
						$errMsg = $this->messages['pwdMustChange'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// expiration date
				if ($rawAccounts[$i][$ids['sambaSamAccount_expireDate']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_expireDate']], 'date')) {
						$parts = explode("-", $rawAccounts[$i][$ids['sambaSamAccount_expireDate']]);
						$time = mktime(0, 0, 0, intval($parts[1]), intval($parts[0]), intval($parts[2]));
						$partialAccounts[$i]['sambaKickoffTime'] = $time;
					}
					else {
						$errMsg = $this->messages['expireDate'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// home drive
				if ($rawAccounts[$i][$ids['sambaSamAccount_homeDrive']] != "") {
					if (preg_match("/[d-z]:/i", $rawAccounts[$i][$ids['sambaSamAccount_homeDrive']])) {
						$partialAccounts[$i]['sambaHomeDrive'] = $rawAccounts[$i][$ids['sambaSamAccount_homeDrive']];
					}
					else {
						$errMsg = $this->messages['homeDrive'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// home path
				if ($rawAccounts[$i][$ids['sambaSamAccount_homePath']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_homePath']], 'UNC')) {
						$partialAccounts[$i]['sambaHomePath'] = $rawAccounts[$i][$ids['sambaSamAccount_homePath']];
					}
					else {
						$errMsg = $this->messages['homePath'][2];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// profile path
				if ($rawAccounts[$i][$ids['sambaSamAccount_profilePath']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_profilePath']], 'UNC')) {
						$partialAccounts[$i]['sambaProfilePath'] = $rawAccounts[$i][$ids['sambaSamAccount_profilePath']];
					}
					else {
						$errMsg = $this->messages['profilePath'][2];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// logon script
				if ($rawAccounts[$i][$ids['sambaSamAccount_logonScript']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_logonScript']], 'logonscript')) {
						$partialAccounts[$i]['sambaLogonScript'] = $rawAccounts[$i][$ids['sambaSamAccount_logonScript']];
					}
					else {
						$errMsg = $this->messages['logonScript'][2];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// workstations
				if ($rawAccounts[$i][$ids['sambaSamAccount_workstations']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_workstations']], 'workstations')) {
						$partialAccounts[$i]['sambaUserWorkstations'] = $rawAccounts[$i][$ids['sambaSamAccount_workstations']];
					}
					else {
						$errMsg = $this->messages['workstations'][1];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				// domain
				$domIndex = -1;
				for ($d = 0; $d < sizeof($domains); $d++) {
					if ($domains[$d]->name == $rawAccounts[$i][$ids['sambaSamAccount_domain']]) {
						$domIndex = $d;
						break;
					}
				}
				if ($domIndex > -1) {
					$partialAccounts[$i]['sambaDomainName'] = $domains[$domIndex]->name;
					$partialAccounts[$i]['sambaSID'] = $domains[$domIndex]->SID;
				}
				else {
					$errMsg = $this->messages['domain'][0];
					array_push($errMsg, array($i));
					$errors[] = $errMsg;
				}
				// group
				if ($rawAccounts[$i][$ids['sambaSamAccount_group']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_group']], 'groupname')
					&& (isset($groups_cn[$rawAccounts[$i][$ids['sambaSamAccount_group']]]))) {
						if (isset($groups_cn[$rawAccounts[$i][$ids['sambaSamAccount_group']]]['SID'])) {
							$partialAccounts[$i]['sambaPrimaryGroupSID'] = $groups_cn[$rawAccounts[$i][$ids['sambaSamAccount_group']]]['SID'];
						}
						else {
							$partialAccounts[$i]['sambaPrimaryGroupSID'] = $domains[$domIndex]->SID . '-' .
							($groups_cn[$rawAccounts[$i][$ids['sambaSamAccount_group']]]['gid'] * 2 +
							 $domains[$domIndex]->RIDbase + 1);
						}
					}
					elseif (in_array($rawAccounts[$i][$ids['sambaSamAccount_group']], array_keys($this->rids))) {
						$partialAccounts[$i]['sambaPrimaryGroupSID'] = $domains[$domIndex]->SID . '-' . $this->rids[$rawAccounts[$i][$ids['sambaSamAccount_group']]];
					}
					else {
						$errMsg = $this->messages['group'][0];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				else {
					// default domain users
					$partialAccounts[$i]['sambaPrimaryGroupSID'] = $domains[$domIndex]->SID . '-' . $this->rids[_('Domain users')];
				}
				// special user
				if ($rawAccounts[$i][$ids['sambaSamAccount_rid']] != "") {
					if (in_array($rawAccounts[$i][$ids['sambaSamAccount_rid']], array_keys($this->rids))) {
						$partialAccounts[$i]['sambaSID'] .= '-' . $this->rids[$rawAccounts[$i][$ids['sambaSamAccount_rid']]];
					}
					elseif (get_preg($rawAccounts[$i][$ids['sambaSamAccount_rid']], 'digit')) {
						$partialAccounts[$i]['sambaSID'] .= '-' . $rawAccounts[$i][$ids['sambaSamAccount_rid']];
					}
					else {
						$errMsg = $this->messages['rid'][2];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				else {
					// default RID uid*2 + RIDBase
					$partialAccounts[$i]['sambaSID'] .= '-' . ($partialAccounts[$i]['uidNumber']*2 + $domains[$domIndex]->RIDbase);
				}
				// logon hours
				if ($rawAccounts[$i][$ids['sambaSamAccount_logonHours']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_logonHours']], 'sambaLogonHours')) {
						$partialAccounts[$i]['sambaLogonHours'] = $rawAccounts[$i][$ids['sambaSamAccount_logonHours']];
					}
					else {
						$errMsg = $this->messages['logonHours'][1];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				else {
					$partialAccounts[$i]['sambaLogonHours'] = "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF";
				}
			}
		}
		else {  // hosts
			for ($i = 0; $i < sizeof($rawAccounts); $i++) {
				if (!in_array("sambaSamAccount", $partialAccounts[$i]['objectClass'])) $partialAccounts[$i]['objectClass'][] = "sambaSamAccount";
				// domain
				$domIndex = -1;
				for ($d = 0; $d < sizeof($domains); $d++) {
					if ($domains[$d]->name == $rawAccounts[$i][$ids['sambaSamAccount_domain']]) {
						$domIndex = $d;
						break;
					}
				}
				if ($domIndex > -1) {
					$partialAccounts[$i]['sambaDomainName'] = $domains[$domIndex]->name;
					$partialAccounts[$i]['sambaSID'] = $domains[$domIndex]->SID;
					$partialAccounts[$i]['sambaPrimaryGroupSID'] = $domains[$domIndex]->SID . " - 515";
				}
				else {
					$errMsg = $this->messages['domain'][0];
					array_push($errMsg, array($i));
					$errors[] = $errMsg;
				}
				// RID
				if ($rawAccounts[$i][$ids['sambaSamAccount_rid']] != "") {
					if (get_preg($rawAccounts[$i][$ids['sambaSamAccount_rid']], 'digit')) {
						$partialAccounts[$i]['sambaSID'] .= '-' . $rawAccounts[$i][$ids['sambaSamAccount_rid']];
					}
					else {
						$errMsg = $this->messages['rid'][3];
						array_push($errMsg, array($i));
						$errors[] = $errMsg;
					}
				}
				else {
					// default RID uid*2 + RIDBase
					$partialAccounts[$i]['sambaSID'] .= '-' . ($partialAccounts[$i]['uidNumber']*2 + $domains[$domIndex]->RIDbase);
				}
				// passwords ( = host name)
				$partialAccounts[$i]['sambaPwdLastSet'] = time();
				if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
					$partialAccounts[$i]['sambaLMPassword'] = lmPassword(substr($partialAccounts[$i]['uid'], 0, sizeof($partialAccounts[$i]['uid']) - 1));
				}
				$partialAccounts[$i]['sambaNTPassword'] = ntPassword(substr($partialAccounts[$i]['uid'], 0, sizeof($partialAccounts[$i]['uid']) - 1));
				// flags
				$partialAccounts[$i]['sambaAcctFlags'] = "[W          ]";
			}
		}
		return $errors;
	}

	/**
	 * Checks if all input values are correct and returns the LDAP commands which should be executed.
	 *
	 * @param string $fields input fields
	 * @param array $attributes LDAP attributes
	 * @return array messages and LDAP commands (array('messages' => array(), 'add' => array(), 'del' => array(), 'mod' => array()))
	 */
	function checkSelfServiceOptions($fields, $attributes) {
		$return = array('messages' => array(), 'add' => array(), 'del' => array(), 'mod' => array());
		if (!in_array_ignore_case('sambaSamAccount', $attributes['objectClass'])) {
			return $return;
		}
		if (isset($_POST['posixAccount_password']) && ($_POST['posixAccount_password'] != '')) {
			if ($_POST['posixAccount_password'] != $_POST['posixAccount_password2']) {
				return $return;
			}
			else {
				if (!get_preg($_POST['posixAccount_password'], 'password')) {
					return $return;
				}
				else {
					$setPassword = false;
					// sync password
					if (in_array('syncNTPassword', $fields)) {
						$return['mod']['sambaNTPassword'][0] = ntPassword($_POST['posixAccount_password']);
						$setPassword = true;
					}
					if (in_array('syncLMPassword', $fields)) {
						$return['mod']['sambaLMPassword'][0] = lmPassword($_POST['posixAccount_password']);
						$setPassword = true;
					}
					if ($setPassword) {
						if (in_array('syncSambaPwdLastSet', $fields)) {
							$return['mod']['sambaPwdLastSet'][0] = time();
						}
					}
					if (in_array('syncSambaPwdMustChange', $fields) || in_array('syncSambaPwdCanChange', $fields)) {
						$sambaDomains = search_domains($_SESSION['ldapHandle'], $this->selfServiceSettings->LDAPSuffix);
						if (($sambaDomains == null) || (sizeof($sambaDomains) == 0)) {
							$return['messages'][] = array("ERROR", _('Unable to sync the time when the user can/must change his password because no domain was found.'), '');
							return $return;
						}
						if (!isset($attributes['sambaSID'][0]) || $attributes['sambaSID'][0] == '') {
							$return['messages'][] = array("ERROR", _('Unable to read sambaSID attribute.'), '');
							return $return;
						}
						$domainSID = substr($attributes['sambaSID'][0], 0, strrpos($attributes['sambaSID'][0], "-"));
						$sel_domain = null;
						for ($i = 0; $i < count($sambaDomains); $i++ ) {
							if ($domainSID == $sambaDomains[$i]->SID) {
								$sel_domain = $sambaDomains[$i];
							}
						}
						if ($sel_domain == null) {
							$return['messages'][] = array("ERROR", _('Unable to sync the time when the user can/must change his password because no domain was found.'), $domainSID);
							return $return;
						}
						if (in_array('syncSambaPwdCanChange', $fields)) {
							if (($sel_domain != null) && (isset($sel_domain->maxPwdAge))) {
								$return['mod']['sambaPwdCanChange'][0] = time() + $sel_domain->minPwdAge;
							}
						}
						if (in_array('syncSambaPwdMustChange', $fields)) {
							if (($sel_domain != null) && (isset($sel_domain->maxPwdAge))) {
								$return['mod']['sambaPwdMustChange'][0] = time() + $sel_domain->maxPwdAge;
							}
						}
					}
				}
			}
		}
		return $return;
	}

	/**
	 * This method specifies if a module manages password attributes.
	 * @see passwordService::managesPasswordAttributes
	 *
	 * @return boolean true if this module manages password attributes
	 */
	public function managesPasswordAttributes() {
		if ($this->get_scope() == "user") {
			return true;
		}
	}

	/**
	 * This function is called whenever the password should be changed. Account modules
	 * must change their password attributes only if the modules list contains their module name.
	 *
	 * @param String $password new password
	 * @param $modules list of modules for which the password should be changed
	 * @return array list of error messages if any as parameter array for StatusMessage
	 *               e.g. return arrray(array('ERROR', 'Password change failed.'))
	 * @see passwordService::passwordChangeRequested
	 */
	public function passwordChangeRequested($password, $modules) {
		if (!in_array(get_class($this), $modules)) {
			return array();
		}
		if (isset($this->moduleSettings['sambaSamAccount_lmHash'][0]) && ($this->moduleSettings['sambaSamAccount_lmHash'][0] == 'no')) {
			$this->attributes['sambaLMPassword'][0] = lmPassword($password);
		}
		$this->attributes['sambaNTPassword'][0] = ntPassword($password);
		$this->attributes['sambaPwdLastSet'][0] = time();
		return array();
	}
		
	/**
	 * Returns the group name of the group with the given group ID.
	 *
	 * @param String $groupID group ID
	 * @return String group name
	 */
	private function getGroupName($groupID) {
		$results = searchLDAPByAttribute('gidNumber', $groupID, 'posixGroup', array('cn'), array('group'));
		if ((sizeof($results) > 0) && isset($results[0]['cn'][0])) {
			return $results[0]['cn'][0];
		}
		return null;
	}
	
}

?>
