/*
 * $Id: kl_copt.c,v 1.1 2004/12/21 23:26:23 tjm Exp $
 *
 * This file is part of libutil.
 * A library which provides auxiliary functions.
 * libutil is part of lkcdutils -- utilities for Linux kernel crash dumps.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2003, 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <kl_lib.h>

int copt_ind = 1;
int copt_error = 0;
char *copt_arg = 0;
static char *copt = 0;

/*
 * reset_copt()
 */
void
reset_copt(void)
{
	copt_ind = 1;
	copt_error = 0;
	copt_arg = 0;
	copt = 0;
}

/*
 * regular_copt()
 */
static int
regular_copt(char *str)
{
	if (str && ((str[0] == '-') && (str[1] != '-'))) {
		return(1);
	}
	return(0);
}

/*
 * extended_copt()
 */
static int
extended_copt(char *str)
{
	if (!strncmp(str, "--", 2)) {
		return(1);
	}
	return(0);
}

/*
 * valid_copt()
 */
static int
valid_copt(int c, char *fmtstr)
{
	char *cp;

	cp = strchr(fmtstr, c);
	if (cp) {
		return(1);
	}
	return(-1);
}

/*
 * copt_needs_arg()
 */
static int
copt_needs_arg(int c, char *fmtstr)
{
	char *cp;

        if((cp = strchr(fmtstr, c))) {
		if (*(cp + 1) && (*(cp + 1) == ':')) {
                	return(1);
		}
        }
        return(0);
}

/*
 * is_copt()
 */
int
is_copt(char *str)
{
	if (*str == '-') {
		return(1);
	}
	return(0);
}

/*
 * process_copt()
 */
static int
process_copt(int argc, char *argv[], const char *fmtstr)
{
	int c;

	if (!copt) {
		return(0);
	}
	c = *copt;
	copt++;

	/* Check to see if this is a valid option letter
	 */
	if (!valid_copt(c, (char *)fmtstr)) {
		if (fmtstr[0] == ':') {
			return(c);
		} else {
			return((int)'?');
		}
	}
	if (copt_needs_arg(c, (char *)fmtstr)) {
		if (!(*copt)) {
			copt = (char *)NULL;
			copt_ind++;
			if (copt_ind == argc) {
				if (fmtstr[0] == ':') {
					return((int)':');
				} else {
					return((int)'?');
				}
			}
			if (argv[copt_ind][0] == '-') {
				return(c);
			}
			copt_arg = argv[copt_ind];
		} else {
			copt_arg = copt;
			copt = (char *)NULL;
		}
		copt_ind++;
	}
	return(c);
}

/*
 * get_copt()
 */
int
get_copt(int argc, char *argv[], const char *fmtstr, char **copt_str)
{
	int c;

	/* start out with a NULL copt_str pointer. It will only end
	 * up non-null if the command line argument starts with a
	 * double hyphen ("--"). When a double hyphen copt_str is
	 * encountered, a 0 value is returned. Note that a double 
	 * hyphen argument that does not have an copt_str should be 
	 * considered the end of the argement section of the command 
	 * line (a -1 value is returned).
	 */
	*copt_str = 0;
again:
	if (copt_ind == argc) {
		return(-1);
	}

	if (extended_copt(argv[copt_ind])) {
		/* This is a double hyphen option with an associated
		 * copt_str (or, it's the end of the options section).
		 */
		if (argv[copt_ind][2] == 0) {
			/* No option string. This means that we are at
			 * the end of the command line options. Just
			 * return -1, after stepping over the current
			 * arg;
			 */
			copt_ind++;
			return(-1);
		}

		/* Get the pointer to the copt_str and then return 0 
		 * to indicate that we are NOT at the end of the 
		 * arguments section, but there is no copt character.
		 */
		*copt_str = &argv[copt_ind][2];

		/* Bump our copt index
		 */
		copt_ind++;
		return(0);
	} else if (copt) {
		if (!(*copt)) {
			copt = (char *)NULL;
			copt_ind++;
			goto again;
		}
		c = process_copt(argc, argv, fmtstr);
	} else if (regular_copt(argv[copt_ind])) {
		copt = &argv[copt_ind][1];
		c = process_copt(argc, argv, fmtstr);
	} else {
		return(-1);
	}
	return(c);
}
