/*
 * $Id: kl_kern_ppc64.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

int kl_has_highmem = FALSE;

/*
 * Name: kl_vtop_ppc64()
 * Func: Performs simple virtual to physical address translation
 * without checking mmap memory.
 * 
 */
int
kl_vtop_ppc64(kaddr_t vaddr, kaddr_t *paddr)
{
	if (KL_KADDR_IS_PHYSICAL(vaddr)) {
		*paddr = (kaddr_t)(vaddr - KL_PAGE_OFFSET);
	} else {
		*paddr = (kaddr_t)vaddr;
	}
	return(0);
}

/* 
 * kl_kernelstack_ppc64()
 */
kaddr_t
kl_kernelstack_ppc64(kaddr_t task)
{
	kaddr_t saddr = 0, thread_info;
	void *tsp;

	if ((tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP))) {
		kl_get_task_struct(task, 2, tsp);
		if (!KL_ERROR) {
			if (LINUX_2_6_X(KL_LINUX_RELEASE)) {
				thread_info = kl_kaddr(tsp, "task_struct", 
					"thread_info");
				saddr = thread_info + KL_KSTACK_SIZE_PPC64;
			} else {
				saddr = (task + KL_KSTACK_SIZE_PPC64);
			}
		}
		kl_free_block(tsp);
	}
	return(saddr);
} 

kaddr_t
kl_fix_vaddr_ppc64(kaddr_t vaddr, size_t sz)
{
	kl_dump_header_ppc64_t dha;
	kaddr_t addr;
	int i;
	
	if (kl_get_dump_header_ppc64(&dha)){
		return vaddr;
	}

	/*
	 * We look thru the saved snapshots of the task structures and if
	 * the requested memory from vaddr thru vaddr + sz is within the
	 * snapshot then we return the requested address within the snapshot.
	 *
	 * If this is a request for the saved task struct then vaddr should
	 * be page aligned. Otherwise original vaddr is returned even
	 * if a part of the range vaddr to vaddr+sz falls within the range
	 * of saved task_struct+stack.
	 */
	for (i = 0; i < KL_GET_UINT32(&dha.smp_num_cpus); i++) {
		if (dha.smp_regs[i].nip < KL_PAGE_OFFSET){
			/* if task is in user space,
			 * no need to look at saved stack */
			continue; 
		}
		if (LINUX_2_6_X(KL_LINUX_RELEASE)) {
			addr = KL_GET_PTR(&dha.stack_ptr[i]);
		} else if (LINUX_2_4_X(KL_LINUX_RELEASE)) {
			addr = KL_GET_PTR(&dha.smp_current_task[i]);
			if(vaddr == addr) {
				return vaddr;
			}
		} else {
			/* Unknown linux release, don't bother with snapshots */
			return vaddr;
		}
		if (vaddr >= addr && vaddr + sz <=  addr + KL_KSTACK_SIZE_PPC64)
			return (kaddr_t)(dha.stack[i] + (vaddr - addr));
	}
	return vaddr;
}
