/*
 * $id: kl_bfd.c,v 1.1.2.3 2002/08/29 16:07:26 aherrman Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>
#include <bfd.h>

static int kl_check_bfd_arch(bfd*);

#define _DBG_COMPONENT KL_DBGCOMP_BFD
int bfd_debug = 0;

/*
 * static function declarations
 */
static void _kl_print_symbol_flags(asymbol**, int);

/*
 * kl_print_symbols()
 */
static int
kl_read_symbols(bfd *abfd, maplist_t *ml)
{
	int ret, type;
	syment_t *cur_syment = (syment_t *)NULL;
	syment_t *syment_list = (syment_t *)NULL;
	symtab_t *stp;
	symbol_info bfd_sym_info;
	kl_modinfo_t *modinfo = NULL;

	kaddr_t addr;
	long storage_needed;
	asymbol **symbol_table;
	long number_of_symbols;
	long i;
	int discarded_data_symbols = 0;
	int discarded_bss_symbols = 0;
#define SEC_TYPE_STRLEN 1024

	if(ml->modname){
		modinfo = kl_lkup_modinfo(ml->modname);
	}

	stp = (symtab_t *)calloc(1, sizeof(symtab_t));
	addr = 0;

	storage_needed = bfd_get_symtab_upper_bound (abfd);

	if (storage_needed < 0){
		kl_free_symtab(stp);
		return(1);
	}

	if (storage_needed == 0) {
		kl_free_symtab(stp);
		return (1);
	}

	symbol_table = (asymbol **) kl_alloc_block(storage_needed, K_TEMP);
	number_of_symbols = bfd_canonicalize_symtab (abfd, symbol_table);

	if (number_of_symbols < 0){
		kl_free_symtab(stp);
		kl_free_block(symbol_table);
		return(1);
	}

	if(bfd_debug){
		_kl_print_symbol_flags(symbol_table, number_of_symbols);
	}

	for (i = 0; i < number_of_symbols; i++) {
		bfd_get_symbol_info(abfd, symbol_table[i], &bfd_sym_info);

		/* discard unwanted symbols */
		if(symbol_table[i]->flags & (BSF_FILE | BSF_SECTION_SYM)){
			continue;
		}
		/* XXX check whether we know the start of
		   the section for this type */
		if(((bfd_sym_info.type == 'b') || (bfd_sym_info.type == 'B') ||
		    (bfd_sym_info.type == 's') || (bfd_sym_info.type == 'S')) &&
		   (modinfo && (modinfo->bss_len == 0))){
			discarded_bss_symbols = 1;
			kl_trace2(0, "discarded bss symbol \"%s\"\n",
				  (bfd_sym_info.name != 0) ?
				  bfd_sym_info.name : "\0");
			continue;
		}

		/* check whether a data symbol belongs to the data section */
		if(((bfd_sym_info.type == 'd') || (bfd_sym_info.type == 'D')) &&
		   (strcmp(symbol_table[i]->section->name, ".data") != 0)){
			discarded_data_symbols = 1;
			kl_trace2(0, "discarded data symbol \"%s\"\n",
				  (bfd_sym_info.name != 0) ?
				  bfd_sym_info.name : "\0");
			continue;
		}
			
		addr = bfd_sym_info.value;
		ret = kl_convert_symbol(&addr, &type, bfd_sym_info.type,
					modinfo);
		if(ret == 1){
			addr = 0;
			continue;
		}

/* 		kl_trace3(KLE_F_NOORIGIN, "%llx %c %s\n", addr, */
/* 			  bfd_sym_info.type, bfd_sym_info.name); */

		stp->symcnt++;
		if (cur_syment) {
			cur_syment->s_next =
				kl_alloc_syment(addr, type,
						bfd_sym_info.name);
			cur_syment = cur_syment->s_next;
		} else {
			syment_list = kl_alloc_syment(addr, type,
						      bfd_sym_info.name);
			cur_syment = syment_list;
		}
		addr = 0;
	}
	
	if(discarded_data_symbols){
		kl_error("warning: discarded data symbols not belonging to "
			 "\".data\" section\n");
	}

	if(discarded_bss_symbols){
		kl_error("warning: discarded bss symbols due to missing "
			 "__insmod_modulename_S.bss_Llength symbol in ksyms\n");
	}

	if(kl_insert_artificial_symbols(stp, &cur_syment, modinfo)){
		/* XXX - error code/error message */
		kl_free_syment_list(syment_list);
		kl_free_symtab(stp);
		return(1);
	}
	kl_insert_symbols(stp, syment_list);

	ml->syminfo=stp;
	kl_free_block(symbol_table);
	return (0);
}

/*
 * read syminfo from elf object file (ignore archives)
 */
int
kl_read_bfd_syminfo(maplist_t *ml)
{
	bfd *archive = NULL;
	bfd *object = NULL;

	kl_reset_error();
	if((kl_open_elf(ml->mapfile, &object, &archive)) == -1){
		/* we could not open a bfd for mapfile */
		return(1);
	}
	if(archive && object){
		/* ml->mapfil is an archive and no object */
		KL_ERROR = KLE_ARCHIVE_FILE;
		bfd_close(archive);
		bfd_close(object);
		return(1);
	}

	if(kl_read_symbols(object, ml)){
		return(1);
	}

	if(object)
		bfd_close(object);

	if(archive)
		bfd_close(archive);

	return(0);
}


/*
 * return 0 iff arch/mach of bfd do not match dump architecture, 1 otherwise
 */
static int
kl_check_bfd_arch(bfd *abfd)
{
	enum bfd_architecture arch;
	unsigned long mach;
	int match=0;
/* 	const bfd_arch_info_type *arch_info; */

/* 	arch_info = bfd_get_arch_info(abfd); */
	arch = bfd_get_arch(abfd);
	mach = bfd_get_mach(abfd);
	switch(KL_ARCH){
		case KL_ARCH_ALPHA:
			match = (arch == bfd_arch_alpha);
			break;
		case KL_ARCH_ARM:
			/* FIXME: Just a workaround */
			arch = bfd_arch_arm;
			match = (arch == bfd_arch_arm);
			break;
		case KL_ARCH_I386:
			match = (arch == bfd_arch_i386);
			break;
		case KL_ARCH_IA64:
			match = (arch == bfd_arch_ia64) &&
				(mach == bfd_mach_ia64_elf64);
			break;
		case KL_ARCH_S390:
#if defined(bfd_mach_s390_esa)
			match = (arch == bfd_arch_s390) &&
				(mach == bfd_mach_s390_esa);
#elif defined(bfd_mach_s390_31)
			match = (arch == bfd_arch_s390) &&
				(mach == bfd_mach_s390_31);
#endif
			break;
		case KL_ARCH_S390X:
#if defined(bfd_mach_s390_esame)
			match = (arch == bfd_arch_s390) &&
				(mach == bfd_mach_s390_esame);
#elif defined(bfd_mach_s390_64)
			match = (arch == bfd_arch_s390) &&
				(mach == bfd_mach_s390_64);
#endif
			break;
		case KL_ARCH_PPC64:
			match = (arch == bfd_arch_powerpc);
			break;
		case KL_ARCH_X86_64:
			match = (arch == bfd_arch_i386);
			break;
	}

	if((arch == bfd_arch_unknown) ||
	   (arch == bfd_arch_obscure)){
		/* XXX - set error code */
		kl_trace1(0, "Arch of object file not supported.\n");
	} else if(!match){
		/* XXX - set error code */
		kl_trace1(0, "Arch of object file does not match dump arch.\n");
	}

	return(match);
}

/*
 * If filename is an archive file the bfd* of the archive is returned in
 * archive and the bfd* of the first/next object is returned in object.
 * To iterate through all objects in an archive file, the values returned
 * in object and archive must be passed to the next call of kl_open_elf().
 * return values: -1, on error,  KL_ERROR is set to KLE_BAD_ELF_FILE
 *                 0, on success (object!=NULL, archive==NULL -> filename is
 *                    an object file, archive!=NULL -> filename is
 *                    an archive file)
 *                 1, no more objects in archive file (object==archive==NULL)
 */
int
kl_open_elf(char *filename, bfd **object, bfd **archive)
{
	/* bfd* of an already opened archive file
	 */
	if(!(*archive)){
		if(*object){
			goto finished;
		}
		switch(KL_ARCH){
		case KL_ARCH_ALPHA:
			*archive = bfd_openr(filename, "elf64-little");
			break;
		case KL_ARCH_ARM:
			*archive = bfd_openr(filename, "elf32-little");
			//*archive = bfd_openr(filename, "default");
			break;
		case KL_ARCH_I386:
			*archive = bfd_openr(filename, "elf32-little");
			break;
		case KL_ARCH_IA64:
			*archive = bfd_openr(filename, "elf64-little");
			break;
		case KL_ARCH_S390:
			*archive = bfd_openr(filename, "elf32-big");
			break;
		case KL_ARCH_S390X:
			*archive = bfd_openr(filename, "elf64-big");
			break;
		case KL_ARCH_PPC64:
			*archive = bfd_openr(filename, "elf64-big");
			break;
		case KL_ARCH_X86_64:
			*archive = bfd_openr(filename, "elf64-little");
			break;
		default:
			kl_trace1(0, "Dump architecture not supported.\n");
			/* XXX set error code/write error message */
			goto error;
		}
		if (!(*archive)){
			kl_check_bfd_error(bfd_error_no_error);
			kl_trace1(0, "Could not open file: %s\n", filename);
			/* XXX set error code/write error message */
			goto error;
		}
	}


	/* Check to see if this namelist is an archive. If it is, then
	 * we need to cycle through the individual object files and
	 * process them one-by-one. 
	 */
	if (bfd_check_format(*archive, bfd_archive) == TRUE) {
		/* clear bfd_error, if we use more than one target
		 * bfd_error_wrong_format might be set
		 */
		bfd_set_error(bfd_error_no_error);

		*object = bfd_openr_next_archived_file(*archive, *object);
		if(!(*object) && 
		   kl_check_bfd_error(bfd_error_no_more_archived_files)){
			/* reached end of archive */
			goto finished;
		}
			
		if (!bfd_check_format(*object, bfd_object)) {
			kl_check_bfd_error(bfd_error_no_error);
			/* XXX set error code/write error message */
			kl_trace1(0, "Could not open object in archive "
				  "file.\n");
			goto error;
		}
		if(!kl_check_bfd_arch(*object)){
			kl_trace1(0, "Could not open object in archive "
				  "file.\n");
			goto error;
		}
	} else {
		kl_check_bfd_error(bfd_error_file_not_recognized);
		if (!bfd_check_format(*archive, bfd_object)) {
			kl_check_bfd_error(bfd_error_no_error);
			/* XXX set error code/write error message */
			kl_trace1(0, "Could not open object file.\n");
			goto error;
		}
		if(!kl_check_bfd_arch(*archive)){
			kl_trace1(0, "Could not open object file.\n");
			goto error;
		}
		*object = *archive;
		*archive = NULL;
	}
	return(0);
 error:
	KL_ERROR=KLE_BAD_ELF_FILE;
	*object = *archive = NULL;
	return(-1);
 finished:
	*object = *archive = NULL;
	return(1);
	
}

/*
 * return 1 iff error matches given error_tag, 0 otherwise
 */
int
kl_check_bfd_error(bfd_error_type error_tag)
{
	bfd_error_type error_type;
	const char *errormsg;

	error_type = bfd_get_error();

	if(error_type != error_tag){
		switch(error_type){
		case bfd_error_no_error: /* nothing to do, not passed here */
			break;
		case bfd_error_system_call: /* check errno */
			errormsg = strerror(errno);
			errno = 0;
			break;
		default: /* any other bfd error type */
			errormsg = bfd_errmsg(error_type);
		}
		if(error_type){
			kl_trace1(0, "libbfd: %s\n", errormsg);
		}
	}

	/* reset error */
	bfd_set_error(bfd_error_no_error);

	return(error_tag == error_type);
}

/*
 * print symbol flags in readable form
 */
static void
_kl_print_symbol_flags(asymbol **symbol_table, int num)
{
	int i;
	char *sec_type, *sec_type_orig;
	sec_type_orig = kl_alloc_block(SEC_TYPE_STRLEN, K_TEMP);

	for (i = 0; i < num; i++) {
		sec_type = sec_type_orig;
		if(!(strcmp(symbol_table[i]->section->name,
			    BFD_ABS_SECTION_NAME))) {
			sec_type += sprintf(sec_type, "ABS ");
		} else if(symbol_table[i]->section->flags & SEC_ALLOC) {
			sec_type += sprintf(sec_type, "ALLOC ");
		} if(symbol_table[i]->section->flags & SEC_LOAD) {
			sec_type += sprintf(sec_type, "LOAD ");
		} if(symbol_table[i]->section->flags & SEC_RELOC) {
			sec_type += sprintf(sec_type, "RELOC ");
		} if(symbol_table[i]->section->flags & SEC_READONLY) {
			sec_type += sprintf(sec_type, "READONLY ");
		} if(symbol_table[i]->section->flags & SEC_CODE) {
			sec_type += sprintf(sec_type, "CODE ");
		} if(symbol_table[i]->section->flags & SEC_DATA) {
			sec_type += sprintf(sec_type, "DATA ");
		} if(symbol_table[i]->section->flags & SEC_ROM) {
			sec_type += sprintf(sec_type, "ROM ");
		} if(symbol_table[i]->section->flags & SEC_DEBUGGING) {
			sec_type += sprintf(sec_type, "DEBUGGING ");
		} if(symbol_table[i]->section->flags & SEC_SMALL_DATA) {
			sec_type += sprintf(sec_type, "SMALL_DATA ");
		}
		kl_trace4(KLE_F_NOORIGIN, "%0-32s %#8x %s\n",
			  symbol_table[i]->name, symbol_table[i]->flags,
			  sec_type_orig);
	}
	kl_free_block(sec_type_orig);
	return;
}
