/*
 * $Id: trace_ppc64.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright (C) 2003, 2004 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include "dis-asm.h"
#include <lcrash.h>

/* ok...let's get off the ground... */
#define NOT_IMPLEMENTED(s) fprintf(stderr, "ppc64: %s not implemented at %s:%d\n", (s), __FILE__, __LINE__)

/*
 * function definitions
 */

/* 
 * setup_trace_rec_ppc64()
 */
int
setup_trace_rec_ppc64(kaddr_t saddr, kaddr_t task, int flag, trace_t *trace)
{
	int aflag = K_TEMP;

	kl_reset_error();

	if (flag & C_PERM) {
		aflag = K_PERM;
	}
	if (task) {
		trace->tsp = kl_alloc_block(TASK_STRUCT_SZ, aflag);
		if (kl_get_task_struct(task, 2, trace->tsp)) {
			kl_free_block(trace->tsp);
			trace->tsp = NULL;
			return(1);
		}
	}
	trace->stack[0].type = S_KERNELSTACK;
	trace->stack[0].size = STACK_SIZE_PPC64;

	/* Get the base address of the stack
	 */
	trace->stack[0].addr = saddr - trace->stack[0].size;
	trace->stack[0].ptr = kl_alloc_block(STACK_SIZE_PPC64, aflag);
	if (KL_ERROR) {
		clean_trace_rec(trace);
		return(1);
	}
	GET_BLOCK(trace->stack[0].addr, STACK_SIZE_PPC64, trace->stack[0].ptr);
	if (KL_ERROR) {
		clean_trace_rec(trace);
		return(1);
	}
	return(0);
}

static int
get_frame_size(kaddr_t sp, kaddr_t newsp, kaddr_t saddr)
{
	if (newsp && newsp < sp) {
		/* frame is in user space. */
		return(0);
	} else {
		return((newsp ? newsp : saddr) - sp);
	}
}
	
/*
 * find_trace_ppc64()
 *
 *   Given a starting pc (start_cp), starting stack pointer (start_sp),
 *   and stack address, check to see if a valid trace is possible. A
 *   trace is considered valid if no errors are encountered (bad PC,
 *   bad SP, etc.) Certain errors are tolorated however. For example,
 *   if the current stack frame is an exception frame (e.g., VEC_*),
 *   go ahead and return success -- even if PC and SP obtained from
 *   the exception frame are bad (a partial trace is better than no
 *   trace)..
 *   
 *   Return zero if no valid trace was found. Otherwise, return the
 *   number of frames found. If the C_ALL flag is passed in, then
 *   return a trace even if it is a subtrace of a trace that was
 *   previously found.
 *   
 *   Parameters:
 *   
 *   start_pc       starting program counter
 *   start_sp       starting stack pointer
 *   check_pc       if non-NULL, check to see if check_pc/check_sp
 *   check_sp       are a sub-trace of trace beginning with spc/ssp
 *   trace          structure containing all trace related info (frames,
 *                  pages, page/frame counts, etc.
 *   flags
 *   
 *  (Ref: 64-bit PowerPC ELF ABI Spplement; Ian Lance Taylor, Zembu Labs).
 *   A PPC64 stack frame looks like this:
 *  
 *  High Address
 *   .-> Back Chain (etc...)
 *   |   FP reg save area
 *   |   GP reg save area
 *   |   Local var space
 *   |   Parameter save area    (SP+48)
 *   |   TOC save area          (SP+40)
 *   |   link editor doubleword (SP+32)
 *   |   compiler doubleword    (SP+24)
 *   |  LR save                        (SP+16)
 *   |   CR save                        (SP+8)
 *   `- Back Chain              <-- sp (SP+0)
 *   
 *   Note that the LR (ret addr) may not be saved in the current frame if
 *   no functions have been called from the current function.
 */
 /* HACK: put an initial lr in this var for find_trace().  It will be
  * cleared during the trace.  Need to expand the interface.
  */
int
find_trace_ppc64(
	kaddr_t start_pc,
	kaddr_t start_sp,
	kaddr_t check_pc,
	kaddr_t check_sp,
	trace_t *trace,
	int flags)
{
	kaddr_t pc, sp, lr, newpc, newsp, newlr;
	kaddr_t sbase, saddr, oldsp;
	uaddr_t sbp, asp;
	int curstkidx = 0, frm_size;
	int nip_offset = kl_member_offset("pt_regs", "nip");
	int lr_offset = kl_member_offset("pt_regs", "link");
	int r1_offset = sizeof(KL_PPC_REG_PPC64); /* Ok...cheat */
	
	kaddr_t func_addr;
	sframe_t *curframe;
	char *func_name;
	
	sbp = (uaddr_t)trace->stack[curstkidx].ptr;
	sbase = trace->stack[curstkidx].addr;
	saddr = sbase + trace->stack[curstkidx].size;
	/* NOTE: the stack should be contained between sbase and saddr,
	 * however the traceback may go through the per-cpu interrupt
	 * stack.  So for now we don't do all the bounds checks.
	 */
	pc = start_pc;
	oldsp = sp = start_sp;
	lr = 0; /* hack...need to pass in initial lr reg */
	func_name = kl_funcname(pc);
	if (sp) {
		frm_size = get_frame_size(sp, KL_VREAD_PTR(sp), saddr);
	}
	 /* On each iteration of this loop we process a single frame.
	  * sp points to the stack frame (i.e. a frame pointer),
	  * pc is the current nip for the frame,
	  * lr is the return address from the frame,
	  * func_name is kl_funcname(pc).
	  */
	while (sp) {
		//fprintf(stderr, "find_trace: pc=0x%Lx lr=0x%Lx sp=0x%Lx\n", 
		//			   pc, lr, sp); 
		/* LOOP TRAP! Make sure we are not just looping on the
		 * same frame forever.
		 */
		if ((trace->nframes > 1) &&
			(curframe->funcname == curframe->prev->funcname) &&
			(curframe->sp == curframe->prev->sp)) {
			curframe->error = 1;
			return(trace->nframes);
		}

		/* Allocate space for a stack frame rec
		 */
		curframe = alloc_sframe(trace, flags);
		if ((func_addr = kl_funcaddr(pc)) == 0) {
			/*
			 * Got the invalid PC. But, still search in the stack
			 * and get other symbols. 
			 */
			curframe->error = KLE_BAD_PC;
		}
		/* Check to see if check_pc/check_sp points to a sub-trace
		 * of spc/ssp. If it does then don't return a trace (unless
		 * C_ALL). Make sure we free the curframe block since we
		 * wont be linking it in to the trace rec.
		 */
		if (check_pc && ((pc == check_pc) && (sp == check_sp))) {
			kl_free_block((void *)curframe);
			if (flags & C_ALL) {
				return(trace->nframes);
			} else {
				return(0);
			}
		}
	
		/* Finally record this frame. */
		asp = (uaddr_t)(sbp + (STACK_SIZE_PPC64 - (saddr - sp)));
		UPDATE_FRAME(func_name, pc, lr, sp, 0, asp, 0, 0, frm_size);
	
		if (sp && sp < oldsp) {
#ifdef DEBUG
			if (sp & 0xffffffff00000000UL) == 0)
				fprintf(stderr, "Stack drops into 32bit user space %0Lx\n", sp);
			else
				fprintf(stderr, "Corrupt stack. Next back chain is %0Lx\n", sp);
#endif
			kl_reset_error();
			return(trace->nframes);
                }
		oldsp = sp;

		/* Gather starting information for the next frame */
		if (lr)
			newpc = lr;
		else
			newpc = KL_VREAD_PTR(sp+16);

		newsp = KL_VREAD_PTR(sp); /* follow back chain */
		if (KL_ERROR)
			newsp = 0;
		newlr = KL_VREAD_PTR(newsp+16);
		if (KL_ERROR)
			newlr = 0;
		frm_size = get_frame_size(sp, newsp, saddr);

		/* Handle special case frames here. */
		if ((func_name = kl_funcname(pc)) != NULL) {
			int interrupt_frame = 0;
			if (strcmp(func_name, ".__switch_to") == 0) {
				/* NOTE: _switch_to() calls _switch() which
				 * is asm.  _switch leaves pc == lr.
				 * Working through this frame is tricky,
				 * and this mess isn't going to help if we
				 * actually dumped here.  Most likely the
				 * analyzer is trying to backtrace a task.
				 */
				newpc = pc;
				sp = newsp;
				newsp = KL_VREAD_PTR(sp);
				sp = newsp; /* skip the _switch_to() frame */
				newpc = KL_VREAD_PTR(sp+16);
				newsp = KL_VREAD_PTR(sp);
				newlr = KL_VREAD_PTR(newsp + 16);
				frm_size = get_frame_size(sp, newsp, saddr);
			} else if (!strcmp(func_name, ".ret_from_syscall_1") ||
				!strcmp(func_name, ".ret_from_except")) {
				interrupt_frame = 1;
			}
			
			if (interrupt_frame) {
				newpc = KL_VREAD_PTR(sp + 
					KL_STACK_FRAME_OVERHEAD_PPC64 + 
					nip_offset);
				newsp = KL_VREAD_PTR(sp + 
					KL_STACK_FRAME_OVERHEAD_PPC64 + 
					r1_offset);
				newlr = KL_VREAD_PTR(sp + 
					KL_STACK_FRAME_OVERHEAD_PPC64 +
					lr_offset);
				frm_size = get_frame_size(sp, newsp, saddr);
			}
		}
		pc = newpc;
		sp = newsp;
		lr = newlr;
		func_name = kl_funcname(pc);
		/* ToDo: handle the interrupt stack */
	}
	return(trace->nframes);
}

/*
 * print_trace_ppc64()
 */
void
print_trace_ppc64(trace_t *trace, int flags, FILE *ofp)
{
	int offset;
	sframe_t *frmp;

	if ((frmp = trace->frame)) {
		do {
			fprintf(ofp, "%2d 0x%"FMT64"x %s", 
					frmp->level, frmp->pc, frmp->funcname);
			offset = pc_offset(frmp->pc);
			if (offset > 0) {
				fprintf(ofp, "+0x%x", offset);
			} else if (offset < 0) {
				fprintf(ofp, "+<ERROR>");
			}
			if (frmp->error == KLE_BAD_PC) {
				fprintf(ofp, "  (unreliable)");
			}
			fprintf(ofp, "\n");
			if (flags & C_FULL) {
				fprintf(ofp, "\n");
				fprintf(ofp, "   RA=0x%"FMT64"x, SP=0x%"FMT64"x, "
				 	"SIZE=%d\n\n", frmp->ra, frmp->sp,
				 	frmp->frame_size); 
				dump_stack_frame(trace, frmp, ofp);
			}
			if (frmp->error == 1) {
				fprintf(ofp, "TRACE ERROR 0x%"FMT64"x\n", 
					frmp->error);
			}
			frmp = frmp->next;
		} while (frmp != trace->frame);
	}
}

/*
 * task_trace_ppc64()
 */
int
task_trace_ppc64(kaddr_t task, int flags, FILE *ofp)
{
	void *tsp;
	trace_t *trace;

        if (!(tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP))) {
		return(1);
	}
	if (kl_get_task_struct(task, 2, tsp)) {
		kl_free_block(tsp);
		return(1);
	}
	trace = (trace_t *)alloc_trace_rec(C_TEMP);
	if (!trace) {
		fprintf(KL_ERRORFP, "Could not alloc trace rec!\n");
		return(1);
	} else {
		kaddr_t saddr, nip, lr, ksp;
		saddr = kl_kernelstack_ppc64(task);
		if (kl_smp_dumptask_ppc64(task)) {
			nip = kl_dumpnip_ppc64(task);
			ksp = kl_dumpgpr_ppc64(task);
		} else {
			uaddr_t *thread;
			int nip_offset = kl_member_offset("pt_regs", "nip");
			int lr_offset = kl_member_offset("pt_regs", "link");
		
			thread = K_PTR(tsp, "task_struct", "thread");
			ksp = KL_UINT(thread, "thread_struct", "ksp");
			nip = KL_VREAD_PTR(ksp + KL_STACK_FRAME_OVERHEAD_PPC64 
				+ nip_offset);
			lr = KL_VREAD_PTR(ksp + KL_STACK_FRAME_OVERHEAD_PPC64 
				+ lr_offset);
		}
		if (ksp < KL_PAGE_OFFSET || nip < KL_PAGE_OFFSET) {
			/* ToDo: make an "is_user_task(tsp) func that
			 *  studies the msr */
			fprintf(KL_ERRORFP, "Task in user space, No backtrace\n");
			return 1;
		}
		setup_trace_rec_ppc64(saddr, 0, 0, trace);
		if (KL_ERROR) {
			fprintf(KL_ERRORFP, "Error setting up trace rec!\n");
			free_trace_rec(trace);
			return(1);
		}
		find_trace_ppc64(nip, ksp, 0, 0, trace, 0);
		trace_banner(ofp);
		fprintf(ofp, "STACK TRACE FOR TASK: 0x%"FMT64"x", task);
		if (KL_TYPEINFO()) {
			fprintf(ofp, "(%s)\n\n", 
				(char *)K_PTR(tsp, "task_struct", "comm"));
		} else {
			fprintf(ofp, "(%s)\n\n", 
				(char *)K_PTR(tsp, "task_struct", "comm"));
		}
		print_trace_ppc64(trace, flags, ofp);
	}
	kl_free_block(tsp);
	free_trace_rec(trace);
	return(0);
}

/*
 * print_traces_ppc64()
 *
 *   Output a list of all valid code addresses contained in a stack
 *   along with their function name and stack location.
 */
int
print_traces_ppc64(kaddr_t saddr, int level, int flags, FILE *ofp)
{
	NOT_IMPLEMENTED("do_list_ppc64");
#if 0
	int nfrms;
	trace_t *trace;
	kaddr_t nip,ksp, addr, sbase;
	
	if (!(trace = (trace_t *)alloc_trace_rec(K_TEMP))) {
		fprintf(KL_ERRORFP, "Could not alloc trace rec!\n");
		return(1);
	}
	setup_trace_rec_ppc64(saddr, 0, 0, trace);
	if (!get_valid_top_stack_frame(trace, &ksp, &nip)) {
		fprintf(KL_ERRORFP, "Could not find top stack frame\n");
		return(1);
	}
	nfrms = find_trace_ppc64(nip, ksp, 0, 0, trace, 0);
	if (nfrms) {
		if ((nfrms >= level) && (!trace->frame->prev->error ||
			(flags & C_ALL))) {
			fprintf(ofp, "\nPC=");
			print_kaddr(nip, ofp, 0);
			fprintf(ofp, "  SP=");
			print_kaddr(ksp, ofp, 0);
			fprintf(ofp, "  SADDR=");
			print_kaddr(saddr, ofp, 0);
			fprintf(ofp, "\n");
			trace_banner(ofp);
			print_trace_ppc64(trace, flags, ofp);
			trace_banner(ofp);
		}
		free_sframes(trace);
	}
#endif
	return(0);
}

/*
 * do_list_ppc64()
 *
 *   Output a list of all valid code addresses contained in a stack
 *   along with their function name and stack location.
 */
int
do_list_ppc64(kaddr_t saddr, int size, FILE *ofp)
{
	NOT_IMPLEMENTED("do_list_ppc64");
	return 0;
}

/*
 * dumptask_trace_ppc64()
 */
int
dumptask_trace_ppc64(kaddr_t curtask,
	int flags,
	FILE *ofp)
{
	kaddr_t nip, ksp, saddr;
	void *tsp;
	trace_t *trace;
	int i;
	kl_dump_header_ppc64_t dha;

	if (kl_get_dump_header_ppc64(&dha)) {
		/* fixme: set error code */
		return(1);
	}

	nip = ksp = 0;
	for (i = 0; i < KL_GET_UINT32(&dha.smp_num_cpus); i++) {
		if (curtask == (kaddr_t)KL_GET_UINT64(&dha.smp_current_task[i])) {
			nip = (kaddr_t)KL_GET_UINT64(&dha.smp_regs[i].nip);
			ksp = (kaddr_t)KL_GET_UINT64(&dha.smp_regs[i].gpr[1]);
			break;
		}
	}
	fprintf(ofp, "nip=%"FMT64"x, ksp=%"FMT64"x\n", nip, ksp);
	
	tsp = kl_alloc_block(TASK_STRUCT_SZ, K_TEMP);
	if (!tsp) {
		return(1);
	}
	if (kl_get_task_struct(curtask, 2, tsp)) {
		kl_free_block(tsp);
		return(1);
	}
	if (!(trace = alloc_trace_rec(K_TEMP))) {
		fprintf(KL_ERRORFP, "Could not alloc trace rec!\n");
	} else {
		saddr = kl_kernelstack_ppc64(curtask);
		fprintf(ofp, "kernelstack saddr=%"FMT64"x\n", saddr);
		setup_trace_rec_ppc64(saddr, 0, 0, trace);
		find_trace_ppc64(nip, ksp, 0, 0, trace, 0);
		trace_banner(ofp);
		fprintf(ofp, "STACK TRACE FOR TASK: 0x%"FMT64"x (%s)\n\n",
			curtask, (char*)K_PTR(tsp, "task_struct", "comm"));
		print_trace_ppc64(trace, flags, ofp);
		trace_banner(ofp);
		free_trace_rec(trace);
	}
	return(0);
}

/*
 * trace_init_ppc64()
 */
int
trace_init_ppc64(void)
{
	SETUP_TRACE_REC = setup_trace_rec_ppc64;
	PRINT_TRACE     = print_trace_ppc64;
	TASK_TRACE      = task_trace_ppc64;
	PRINT_TRACES    = print_traces_ppc64;
	FIND_TRACE      = find_trace_ppc64;
	DUMPTASK_TRACE  = dumptask_trace_ppc64;
	DO_LIST         = do_list_ppc64;
	STACK_SEGMENTS  = STACK_SEGMENTS_PPC64;
	STACK_SIZE      = STACK_SIZE_PPC64;
	return(0);
}
