/*
 * $Id: trace_alpha.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/*
 * FIXME: find_task_trace() -- currently no alpha implementation
 */
#ifndef ALPHA_FULLY_IMPLEMENTED
#error "CODE REQUIRES PORT TO HOST ARCHITECTURE"
#else
static int
find_task_trace_alpha(trace_t *trace, int flags)
{
	return(0);
}
#endif
/* 
 * setup_trace_rec_alpha()
 */
int
setup_trace_rec_alpha(kaddr_t saddr, kaddr_t task, int flag, trace_t *trace)
{
	int aflag = K_TEMP;
	kl_reset_error();

	if (flag & C_PERM) {
		aflag = K_PERM;
	}
	if (task) {
		trace->task = task;
		trace->tsp = kl_alloc_block(TASK_STRUCT_SZ, aflag);
		if (kl_get_task_struct(task, 2, trace->tsp)) {
			kl_free_block(trace->tsp);
			trace->tsp = (struct task_struct *)NULL;
			return(1);
		}
	}
	trace->stack[0].type = S_KERNELSTACK;
	trace->stack[0].size = STACK_SIZE;

	/* Get the base address of the stack
	 */
	trace->stack[0].addr = saddr - trace->stack[0].size;
	trace->stack[0].ptr = kl_alloc_block(STACK_SIZE, aflag);
	if (KL_ERROR) {
		clean_trace_rec(trace);
		return(1);
	}
	GET_BLOCK(trace->stack[0].addr, STACK_SIZE, trace->stack[0].ptr);
	if (KL_ERROR) {
		clean_trace_rec(trace);
		return(1);
	}
	return(0);
}


/*
 * print_trace_alpha()
 */
void
print_trace_alpha(trace_t *trace, int flags, FILE *ofp)
{
	int offset;
	sframe_t *frmp;

	if ((frmp = trace->frame)) {
		do {
			fprintf(ofp, "%2d %s", frmp->level, frmp->funcname);
			offset = pc_offset(frmp->pc);
			if (offset > 0) {
				fprintf(ofp, "+%d", offset);
			} else if (offset < 0) {
				fprintf(ofp, "+<ERROR>");
			}
			fprintf(ofp, " [0x%llx]\n", frmp->pc);
			if (flags & C_FULL) {
				fprintf(ofp, "\n");
				fprintf(ofp, "   RA=0x%llx, SP=0x%llx, "
					"FP=0x%llx, SIZE=%d\n\n", 
					frmp->ra, frmp->sp, 
					frmp->fp, frmp->frame_size);
#ifdef FRMSIZE_DBG
				fprintf(ofp, "\n  FRAMESIZE=%d\n\n",
					get_framesize(frmp->pc));
#endif
			}
			if (frmp->error) {
				fprintf(ofp, "TRACE ERROR 0x%llx\n", 
					frmp->error);
			}
			frmp = frmp->next;
		} while (frmp != trace->frame);
	}
}

/*
 * task_trace_alpha()
 */
int
task_trace_alpha(kaddr_t task, int flags, FILE *ofp)
{
	kaddr_t saddr;
	trace_t *trace;

	if (!(trace = (trace_t *)alloc_trace_rec(C_TEMP))) {
		fprintf(KL_ERRORFP, "Could not alloc trace rec!\n");
		return(1);
	} else {
		saddr = kl_kernelstack_alpha(task);
		setup_trace_rec_alpha(saddr, task, 0, trace);
		if (KL_ERROR) {
			fprintf(KL_ERRORFP, "Error setting up trace rec!\n");
			free_trace_rec(trace);
			return(1);
		}
#ifndef ALPHA_FULLY_IMPLEMENTED
#error "CODE REQUIRES PORT TO HOST ARCHITECTURE"
#else
		find_task_trace_alpha(trace, 0);
		trace_banner(ofp);
		fprintf(ofp, "STACK TRACE FOR TASK: ");
		print_kaddr(task, ofp, 0);
		fprintf(ofp, " (%s)\n\n",
			(char*)K_PTR(trace->tsp,"task_struct","comm"));
		print_trace_alpha(trace, flags, ofp);
#endif
	}
	free_trace_rec(trace);
	return(0);
}

/*
 * trace_init_alpha()
 */
int
trace_init_alpha(void)
{
	SETUP_TRACE_REC = setup_trace_rec_alpha;
	PRINT_TRACE = print_trace_alpha;
	TASK_TRACE = task_trace_alpha;
	STACK_SEGMENTS = STACK_SEGMENTS_ALPHA;
	STACK_SIZE = STACK_SIZE_ALPHA;
	return(0);
}
