/*
 * $Id: dis_ppc64.c,v 1.1 2004/12/21 23:26:17 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <strings.h>
#include "dis-asm.h"

/*
 * do_dis_ppc64()
 */
kaddr_t
do_dis_ppc64(kaddr_t value, int lines, FILE *ofp)
{
       bfd_vma pc;
       int i;

       set_dis_ofp(ofp);

        pc = value;
       for (i = 0; i < lines; i++) {
               dis_printintaddr(pc, &DISINFO, 1);
               pc += print_insn_big_powerpc(pc, &DISINFO);
               DISINFO.fprintf_func(ofp, "\n");
       }
       return((kaddr_t)pc);
}

/*
 * print_instr_ppc64()
 */
int
print_instr_ppc64(kaddr_t pc, FILE *ofp, int flag)
{
	return(do_dis_ppc64(pc, 1, ofp) - pc);
}

/*
 * dump_instr_ppc64 () -- architecture specific instruction dump routine
 */
void
dump_instr_ppc64(kaddr_t addr, uint64_t count, int flags, FILE *ofp)
{
	unsigned char template;
	uint64_t t0, t1, slot0, slot1, slot2;
	kaddr_t bundlep;
	bfd_byte bundle[16];

	bundlep = (addr & 0xfffffffffffffff0);
	GET_BLOCK(bundlep, sizeof(bundle), bundle);
	t0 = bfd_getl64(bundle);
	t1 = bfd_getl64(bundle + 8);
	template = (t0 >> 1) & 0xf;
	slot0 = (t0 >>  5) & 0x1ffffffffffLL;
	slot1 = ((t0 >> 46) & 0x3ffff) | ((t1 & 0x7fffff) << 18);
	slot2 = (t1 >> 23) & 0x1ffffffffffLL;

	fprintf(ofp, "\nbundlep  : 0x%"FMTPTR"x\n", bundlep);
	fprintf(ofp, "template : %02x\n", template);
	fprintf(ofp, "   slot0 : %011"FMTPTR"x\n", slot0);
	fprintf(ofp, "   slot1 : %011"FMTPTR"x\n", slot1);
	fprintf(ofp, "   slot2 : %011"FMTPTR"x\n", slot2);
}

/*
 * print_instr_stream_ppc64()
 * v           virtual addr to dump
 * bcount      num instructions before v to dump
 * acount      num instructions after v to dump
 */
kaddr_t
print_instr_stream_ppc64(kaddr_t v, int bcount, int acount, int flags, FILE *ofp)
{
	int count;
	kaddr_t pc = v;

	/* Make sure that output goes to the right place
	 */
	set_dis_ofp(ofp);

	pc &= ~3;       /* 4-byte aligned */
	pc -= bcount*4;
	count = acount + bcount + 1;
	pc = do_dis_ppc64(pc, count, ofp);
	return(pc);
}

/*
 * In PPC64 , all actual function symbols starts with '.'. Therefore,
 * the given function name symbol type is defined as global data which
 * has the address of actual function symbol.
 */
syment_t *
get_func_name_ppc64(char *fname)
{
	syment_t *sp;
	kaddr_t func_addr;
	char *actname;

	sp = kl_lkup_symname(fname);
	if ((fname[0] == '.') || !sp) {
		/*
		 * Specified the proper symbol or could not find the symbol.
		 */
		return sp;
	}
	func_addr = KL_VREAD_PTR(sp->s_addr);
	if (KL_ERROR) {
		return NULL;
	}
	actname = kl_funcname(func_addr);
	sp = kl_lkup_symname(actname);
	return sp;
}

/*
 * dis_init_ppc64() - init arch specific stuff for disassembling
 */
int
dis_init_ppc64(FILE *ofp, int dumparch)
{
	PRINT_INSTR        = print_instr_ppc64;
	PRINT_INSTR_STREAM = print_instr_stream_ppc64;
	DUMP_INSTR         = dump_instr_ppc64;

	DISINFO.fprintf_func            = dis_fprintf;
	DISINFO.stream                  = ofp;
	DISINFO.application_data        = NULL;
	DISINFO.flavour                 = bfd_target_elf_flavour;
	DISINFO.arch                    = bfd_arch_powerpc;
	DISINFO.mach                    = bfd_mach_ppc_620;
	DISINFO.endian                  = BFD_ENDIAN_BIG;
	DISINFO.flags                   = 0;
	DISINFO.private_data            = NULL;
	DISINFO.read_memory_func        = getidmem;
	DISINFO.print_address_func      = dis_printaddr;
	DISINFO.symbol_at_address_func  = dis_getsym;
	DISINFO.buffer                  = NULL;
	DISINFO.buffer_vma              = 0;
	DISINFO.buffer_length           = 0;
	DISINFO.bytes_per_line          = 0;
	DISINFO.bytes_per_chunk         = 0;
	DISINFO.display_endian          = BFD_ENDIAN_BIG;
	DISINFO.insn_info_valid         = 0;
	DISINFO.branch_delay_insns      = 0;
	DISINFO.data_size               = 0;
	DISINFO.insn_type               = 0;
	DISINFO.target                  = 0;
	DISINFO.target2                 = 0;

	return(0);
}
