/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_posix_enf.mod posix enforcement plugin

    \section posixsynopsis SYNOPSIS
    \b lcmaps_posix_enf.mod [-maxuid|-MAXUID \<number of uids\>] [-maxpgid|-MAXPGID
    \<number of primary gids\>] [-maxsgid|-MAXSGID \<number of secondary gids\>]
 
    \section posixdesc DESCRIPTION

The Posix Enforcement plugin will enforce (apply) the gathered credentials
that are stacked in the datastructure of the Plugin Manager.
The plugin will get the credential information that is gathered by one or
more Acquisition plugins. This implies that at least one Acquisition should have been
run prior to this Enforcement.
All of the gathered information will be checked by looking into the 'passwd'
file of the system. These files have information about all registered system
account and its user groups.

The Posix Enforcent plugin does not validate the secondary GIDs. It does
check the existance of the GID and the UID.
They must exist although it is not needed that the GID and UID are a pair of
each other.

The (BSD/POSIX) functions setreuid(), setregid() and setgroups() are used to change
the privileges of the process from root to that of a local user.

    \section posixoptions OPTIONS
    \subsection posixoptie1 -MAXUID \<number of uids\>
        See \ref posixoptie2 "-maxuid"

    \subsection posixoptie2 -maxuid <number of uids>
        In principle, this will set the maximum number of allowed UIDs that this plugin will handle,
        but at the moment only the first UID found will be enforced;
        the others will discarded.
        By setting the value to a maximum there will be a failure raised when the amount of UIDs
        exceed the set maximum. Without this value the plugin will continue
        and will enforce only the first found value in the credential data structure.

    \subsection posixoptie3 -MAXPGID \<number of primary gids\>
        See \ref posixoptie4 "-maxpgid"

    \subsection posixoptie4 -maxpgid <number of primary gids>
        This will set the maximum number of allowed Primary GIDs that this plugin will handle, similar to 
        \ref posixoptie2 "-maxuid". Also here only the first primary GID found will be taken into account.

    \subsection posixoptie5 -MAXSGID \<number of secondary gids\>
        See \ref posixoptie6 "-maxsgid"

    \subsection posixoptie6 -maxsgid <number of secondary gids>
        This will set the maximum allowed Secondary GIDs that this plugin will handle.
        This number is limited by the system (NGROUPS) and is usually 32. If the plugin cannot
        determine the system value, it limits itself to 32.

    \subsection posixoptie7 -set_only_euid [yes|no]
        The result of setting this option to 'yes' is that only the effective uid is set.
        In other words, it is still possible to regain root (uid) privileges for the process.
        This is definitely undesirable if this module is used from a process like the gatekeeper,
        since it would be possible for user jobs to get root privileges !
        \b THIS IS A DANGEROUS OPTION, PLEASE KNOW WHAT YOU'RE DOING !
        By default, this option is set to 'no'. Possibly this option should be set if the module is used by
        gridFTP, since this service does not spawn user jobs and has to regain root pivileges at the end.

    \subsection posixoptie8 -set_only_egid [yes|no]
        Analogue to the previous option the result of setting this option to 'yes' is that only the
        effective (primary) gid is set.
        In other words, it is still possible to regain root (gid) privileges for the process.
        This is definitely undesirable if this module is used from a process like the gatekeeper,
        since it would be possible for user jobs to get root privileges !
        \b THIS IS A DANGEROUS OPTION, PLEASE KNOW WHAT YOU'RE DOING !
        By default, this option is set to 'no'. Possibly this option should be set if the module is used by
        gridFTP, since this service does not spawn user jobs and has to regain root pivileges at the end.

    \section posixreturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure

    \section posixerrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)
 
    \section posixSeeAlso SEE ALSO
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_posix.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code for the posix process enforcement LCMAPS plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include "lcmaps_plugins_basic_config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <grp.h>
#include <ctype.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

#define MAX_UNDEFINED ((int)(-1))

#ifndef NGROUPS
    #ifdef NGROUPS_MAX
        #define NGROUPS NGROUPS_MAX
    #else
        #define NGROUPS 32 
    #endif
#endif

/* 
 * MAX_NGROUPS has also been used. I couldn't see in the source if it was
 * a self made define in the code or that it was OS specific.
 */

#ifndef MAX_LOG_BUFFER_SIZE
    #define MAX_LOG_BUFFER_SIZE 500
#endif

/******************************************************************************
                          Module specific prototypes
******************************************************************************/

static char * cgul_process_identity_oneline(void);
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);


/******************************************************************************
                       Define module specific variables
******************************************************************************/

static int maxuid  = MAX_UNDEFINED;
static int maxpgid = MAX_UNDEFINED;
static int maxsgid = MAX_UNDEFINED;
static int set_only_euid = 0;
static int set_only_egid = 0;



char * cgul_process_identity_oneline(void)
{
    int t = 0;
    int stored_errno = 0;
    char * line_buf = NULL;
    struct passwd * p = NULL;
    struct group  * g = NULL;
    int ngroups = 0;
    gid_t * list = NULL;

    uid_t uid  = getuid();
    uid_t euid = geteuid();
    gid_t gid  = getgid();
    gid_t egid = getegid();


    /* Allocate write buffer, must be free'd */
    if (!(line_buf = calloc (sizeof (char), 1024 + 1)))
        return NULL;

    /* Real User ID */
    if (p = getpwuid (uid))
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), "uid=%d(%s)", uid, p -> pw_name);
    else
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), "uid=%d", uid);


    /* Effective User ID */
    if (p = getpwuid (euid))
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",euid=%d(%s)", euid, p -> pw_name);
    else
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",euid=%d", euid);


    /* Real Group ID */
    if (g = getgrgid (gid))
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",gid=%d(%s)", gid, g -> gr_name);
    else
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",gid=%d", gid);


    /* Effective Group ID */
    if (g = getgrgid (egid))
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",egid=%d(%s)", egid, g -> gr_name);
    else
        snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",egid=%d", egid);


    /* Get the number of secondary group IDs for root */
    if ((ngroups = getgroups(0, NULL)) < 0)
    {
        stored_errno = errno;
        free (line_buf);
        errno = stored_errno;
        return NULL;
    }
    list = (gid_t *) malloc(ngroups * sizeof(gid_t));

    /* Get the secondary group IDs for root */
    if (getgroups(ngroups, list) < 0)
    {
        stored_errno = errno;
        free (line_buf);
        errno = stored_errno;
        return NULL;
    }

    /* Make a oneliner out of all the secondary GIDs */
    for (t = 0; t < ngroups; t++)
    {
        /* Secondary Group ID */
        if (g = getgrgid (list[t]))
            snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",sgid=%d(%s)", list[t], g -> gr_name);
        else
            snprintf (&line_buf[strlen(line_buf)], 1024 - strlen(line_buf), ",sgid=%d", list[t]);
    }

    /* Memory liberation */
    if (list) free(list);
    list=(gid_t *) NULL;

    return line_buf;
}



/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    int i, j;
    char * logstr = "lcmaps_plugin_posix_enf-plugin_initialize()";


    lcmaps_log_debug(5,"%s: passed arguments:\n", logstr);
    for (i=0; i < argc; i++)
    {
       lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++)
    {
        /* setting maxuid parameter from init plugin arguments */
        if ( (((strcmp(argv[i], "-maxuid") == 0) ||
             (strcmp(argv[i], "-MAXUID") == 0)) &&
             (maxuid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"-maxuid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 { 
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxuid = -1;
                         goto fail_posix;
                     }
                 }
                 maxuid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_posix;
            }
            i++;
        }

        /* setting maxpgid parameter from init plugin arguments */
        else if ( (((strcmp(argv[i], "-maxpgid") == 0) ||
             (strcmp(argv[i], "-MAXPGID") == 0)) &&
             (maxpgid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"-maxpgid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is at least one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxpgid = -1;
                         goto fail_posix;
                     }
                 }
                 maxpgid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_posix;
            }
            i++;
        }

        /* setting maxsgid parameter from init plugin arguments */
        else if  ( (((strcmp(argv[i], "-maxsgid") == 0) ||
             (strcmp(argv[i], "-MAXSGID") == 0)) &&
             (maxsgid == MAX_UNDEFINED))
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 lcmaps_log_debug(5,"%s: Checking if argument behind \"-maxsgid\" is a number\n", logstr);
                 for (j = 0; j < strlen(argv[i + 1]); j++)
                 {
                     if (!isdigit((argv[i + 1])[j]))
                     {
                         lcmaps_log(LOG_ERR,"%s: Error: maybe found some digits, but there is atleast one char corrupting this parameter: %s\n", logstr, argv[i + 1]);
                         maxsgid = -1;
                         goto fail_posix;
                     }
                 }
                 maxsgid = atoi(argv[i + 1]);
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_posix;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-set_only_euid") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     set_only_euid = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     set_only_euid = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     goto fail_posix;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_posix;
            }
            i++;
        }
        else if  ( (strcmp(argv[i], "-set_only_egid") == 0)
             && (i + 1 < argc) )
        {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0))
            {
                 if (strcmp(argv[i+1],"yes") == 0)
                 {
                     set_only_egid = 1;
                 }
                 else if (strcmp(argv[i+1],"no") == 0)
                 {
                     set_only_egid = 0;
                 }
                 else
                 {
                     lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n", logstr, argv[i]);
                     goto fail_posix;
                 }
            }
            else
            {
                lcmaps_log(LOG_ERR,"%s: no argument found for %s (failure)\n", logstr, argv[i]);
                goto fail_posix;
            }
            i++;
        }
    }

    if (maxsgid > NGROUPS)
    {
        lcmaps_log(LOG_ERR,"%s: Error: The prefered set maximum of %d Secondary Gid's exceeds the system maximum of NGROUPS witch is set to %d on this system\n", logstr, maxsgid, NGROUPS);
        goto fail_posix;
    }
    else if (maxsgid == MAX_UNDEFINED)
    {
        lcmaps_log(LOG_INFO,"%s: Auto set maximum Secondary Gid's to system maximum of NGROUPS witch is set to %d on this system\n", logstr, NGROUPS);
    }


    lcmaps_log_debug(4,"%s: Summary init maxuid  : %d\n", logstr, maxuid);
    lcmaps_log_debug(4,"%s: Summary init maxpgid : %d\n", logstr, maxpgid);
    lcmaps_log_debug(4,"%s: Summary init maxsgid : %d\n", logstr, maxsgid);

    return LCMAPS_MOD_SUCCESS;


 fail_posix:
    return LCMAPS_MOD_FAIL; 
} 

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char * logstr = "lcmaps_plugin_posix_enf-plugin_introspect()";  

    static lcmaps_argument_t argList[] = {
        {NULL           ,       NULL            , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)

{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
+Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char * logstr = "lcmaps_plugin_posix_enf-plugin_run()";  
    char * buffer = NULL;
    int                 i, t; 
    gid_t *             list        = NULL;
    int                 ngroups     = 0;
 
    char  *             dn          = NULL;
    uid_t *             uid         = NULL;
    int                 cntUid;
    gid_t *             priGid      = NULL;
    int                 cntPriGid;
    gid_t *             secGid      = NULL;
    int                 cntSecGid;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_posix_enf-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_posix_enf-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_posix_enf-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_posix;
    }

    uid    = getCredentialData(UID,     &cntUid);
    priGid = getCredentialData(PRI_GID, &cntPriGid);
    secGid = getCredentialData(SEC_GID, &cntSecGid);
    /* lcmaps_log_debug(2,"%s: number of uids: %d, priGids: %d, secGids: %d\n", logstr, cntUid,cntPriGid,cntSecGid); */

    /* Check amount of uid's, pri-gid's and sec-gid's with the set maximum */
    if (maxuid != MAX_UNDEFINED)
    {
        /* lcmaps_log_debug(2,"%s: max number of uids: %d\n", logstr, maxuid); */
        if (cntUid > maxuid)
        {
            lcmaps_log(LOG_ERR, "%s: Error: The set amount of uid's gathered exceeds the maximum of %d uid('s) by %d\n", logstr, maxuid, (cntUid - maxuid));
            goto fail_posix;
        }
    }
    if (maxpgid != MAX_UNDEFINED)
    {
        /* lcmaps_log_debug(2,"%s: max number of primary gid('s): %d\n", logstr, maxpgid); */
        if (cntPriGid > maxpgid)
        {
            lcmaps_log(LOG_ERR, "%s: Error: The set amount of primary gid's gathered exceeds the maximum of %d primary gid('s) by %d\n", logstr, maxpgid, (cntPriGid - maxpgid));
            goto fail_posix;
        }
    }
    if (maxsgid != MAX_UNDEFINED)
    {
        /* lcmaps_log_debug(2,"%s: max number of secondary gid's: %d\n", logstr, maxsgid); */
        if (cntSecGid > maxsgid)
        {
            lcmaps_log(LOG_ERR, "%s: Error: The set amount of secondary gid's gathered exceeds the maximum of %d secunadary gid's by %d\n", logstr, maxsgid, (cntSecGid - maxsgid));
            goto fail_posix;
        }
    }


    /* You must have root privileges to continue making the identity switch */
    if (geteuid() != 0)
    {
        lcmaps_log(LOG_ERR, "%s: The tool or service doesn't have root privileges (any more). -> current real UID %d, effective UID %d\n", logstr, getuid(), geteuid());
        goto fail_posix;
    }


    /* It doesn't make sense to record the previous ID if we're in a root started service or tool */
    /* We skip the recording of the current ID, when root */
    if (getuid() != 0)
    {
        /* Make a pre ID switching log message */
        buffer = cgul_process_identity_oneline();
        lcmaps_log (LOG_INFO, "%s:  pre-id-switch: %s\n", logstr, buffer);

        /* Memory liberation */
        free(buffer);
        buffer = NULL;
    }


    /* Start ID Switching */


    /* Set new primary group ID */
    if (cntPriGid > 0)
    {
        if (set_only_egid)
            if (setregid(-1, priGid[0]) != 0)
            {
                lcmaps_log(LOG_ERR, "%s: cannot set effective gid by setregid(): %s\n", logstr, strerror(errno));
            }
            else
            {
                lcmaps_log_debug(1,"%s: Setting only effective primary gid to %d\n", logstr, (int) priGid[0]);
            }
        else
            if (setregid(priGid[0], priGid[0]) != 0)
            {
                lcmaps_log(LOG_ERR, "%s: cannot set real and effective setregid(): %s\n", logstr, strerror(errno));
            }
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: No primary group IDs found, need at least 1 !\n", logstr);
        goto fail_posix;
    }



    /* Set the new secondary group IDs */
    if (setgroups(cntSecGid, secGid)!=0)
    {
        switch (errno)
        {
            case EFAULT :
                {
                    lcmaps_log_debug(LOG_NOTICE, "%s: Not that fatal but serious error: %s\n", logstr, strerror(errno));
                    goto fail_posix;
                    break;
                }
            case EPERM :
                {
                    lcmaps_log(LOG_ERR, "%s: You are not ROOT: %s\n", logstr, strerror(errno));
                    goto fail_posix;
                    break;
                }
            case EINVAL :
                {
                    for (i = 0; i < cntSecGid; i++)
                    {
                        lcmaps_log_debug(1,"%s: > i = %d with %d\n", logstr, i, secGid[i]);
                    }
                    lcmaps_log_debug(1, "%s: Invalid GID list\n", logstr);
                    lcmaps_log(LOG_ERR, "%s: %s\n", logstr, strerror(errno));
                    goto fail_posix;
                    break;
                }
            default :
                {
                    lcmaps_log(LOG_ERR, "%s: Unspecified error in setgroups()\n", logstr);
                    goto fail_posix;
                }
        }
    }

    /* Set the new user ID */
    if (cntUid > 0)
    {
        if (set_only_euid)
            if (setreuid(-1, uid[0]) != 0)
            {
                lcmaps_log(LOG_ERR, "%s: cannot set effective uid by setreuid(): %s\n", logstr, strerror(errno));
            }
            else
            {
                lcmaps_log_debug(1,"%s: Setting only effective uid to %d\n", logstr, (int) uid[0]);
            }
        else
            if (setreuid(uid[0], uid[0]) != 0)
            {
                lcmaps_log(LOG_ERR, "%s: cannot setreuid(): %s\n", logstr, strerror(errno));
            }
    }
    else
    {
        lcmaps_log(LOG_ERR, "%s: No user IDs found, need at least 1 !\n", logstr);
        goto fail_posix;
    }



    /*
     * Do Not End This POSIX Enforment with ROOT privileges
     * (depending on the set_only_euid and set_only_egid options)
     */
    if ( (!set_only_euid) && ((getuid() == 0) || (geteuid() == 0)) )
    {
        lcmaps_log(LOG_ERR, "%s: real and or effective uid == 0, which is not allowed at this stage\n", logstr);
        goto fail_posix;
    }
    else if (geteuid() == 0)
    {
        lcmaps_log(LOG_ERR, "%s: effective uid == 0, which is not allowed at this stage\n", logstr);
        goto fail_posix;
    }

    if ( (!set_only_egid) && ((getgid() == 0) || (getegid() == 0)) )
    {
        lcmaps_log(LOG_ERR, "%s: real and or effective gid == 0, which is not allowed at this stage\n", logstr);
        goto fail_posix;
    }
    else if (getegid() == 0)
    {
        lcmaps_log(LOG_ERR, "%s: effective gid == 0, which is not allowed at this stage\n", logstr);
        goto fail_posix;
    }


    /* Done with ID Switching */


    /* Make aa post ID switching log message */
    buffer = cgul_process_identity_oneline();
    lcmaps_log (LOG_INFO, "%s: post-id-switch: %s\n", logstr, buffer);

    /* Memory liberation */
    free(buffer);
    buffer = NULL;


    /* succes */
 success_posix:
    if (list) free(list);
    lcmaps_log(LOG_INFO,"%s: posix_enf plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_posix:
    if (list) free(list);
    lcmaps_log(LOG_INFO,"%s: posix_enf plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char * logstr = "lcmaps_plugin_posix_enf-plugin_introspect()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps-plugins-basic/src/posix_enf/lcmaps_posix.c,v $
    $Date: 2010-02-25 09:30:34 $
    $Revision: 1.13 $
    $Author: okoeroo $
******************************************************************************/
