/**
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2012-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */


/*!
    \page lcmaps_ban_dn.mod ban_dn plugin

    \section ban_dnsection1 SYNOPSIS
    \b lcmaps_ban_dn.mod
        [-banmapfile \<location banmapfile\>]

    \section ban_dnsection2 DESCRIPTION

    This plugin is an Banning Plugin and will provide the LCMAPS system a credential banning feature based on the Distinghuished Name (DN).
    A banmapfile will be read, if the DN is listed, this means the DN is banned and the plug-in will register a negative result. This means a LCMAPS_MOD_FAIL.

    \section ban_dnoptions OPTIONS
    \subsection ban_dnoptie1 -banmapfile <banmapfile>
        When this option is set it will override the default path of the banmapfile.
        It is advised to use an absolute path to the banmapfile to avoid usage of the wrong file(path).

\section ban_dnReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section ban_dnErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

\section ban_dnSeeAlso SEE ALSO
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod"
*/

/*!
    \file   lcmaps_ban_dn.c
    \brief  Interface to the LCMAPS plugins
    \author Oscar Koeroo

    This file contains the code for ban_dn plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/*****************************************************************************
                            Include header files
******************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <sys/stat.h>

#include "lcmaps_plugins_basic_config.h"
#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>
#include "lcmaps_gridlist.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define PLUGIN_RUN      0
#define PLUGIN_VERIFY   1

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static int plugin_run_or_verify(int, lcmaps_argument_t *, int);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

static char *banmapfile = NULL;
static int   disable_wildcardmatching = 0;


/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(
        int argc,
        char ** argv
)
{
    char * logstr = "lcmaps_plugin_ban_dn-plugin_initialize()";
    int i;
    struct stat ban_stat;

    lcmaps_log_debug(5,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++) {
        lcmaps_log_debug(5,"%s: arg %d is %s\n", logstr, i, argv[i]);
    }

    /*
     * the first will be the thing to edit/select (gridmap(file))
     * the second will be the path && filename of the banmapfile
     */

    /*
     * Parse arguments, argv[0] = name of plugin, so start with i = 1
     */
    for (i = 1; i < argc; i++) {
        if ( ((strcmp(argv[i], "-gridmap") == 0) ||
              (strcmp(argv[i], "-GRIDMAP") == 0) ||
              (strcmp(argv[i], "-gridmapfile") == 0) ||
              (strcmp(argv[i], "-banmapfile") == 0) ||
              (strcmp(argv[i], "-BANMAPFILE") == 0) ||
              (strcmp(argv[i], "-ban_mapfile") == 0) ||
              (strcmp(argv[i], "-BAN_MAPFILE") == 0) ||
              (strcmp(argv[i], "-GRIDMAPFILE") == 0))
             && (i + 1 < argc)) {
            if ((argv[i + 1] != NULL) && (strlen(argv[i + 1]) > 0)) {
                if ( (banmapfile = strdup(argv[i + 1])) == NULL)
		    goto fail_init_module;
                if (stat(banmapfile, &ban_stat))    {
		    lcmaps_log(LOG_ERR, "%s: Error in reading the ban file \"%s\".\n", logstr, banmapfile);
		    goto fail_init_module;
		}
		lcmaps_log(LOG_DEBUG, "%s: Using ban file \"%s\".\n", logstr, banmapfile);
            }
            i++;
        } else if ((strcmp(argv[i], "-disablewildcard") == 0) ||
                   (strcmp(argv[i], "--disablewildcard") == 0) ||
                   (strcmp(argv[i], "-disablewildcards") == 0) ||
                   (strcmp(argv[i], "--disablewildcards") == 0)) {
            disable_wildcardmatching = 1;
        } else {
            lcmaps_log(LOG_ERR,"%s: Error in initialization parameter: %s (failure)\n", logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Post check */
    if (!banmapfile) {
        lcmaps_log(LOG_ERR,"%s: Error: No ban (grid)mapfile was provided. Use the option -banmapfile <banmapfile> to set one.\n", logstr);
        goto fail_init_module;
    }

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    if (banmapfile) {
        free(banmapfile);
        banmapfile = NULL;
    }
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(
        int * argc,
        lcmaps_argument_t ** argv
)
{
    char * logstr = "lcmaps_plugin_ban_dn-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
        {"user_dn"      ,       "char *"        , 1,   NULL},
        {NULL           ,       NULL            , -1,   NULL}
    };

    lcmaps_log_debug(4,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log_debug(5,"%s: address first argument: 0x%x\n", logstr, argList);

    return LCMAPS_MOD_SUCCESS;
}


/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().
    In this particular case "plugin_verify()" is identical to "plugin_run()"

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(
        int argc,
        lcmaps_argument_t * argv
)
{
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

static int plugin_run_or_verify(
        int argc,
        lcmaps_argument_t * argv,
        int lcmaps_mode
)
{
    char *              logstr      = "lcmaps_plugin_ban_dn-plugin_run()";
    char *              dn          = NULL;
    int                 dn_cnt      = 0;
    char *              username    = NULL;
    struct passwd       *user_info  = NULL;
    int                 i           = 0;
    int                 cnt_sec_gid = 0;
    gid_t *             sec_gid     = NULL;
    int                 rc          = 0;

    /*
     * The beginning
     */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = "lcmaps_plugin_ban_dn-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = "lcmaps_plugin_ban_dn-plugin_verify()";
    else
    {
        lcmaps_log(LOG_ERR, "lcmaps_plugin_ban_dn-plugin_run_or_verify(): attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_ban_dn;
    }

    /*
     * Try to get the ordered values:
     */
    if ( ( dn = *(char **) lcmaps_getArgValue("user_dn", "char *", argc, argv) ) )
    {
        lcmaps_log_debug(1,"%s: found dn: %s\n", logstr, dn);

        /* Do I already have a DN registered in LCMAPS? If not registered, push it to the end-result registry */
        getCredentialData (DN, &dn_cnt);
        if (dn_cnt == 0)
        {
            lcmaps_log_debug (5, "%s: Adding DN: %s\n", logstr, dn);
            addCredentialData(DN, &dn);
        }
    }
    else
        lcmaps_log(LOG_ERR,"%s: could not get value of dn !\n", logstr);


    /*
     * Check the banmapfile
     */

    if ((banmapfile != NULL) && (strlen(banmapfile) > 0))
        lcmaps_log(LOG_INFO, "%s: banmapfile is: %s\n", logstr, banmapfile);
    else
    {
        if (banmapfile) {
	    free(banmapfile);
	    banmapfile = NULL;
	}
        lcmaps_log(LOG_ERR, "%s: Error: No banmapfile assigned.\n", logstr);
        goto fail_ban_dn;
    }

    /*
     * Try to find the dn in the banmapfile
     */

    if (disable_wildcardmatching) {
        rc = lcmaps_gridlist(dn, &username, banmapfile, MATCH_ONLY_DN, NULL, NULL);
    } else {
        rc = lcmaps_gridlist(dn, &username, banmapfile, MATCH_WILD_CHARS|MATCH_ONLY_DN, NULL, NULL);
    }

    switch (rc) {
        case LCMAPS_MOD_NOFILE :
            lcmaps_log(LOG_ERR, "%s: Could not find the banmapfile %s\n", logstr, banmapfile);
            goto fail_ban_dn;
        case LCMAPS_MOD_ENTRY :
            lcmaps_log(LOG_NOTICE,"%s: DN \"%s\" is found on the ban list.\n", logstr, dn);
            goto fail_ban_dn;
        case LCMAPS_MOD_NOENTRY :
            lcmaps_log_debug(1,"%s: No entry found in the ban list. Nothing banned, moving along.\n", logstr);
            goto success_ban_dn;
        default :
            lcmaps_log(LOG_ERR,"%s: Something unexpected went wrong. lcmaps_gridlist() returned %d\n", logstr, rc);
            goto fail_ban_dn;
    }


    /* succes */
 success_ban_dn:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: ban_dn plugin succeeded\n", logstr);
    return LCMAPS_MOD_SUCCESS;

 fail_ban_dn:
    if (username) free(username);
    lcmaps_log(LOG_INFO,"%s: ban_dn plugin failed\n", logstr);
    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate()
{
    char * logstr = "lcmaps_plugin_ban_dn-plugin_terminate()";

    lcmaps_log_debug(4,"%s: terminating\n", logstr);

    if (banmapfile) {
        free(banmapfile);
        banmapfile=NULL;
    }

    return LCMAPS_MOD_SUCCESS;
}

