/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcas_log.h
    \brief  Logging API for the LCAS plugins and LCAS itself
    \author Martijn Steenbakkers for the EU DataGrid.

    This header contains the declarations of the LCAS logging functions
    The LCAS plugins can use this API to write output to the LCAS logging
    devices.
    -# lcas_log(): Log to LCAS logging devices.
    -# lcas_log_debug(): Produce debugging output.
    \ingroup APIforLcasPlugins
*/

#ifndef LCAS_LOG_H
#define LCAS_LOG_H

/******************************************************************************
                             Include header files
******************************************************************************/
#include <syslog.h>

/******************************************************************************
 *                            Module definition
 *****************************************************************************/
/******************************************************************************
Function:       lcas_log()
Description:    Log information to file and or syslog
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log(
        int prty,
        char * fmt,
        ...
        )
    \brief log information
    
    This function does the logging for the LCAS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty syslog priority (if 0 don't syslog).
    \param fmt  string format
    \param ...  variable argument list

    \retval 0 succes.
    \retval 1 failure.
*/
extern int lcas_log(
        int prty,
        char * fmt,
        ...
);

/******************************************************************************
Function:       lcas_log_debug()
Description:    Print debugging information
Parameters:
                debug_lvl: debugging level
                fmt:       string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_debug(
        int debug_lvl,
        char * fmt,
        ...
        )
    \brief Print debugging information

    This function prints debugging information (using lcas_log with priority 0)
    provided debug_lvl <= DEBUG_LEVEL (default is 0).

    \param debug_lvl debugging level
    \param fmt       string format
    \param ...       variable argument list

    \retval 0 succes.
    \retval 1 failure.
*/
extern int lcas_log_debug(
        int debug_lvl,
        char * fmt,
        ...
);

/******************************************************************************
Function:       lcas_log_time()
Description:    Log information to file and or syslog with a timestamp
Parameters:
                prty:    syslog priority (if 0 don't syslog)
                fmt:     string format
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_time(
        int prty,
        char * fmt,
        ...
        )
    \brief log information with timestamp
 
    This function logs information with a timestamp for LCAS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.
 
    \param prty syslog priority (if 0 don't syslog).
    \param fmt  string format
    \param ...  variable argument list
 
    \retval 0 succes.
    \retval 1 failure.
*/
extern int lcas_log_time(
        int prty, 
        char * fmt, 
        ...
);

/******************************************************************************
Function:       lcas_log_a_string()
Description:    Log a string according to the passed format to file and or syslog
Parameters:
                prty:       syslog priority (if 0 don't syslog)
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string(
        int prty,
        char * fmt,
        char * the_string
        )
    \brief log information
    
    This function logs information for LCAS and its plugins.
    Syslog() is called with the specified priority. No syslog() is done if the
    priority is 0.

    \param prty         syslog priority (if 0 don't syslog).
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
extern int lcas_log_a_string(
        int prty,
        char * fmt,
        char * the_string
);

/******************************************************************************
Function:       lcas_log_a_string_debug()
Description:    Print debugging information
Parameters:
                debug_lvl:  debugging level
                fmt:        string format
                the_string: the string
Returns:        0 succes
                1 failure
******************************************************************************/
/*!
    \fn lcas_log_a_string_debug(
        int debug_lvl,
        char * fmt,
        char * the_string
        )
    \brief Print debugging information

    This function prints debugging information (using lcas_log with priority 0)
    provided debug_lvl <= DEBUG_LEVEL (default is 0).

    \param debug_lvl    debugging level
    \param fmt          string format
    \param the_string   the string

    \retval 0 succes.
    \retval 1 failure.
*/
extern int lcas_log_a_string_debug(
        int debug_lvl,
        char * fmt,
        char * the_string
);

/******************************************************************************
Function:       lcas_get_debug_level()
Description:    Retrieve the debug_level
Parameters:
Returns:        the debug_level
******************************************************************************/
/*!
    \fn lcas_get_debug_level(
        )
    \brief Retrieve the debug_level
    \return the debug_level
*/
extern int lcas_get_debug_level(void);

#endif /* LCAS_LOG_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcas/interface/lcas_log.h,v $
    $Date: 2010-05-03 10:43:18 $
    $Revision: 1.5 $
    $Author: okoeroo $
******************************************************************************/
