package latexDraw.util;

import java.awt.Color;
import java.util.Vector;

import latexDraw.figures.Dot;
import latexDraw.figures.Draw;
import latexDraw.figures.Figure;
import latexDraw.figures.Line;
import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.dialog.ExceptionFrameDialog;


/** 
 * A simple way to implement undo/redo management (contrary to java :s)<br>
 * <br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE.  See the GNU General Public License for more details.<br>
 *<br>
 * 
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class UndoRedoManager 
{
	/** This label corresponds to a change of parameters of a figure */
	public static final String LABEL_CHANGE_PARAM = LaTeXDrawLang.getOthersString("UndoRedoManager.changeParams"); //$NON-NLS-1$

	/** This label corresponds to a change in the disposition of 
	 * some figures (i.e. put in background, foreground, ...)*/
	public static final String LABEL_CHANGE_DISPOSITION = LaTeXDrawLang.getOthersString("UndoRedoManager.changeDispo"); //$NON-NLS-1$
	
	public static final String LABEL_TRANSFORM  = LaTeXDrawLang.getString1_8("UndoRedoManager.0"); //$NON-NLS-1$
	
	public static final String LABEL_ALIGN  = LaTeXDrawLang.getOthersString("UndoRedoManager.align"); //$NON-NLS-1$
	
	public static final String LABEL_DISTRIB  = LaTeXDrawLang.getOthersString("UndoRedoManager.distrib"); //$NON-NLS-1$
	
	/** This label corresponds to a change of position of a figure */
	public static final String LABEL_CHANGE_MOVE  = LaTeXDrawLang.getOthersString("UndoRedoManager.move"); //$NON-NLS-1$
	
	/** This label corresponds to the deletion of a figure */
	public static final String LABEL_DEL          = LaTeXDrawLang.getOthersString("UndoRedoManager.delete"); //$NON-NLS-1$
	
	/** This label corresponds to the creation of a figure */
	public static final String LABEL_CREATE       = LaTeXDrawLang.getOthersString("UndoRedoManager.create"); //$NON-NLS-1$
	
	/** This label corresponds to a join of several figures */
	public static final String LABEL_JOIN = LaTeXDrawLang.getOthersString("UndoRedoManager.join"); //$NON-NLS-1$
	
	/** This label corresponds to a separation of several joined figures */
	public static final String LABEL_SEPARATE = LaTeXDrawLang.getOthersString("UndoRedoManager.seperate"); //$NON-NLS-1$
	
	/** This label corresponds to a change of the attribute fill
	 * of the class figure */
	public static final String LABEL_CHANGE_FILL  = LaTeXDrawLang.getOthersString("UndoRedoManager.filled"); //$NON-NLS-1$

	/** This label corresponds to a change of colour of the interior
	 * of the figure */
	public static final String LABEL_CHANGE_INT_COL = LaTeXDrawLang.getOthersString("UndoRedoManager.intCol"); //$NON-NLS-1$
	
	/** This label corresponds to a change of colour of the borders
	 * of the figure */
	public static final String LABEL_CHANGE_BORD_COL = LaTeXDrawLang.getOthersString("UndoRedoManager.bordCol"); //$NON-NLS-1$
	
	/** This label corresponds to a change of thickness of the figure */
	public static final String LABEL_CHANGE_THICKNESS = LaTeXDrawLang.getOthersString("UndoRedoManager.thick"); //$NON-NLS-1$
	
	/** This label corresponds to a change of style the figure */
	public static final String LABEL_CHANGE_STYLE = LaTeXDrawLang.getOthersString("UndoRedoManager.style"); //$NON-NLS-1$
	
	/** This label corresponds to a change of arrow style of the line */
	public static final String LABEL_CHANGE_LARROW_STYLE = LaTeXDrawLang.getOthersString("UndoRedoManager.lArrow"); //$NON-NLS-1$
	
	/** This label corresponds to a change of arrow style of the line */
	public static final String LABEL_CHANGE_RARROW_STYLE = LaTeXDrawLang.getOthersString("UndoRedoManager.rArrow"); //$NON-NLS-1$
	
	/** This label corresponds to a change of the style of the dot */
	public static final String LABEL_CHANGE_DOT_STYLE = LaTeXDrawLang.getOthersString("UndoRedoManager.dotStyle"); //$NON-NLS-1$
	
	/** The label display when there is no redo possible */
	public static final String EMPTY_REDO = LaTeXDrawLang.getOthersString("UndoRedoManager.noRedo"); //$NON-NLS-1$
	
	/** The label display when there is no undo action */
	public static final String EMPTY_UNDO = LaTeXDrawLang.getOthersString("UndoRedoManager.noUndo"); //$NON-NLS-1$
	
	/** This vector contains all labels of redo actions */
	protected Vector<String> textRedo;
	
	/** This vector contains positions of figures, in the vector figures,
	 * concerned by the redo */
	protected Vector<int[]> idRedo;
	
	/** This vector contains all figures, values, ... concerned by the
	 * redo */
	protected Vector<Object> objectRedo;
	
	/** This vector contains all labels of undo actions */
	protected Vector<String> textUndo;
	
	/** This vector contains positions of figures, in the vector figures,
	 * concerned by the undo */
	protected Vector<int[]> idUndo;
	
	/** This vector contains all figures, values, ... concerned by the
	 * undo */
	protected Vector<Object> objectUndo;
	
	/** This vector contains all the figures of the drawing : in this case, this
	 * vector is pointer of the vector "figures" of the class DrawContainer */
	protected Vector<Figure> figures;
	
	/** The current drawing which contains the figures */
	protected DrawContainer draw;
	
	/** The main frame (which contains buttons, ...) */
	protected LaTeXDrawFrame mainFrame;
	
	
	
	
	
	/**
	 * The constructor by default
	 * @param f The vector containing all the figures of the drawing
	 */
	public UndoRedoManager(Vector<Figure> f, DrawContainer d, LaTeXDrawFrame frame)
	{
		textRedo   = new Vector<String>();
		idRedo     = new Vector<int[]>();
		objectRedo = new Vector<Object>();
		textUndo   = new Vector<String>();
		idUndo     = new Vector<int[]>();
		objectUndo = new Vector<Object>();
		
		mainFrame = frame;
		figures = f;
		draw = d;
	}
	
	
	
	public void setFigures(Vector<Figure> f)
	{
		if(f!=null)
			figures = f;
	}
	
	
	
	
	/**
	 * Allows to get the name of the next undo actions
	 * @return The name of the next undo actions
	 */
	public String getTextNextUndo()
	{
		if(textUndo.isEmpty())
			return null;
		return textUndo.lastElement();
	}
	

	
	

	/**
	 * Allows to get the name of the next redo actions
	 * @return The name of the next redo actions
	 */
	public String getTextNextRedo()
	{
		if(textRedo.isEmpty())
			return null;
		return textRedo.lastElement();
	}
	
	
	
	
	/**
	 * Allows to add a event to manage by undo/redo
	 * @param label The kind of event (add, deletion, rotation, ...)
	 * @param id The position of the concerned figure in the vector "figures"
	 * @param o For some actions (like deletion) we need to have to concerned
	 * figure in order to keep it until a redo
	 */
	public void add(String label, int id, Object o)
	{
		int[] id2 = new int[1];
		id2[0] = id;
		add(label, id2, o, false);
	}
	
	
	
	
	
	/**
	 * Allows to add a event to manage by undo/redo
	 * @param label The kind of event (add, deletion, rotation, ...)
	 * @param id The indices of the concerned figures in the vector "figures"
	 * @param o For some actions (like deletion) we need to have to concerned
	 * figure in order to keep it until a redo.
	 * @param fromRedo True if the call of the function comes from the method redo (useful to clear the redo).
	 */
	public void add(String label, int[] id, Object o, boolean fromRedo)
	{
		try
		{
			if(!fromRedo)
				clearRedo();
			
			for(int i=0, size=id.length; i<size; i++)
				if(id[i]<0)
					throw new ArrayIndexOutOfBoundsException(id[i]);

			if(id.length==0)
				throw new IllegalArgumentException();

			if(isValidLabel(label))
			{
				if(label.equals(LABEL_CREATE) || label.equals(LABEL_CHANGE_FILL)
						|| label.equals(LABEL_JOIN))
				{
					textUndo.add(label);
					idUndo.add(id);
				}else
				if(label.equals(LABEL_DEL) || label.equals(LABEL_CHANGE_RARROW_STYLE) || 
					label.equals(LABEL_CHANGE_LARROW_STYLE) || 
					label.equals(LABEL_CHANGE_BORD_COL) || label.equals(LABEL_CHANGE_DOT_STYLE) || 
					label.equals(LABEL_CHANGE_THICKNESS) || label.equals(LABEL_CHANGE_INT_COL) || 
					label.equals(LABEL_CHANGE_MOVE) || label.equals(LABEL_CHANGE_PARAM) || 
					label.equals(LABEL_CHANGE_STYLE) || label.equals(LABEL_CHANGE_DISPOSITION) ||
					label.equals(LABEL_SEPARATE) || label.equals(LABEL_TRANSFORM) || label.equals(LABEL_ALIGN) || 
					label.equals(LABEL_DISTRIB))
				{
					textUndo.add(label);
					idUndo.add(id);
					objectUndo.add(o);					
				}
				else
					throw new IllegalArgumentException();
			}//if(isValidLabel(label))
			else
				throw new IllegalArgumentException();
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}

	}
	
	
	
	
	/**
	 * Undo an action
	 */
	public void undo()
	{
		try
		{
			if(!textUndo.isEmpty() && !idUndo.isEmpty())
			{
				String label = textUndo.remove(textUndo.size()-1);
				int[] id = idUndo.remove(idUndo.size()-1);
				int i;
				
				if(label.equals(LABEL_CHANGE_DISPOSITION))
				{
					Object o = objectUndo.remove(objectUndo.size()-1);	
					
					if(!(o instanceof int[]))
						throw new ClassCastException(o.getClass().getName());
					
					if(((int[])o).length!=id.length)
						throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);						
					
					int[] oldId =  (int[])o;
					Figure f;
					for(i=0; i<id.length; i++)
					{
						f = figures.remove(id[i]);
						figures.add(oldId[i], f);
					}
					objectRedo.add(id);
					id = oldId;					
				}else				
				if(label.equals(LABEL_CREATE))
				{
					if(draw.getSelected().contains(figures.elementAt(id[0])))
					{
						mainFrame.setSelection(false);
						draw.setSelected(null);
					}
					objectRedo.add(figures.remove(id[0]));
				}
				else
				if(label.equals(LABEL_DEL))
				{
					Object o = objectUndo.remove(objectUndo.size()-1);	
					
					if(!(o instanceof Figure))
						throw new ClassCastException(o.getClass().getName());
					
					draw.getSelected().setSelected(false);
					draw.getSelected().clear();
					if(id.length==1)
					{
						Figure f;
						if(o instanceof Draw) f = ((Draw)o).getFigureAt(0);
						else f = (Figure)o;
						figures.add(id[0], f);		
						f.setSelected(true);
						draw.setSelected(f);
						mainFrame.setSelection(true);
					}
					else
					{
						if(!(o instanceof Draw))
							throw new ClassCastException();
						Draw d = (Draw)o;	
						
						if(id.length!=d.size())
							throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);
						
						for(i=0; i<id.length; i++)
							figures.add(id[i], d.getFigureAt(i));
						
						d.setSelected(true);
						draw.setSelected(d);
						mainFrame.setSelection(true);
					}
				}
				else
				if(label.equals(LABEL_CHANGE_FILL))
					figures.elementAt(id[0]).setIsFilled(!figures.elementAt(id[0]).isFilled());		
				else
				if(label.equals(LABEL_CHANGE_RARROW_STYLE))
				{
					Line l = (Line)figures.elementAt(id[0]);
					String style = l.getArrow2Style();
					l.setArrow2Style((String)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(style);
				}
				else
				if(label.equals(LABEL_CHANGE_LARROW_STYLE))
				{
					Line l = (Line)figures.elementAt(id[0]);
					String style = l.getArrow1Style();
					l.setArrow1Style((String)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(style);
				}
				else
				if(label.equals(LABEL_CHANGE_DOT_STYLE))
				{
					Dot d = (Dot)figures.elementAt(id[0]);
					String style = d.getCurrentStyle();
					d.setCurrentStyle((String)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(style);
				}
				else
				if(label.equals(LABEL_CHANGE_INT_COL))
				{
					Figure f = figures.elementAt(id[0]);
					Color c = f.getInteriorColor();
					f.setInteriorColor((Color)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(c);
				}
				else
				if(label.equals(LABEL_CHANGE_BORD_COL))
				{
					Figure f = figures.elementAt(id[0]);
					Color c = f.getLinesColor();
					f.setLinesColor((Color)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(c);
				}else
				if(label.equals(LABEL_CHANGE_MOVE) || label.equals(LABEL_TRANSFORM) || label.equals(LABEL_ALIGN) ||
					label.equals(LABEL_DISTRIB))
				{
					Object o = objectUndo.remove(objectUndo.size()-1);
					if(!(o instanceof Draw)) throw new ClassCastException();

					Draw fOld = (Draw)o;
		
					if(id.length != fOld.size()) 
						throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);
						
					Draw d = new Draw(false, false);
					int size = fOld.size();
					
					for(i=0; i<size; i++)
					{
						d.addFigure(figures.remove(id[i]));
						figures.add(id[i], fOld.getFigureAt(i));
					}

					draw.getSelected().setSelected(false);
					draw.getSelected().clear();
					fOld.setSelected(true);
					draw.setSelected(fOld);
					objectRedo.add(d);
				}else
				if(label.equals(LABEL_CHANGE_PARAM))
				{
					Object o = objectUndo.remove(objectUndo.size()-1);

					if(!(o instanceof Figure))
							throw new ClassCastException();
					
					Figure fOld;
					if(o instanceof Draw)
						fOld = ((Draw)o).getFigureAt(0);
					else
						fOld = (Figure)o;
						
					Figure f = figures.elementAt(id[0]);
					figures.remove(id[0]);
					figures.add(id[0], fOld);
					
					draw.getSelected().setSelected(false);
					draw.getSelected().clear();
					fOld.setSelected(true);
					draw.setSelected(fOld);

					objectRedo.add(f);
				}
				else
				if(label.equals(LABEL_CHANGE_STYLE))
				{
					Figure f = figures.elementAt(id[0]);
					String style = f.getLineStyle();
					f.setLineStyle((String)objectUndo.remove(objectUndo.size()-1));
					objectRedo.add(style);
				}
				else
				if(label.equals(LABEL_CHANGE_THICKNESS))
				{
					Figure f = figures.elementAt(id[0]);
					float thickness;
					if(f instanceof Dot)
					{
						thickness = ((Dot)f).getWidth();
						  ((Dot)f).setWidth((Float)objectUndo.remove(objectUndo.size()-1));
					}
					else  
					{
						thickness = f.getThickness();
						f.setThickness((Float)objectUndo.remove(objectUndo.size()-1));
					}
					objectRedo.add(thickness);
				}else
				if(label.equals(LABEL_JOIN))
				{	
					int min = id[0];
					i = 1;
					
					while(i<id.length)
					{
						if(id[i]<min)
							min = id[i];
						i++;
					}
				
					Draw d = (Draw)figures.remove(min);
					
					for(i=id.length-1; i>=0; i--)
						if(figures.size()==0 || id[i]==figures.size())
							figures.add(d.getFigureAt(i));
						else
							figures.add(id[i], d.getFigureAt(i));
					
					d.setDrawFigures(false);
					draw.setSelected(d);
					objectRedo.add(d.clone());
				}else
				if(label.equals(LABEL_SEPARATE))
				{	
					Object o = objectUndo.remove(objectUndo.size()-1);
					
					if(!(o instanceof Draw))
						throw new ClassCastException();
					
					Draw d = (Draw)o;
					
					int size = d.size();
					for(i=0; i<size; i++)
						figures.remove(id[0]);
					d.setDrawFigures(true);
					figures.add(id[0], d);
					draw.setSelected(d);
				}
				
				idRedo.add(id);
				textRedo.add(label);
			}
			else throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	
	/**
	 * Allows to know if there is actions in the redo stack
	 * @return True if there is actions in the redo stack 
	 */
	public boolean isEmptyRedo()
	{
		if(textRedo==null) return true;
		return textRedo.isEmpty();
	}
	
	
	

	
	/**
	 * Allows to know if there is actions in the undo stack
	 * @return True if there is actions in the undo stack 
	 */
	public boolean isEmptyUndo()
	{
		if(textUndo==null) return true;
		return textUndo.isEmpty();
	}
	
	
	
	
	
	/**
	 * Redo an action
	 */
	public void redo()
	{
		try
		{
			if(!textRedo.isEmpty() && !idRedo.isEmpty())
			{
				String label = textRedo.remove(textRedo.size()-1);
				int[] id = idRedo.remove(idRedo.size()-1);
				int i;
				
				if(label.equals(LABEL_CHANGE_DISPOSITION))
				{
					Object o = objectRedo.remove(objectRedo.size()-1);	
					
					if(!(o instanceof int[]))
						throw new ClassCastException(o.getClass().getName());
					
					if(((int[])o).length!=id.length)
						throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);						
					
					int[] oldId =  (int[])o;
					Figure f;
					for(i=id.length-1; i>=0; i--)
					{
						f = figures.remove(id[i]);
						figures.add(oldId[i], f);
					}								
					add(LABEL_CHANGE_DISPOSITION, oldId, id, true);
					
				}else
				if(label.equals(LABEL_CREATE))
				{			
					Figure f = (Figure) objectRedo.remove(objectRedo.size()-1);
					
					if(f.isSelected())
						if(!draw.getSelected().contains(f))
							draw.setSelected(f);
						else f.onRelease();
					
					figures.add(id[0], f);
					add(LABEL_CREATE, id, null, true);
				}else
				if(label.equals(LABEL_DEL))
				{
					Draw d = new Draw(false, false); 
					Draw listSelected = draw.getSelected();					

					for(i=id.length-1; i>=0; i--)
					{
						Figure f = figures.remove(id[i]);
						if(listSelected.contains(f))
							listSelected.removeFigure(f);
						d.addFigure(f);
					}
		
					for(i=id.length-2; i>=0;  i--)
						d.addFigure(d.removeFigureAt(i));
				
					add(LABEL_DEL, id, d, true);
				}
				else
				if(label.equals(LABEL_CHANGE_FILL))
				{
					figures.elementAt(id[0]).setIsFilled(!
							figures.elementAt(id[0]).isFilled());
					add(LABEL_CHANGE_FILL, id, null, true);	
				}
				else
				if(label.equals(LABEL_CHANGE_RARROW_STYLE))
				{
					Line l = (Line)figures.elementAt(id[0]);
					String style = l.getArrow2Style();
					l.setArrow2Style(
							(String)objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_RARROW_STYLE, id, style, true);
				}
				else
				if(label.equals(LABEL_CHANGE_LARROW_STYLE))
				{
					Line l = (Line)figures.elementAt(id[0]);
					String style = l.getArrow1Style();
					l.setArrow1Style(
							(String)objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_LARROW_STYLE, id, style, true);
				}
				else
				if(label.equals(LABEL_CHANGE_BORD_COL))
				{
					Figure f = figures.elementAt(id[0]);
					Color c = f.getLinesColor();
					f.setLinesColor((Color)objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_BORD_COL, id, c, true);
				}
				else
				if(label.equals(LABEL_CHANGE_INT_COL))
				{
					Figure f = figures.elementAt(id[0]);
					Color c = f.getInteriorColor();
					f.setInteriorColor(
							(Color) objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_INT_COL, id, c, true);
				}
				else
				if(label.equals(LABEL_CHANGE_STYLE))
				{
					Figure f = figures.elementAt(id[0]);
					String style = f.getLineStyle();
					f.setLineStyle((String)objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_STYLE,id,style, true);
				}
				else
				if(label.equals(LABEL_CHANGE_MOVE) || label.equals(LABEL_TRANSFORM) || label.equals(LABEL_ALIGN) ||
					label.equals(LABEL_DISTRIB))
				{
					Object o = objectRedo.remove(objectRedo.size()-1);
					if(!(o instanceof Draw)) throw new ClassCastException();

					Draw fOld = (Draw)o;
				
					if(id.length != fOld.size()) 
						throw new LaTeXDrawException(LaTeXDrawException.INCORRECT_VALUE);
						
					Draw d = new Draw(false, false);
					for(i=0; i<fOld.size(); i++)
					{
						Figure f = figures.elementAt(id[i]);
						d.addFigure(f);
						figures.remove(id[i]);
						figures.add(id[i], fOld.getFigureAt(i));
					}

					draw.getSelected().setSelected(false);
					draw.getSelected().clear();
					fOld.setSelected(true);
					draw.setSelected(fOld);
					add(label.equals(LABEL_CHANGE_MOVE) ? LABEL_CHANGE_MOVE : LABEL_TRANSFORM, id, d, true);
				}else
				if(label.equals(LABEL_CHANGE_PARAM))
				{
					Figure f = figures.elementAt(id[0]);
					figures.remove(id[0]);
					Figure f2 = (Figure) objectRedo.remove(objectRedo.size()-1);
					figures.add(id[0], f2);
					draw.getSelected().setSelected(false);
					draw.getSelected().clear();
					
					add(LABEL_CHANGE_PARAM,id,f, true);
				}
				else
				if(label.equals(LABEL_CHANGE_DOT_STYLE))
				{
					Dot d = (Dot)figures.elementAt(id[0]);
					String style = d.getCurrentStyle();
					d.setCurrentStyle((String)objectRedo.remove(objectRedo.size()-1));
					add(LABEL_CHANGE_DOT_STYLE,id,style, true);
				}
				else
				if(label.equals(LABEL_CHANGE_THICKNESS))
				{
					Figure f = figures.elementAt(id[0]);
					float thickness;
					if(f instanceof Dot)
					{
						thickness = ((Dot)f).getWidth();
						((Dot)f).setWidth((Float)objectRedo.remove(objectRedo.size()-1));
					}
					else 
					{
						thickness = f.getThickness();
						f.setThickness((Float)objectRedo.remove(objectRedo.size()-1));
					}
					
					add(LABEL_CHANGE_THICKNESS,id,thickness, true);
				}else
				if(label.equals(LABEL_SEPARATE))
				{	
					Draw d = (Draw)figures.remove(id[0]);
					int size = d.size();
					for(i=size-1; i>=0; i--)
						figures.add(id[0], d.getFigureAt(i));
					
					d.setDrawFigures(false);
					draw.setSelected(d);
					add(LABEL_SEPARATE, id, d.clone(), true);
					
				}else
				if(label.equals(LABEL_JOIN))
				{	
					Object o = objectRedo.remove(objectRedo.size()-1);
					int min = id[0];
					i = 1;
					
					while(i<id.length)
					{
						if(id[i]<min)
							min = id[i];
						i++;
					}
					
					if(!(o instanceof Draw))
						throw new ClassCastException();
					
					Draw d = (Draw)o;
					
					for(i=0; i<id.length; i++)
						figures.remove(id[i]);
					
					d.setDrawFigures(true);
					figures.add(min, d);
					draw.setSelected(d);
					add(LABEL_JOIN, id, null, true);
				}
						
			}
			
			Draw select = mainFrame.getDrawPanel().getDraw().getSelected();
			mainFrame.setSelection(select!=null && !select.isEmpty());
		}catch(Exception ex)
		{
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}
	
	
	
	
	
	/**
	 * Allows to know if the label is valid
	 * @param label The label to check
	 * @return True : is the label is valid
	 */
	public boolean isValidLabel(String label)
	{
		return label.equals(LABEL_CREATE) || label.equals(LABEL_CHANGE_MOVE) ||
				label.equals(LABEL_CHANGE_PARAM) || label.equals(LABEL_DEL) ||
				label.equals(LABEL_CHANGE_BORD_COL) || label.equals(LABEL_CHANGE_THICKNESS) ||
				label.equals(LABEL_CHANGE_DOT_STYLE) || label.equals(LABEL_CHANGE_STYLE) ||
				label.equals(LABEL_CHANGE_INT_COL) || label.equals(LABEL_CHANGE_LARROW_STYLE) ||
				label.equals(LABEL_CHANGE_RARROW_STYLE) || label.equals(LABEL_CHANGE_FILL) ||
				label.equals(LABEL_CHANGE_DISPOSITION) || label.equals(LABEL_JOIN) ||
				label.equals(LABEL_SEPARATE) || label.equals(LABEL_TRANSFORM) || label.equals(LABEL_ALIGN) ||
				label.equals(LABEL_DISTRIB);
	}

	
	
	/**
	 * Clean the manager.
	 * @since 2.0.0
	 */
	public void clear()
	{
		idUndo.clear();
		objectUndo.clear();
		textUndo.clear();
		clearRedo();
	}
	
	
	/**
	 * Allow to clear the redo when an action, different than undo occurred.
	 */
	private void clearRedo()
	{
		idRedo.clear();
		objectRedo.clear();
		textRedo.clear();
		mainFrame.updateUndoRedo();
	}
}
