package latexDraw.ui.codeEditorPane;

import static latexDraw.ui.CodePanel.LABEL_COPY_SELECTION_CODE;

import java.awt.event.FocusEvent;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;

import javax.swing.JButton;
import javax.swing.JEditorPane;

import latexDraw.ui.CodePanel;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * Defines an abstract model of code editor pane.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 06/03/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public abstract class AbstractCodeEditorPane extends JEditorPane implements MouseListener
{
	private static final long	serialVersionUID	= 1L;

	protected CodePanel codePanel;
	
	/** Corresponds to the beginning of the code of the figure: \begin{pspicture}... */
	protected String begin;
	
	/** Corresponds to the body of the code : the definitions of the figures (lines, dots, ...)*/
	protected String body;
	
	/** Defines the start of the code. */
	protected String beginCodeTag;
	
	/** Corresponds to the end of the code of the figure. */
	protected String endCodeTag;
	
	
	
	/**
	 * The constructor.
	 * @param codePanel The associated codePanel.
	 */
	public AbstractCodeEditorPane(CodePanel codePanel)
	{
		super();
		setEditable(false);
		addMouseListener(this);
		setDragEnabled(true);
		this.codePanel = codePanel;
		begin = "";//$NON-NLS-1$
		body = "";//$NON-NLS-1$
		setBeginCodeTag();
		setEndCodeTag();
	}
	
	
	
	
	/**
	 * Update the buttons managing the copy of the code.
	 * @since 1.9
	 */
	public void updateButtonsCopy()
	{
		String text = getSelectedText();
		boolean enable = text!=null && !text.equals("");//$NON-NLS-1$
		codePanel.getButtonCopySel().setEnabled(enable); 
	}
	
	
	
	public void mouseClicked(MouseEvent e)
	{
		updateButtonsCopy();
	}




	public void mousePressed(MouseEvent e)
	{
		/* No code required */		
	}




	public void mouseReleased(MouseEvent e)
	{
		updateButtonsCopy();
	}




	public void mouseEntered(MouseEvent e)
	{
		/* No code required */
	}




	public void mouseExited(MouseEvent e)
	{
		/* No code required */
	}



	@Override
	protected void processFocusEvent(FocusEvent e)
	{
		super.processFocusEvent(e);
	
		if(e.getID()==FocusEvent.FOCUS_LOST)
		{
			Object o = e.getOppositeComponent();

			if(!(o instanceof JButton && ((JButton)o).getActionCommand().equals(LABEL_COPY_SELECTION_CODE)))
			{
				select(0, 0);
				codePanel.getButtonCopySel().setEnabled(false);
			}
		}
	}
	
	
	
	/**
	 * Update the text of the panel.
	 */
	public abstract void updateText();
	
	
	
	/**
	 * Set the beginning of the code generated.
	 * @param SW The top right point.
	 * @param NE The bottom left point.
	 * @param pixPerCm The definition level of the drawing.
	 * @param origin The origin point of the system of coordinates.
	 */
	public abstract void setStart(LaTeXDrawPoint2D SW, LaTeXDrawPoint2D NE, float pixPerCm, LaTeXDrawPoint2D origin);
	
	
	
	/**
	 * Set the body (between the beginning and the end) of the code.
	 * @param code The new code.
	 */
	public void setBody(String code)
	{
		body = code;
		updateText();
	}
	
	
	/**
	 * Empty the panel.
	 */
	public void setEmpty()
	{
		body=""; //$NON-NLS-1$
		setText("");//$NON-NLS-1$
	}

	
	
	/**
	 * Set the tag which will be defined the start of the code.
	 * @since 2.0.0
	 */
	protected abstract void setBeginCodeTag();
	
	
	/**
	 * Set the tag which will be defined the end of the code.
	 * @since 2.0.0
	 */
	protected abstract void setEndCodeTag();
}
