package latexDraw.generators.svg;

import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.geom.Point2D;
import java.text.ParseException;
import java.util.Vector;

import latexDraw.figures.*;
import latexDraw.parsers.latexdraw.PointsParser;
import latexDraw.parsers.svg.*;
import latexDraw.parsers.svg.elements.SVGElement;
import latexDraw.parsers.svg.elements.SVGGElement;
import latexDraw.parsers.svg.elements.SVGLineElement;
import latexDraw.parsers.svg.elements.SVGTextElement;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.util.LaTeXDrawNamespace;
import latexDraw.util.LaTeXDrawPoint2D;

/**
 * Defines a SVG generator for an axe.<br>
 *<br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 11/11/07<br>
 * @author Arnaud BLOUIN<br>
 * @version 0.1<br>
 */
public class LAxeSVGGenerator extends LShapeSVGGenerator
{

	public LAxeSVGGenerator(Axe f)
	{
		super(f);
	}

	
	
	public LAxeSVGGenerator(SVGGElement elt)
	{
		this(elt, true);
	}
	

	/**
	 * Creates axes from a latexdraw-SVG element.
	 * @param elt The source element.
	 * @param withTransformation If true, the SVG transformations will be applied.
	 * @since 2.0.0
	 */
	public LAxeSVGGenerator(SVGGElement elt, boolean withTransformation)
	{
		this(new Axe(new LaTeXDrawPoint2D(), false));

		if(elt==null)
			throw new IllegalArgumentException();
		
		setNumber(elt);
		setSVGParameters(elt);
		
		Vector<Point2D> values;
		String pref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':';
		String str;
		Axe axe = (Axe)getShape();
		
		axe.setYLabelWest(Boolean.valueOf(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_IS_WEST)).booleanValue());
		axe.setXLabelSouth(Boolean.valueOf(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_IS_SOUTH)).booleanValue());
		axe.setShowOrigin(Boolean.valueOf(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_SHOW_ORIGIN)).booleanValue());
		axe.setAxesStyleToken(elt.getAttribute(pref+LaTeXDrawNamespace.XML_STYLE));
		axe.setTicksDisplayedToken(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_SHOW_TICKS));
		axe.setTicksStyleToken(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_TICKS_STYLE));
		axe.setLabelsDisplayedToken(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_LABELS_STYLE));
		str = elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_TICKS_SIZE);
		
		if(str!=null)
			try{ axe.setTicksSize(Double.valueOf(str)); }
			catch(NumberFormatException e) { /* */ }
			
		try
		{
			values = PointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_END));
			
			if(values!=null && values.size()>0)
			{ 
				axe.setGridEndX((int)values.elementAt(0).getX());  
				axe.setGridEndY((int)values.elementAt(0).getY());  
			}
		}
		catch(ParseException e) { e.printStackTrace(); }
		catch(NumberFormatException e) { e.printStackTrace(); }
		
		try
		{
			values = PointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_START));
			
			if(values!=null && values.size()>0)
			{ 
				axe.setGridStartX((int)values.elementAt(0).getX());  
				axe.setGridStartY((int)values.elementAt(0).getY());  
			}
		}
		catch(ParseException e) { e.printStackTrace(); }
		catch(NumberFormatException e) { e.printStackTrace(); }
		
		try
		{
			values = PointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_GRID_ORIGIN));
		
			if(values!=null && values.size()>0)
			{ 
				axe.setOriginX((int)values.elementAt(0).getX());  
				axe.setOriginY((int)values.elementAt(0).getY());  
			}
		}
		catch(ParseException e) { e.printStackTrace(); }
		catch(NumberFormatException e) { e.printStackTrace(); }
		
		try
		{
			values = PointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_INCREMENT));
				
			if(values!=null && values.size()>0)
			{ 
				axe.setIncrementX(values.elementAt(0).getX());  
				axe.setIncrementY(values.elementAt(0).getY());  
			}
		}
		catch(ParseException e) { e.printStackTrace(); }
		catch(NumberFormatException e) { e.printStackTrace(); }
			
		try
		{
			values = PointsParser.getPoints(elt.getAttribute(pref+LaTeXDrawNamespace.XML_AXE_DIST_LABELS));
				
			if(values!=null && values.size()>0)
			{ 
				axe.setDistLabelsX(values.elementAt(0).getX());  
				axe.setDistLabelsY(values.elementAt(0).getY());  
			}
		}
		catch(ParseException e) { e.printStackTrace(); }
		catch(NumberFormatException e) { e.printStackTrace(); }
		
		/* Looking for the two axe in order to get the position of the axes. */
		SVGNodeList nl = elt.getChildren(SVGElements.SVG_G);
		int i=0, size = nl.getLength();
		SVGGElement l1=null, l2=null;
		SVGElement element;
		
		while((l1==null || l2==null) && i<size)
		{
			element = nl.item(i);
			
			if(element instanceof SVGGElement)
				if(l1==null)
					l1 = (SVGGElement)element;
				else
					l2 = (SVGGElement)element;
			
			i++;
		}
		
		if(l1!=null && l2!=null)
			try
			{
				Line la = (Line)new LLineSVGGenerator(l1, false).shape;
				Line lb = (Line)new LLineSVGGenerator(l2, false).shape;
				
				axe.setPosition(new LaTeXDrawPoint2D(lb.getPt1().x, la.getPt1().y));
				axe.setLineStyle(la.getLineStyle());
				
				if(axe.isXLabelSouth())
				{
					axe.setArrow1Style(la.getArrow1Style());
					axe.setArrow2Style(la.getArrow2Style());
				}
				else
				{
					axe.setArrow1Style(la.getArrow2Style());
					axe.setArrow2Style(la.getArrow1Style());
				}
				
				la = null;
				lb = null;
			}
			catch(IllegalArgumentException e) { e.printStackTrace(); }

		homogeniseArrows(axe.getArrowHead1(), axe.getArrowHead2());
		axe.updateGravityCenter();
		axe.updateBorders();
		applyTransformations(elt);
		
		l1 		= null;
		l2 		= null;
		element = null;
		nl 		= null;
		values 	= null;
	}

	

	@Override
	public SVGElement toSVG(SVGDocument doc)
	{
		if(doc==null)
			return null;
		
		Axe axe = (Axe)getShape();
		String pref = LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':';
		SVGElement root = new SVGGElement(doc);
		setThickness(root, axe.getThickness(), false, 0.);
		root.setStroke(axe.getLinesColor());
		
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_IS_WEST, String.valueOf(axe.isYLabelWest()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_IS_SOUTH, String.valueOf(axe.isXLabelSouth()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_STYLE, axe.getAxesStyleToken());
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_START, axe.getGridStart().x + " " + axe.getGridStart().y); //$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_END, axe.getGridEnd().x + " " + axe.getGridEnd().y);//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_GRID_ORIGIN, axe.getOrigin().x + " " + axe.getOrigin().y);//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_INCREMENT, axe.getIncrement().x + " " + axe.getIncrement().y);//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_DIST_LABELS, axe.getDistLabelsX() + " " + axe.getDistLabelsY());//$NON-NLS-1$
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_TICKS_SIZE, String.valueOf(axe.getTicksSize()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_SHOW_ORIGIN, String.valueOf(axe.isShowOrigin()));
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_SHOW_TICKS, axe.getTicksDisplayedToken());
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_LABELS_STYLE, axe.getLabelsDisplayedToken());
		root.setAttribute(pref+LaTeXDrawNamespace.XML_AXE_TICKS_STYLE, axe.getTicksStyleToken());
		
		createSVGAxe(root, doc);
		
		return root;
	}
	
	
	
	
	protected void createSVGAxe(SVGElement elt, SVGDocument document)
	{
		if(elt==null || document==null)
			return ;
		
		Axe axe = (Axe)shape;
		double minX, maxX, minY, maxY, maxX3, minX3, maxY3, minY3;
		Point2D.Double increment = axe.getIncrement();
		Point2D.Double gridEnd   = axe.getGridEnd();
		Point2D.Double gridStart = axe.getGridStart();
		Point2D.Double position  = axe.getPosition();
		Point2D.Double origin    = axe.getOrigin();
		double distLabelsX = axe.getDistLabelsX();
		double distLabelsY = axe.getDistLabelsY();
		double gapX = distLabelsX==0. ? Figure.PPC : (distLabelsX/increment.x)*Figure.PPC, i; 
		double gapY = distLabelsY==0. ? Figure.PPC : (distLabelsY/increment.y)*Figure.PPC;
		double rotationAngle = shape.getRotationAngle();
		double lgth;
		int ticksStyle = axe.getTicksStyle();
		int axesStyle  = axe.getAxesStyle();
		int ticksDisplayed  = axe.getTicksDisplayed();
		int labelsDisplayed  = axe.getLabelsDisplayed();
		double ticksSize  = axe.getTicksSize();
		boolean isYLabelWest  = axe.isYLabelWest();
		boolean isXLabelSouth = axe.isXLabelSouth();
		boolean showOrigin    = axe.isShowOrigin();
		FontMetrics fontMetrics = axe.getFontMetrics();
		Font font = axe.getFont();
		ArrowHead arrowHead1 = axe.getArrowHead1();
		ArrowHead arrowHead2 = axe.getArrowHead2();
		boolean ticksBot = ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE || ticksStyle==Axe.TICKS_STYLE_FULL_VALUE;
		double thickness = shape.getThickness();
		int j;

        elt.setAttribute(LaTeXDrawNamespace.LATEXDRAW_NAMESPACE+':'+LaTeXDrawNamespace.XML_TYPE, LaTeXDrawNamespace.XML_TYPE_AXE);
        elt.setAttribute(SVGAttributes.SVG_ID, getSVGID());
		
		if(gridEnd.x<gridStart.x)
		{
			minX = gridEnd.x;
			maxX = gridStart.x;
		}
		else
		{
			minX = gridStart.x;
			maxX = gridEnd.x;
		}
		
		if(gridEnd.getY()<gridStart.getY())
		{
			minY = gridEnd.getY();
			maxY = gridStart.getY();
		}
		else
		{
			minY = gridStart.getY();
			maxY = gridEnd.getY();
		}
		
		if(distLabelsX!=0.)
		{
			if(maxX!=0.)
			{
				maxX = maxX/distLabelsX;
				if(maxX==0.)
					maxX=0.1;
			}
			
			if(minX!=0)
			{
				minX = minX/distLabelsX;
				if(minX==0.)
					minX=0.1;
			}
		}
		
		if(distLabelsY!=0.)
		{
			if(maxY!=0.)
			{
				maxY = maxY/distLabelsY;
				if(maxY==0.)
					maxY=0.1;
			}
			
			if(minY!=0.)
			{
				minY = minY/distLabelsY;
				if(minY==0.)
					minY=0.1;
			}
		}
		
		boolean arrow1Ok = !arrowHead1.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE);
		boolean arrow2Ok = !arrowHead2.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE);
		
		minX3 = axesStyle==Axe.AXES_STYLE_AXES_VALUE && ((arrow1Ok && isYLabelWest) || 
				(arrow2Ok && !isYLabelWest)) && gridStart.x!=0 && (distLabelsX==0. || distLabelsX==1.)? minX+1 : minX; 		
		
		minY3 = axesStyle==Axe.AXES_STYLE_AXES_VALUE && ((arrow1Ok && isXLabelSouth) || 
				(arrow2Ok && !isXLabelSouth)) && gridStart.getY()!=0 && (distLabelsY==0. || distLabelsY==1.)? minY+1 : minY; 			 
					 
		maxX3 = axesStyle==Axe.AXES_STYLE_AXES_VALUE &&
		 		((arrow2Ok && isYLabelWest) || (arrow1Ok && !isYLabelWest)) && (distLabelsX==0. || distLabelsX==1.)? maxX-1 : maxX; 
		
		maxY3 = axesStyle==Axe.AXES_STYLE_AXES_VALUE &&
		 		((arrow2Ok && isXLabelSouth) || (arrow1Ok && !isXLabelSouth)) && (distLabelsY==0. || distLabelsY==1.)? maxY-1 : maxY; 	
		
		minX3 = minX>0 ? 0 : minX3;
		maxX3 = maxX<0 ? 0 : maxX3;
		minY3 = minY>0 ? 0 : minY3;
		maxY3 = maxY<0 ? 0 : maxY3;
		
		switch(axesStyle)
		{
			case Axe.AXES_STYLE_AXES_VALUE:
				
				boolean arrowOk = (float)minX!=(float)maxX || (float)maxX!=(float)minY || (float)minY!=(float)maxY;
				Line xLine = new Line(new LaTeXDrawPoint2D(position.x+gridStart.x*Figure.PPC, position.getY()), 
										new LaTeXDrawPoint2D(position.x+gridEnd.x*Figure.PPC, position.getY()), true);
				Line yLine = new Line(new LaTeXDrawPoint2D(position.x, position.getY()-gridEnd.getY()*Figure.PPC), 
										new LaTeXDrawPoint2D(position.x, position.getY()-gridStart.getY()*Figure.PPC), true);

				xLine.setLinesColor(shape.getLinesColor());
				xLine.setLineStyle(shape.getLineStyle());
				xLine.setThickness((float)thickness);
				yLine.setLinesColor(shape.getLinesColor());
				yLine.setLineStyle(shape.getLineStyle());
				yLine.setThickness((float)thickness);
				
				if(!arrowHead1.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE) && arrowOk)
				{
					String arrowStyle = arrowHead1.getArrowStyle();
					lgth = arrowHead1.getArrowHeadLength();
					boolean isArrow = arrowHead1.needReduceLine();
					
					if((minX!=0 && isYLabelWest) || (maxX!=0 && !isYLabelWest) || maxY==minY)
					{
						if(isArrow && !isYLabelWest)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(isArrow)
							if(isYLabelWest)
								 xLine.getPt1().x-=lgth;
							else xLine.getPt2().x+=lgth;
						
						if(isYLabelWest)
							xLine.setArrow1Style(arrowStyle);
						else
							xLine.setArrow2Style(arrowStyle);
					}
					
					if((minY!=0 && isXLabelSouth) || (maxY!=0 && !isXLabelSouth) || maxX==minX)
					{
						arrowStyle = arrowHead1.getArrowStyle();
						
						if(isArrow && isXLabelSouth)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(isArrow)
							if(isXLabelSouth)
								 yLine.getPt2().y+=lgth;
							else yLine.getPt1().y-=lgth;
						
						if(isXLabelSouth)
							yLine.setArrow2Style(arrowStyle);
						else
							yLine.setArrow1Style(arrowStyle);
					}
				}
				
				if(!arrowHead2.getArrowStyle().equals(PSTricksConstants.NONEARROW_STYLE) && arrowOk)
				{
					String arrowStyle = arrowHead2.getArrowStyle();
					lgth = arrowHead2.getArrowHeadLength();
					boolean isArrow = arrowHead2.needReduceLine();
					
					if((maxY!=0 && isXLabelSouth) || (minY!=0 && !isXLabelSouth) || maxY==minY)
					{
						if(isArrow && !isXLabelSouth)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(maxY==minY && minY==0)
						{
							if(isArrow)
								if(maxX>0)
									 xLine.getPt1().y-=lgth;
								else xLine.getPt2().y+=lgth;
							
							if(maxX>0)
								xLine.setArrow1Style(arrowStyle);
							else
								xLine.setArrow2Style(arrowStyle);
						}
						else
						{
							if(isArrow)
								if(isXLabelSouth)
									 yLine.getPt1().y-=lgth;
								else yLine.getPt2().y+=lgth;
							
							if(isXLabelSouth)
								yLine.setArrow1Style(arrowStyle);
							else
								yLine.setArrow2Style(arrowStyle);
						}
					}
					
					if((maxX!=0 && isYLabelWest) || (minX!=0 && !isYLabelWest) || maxX==minX)
					{
						arrowStyle = arrowHead2.getArrowStyle();
						
						if(isArrow && isYLabelWest)
						{
							if(arrowStyle.equals(PSTricksConstants.RARROW_STYLE))
								arrowStyle = PSTricksConstants.LARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LARROW_STYLE))
								arrowStyle = PSTricksConstants.RARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DLARROW_STYLE))
								arrowStyle = PSTricksConstants.DRARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.DRARROW_STYLE))
								arrowStyle = PSTricksConstants.DLARROW_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LRBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RRBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.RSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.LSBRACKET_STYLE;
							else if(arrowStyle.equals(PSTricksConstants.LSBRACKET_STYLE))
								arrowStyle = PSTricksConstants.RSBRACKET_STYLE;
						}
						
						if(maxX==minX && minX==0)
						{
							if(isArrow)
								if(maxY>0)
									 yLine.getPt2().x+=lgth;
								else xLine.getPt1().x-=lgth;
							
							if(maxY>0)
								yLine.setArrow2Style(arrowStyle);
							else
								yLine.setArrow1Style(arrowStyle);
						}
						else
						{
							if(isArrow)
								if(isYLabelWest)
									 xLine.getPt2().x+=lgth;
								else xLine.getPt1().x-=lgth;
							
							if(isYLabelWest)
								xLine.setArrow2Style(arrowStyle);
							else
								xLine.setArrow1Style(arrowStyle);
						}
					}
				}

				elt.appendChild(new LLineSVGGenerator(xLine).toSVG(document));
				elt.appendChild(new LLineSVGGenerator(yLine).toSVG(document));
				
				break;
				
			case Axe.AXES_STYLE_FRAME_VALUE:
				
				if(gridEnd.x>0 || gridEnd.y>0)
				{
					double xMax = position.x+gridEnd.x*Figure.PPC;
					double yMax = position.y-gridEnd.y*Figure.PPC;
					
					LaTeXDrawRectangle r = new LaTeXDrawRectangle(
							new LaTeXDrawPoint2D(position.x, gridEnd.y>0?yMax:position.y), 
							new LaTeXDrawPoint2D(gridEnd.x>0?xMax:position.x, position.y), false);
					
					r.setBordersPosition(PSTricksConstants.BORDERS_MIDDLE);
					r.setLinesColor(shape.getLinesColor());
					r.setLineStyle(shape.getLineStyle());
					r.setThickness((float)thickness);
					elt.appendChild(new LRectangleSVGGenerator(r).toSVG(document));
				}
				break;
		}
		
		double endX = distLabelsX!=0. ? maxX3 : maxX3/increment.x;
		double endY = distLabelsY!=0. ? maxY3 : maxY3/increment.getY();
		double startX = distLabelsX!=0. ? minX3 : minX3/increment.x;
		double startY = distLabelsY!=0. ? minY3 : minY3/increment.getY();
		
		if(ticksDisplayed!=Axe.LABELS_DISPLAYED_NON_VALUE)// We draw the ticks
		{
			if(ticksStyle==Axe.TICKS_STYLE_FULL_VALUE || (ticksStyle==Axe.TICKS_STYLE_TOP_VALUE && maxY>0) ||
			  (ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE && maxY<=0))
				if(ticksDisplayed==Axe.LABELS_DISPLAYED_X_VALUE || ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE)
				{
					SVGElement ticksX = new SVGGElement(document);
					Line l = new Line(false);
					l.setThickness((float)thickness);
					l.setLinesColor(shape.getLinesColor());
					
					for(j=1; j<=endX; j++)
						try{ ticksX.appendChild(new SVGLineElement(position.x+j*gapX*increment.x, position.y,
								            position.x+j*gapX*increment.x, position.y-ticksSize-thickness/2., document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					for(j=-1; j>=startX; j--)
						try{ ticksX.appendChild(new SVGLineElement(position.x+j*gapX*increment.x, position.y,
											position.x+j*gapX*increment.x, position.y-ticksSize-thickness/2., document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					elt.appendChild(ticksX);
				}
			
			if(ticksStyle==Axe.TICKS_STYLE_FULL_VALUE || (ticksStyle==Axe.TICKS_STYLE_TOP_VALUE && maxX>0) ||
			  (ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE && maxX<=0))
				if(ticksDisplayed==Axe.LABELS_DISPLAYED_Y_VALUE || ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE)
				{
					SVGElement ticksY = new SVGGElement(document);
					
					for(j=1; j<=endY; j++)
						try{ ticksY.appendChild(new SVGLineElement(position.x, position.y-j*gapY*increment.getY(),
								  			position.x+ticksSize+thickness/2., position.y-j*gapY*increment.getY(), document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					for(j=-1; j>=startY; j--)
						try{ ticksY.appendChild(new SVGLineElement(position.x, position.y-j*gapY*increment.getY(), 
											position.x+ticksSize+thickness/2., position.y-j*gapY*increment.getY(), document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					elt.appendChild(ticksY);
				}
			
			if(ticksStyle==Axe.TICKS_STYLE_FULL_VALUE || (ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE && maxY>0) ||
			  (ticksStyle==Axe.TICKS_STYLE_TOP_VALUE && maxY<=0))
				if(ticksDisplayed==Axe.LABELS_DISPLAYED_X_VALUE || ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE)
				{
					SVGElement ticksX = new SVGGElement(document);
					
					for(j=1; j<=endX; j++)
						try{ ticksX.appendChild(new SVGLineElement(position.x+j*gapX*increment.x, position.y, 
											position.x+j*gapX*increment.x, position.y+ticksSize+thickness/2., document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					for(j=-1; j>=startX; j--)
						try{ ticksX.appendChild(new SVGLineElement(position.x+j*gapX*increment.x, position.y,
											position.x+j*gapX*increment.x, position.y+ticksSize+thickness/2., document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					elt.appendChild(ticksX);
				}
			
			if(ticksStyle==Axe.TICKS_STYLE_FULL_VALUE || (ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE && maxX>0) ||
				(ticksStyle==Axe.TICKS_STYLE_TOP_VALUE && maxX<=0))
				if(ticksDisplayed==Axe.LABELS_DISPLAYED_Y_VALUE || ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE)
				{
					SVGElement ticksY = new SVGGElement(document);
					
					for(j=1; j<=endY; j++)
						try{ ticksY.appendChild(new SVGLineElement(position.x, position.y-j*gapY*increment.getY(), 
											position.x-ticksSize-thickness/2., position.y-j*gapY*increment.getY(), document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					for(j=-1; j>=startY; j--)
						try{ ticksY.appendChild(new SVGLineElement(position.x, position.y-j*gapY*increment.getY(), 
											position.x-ticksSize-thickness/2., position.y-j*gapY*increment.getY(), document)); }
						catch(MalformedSVGDocument e) { /* */ }
					
					elt.appendChild(ticksY);
				}
		}

		if(labelsDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE || labelsDisplayed==Axe.LABELS_DISPLAYED_X_VALUE)
		{// We show the labels on the X-axe.
			float height 	= fontMetrics.getAscent();
			double gap 		= ((ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE||
							  ticksDisplayed==Axe.LABELS_DISPLAYED_X_VALUE)&&
							((isXLabelSouth && (ticksStyle==Axe.TICKS_STYLE_BOTTOM_VALUE || ticksStyle==Axe.TICKS_STYLE_FULL_VALUE)) ||
							 (!isXLabelSouth && (ticksStyle==Axe.TICKS_STYLE_TOP_VALUE || ticksStyle==Axe.TICKS_STYLE_FULL_VALUE)))?
							  ticksSize:0)+thickness/2.+Axe.GAP_LABEL;
			double sep		= maxY<=0 || !isXLabelSouth? -gap-Axe.GAP_LABEL : gap+height;
			String str;

			if(((isXLabelSouth && gridStart.y>=0) || (!isXLabelSouth && 
				gridEnd.y<=0)) && (gridStart.x!=gridEnd.x || gridStart.x!=0) && gridStart.y<=0 && showOrigin)
			{
				SVGElement text = new SVGTextElement(document);
				text.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
				text.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
				text.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
				text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x-fontMetrics.stringWidth(String.valueOf((int)origin.x))/2.)));
				text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+sep)));
				text.setTextContent(String.valueOf((int)origin.x));
				elt.appendChild(text);
			}
			
			SVGElement texts = new SVGGElement(document);
			texts.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
			texts.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
			texts.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
			SVGElement text;
			
			for(i=increment.x, j=1; j<=endX; i+=increment.x, j++)
			{
				text = new SVGTextElement(document);
				str = (((int)increment.x)==increment.x?String.valueOf((int)(i+origin.x)) : String.valueOf(i+origin.x));
				text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x+j*gapX*increment.x-fontMetrics.stringWidth(str)/2.)));
				text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+sep)));
				text.setTextContent(str);
				texts.appendChild(text);
			}
			
			for(i=-increment.x, j=-1; j>=startX; i-=increment.x, j--)
			{
				text = new SVGTextElement(document);
				str = (((int)increment.x)==increment.x?String.valueOf((int)(i+origin.x)) : String.valueOf(i+origin.x));
				text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x+j*gapX*increment.x-fontMetrics.stringWidth(str)/2.)));
				text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+sep)));
				text.setTextContent(str);
				texts.appendChild(text);
			}
			
			elt.appendChild(texts);
		}
		
		if(labelsDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE || labelsDisplayed==Axe.LABELS_DISPLAYED_Y_VALUE)
		{// We show the labels on the Y-axe.
			float height = fontMetrics.getAscent();
			String str;

			if(maxX>0 && isYLabelWest)
			{
				double gap	= (ticksBot && (ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE ||
								ticksDisplayed==Axe.LABELS_DISPLAYED_Y_VALUE)?ticksSize:0)+thickness/2.;
			
				if(gridStart.x==0 && (gridStart.y!=gridEnd.y || gridStart.y!=0) && showOrigin)
				{
					SVGElement text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
					text.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
					text.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x-gap-fontMetrics.stringWidth(String.valueOf((int)origin.y))-Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.)));
					text.setTextContent(String.valueOf((int)origin.y));
					elt.appendChild(text);
				}
				
				SVGElement texts = new SVGGElement(document);
				texts.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
				texts.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
				texts.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
				SVGElement text;
				
				for(i=increment.getY(), j=1; j<=endY; i+=increment.getY(), j++)
				{
					text = new SVGTextElement(document);
					str = ((int)increment.getY())==increment.getY()?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x-gap-fontMetrics.stringWidth(str)-Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.-j*gapY*increment.getY())));
					text.setTextContent(str);
					texts.appendChild(text);
				}
				
				for(i=-increment.getY(), j=-1; j>=startY; i-=increment.getY(), j--)
				{
					text = new SVGTextElement(document);
					str = ((int)increment.getY())==increment.getY()?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x-gap-fontMetrics.stringWidth(str)-Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.-j*gapY*increment.getY())));
					text.setTextContent(str);
					texts.appendChild(text);
				}
				
				elt.appendChild(texts);
			}
			else
			{
				double gap	= ((!ticksBot || ticksStyle==Axe.TICKS_STYLE_FULL_VALUE) && (ticksDisplayed==Axe.LABELS_DISPLAYED_ALL_VALUE||
								ticksDisplayed==Axe.LABELS_DISPLAYED_Y_VALUE)?ticksSize:0)+thickness/2.;
				
				if((!isYLabelWest && gridEnd.x<=0) || (isYLabelWest && gridStart.x>=0) && (gridStart.y!=gridEnd.y  || gridStart.y!=0) && showOrigin)
				{
					SVGElement text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
					text.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
					text.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x+gap+Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.)));
					text.setTextContent(String.valueOf((int)origin.y));
					elt.appendChild(text);
				}
				
				SVGElement texts = new SVGGElement(document);
				texts.setAttribute(SVGAttributes.SVG_STROKE, CSSColors.CSS_BLACK_NAME);
				texts.setAttribute(SVGAttributes.SVG_FONT_SIZE, String.valueOf(font.getSize()));
				texts.setAttribute(SVGAttributes.SVG_STROKE_WIDTH, "1");//$NON-NLS-1$
				SVGElement text;
				
				for(i=increment.getY(), j=1; j<=endY; i+=increment.getY(), j++)
				{
					text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x+gap+Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.-j*gapY*increment.getY())));
					text.setTextContent(((int)increment.getY())==increment.getY()?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y));
					texts.appendChild(text);
				}
				
				for(i=-increment.getY(), j=-1; j>=startY; i-=increment.getY(), j--)
				{
					text = new SVGTextElement(document);
					text.setAttribute(SVGAttributes.SVG_X, String.valueOf((int)(position.x+gap+Axe.GAP_LABEL)));
					text.setAttribute(SVGAttributes.SVG_Y, String.valueOf((int)(position.y+height/2.-j*gapY*increment.getY())));
					text.setTextContent(((int)increment.getY())==increment.getY()?String.valueOf((int)(i+origin.y)):String.valueOf(i+origin.y));
					texts.appendChild(text);
				}
				
				elt.appendChild(texts);
			}
		}
		
		if(rotationAngle%(Math.PI*2)!=0)
			setSVGRotationAttribute(elt);
	}
}
