package latexDraw.figures;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.Ellipse2D;
import java.io.IOException;
import java.io.ObjectInputStream;

import latexDraw.psTricks.DviPsColors;
import latexDraw.psTricks.PSTricksConstants;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.util.LaTeXDrawPoint2D;


/** 
 * This class defines a circle.<br>
 *<br>
 * This file is part of LaTeXDraw<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 01/20/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class Circle extends Ellipse
{
	private static final long serialVersionUID = 1L;


	/**
	 * @param f The figure to copy.
	 * @param sameNumber True if the figure will have the same number of the copy.
	 */
	public Circle(Figure f, boolean sameNumber)
	{
		super(f, sameNumber);
		
		double b = Math.abs(borders.getPoint(0).y-borders.getPoint(2).y)/2.;
		double a = Math.abs(borders.getPoint(0).x-borders.getPoint(1).x)/2.;
		double radius = Math.max(b, a);
		LaTeXDrawPoint2D cg = borders.getGravityCenter();
		
		borders = new Square(new LaTeXDrawPoint2D(cg.x-radius, cg.y-radius), 
							new LaTeXDrawPoint2D(cg.x+radius, cg.y-radius),
							new LaTeXDrawPoint2D(cg.x-radius, cg.y+radius), 
							new LaTeXDrawPoint2D(cg.x+radius, cg.y+radius), false);
		updateGravityCenter();
		updateShape();
	}
	
	
	
	/**
	 * The constructor by default
	 */
	public Circle(boolean increaseMeter)
	{
		this(new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D(), new LaTeXDrawPoint2D() , increaseMeter);
	}	
	

	
	/** 
	 * The constructor using two points
	 * @param pt1 The top left point of the rectangle containing the ellipse
	 * @param pt2 The bottom right point of the rectangle containing the ellipse
	 */
	public Circle(LaTeXDrawPoint2D pt1, LaTeXDrawPoint2D pt2, LaTeXDrawPoint2D pt3, LaTeXDrawPoint2D pt4, boolean increaseMeter)
	{
		super(pt1, pt2, pt3, pt4, increaseMeter);
		borders = new Square(pt1, pt2, pt3, pt4, false);
		gravityCenter = borders.getGravityCenter();
		updateGravityCenter();
		shape = getInsideOutsideOrMiddleBorders();
	}
	

		
	
	
	@Override
	public String getCodePSTricks(DrawBorders drawBorders, float ppc)
	{
		LaTeXDrawPoint2D d = drawBorders.getOriginPoint();
		LaTeXDrawPoint2D pt1 = borders.getPoint(0), pt2 = borders.getPoint(1);
		LaTeXDrawPoint2D pt3 = borders.getPoint(2);
		String add = "", addBegin="", addEnd="", fillType=""; //$NON-NLS-1$ //$NON-NLS-3$ //$NON-NLS-2$//$NON-NLS-4$
		boolean isFilledWasChanged = false;
		double x = (pt1.x+pt2.x)/2. - d.x, y = d.y - (pt1.y+pt3.y)/2. ;
		double width = Math.abs(pt1.x-pt2.x);
		
		if(hasShadow)
		{
			fillType+=",shadow=true";//$NON-NLS-1$
			if(Math.toDegrees(shadowAngle)!=PSTricksConstants.DEFAULT_SHADOW_ANGLE)
				fillType+=",shadowangle="+(float)Math.toDegrees(shadowAngle);//$NON-NLS-1$
			
			if(((float)shadowSize)!=((float)DEFAULT_SHADOW_SIZE))
				fillType+=",shadowsize="+(float)(shadowSize/PPC);//$NON-NLS-1$
			
			if(!shadowColor.equals(PSTricksConstants.DEFAULT_SHADOW_COLOR))
			{
				String name = DviPsColors.getColourName(shadowColor);
				if(name==null)
				{
					name = "color"+number+'e';//$NON-NLS-1$
					DviPsColors.addUserColour(shadowColor, name); 
				}
				fillType += ",shadowcolor=" + name; //$NON-NLS-1$
			}
			if(!isFilled)
			{
				isFilled = true;
				isFilledWasChanged = true;
			}
		}
		
		String str = getPSTricksCodeFilling(ppc);
		if(str.length()>0) fillType=fillType+','+str;
		
		str = getPSTricksCodeLine(ppc);
		if(str.length()>0) add=add+','+str;
		
		if(rotationAngle%(Math.PI*2)!=0.)
		{
			double angle = -Math.toDegrees(rotationAngle);
			double cx = (gravityCenter.x-d.x)/ppc;
			double cy = (d.y-gravityCenter.y)/ppc;
			double x2 = -Math.cos(-rotationAngle)*cx+
						Math.sin(-rotationAngle)*cy+cx;
			double y2 =  -Math.sin(-rotationAngle)*cx-
						Math.cos(-rotationAngle)*cy+cy;
			
			if(Math.abs(x2) < 0.001) x2 = 0;
			if(Math.abs(y2) < 0.001) y2 = 0;
			
			addBegin +="\\rput{"+(float)angle+ "}("+(float)x2+','+(float)y2+"){"; //$NON-NLS-2$ //$NON-NLS-1$ //$NON-NLS-3$
			addEnd = "}"; //$NON-NLS-1$
		}
		
		add+=",dimen="+bordersPosition;  //$NON-NLS-1$		
		
		if(hasDoubleBoundary)
		{
			add+=",doubleline=true,doublesep="+(float)(doubleSep/ppc); //$NON-NLS-1$
			
			if(doubleColor!=PSTricksConstants.DEFAULT_DOUBLE_COLOR)
			{
				String name = DviPsColors.getColourName(doubleColor);
				if(name==null)
				{
					name = "color"+number+'d';//$NON-NLS-1$
					DviPsColors.addUserColour(doubleColor, name); 
				}
				add+= ",doublecolor="+name; //$NON-NLS-1$
			}
		}
		
		if(Math.abs(x) < 0.001) x = 0;
		if(Math.abs(y) < 0.001) y = 0;
		if(isFilledWasChanged) isFilled = false;
		
		return addBegin+"\\pscircle[linewidth=" + (thickness/ppc) +  //$NON-NLS-1$
				add + fillType + "]("  //$NON-NLS-1$
				+ (float)(x/ppc) + ',' + (float)(y/ppc) + "){"  //$NON-NLS-1$
				+ (float)((width/2)/ppc) + '}'+addEnd;
	}

	
	
	
	@Override
	public boolean isIn(LaTeXDrawPoint2D p)
	{
		LaTeXDrawPoint2D pt = rotateInvertPoint(p);	
		
		if(isSelected && (borders.dNE.isIn(pt) || borders.dNW.isIn(pt) 
				   || borders.dSE.isIn(pt) || borders.dSW.isIn(pt)))
			return true;
		
		LaTeXDrawPoint2D NW = getTheNWNonRotatedBoundPoint();
		LaTeXDrawPoint2D SE = getTheSENonRotatedBoundPoint();
		Ellipse2D.Double s = new Ellipse2D.Double(NW.x, NW.y, Math.abs(NW.x - SE.x), Math.abs(NW.y - SE.y));

		if(!s.contains(pt))
			return false;

		if(isFilled || hasShadow || hasGradient())
			return true;

		Shape s2;

		if (hasDoubleBoundary)
			s2 = new Ellipse2D.Double(NW.x + thickness * 2 + doubleSep, NW.y + thickness * 2 + doubleSep, 
						Math.abs(NW.x - SE.x)- 4 * thickness - 2 * doubleSep, 
						Math.abs(NW.y- SE.y) - 4 * thickness - 2 * doubleSep);
		else
			s2 = new Ellipse2D.Double(NW.x + thickness, NW.y + thickness, Math.abs(NW.x - SE.x) - 2 * thickness, 
						Math.abs(NW.y- SE.y)- 2 * thickness);

		return !s2.contains(pt);
	}
	
	
	
	
	
	@Override
	public void draw(Graphics2D g, Object antiAlias, Object rendering, Object alphaInter, Object colorRendering)
	{
		LaTeXDrawPoint2D NW = borders.getTheNWPoint(), SE = borders.getTheSEPoint();
		Color formerCol = g.getColor();
		double cx = (NW.x+SE.x)/2., cy = (NW.y+SE.y)/2.;
		double c2x = Math.cos(rotationAngle)*cx - Math.sin(rotationAngle)*cy;
		double c2y = Math.sin(rotationAngle)*cx + Math.cos(rotationAngle)*cy;
		double c3x = Math.cos(-rotationAngle)*(cx-c2x) - Math.sin(-rotationAngle)*(cy-c2y);
		double c3y = Math.sin(-rotationAngle)*(cx-c2x) + Math.cos(-rotationAngle)*(cy-c2y);
		double dx=0, dy=0;
		boolean changeFillStyle = false;

		if(rotationAngle%(Math.PI*2)!=0)
		{		
			g.rotate(rotationAngle);
			g.translate(c3x,c3y);
		}
		
		if(hasShadow)
		{
			LaTeXDrawPoint2D cg = getGravityCenter();
			LaTeXDrawPoint2D shadowCg = (LaTeXDrawPoint2D)cg.clone();
			shadowCg.setLocation(cg.x+shadowSize, cg.y);
			shadowCg = Figure.rotatePoint(shadowCg, cg, shadowAngle);
			dx = shadowCg.x-cg.x;
			dy = cg.y-shadowCg.y;
		}
		
		if(hasDoubleBoundary)
		{
			Shape s0 = shape;
			Shape s[] = getDbleBoundariesOutInOrMiddle(s0);
			Shape s1, s2, s3;

			if(bordersPosition.equals(PSTricksConstants.BORDERS_INSIDE))
			{
				s1 = s0;
				s2 = s[0];
				s3 = s[1];
			}
			else
				if(bordersPosition.equals(PSTricksConstants.BORDERS_MIDDLE))
				{
					s1 = s[0];
					s2 = s0;
					s3 = s[1];
				}
				else
				{
					s1 = s[0];
					s2 = s[1];
					s3 = s0;
				}

			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
			{
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				if(hasShadow)
				{
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					g.draw(s1);
					g.translate(-dx, -dy);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}

				g.setColor(doubleColor);
				g.setStroke(new BasicStroke((float)(doubleSep + thickness), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.draw(s2);
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering,s3);
				g.setColor(linesColor);
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.draw(s1);
				g.draw(s3);
			}
			else
			{
				if(hasShadow)
				{
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.translate(dx, dy);
					g.setColor(shadowColor);
					g.fill(s1);
					g.draw(s1);
					g.translate(-dx, -dy);
					g.setColor(interiorColor);
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
					g.draw(s2);
					
					if(!isFilled)
					{
						changeFillStyle = true;
						isFilled = true;
					}
				}

				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep), BasicStroke.CAP_ROUND,
							BasicStroke.JOIN_MITER, 1.f, new float[] { 0, (float)(thickness*2+doubleSep + dotSep) }, 0));
				else
					g.setStroke(new BasicStroke((float)(thickness*2+doubleSep),
							BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
							new float[] { blackDashLength, whiteDashLength }, 0));
					
				fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, s2);
				g.setColor(linesColor);
				g.draw(s2);
				g.setStroke(new BasicStroke((float)doubleSep, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.setColor(doubleColor);
				g.draw(s2);
			}				
		}
		else
		{
			Shape shape2 = shape;
			if(hasShadow)
			{
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
				g.translate(dx, dy);
				g.setColor(shadowColor);
				g.fill(shape);
				g.draw(shape);
				g.translate(-dx, -dy);
				
				if(!isFilled)
				{
					changeFillStyle = true;
					isFilled = true;
				}
				g.setColor(interiorColor);
				g.draw(shape);
			}

			if(lineStyle.equals(PSTricksConstants.LINE_NONE_STYLE))
				g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER));
			else
				if(lineStyle.equals(PSTricksConstants.LINE_DOTTED_STYLE))
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_ROUND,
							BasicStroke.JOIN_MITER, 1.f, new float[] { 0, thickness + dotSep }, 0));
				else
					g.setStroke(new BasicStroke(thickness, BasicStroke.CAP_BUTT, BasicStroke.JOIN_MITER, 1.f,
							new float[] { blackDashLength, whiteDashLength }, 0));
			
			fillFigure(g, antiAlias, rendering, alphaInter, colorRendering, shape2);
			g.setColor(linesColor);
			g.draw(shape);
		}
		
		g.setColor(formerCol);
		if(changeFillStyle) isFilled = false;
		
		if(rotationAngle%(Math.PI*2)!=0)
		{
			g.translate(-c3x, -c3y);
			g.rotate(-rotationAngle);
		}
		
		if(isSelected)
			borders.draw(g, false, antiAlias, rendering, alphaInter, colorRendering);
	}

	
	
	@Override
	public Object clone() throws CloneNotSupportedException
	{
		return super.clone();
	}

	
	
	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException
	{
		canHaveShadow = true;
		interiorColor = (Color) ois.readObject();
		lineStyle = (String) ois.readObject();
		rotationAngle = ois.readDouble();
		thickness = ois.readFloat();
		isFilled = ois.readBoolean();
		isSelected = ois.readBoolean();
		isOnRotation = ois.readBoolean();
		linesColor = (Color) ois.readObject();
		blackDashLength = ois.readFloat();
		dotSep = ois.readFloat();
		whiteDashLength = ois.readFloat();
		borders = (LaTeXDrawRectangle) ois.readObject();

		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.5")>=0) //$NON-NLS-1$
		{
			hasDoubleBoundary = ois.readBoolean();
			doubleColor = (Color)ois.readObject();
			doubleSep = ois.readDouble();
			bordersPosition = (String)ois.readObject();
			if(!(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6")>=0)) //$NON-NLS-1$
				ois.readBoolean();
			hatchingAngle = ois.readDouble();
			hatchingColor = (Color)ois.readObject();
			hatchingStyle = (String)ois.readObject();
			hatchingWidth = ois.readFloat();
			
			if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.6") < 0)//$NON-NLS-1$
			{
				if(hatchingStyle.equals(DECREPETED_FILL_CROSS))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_CROSSHATCH;
				else if(hatchingStyle.equals(DECREPETED_FILL_HORIZ))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_HLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_VERT))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_VLINES;
				else if(hatchingStyle.equals(DECREPETED_FILL_NO))
					hatchingStyle = PSTricksConstants.TOKEN_FILL_NONE;
			}
		}
		else
		{
			hasDoubleBoundary  = DEFAULT_HAS_DOUBLE_BOUNDARY;
			doubleColor = DEFAULT_DOUBLE_COLOR;
			doubleSep   = DEFAULT_DOUBLESEP;
			bordersPosition = DEFAULT_BORDERS_POSITION;
			hatchingAngle = DEFAULT_HATCH_ANGLE;
			hatchingColor = DEFAULT_HATCH_COL;
			hatchingStyle = DEFAULT_HATCH_STYLE;
			hatchingWidth = DEFAULT_HATCH_WIDTH;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.7")>=0) //$NON-NLS-1$
		{
			hasShadow 	= ois.readBoolean();
			shadowAngle = ois.readDouble();
			shadowSize	= ois.readDouble();
			shadowColor	= (Color)ois.readObject();
			gradientEndColor = (Color)ois.readObject();
			gradientStartColor = (Color)ois.readObject();
			gradientAngle = ois.readDouble();
			gradientMidPoint = ois.readDouble();
		}
		else
		{
			hasShadow 	= DEFAULT_SHADOW_HAS;
			shadowAngle	= DEFAULT_SHADOW_ANGLE;
			shadowSize	= DEFAULT_SHADOW_SIZE;
			shadowColor	= DEFAULT_SHADOW_COLOR;
			gradientEndColor = PSTricksConstants.DEFAULT_GRADIENT_END_COLOR;
			gradientStartColor = PSTricksConstants.DEFAULT_GRADIENT_START_COLOR;
			gradientAngle = DEFAULT_GRADIENT_ANGLE;
			gradientMidPoint = DEFAULT_GRADIENT_MID_POINT;
		}
		
		if(LaTeXDrawFrame.getVersionOfFile().compareTo("1.8")>=0) //$NON-NLS-1$
			hatchingSep = ois.readDouble();
		else
			hatchingSep = DEFAULT_HATCH_SEP;
		
		updateGravityCenter();
		updateStyleOfDelimitors();
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	/**
	 * Compute the intersection points between a circle and a line.
	 * @param coeffA The a of y=ax+b of the second line
	 * @param coeffB The b of y=ax+b of the second line
	 * @return Null if there is no point ; else the two points.
	 */
	public LaTeXDrawPoint2D[] getIntersectionPoints(double coeffA, double coeffB)
	{
		double R = Math.abs(gravityCenter.x-getTheNWPoint().x);
		double c = coeffB*coeffB-R*R;
		double b = 2*coeffA*coeffB;
		double a = (1+coeffA)*(1+coeffA);
		double delta = b*b - 4*a*c;
		
		if(delta<=0) return null;
		
		double x1 = (-b-Math.sqrt(delta))/(2*a);
		double x2 = (-b+Math.sqrt(delta))/(2*a);
						
		LaTeXDrawPoint2D[] pts = new LaTeXDrawPoint2D[2];
		pts[0] = new LaTeXDrawPoint2D(x1, coeffA*x1+coeffB);
		pts[1] = new LaTeXDrawPoint2D(x2, coeffA*x2+coeffB);
		
		return pts;
	}

	
	
	@Override
	public synchronized void setThickness(float value)
	{
		super.setThickness(value);
		shape = getInsideOutsideOrMiddleBorders();
	}
	
	
	
	
	/**
	 * @return The radius of the circle.
	 * @since 1.9
	 */
	public synchronized double getRadius()
	{
		return Math.abs(borders.getPoint(0).x-borders.getPoint(1).x)/2.;
	}
	
	
	
	@Override
	public int hashCode()
	{
		return super.hashCode()^((int)getRadius());
	}
}
