/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#ifdef WIN32
#include <windows.h>
#endif

#include <QtGui>
#include <QTimer>

#include "containerutils.h" //Item enum TODO remove

#include "logger.h"
#include "Settings.h"
#include "tagdialog.h"
#include "WebService.h"
#include "WebService/Request.h"


TagDialog::TagDialog( int defaultTagType, QWidget *parent )
        : QDialog( parent, Qt::Dialog )
{
    ui.setupUi( this );

    QRegExp rx( "[a-zA-Z0-9\\-:, ]{1,255}" );
    QValidator *validator = new QRegExpValidator( rx, this );
    ui.tagEdit->setValidator( validator );

    if ( defaultTagType == -1 )
    {
        ui.tagTypeBox->setCurrentIndex( The::settings().currentUser().lastTagType( 1 ) );
    }
    else
    {
        ui.tagTypeBox->setCurrentIndex( defaultTagType );
    }

    connect( ui.tagTypeBox, SIGNAL(activated( int )), SLOT(tagTypeChanged( int )) );
    connect( ui.okButton, SIGNAL(clicked()), SLOT(okPressed()) );
    connect( ui.cancelButton, SIGNAL(clicked()), SLOT(cancelPressed()) );
    connect( ui.personalTagsList, SIGNAL(itemDoubleClicked( QListWidgetItem* )), SLOT(addTag( QListWidgetItem* )) );
    connect( ui.publicTagsList, SIGNAL(itemDoubleClicked( QListWidgetItem* )), SLOT(addTag( QListWidgetItem* )) );
    connect( ui.tagEdit, SIGNAL(textChanged(QString)), SLOT(validateTag()) );

//////
    UserTagsRequest *r = new UserTagsRequest;
    connect( r, SIGNAL(result( Request* )), SLOT(tagsResult( Request* )) );
    r->start();
}


void
TagDialog::setSong( MetaData metaData )
{
    m_metaData = metaData;
    tagTypeChanged( ui.tagTypeBox->currentIndex() );

    ui.tagTypeBox->setItemText( 0, QString( tr( "Artist: %1" ) ).arg( metaData.artist() ) );
    ui.tagTypeBox->setItemText( 1, QString( tr( "Track: %1 - %2" ) ).arg( metaData.artist() ).arg( metaData.track() ) );
    ui.tagTypeBox->setItemText( 2, QString( tr( "Album: %1 - %2" ) ).arg( metaData.artist() ).arg( metaData.album() ) );

    Request *r = new ArtistTagsRequest( m_metaData.artist() );
    connect( r, SIGNAL(result( Request* )), SLOT(tagsResult( Request* )) );
    r->start();
}


void
TagDialog::setSong( Track track )
{
    m_metaData.setArtist( track.artist() );
    m_metaData.setTrack( track.title() );
    tagTypeChanged( ui.tagTypeBox->currentIndex() );
    
    ui.tagTypeBox->setItemText( 0, QString( tr( "Artist: %1" ) ).arg( track.artist() ) );
    ui.tagTypeBox->setItemText( 1, QString( tr( "Track: %1 - %2" ) ).arg( track.artist() ).arg( track.title() ) );
    ui.tagTypeBox->removeItem( 2 );
    
    Request *r = new ArtistTagsRequest( m_metaData.artist() );
    connect( r, SIGNAL(result( Request* )), SLOT(tagsResult( Request* )) );
    r->start();
    
    // We can't tag album in this case, so if that's in the config, override
    int lastTagType = The::settings().currentUser().lastTagType();
    if ( lastTagType != 2 )
        ui.tagTypeBox->setCurrentIndex( lastTagType );
    else
        ui.tagTypeBox->setCurrentIndex( 0 );
}


void
TagDialog::okPressed()
{
    SetTagRequest *request = new SetTagRequest;
    request->setTag( ui.tagEdit->text() );
    request->setArtist( m_metaData.artist() );

    switch( ui.tagTypeBox->currentIndex() )
    {
        case 0: // artist
            request->setType( ItemArtist );
            break;

        case 1: // track
            request->setType( ItemTrack );
            request->setToken( m_metaData.track() );
            break;

        case 2: // album
            request->setType( ItemAlbum );
            request->setToken( m_metaData.album() );
            break;

        default:
            delete request;
            return;
    }

    request->start();

    UserSettings &user = The::settings().currentUser();
    user.setPersonalTagsListSortOrder( (int)ui.personalTagsList->sortOrder() );
    user.setPublicTagsListSortOrder( (int)ui.publicTagsList->sortOrder() );
}


void
TagDialog::cancelPressed()
{
    UserSettings &user = The::settings().currentUser();
    user.setPersonalTagsListSortOrder( (int)ui.personalTagsList->sortOrder() );
    user.setPublicTagsListSortOrder( (int)ui.publicTagsList->sortOrder() );    
}


void
TagDialog::tagTypeChanged( int type )
{
    ui.tagTypeBox->setEnabled( false );
    ui.tagEdit->setEnabled( false );
    ui.tagEdit->setText( tr( "Loading..." ) );

    UserArtistTagsRequest *tags = 0;

    switch ( type )
    {
        case 0:
            tags = new UserArtistTagsRequest;
            break;

        case 1:
            tags = new UserTrackTagsRequest;
            ((UserTrackTagsRequest*)tags)->setTrack( m_metaData.track() );
            break;

        case 2:
            tags = new UserAlbumTagsRequest;
            ((UserAlbumTagsRequest*)tags)->setAlbum( m_metaData.album() );
            break;
    }

    tags->setArtist( m_metaData.artist() );
    connect( tags, SIGNAL(result( Request* )), SLOT(tagsResult( Request* )) );
    tags->start();

    The::settings().currentUser().setLastTagType( ui.tagTypeBox->currentIndex() );
}


void
TagDialog::tagsResult( Request *request )
{
    TagsRequest *tags = static_cast<TagsRequest*>( request );
    switch (tags->type())
    {
        case TypeArtistTags:
            ui.publicTagsList->clear();
            ui.publicTagsList->addItems( tags->tags() );
            break;

        case TypeUserTags:
            ui.personalTagsList->clear();
            ui.personalTagsList->addItems( tags->tags() );
            break;

        case TypeUserArtistTags:
        case TypeUserAlbumTags:
        case TypeUserTrackTags:
            ui.tagTypeBox->setEnabled( true );
            ui.tagEdit->setEnabled( true );
            ui.tagEdit->setText( tags->tags().operator QStringList().join( ", " ) );
            ui.tagEdit->setFocus();
            break;

        default:
            break;
    }

    ui.okButton->setEnabled( false );
    m_originalTags = ui.tagEdit->text();

    UserSettings &user = The::settings().currentUser();
    ui.personalTagsList->setSortOrder( (Tags::SortOrder) user.personalTagsListSortOrder() );
    ui.publicTagsList->setSortOrder( (Tags::SortOrder) user.publicTagsListSortOrder() );
}


void
TagDialog::addTag( QListWidgetItem *item )
{
    QString itemText = item->text().trimmed();
    QString existingText = ui.tagEdit->text().trimmed();
    QStringList tags;

    foreach( QString tag, ui.tagEdit->text().trimmed().split( "," ) )
        tags << tag.trimmed();

    if ( tags.contains( itemText ) )
        return;

    if ( existingText.isEmpty() )
    {
        ui.tagEdit->setText( itemText );
    }
    else if ( existingText.endsWith( "," ) )
    {
        ui.tagEdit->setText( existingText + " " + itemText );
    }
    else
    {
        ui.tagEdit->setText( existingText + ", " + itemText );
    }
}


void
TagDialog::validateTag()
{
    // Don't disable the Ok button, even if the tag-list is empty,
    // the user might want to remove all tags from an item.
    ui.okButton->setEnabled( m_originalTags != ui.tagEdit->text() );
}
