/***************************************************************************
 *   Copyright (C) 2005 - 2007 by                                          *
 *      Christian Muehlhaeuser, Last.fm Ltd <chris@last.fm>                *
 *      Erik Jaelevik, Last.fm Ltd <erik@last.fm>                          *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA  02111-1307, USA.          *
 ***************************************************************************/

#include "last.fm.h"
#include "containerutils.h" //dataPath()
#include <QPainter>
#include <QPixmap>
#include "Radio.h"
#include "searchextension.h"
#include "watermarkwidget.h"
#include "WebService/Request.h"


// These are pixel sizes
#ifdef Q_WS_MAC
static const int k_fontMin = 13;
#else
static const int k_fontMin = 11;
#endif
static const int k_fontMax = 24;
static const int k_lineHeight = 29;

static const int k_maxItemsTopTags = 35;
static const int k_maxItems = 35;

static const QColor k_tagFontColour = QColor( 0x6d, 0x83, 0xa2 );


SearchExtension::SearchExtension( QWidget *parent ) :
        QWidget( parent ),
        m_type( 0 ), // set artist search as default, despite presenting the toptags
        m_currentCloudType( 1 ), // tags in cloud
        m_topTags( false )
{
    Q_DEBUG_BLOCK;

    /*
        On the Mac, we need to put a spacer above the scroll area to reduce the
        ugliness of the scrollbar not going all the way up to the line. On
        Windows we just set the spacer to 0.
    */

    // The widget inside the scroll area
    WatermarkWidget* searchExtWidget = new WatermarkWidget( this );
    searchExtWidget->setWatermark( dataPath( "watermark.png" ) );

    ui.setupUi( searchExtWidget );

    requestTopTags();

    QPalette palette( QTabWidget().palette() );
    palette.setColor( QPalette::Base, palette.window().color() );

    // Tag cloud
    ui.resultBrowser->setPalette( palette );
    ui.resultBrowser->setItems( QStringList() );
    ui.resultBrowser->setUniformLineHeight( k_lineHeight );
    ui.resultBrowser->setItemsSelectable( true );
    ui.resultBrowser->setAlignment( Qt::AlignLeft | Qt::AlignBottom );
    ui.resultBrowser->setJustified( true );

    // Top search widgets
    ui.searchType->setCurrentIndex( m_type );
    ui.searchType->setEnabled( false );
    ui.searchButton->setEnabled( false );
    ui.playButton->setEnabled( false );

    ui.searchEdit->installEventFilter( this );

    m_spinnerMovie.setFileName( dataPath( "progress.mng" ) );
    m_spinnerMovie.start();
    ui.spinnerLabel->setMovie( &m_spinnerMovie );

  #ifdef Q_WS_MAC
    // Hack some minsizes for the buttons as Qt is being gay on the Mac
    ui.searchButton->setFixedWidth( ui.searchButton->sizeHint().width() + 0 );
    ui.playButton->setFixedWidth( ui.playButton->sizeHint().width() + 0 );
  #endif

    // Scroll area
    m_scrollArea = new QScrollArea( this );

    QPalette p = m_scrollArea->palette();
    p.setColor( QPalette::Window, Qt::white );
    m_scrollArea->setPalette( p );
    m_scrollArea->setHorizontalScrollBarPolicy( Qt::ScrollBarAsNeeded );
    m_scrollArea->setVerticalScrollBarPolicy( Qt::ScrollBarAsNeeded );
    m_scrollArea->setFrameStyle( QFrame::NoFrame );
    m_scrollArea->setWidgetResizable( true );
    m_scrollArea->setWidget( searchExtWidget );

    // Create a vertical layout onto which we'll add a spacer (for Mac) and the
    // scroll area.
    QVBoxLayout* vbl = new QVBoxLayout( this );
    vbl->setMargin( 0 );
    vbl->setSpacing( 0 );
    #ifdef Q_WS_MAC
        // Add space to the spacer above the scroll area and remove the same
        // amount from the one below.
        QSpacerItem* macSpace = new QSpacerItem( 1, 30, QSizePolicy::Fixed, QSizePolicy::Minimum );
        vbl->addItem( macSpace );
//        ui.spacerItem->changeSize( 1, 0, QSizePolicy::Fixed, QSizePolicy::Minimum );
    #endif
    vbl->addWidget( m_scrollArea );

    connect( ui.searchEdit, SIGNAL(returnPressed()), ui.searchButton, SLOT(animateClick()) );
    connect( ui.searchEdit, SIGNAL(textChanged( QString )), SLOT(searchFieldChanged()) );
    connect( ui.searchButton, SIGNAL(clicked()), SLOT(search()) );
    connect( ui.playButton, SIGNAL(clicked() ), SLOT(play()) );
    connect( ui.resultBrowser, SIGNAL(clicked( int )), SLOT(itemClicked( int )) );
}


bool
SearchExtension::eventFilter( QObject* o, QEvent* e )
{
    if (o == ui.searchEdit)
    {
        switch (e->type())
        {
            case QEvent::FocusOut:
            case QEvent::FocusIn:
                ui.searchEdit->update();
                break;

            case QEvent::Paint:
                if (!ui.searchEdit->hasFocus() && ui.searchEdit->text().isEmpty())
                {
                    ui.searchEdit->event( e );

                    QRect r = ui.searchEdit->rect().adjusted( 5, 2, -5, 0 );
                    QPainter p( ui.searchEdit );
                    p.setPen( Qt::gray );
                    p.setFont( ui.searchEdit->font() );
                    p.drawText( r, Qt::AlignVCenter, tr( "Find a station by" ) );
                    ui.searchEdit->setMinimumWidth( p.fontMetrics().width( tr( "Find a station by" ) ) + 12 );

                    return true; //eat event
                }
                break;

            default:
                break;
        }
    }

    return QWidget::eventFilter( o, e );
}

void
SearchExtension::requestTopTags()
{
    ui.spinnerLabel->show();

    m_topTags = true;

    TopTagsRequest *tags = new TopTagsRequest;
    connect( tags, SIGNAL(result( Request* )), SLOT(searchResults( Request* )) );
    tags->start();

    ui.statusLabel->setText( tr( "Generating popular tags..." ) );
}

void
SearchExtension::searchFieldChanged()
{
    ui.playButton->setEnabled( !ui.searchEdit->text().isEmpty() );
}

void
SearchExtension::search()
{
    if ( m_type != ui.searchType->currentIndex() )
    {
        // search mode changed, wipe selections
        clearSelection();
        ui.resultBrowser->clear();
        m_type = ui.searchType->currentIndex();
    }

    const QString searchText = ui.searchEdit->text().trimmed();
    if ( searchText.isEmpty() )
    {
        requestTopTags();
        return;
    }

    if ( searchText.startsWith( "lastfm://" ) )
    {
        play();
        return;
    }

    ui.searchButton->setEnabled( false );
    ui.searchType->setEnabled( false );
    ui.resultBrowser->setItemsSelectable( false );
    ui.spinnerLabel->show();

    Request *request;
    switch( m_type )
    {
        case 0:
            request = new SimilarArtistsRequest( searchText );

            ui.statusLabel->setText( tr( "Generating similar artists..." ) );
            ui.resultBrowser->setItemType( ItemArtist );
            break;

        case 1:
            request = new SimilarTagsRequest( searchText );

            ui.statusLabel->setText( tr( "Generating similar tags..." ) );
            ui.resultBrowser->setItemType( ItemTag );
            break;
    }

    connect( request, SIGNAL(result( Request* )), SLOT(searchResults( Request* )) );
    request->start();

    m_searching = true;
}

void
SearchExtension::searchResults( Request *r )
{
    Q_DEBUG_BLOCK;

    WeightedStringList items;
    QString searchToken;

    switch (r->type()) {
        case TypeSimilarArtists:
            items = static_cast<SimilarArtistsRequest*>(r)->artists();
            searchToken = static_cast<SimilarArtistsRequest*>(r)->artist();
            break;
        case TypeTopTags:
        case TypeSimilarTags:
            items = static_cast<TagsRequest*>(r)->tags();
            break;

        default:
            qWarning() << "Unhandled case" << __PRETTY_FUNCTION__;
    }

//////
    int total = k_maxItems;
    if ( m_topTags )
    {
        // We want more tags for the initial top tags cloud
        m_topTags = false;
        total = k_maxItemsTopTags;
    }
    else
    {
        m_currentCloudType = m_type;
    }

    WeightedStringList alpha_sorted = items.mid( 0, total );

    // They are still sorted by weight when they get here,
    // so element 0 will have the biggest weighting
    int const max = alpha_sorted.isEmpty() ? 0 : alpha_sorted.first().weighting();
    int const min = alpha_sorted.size() < 2 ? 0 : alpha_sorted.last().weighting();

    // this will sort them alphabetically
    qSort( alpha_sorted.begin(), alpha_sorted.end() );

    ui.resultBrowser->clear();
    for ( int i = 0; i < alpha_sorted.count(); i++ )
    {
        QString name = alpha_sorted[i];
        ui.resultBrowser->append( name );

        int w = alpha_sorted[i].weighting();
        float normalised_w = (float) (w - min) / qMax( 1, max - min );
        int size = int(normalised_w * (k_fontMax - k_fontMin) + k_fontMin + 0.5f);

        QFont f( "Arial" );
        f.setBold( true );

        // we qBound because we don't check that maxWeight is valid, and it is
        // sometimes the case that it in fact isn't, thus we get a crazy font sizes
        // tag cloud! :) --mxcl
        f.setPixelSize( qBound( k_fontMin, size, k_fontMax ) );

        ui.resultBrowser->setItemFont( i, f );
        ui.resultBrowser->setItemColor( i, k_tagFontColour );

        // why on earth use a QHash? --mxcl
        QHash<QString, QString> data;
        data.insert( "artist", name ); // eh?
        ui.resultBrowser->setItemData( i, data );
    }

    ui.searchButton->setEnabled( true );
    ui.searchType->setEnabled( true );
    ui.resultBrowser->setItemsSelectable( true );
    ui.statusLabel->clear();
    ui.spinnerLabel->hide();

    if ( items.count() )
    {
        switch ( m_currentCloudType )
        {
            case 0:
                ui.resultBrowser->setJustified( false );
                break;

            case 1:
                ui.resultBrowser->setJustified( true );
                break;
        }
    }
    else
    {
        ui.playButton->setEnabled( false );

        ui.statusLabel->setText( tr( "Sorry, your search didn't return any results." ) );
    }

    if ( ui.searchEdit->text().trimmed().isEmpty() || alpha_sorted.count() == 0 )
    {
        ui.searchButton->setDefault( true );
        ui.playButton->setDefault( false );
        ui.searchButton->setFocus();

        if ( alpha_sorted.count() == 0 )
            ui.searchEdit->setFocus();
    }
    else
    {
        ui.searchButton->setDefault( false );
        ui.playButton->setDefault( true );
        ui.playButton->setFocus();
    }

    m_searching = false;
}


StationUrl
SearchExtension::stationUrl()
{
    QString text = ui.searchEdit->text();

    if (text.startsWith( "lastfm://" ))
        return text; //we assume it's percent encoded as double encoding would break the url

    text = QString::fromUtf8( QUrl::toPercentEncoding( text ) );

    switch (m_type)
    {
        case 0:
            return StationUrl( "lastfm://artist/" + text + "/similarartists" );
        case 1:
            return StationUrl( "lastfm://globaltags/" + text );
    }

    return StationUrl();
}


void
SearchExtension::itemClicked( int index )
{
    if (m_searching)
        return;

    QString item = ui.resultBrowser->items().at( index );

    if ( m_currentCloudType != ui.searchType->currentIndex() )
    {
        ui.searchType->setCurrentIndex( m_currentCloudType );
    }

    ui.searchEdit->setText( item );
    search();
}


void
SearchExtension::play()
{
    ui.searchButton->setDefault( true );
    ui.playButton->setDefault( false );
    ui.searchButton->setFocus();

    The::radio().playStation( stationUrl() );
}


void
SearchExtension::clearSelection()
{
    ui.resultBrowser->clearSelections();
}
