#include <QtCore>
#include "mediadevicewatcher.h"

#include "controlinterface.h"
#include "logger.h"
#include "containerutils.h"
#include "Settings.h"

#ifdef WIN32

#include <QtGui>
#include <windows.h>
#include <dbt.h>

char FirstDriveFromMask (ULONG unitmask);

/*------------------------------------------------------------------
   FirstDriveFromMask (unitmask)

   Description
     Finds the first valid drive letter from a mask of drive letters.
     The mask must be in the format bit 0 = A, bit 1 = B, bit 3 = C, 
     etc. A valid drive letter is defined when the corresponding bit 
     is set to 1.

   Returns the first drive letter that was found.
--------------------------------------------------------------------*/

char FirstDriveFromMask (ULONG unitmask)
{
    char i;

    for (i = 0; i < 26; ++i)
    {
        if (unitmask & 0x1)
            break;
        unitmask = unitmask >> 1;
    }

    return (i + 'A');
}


class DeviceTargetWindow : public QWidget
{
    public:
        DeviceTargetWindow( MediaDeviceWatcher* parent );

        protected:
            virtual bool winEvent( MSG* message, long* result );

        private:
            MediaDeviceWatcher* m_parent;
};


DeviceTargetWindow::DeviceTargetWindow( MediaDeviceWatcher* parent ) :
    QWidget( NULL )
{
    m_parent = parent;

    DEV_BROADCAST_HDR hdr;
    ZeroMemory( &hdr, sizeof(hdr) );
    hdr.dbch_size = sizeof(DEV_BROADCAST_HDR);
    hdr.dbch_devicetype = DBT_DEVTYP_VOLUME;

    HDEVNOTIFY hd = RegisterDeviceNotification(
        winId(),
        reinterpret_cast<void*>(&hdr),
        DEVICE_NOTIFY_WINDOW_HANDLE );
}
#endif


#ifdef WIN32
class ListenerApplication : public QApplication
#else
class ListenerApplication : public QCoreApplication
#endif
{
    public:
        ListenerApplication( int argc, char **argv );
        ~ListenerApplication();

        bool m_quit;

    private:
        ControlInterface* m_controlInterface;
        MediaDeviceWatcher* m_mdw;

#ifdef WIN32
        DeviceTargetWindow* m_wnd;
#endif
};


ListenerApplication::ListenerApplication( int argc, char **argv )
#ifdef WIN32
    : QApplication( argc, argv )
#else
    : QCoreApplication( argc, argv )
#endif
    , m_quit( false )
{
    Logger& logger = Logger::GetLogger();
    logger.Init( savePath( "lastfmhelper.log" ), false );
    logger.SetLevel( Logger::Debug );

    QStringList argList = arguments();
    argList.removeAt( 0 );

    bool alreadyRunning = ControlInterface::sendToInstance();
    if ( alreadyRunning )
    {
        QString params = argList.join( " " );
        if ( !params.isEmpty() )
        {
            ControlInterface::sendToInstance( params );
        }
        m_quit = true;
    }
    else
    {
        if ( argList.contains( "--quit" ) )
        {
            m_quit = true;
        }
    }

    if ( !m_quit )
    {   
        QString version = The::settings().version();
        LOGL( 1, "App version: " << version );
    
        m_controlInterface = new ControlInterface( this );

        m_mdw = new MediaDeviceWatcher();
      #ifdef WIN32
        m_wnd = new DeviceTargetWindow( m_mdw );
      #endif
    }
}


ListenerApplication::~ListenerApplication()
{
    if ( m_quit )
        return;

  #ifdef WIN32
    delete m_wnd;
  #endif
    delete m_mdw;
    delete m_controlInterface;
}


#ifdef WIN32
bool
DeviceTargetWindow::winEvent( MSG * msg, long * result )
{
    /*
    typedef struct MSG {
        HWND        hwnd;
        UINT        message;
        WPARAM      wParam;
        LPARAM      lParam;
        DWORD       time;
        POINT       pt;
    }*/

    if ( msg->message != WM_DEVICECHANGE )
        return false;

    PDEV_BROADCAST_HDR lpdb = (PDEV_BROADCAST_HDR)msg->lParam;

    switch( msg->wParam )
    {
        case DBT_DEVICEARRIVAL:
        {
            if (lpdb -> dbch_devicetype == DBT_DEVTYP_VOLUME)
            {
                PDEV_BROADCAST_VOLUME lpdbv = (PDEV_BROADCAST_VOLUME)lpdb;
                //if (lpdbv -> dbcv_flags & DBTF_MEDIA)
                    qDebug() << "Drive" << FirstDriveFromMask(lpdbv ->dbcv_unitmask) << " media has arrived.";

                m_parent->forceDetection( QString( FirstDriveFromMask(lpdbv ->dbcv_unitmask)) + ":\\" );

                return true;
            }
        }
        break;

        case DBT_DEVICEREMOVECOMPLETE:
        {
            if (lpdb -> dbch_devicetype == DBT_DEVTYP_VOLUME)
            {
                PDEV_BROADCAST_VOLUME lpdbv = (PDEV_BROADCAST_VOLUME)lpdb;
                //if (lpdbv -> dbcv_flags & DBTF_MEDIA)
                    qDebug() << "Drive" << FirstDriveFromMask(lpdbv ->dbcv_unitmask) << " media is gone.";

                return true;
            }
        }
        break;
   }

    return false;
}
#endif


int main( int argc, char **argv )
{
    // used by some Qt stuff, eg QSettings
    // leave first! As Settings object is created quickly
    QCoreApplication::setApplicationName( "Last.fm" );
    QCoreApplication::setOrganizationName( "Last.fm" );
    QCoreApplication::setOrganizationDomain( "last.fm" );

    ListenerApplication app( argc, argv );

    if ( !app.m_quit )
    {
        qDebug() << "Running n waiting!";
        app.exec();
    }

    qDebug() << "Quitting helper";

    return 0;
}
