/*
 *   LADCCA
 *    
 *   Copyright (C) 2002 Robert Ham <rah@bash.sh>
 *    
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <netinet/in.h>
#include <stdio.h>

#include <ladcca/config.h>
#include <ladcca/xmalloc.h>
#include <ladcca/internal.h>

void
cca_config_init (cca_config_t * config)
{
  config->key = NULL;
  config->value = NULL;
  config->value_size = 0;
}

cca_config_t *
cca_config_new ()
{
  cca_config_t * config;
  
  config = cca_malloc (sizeof (cca_config_t));
  cca_config_init (config);
  
  return config;
}

cca_config_t *
cca_config_dup (const cca_config_t * config)
{
  cca_config_t * config_dup;
  
  config_dup = cca_config_new ();
  cca_config_set_key (config_dup, cca_config_get_key (config));
  cca_config_set_value (config_dup, cca_config_get_value (config),
                                    cca_config_get_value_size (config));
  
  return config_dup;
}

cca_config_t *
cca_config_new_with_key (const char * key)
{
  cca_config_t * config;
  config = cca_config_new ();
  cca_config_set_key (config, key);
  return config;
}

void
cca_config_free (cca_config_t * config)
{
  if (config->key)
    {
      free (config->key);
      config->key = NULL;
    }
    
  if (config->value)
    {
      free (config->value);
      config->value = NULL;
    }
  
  config->value_size = 0;
}

void
cca_config_destroy (cca_config_t * config)
{
  cca_config_free (config);
  free (config);
}

const char *
cca_config_get_key (const cca_config_t * config)
{
  return config->key;
}

const void *
cca_config_get_value (const cca_config_t * config)
{
  return config->value;
}

size_t
cca_config_get_value_size (const cca_config_t * config)
{
  return config->value_size;
}

uint32_t
cca_config_get_value_int (const cca_config_t * config)
{
  const uint32_t * value;
  
  value = (const uint32_t *) cca_config_get_value (config);
  
  return ntohl(*value);
}

float
cca_config_get_value_float (const cca_config_t * config)
{
  const float * value;
  
  value = (const float *) cca_config_get_value (config);
  
  return *value;
}

double
cca_config_get_value_double (const cca_config_t * config)
{
  const double * value;
  
  value = (const double *) cca_config_get_value (config);
  
  return *value;
}

const char *
cca_config_get_value_string (const cca_config_t * config)
{
  return (const char *) cca_config_get_value (config);
}

void
cca_config_set_key (cca_config_t * config, const char * key)
{
  /* free the existing key */
  if (config->key)
    free (config->key);
  
  /* copy the new one */
  config->key = cca_strdup (key);
}

void
cca_config_set_value (cca_config_t * config, const void * value, size_t value_size)
{
  /* check whether the value size will survive the network */
  if (ntohl (htonl (value_size)) != value_size)
    {
      fprintf (stderr, "%s: value_size is too big and will not survive the network!\n",
               __FUNCTION__);
    }

  /* free the existing value */
  if (config->value)
    free (config->value);
  
  if (value)
    {
      config->value = cca_malloc (value_size);
      memcpy (config->value, value, value_size);
      
      config->value_size = value_size;
    }
  else
    {
      config->value = NULL;
      config->value_size = (size_t) 0;
    }
}

void
cca_config_set_value_int (cca_config_t * config, uint32_t value)
{
  uint32_t nvalue;
  
  nvalue = htonl (value);
  
  cca_config_set_value (config, &nvalue, sizeof (uint32_t));
}

void
cca_config_set_value_float (cca_config_t * config, float value)
{
  cca_config_set_value (config, &value, sizeof (float));
}

void
cca_config_set_value_double (cca_config_t * config, double value)
{
  cca_config_set_value (config, &value, sizeof (value));
}

void
cca_config_set_value_string (cca_config_t * config, const char * value)
{
  cca_config_set_value (config, value, strlen (value) + 1);
}


/* EOF */
