
#include <rqcachecontrol.h>

#include <rpackagelister.h>

#include <qlistview.h>

#include <rqpackageitem.h>
#include <rqfetchwindow.h>
#include <rqchangeswindow.h>

void RQCacheControl::markPackage(RPackage *pkg, int mark, bool fix)
{
   switch (mark) {
      case MarkKeep:
         pkg->setKeep();
         break;
      case MarkInstall:
         pkg->setInstall();
         break;
      case MarkReInstall:
         pkg->setKeep();
         pkg->setReInstall(true);
         break;
      case MarkRemove:
         pkg->setRemove(false);
         break;
      case MarkPurge:
         pkg->setRemove(true);
         break;
   }
   if (fix && !_lister->check())
      _lister->fixBroken();
}

void RQCacheControl::markPackages(vector<RPackage *> packages,
                                  int mark, bool fix)
{
   if (packages.empty())
      return;

   RPackageLister::pkgState state;
   _lister->saveState(state);

   for (unsigned int i = 0; i != packages.size(); i++)
      markPackage(packages[i], mark, false);

   if (fix && !_lister->check())
      _lister->fixBroken();

   vector<RPackage *> changed;
   _lister->getStateChanges(state, changed, changed, changed,
                            changed, changed, changed, changed, packages);

   bool rejected = false;
   if (!changed.empty()) {
      RQChangesWindow changes(_parent, changed);
      changes.exec();
      if (changes.result() == QDialog::Rejected) {
         _lister->restoreState(state);
         rejected = true;
      }
   }

   // XXX Check for unsuccessful changes?

   if (!rejected)
      _lister->saveUndoState(state);
}



RQListViewControl::RQListViewControl(RPackageLister *lister,
                                     QListView *listView)
   : RQCacheControl(lister, listView), _packageListView(listView),
     _packagePopup(listView), _packageTip(listView), _showIcon(true)
{
   connect(&_packagePopup, SIGNAL(activated(int)),
           this, SLOT(markPackagesFromPopup(int)));
}

void RQListViewControl::clearPackages()
{
   _packageListView->clear();
}

void RQListViewControl::reloadViewPackages()
{
   _packageListView->clear();
   for (int i = 0; i != _lister->viewPackagesSize(); i++) {
      RPackage *pkg = _lister->getViewPackage(i);
      (void)new RQPackageItem(_packageListView, pkg);
   }
}

void RQListViewControl::reloadAllPackages()
{
   _packageListView->clear();
   for (int i = 0; i != _lister->packagesSize(); i++) {
      RPackage *pkg = _lister->getPackage(i);
      (void)new RQPackageItem(_packageListView, pkg);
   }
}

vector<RPackage *> RQListViewControl::getSelected()
{
   vector<RPackage *> packages;
   RQPackageItem *pkgItem = (RQPackageItem*)_packageListView->firstChild();
   while (pkgItem) {
      if (pkgItem->isSelected())
         packages.push_back(pkgItem->pkg);
      pkgItem = (RQPackageItem *)pkgItem->nextSibling();
   }
   return packages;
}

void RQListViewControl::setClickable(bool flag)
{
   if (flag) {
      connect(_packageListView, SIGNAL(contextMenuRequested(QListViewItem *,
                                                            const QPoint &, int)),
              this, SLOT(rightClickedOnPackageList(QListViewItem *,
                                                   const QPoint &, int)));
      connect(_packageListView, SIGNAL(clicked(QListViewItem *,
                                               const QPoint &, int)),
              this, SLOT(leftClickedOnPackageList(QListViewItem *,
                                                  const QPoint &, int)));
   } else {
      disconnect(_packageListView, SIGNAL(contextMenuRequested(QListViewItem *,
                                                               const QPoint &, int)),
                 this, SLOT(rightClickedOnPackageList(QListViewItem *,
                                                      const QPoint &, int)));
      disconnect(_packageListView, SIGNAL(clicked(QListViewItem *,
                                                  const QPoint &, int)),
                 this, SLOT(leftClickedOnPackageList(QListViewItem *,
                                                     const QPoint &, int)));
   }
}

void RQListViewControl::markPackagesFromPopup(int id)
{
   vector<RPackage *> packages = getSelected();
   if (!packages.empty()) {
      switch (id) {
         case RQPackagePopup::IdUnmark:
            markPackages(packages, MarkKeep);
            break;
         case RQPackagePopup::IdInstall:
         case RQPackagePopup::IdUpgrade:
            markPackages(packages, MarkInstall);
            break;
         case RQPackagePopup::IdReinstall:
            markPackages(packages, MarkReInstall);
            break;
         case RQPackagePopup::IdRemove:
            markPackages(packages, MarkRemove);
            break;
      }
      _packageListView->triggerUpdate();
   }
}

void RQListViewControl::rightClickedOnPackageList(QListViewItem *,
                                                  const QPoint &pos, int)
{
   _packagePopup.update(_packageListView);
   _packagePopup.popup(pos, _packagePopup.firstEnabledId());
}

void RQListViewControl::leftClickedOnPackageList(QListViewItem *,
                                                 const QPoint &pos, int col)
{
   if (col == 0) {
      _packagePopup.update(_packageListView);
      _packagePopup.popup(pos, _packagePopup.firstEnabledId());
   }
}

// vim:ts=3:sw=3:et
