/***************************************************************************
                          commands_edit  -  description
                             -------------------
    begin                : Wed Nov 26 2003
    copyright            : (C) 2003 by The KXMLEditor Team
    email                : a_charytoniuk@user.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#include "commands_edit.h"

#include <kdebug.h>

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Cutting element to clipboard			//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXECutCommand::KXECutCommand(KXMLEditorPart *pPart, QDomNode &domNode)
 : KXEDeleteNodeCommand(pPart, domNode)
{
}

KXECutCommand::~KXECutCommand()
{
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Pasting node from clipboard to document		//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEPasteToDocumentCommand::KXEPasteToDocumentCommand(
  KXMLEditorPart *pPart,
  QDomDocument *pDomTargetDoc,
  QDomElement &domSourceElement
  )
 : KXECommand(pPart)
{
  if ( pDomTargetDoc == 0 )
    kdError() << "KXEPasteToDocumentCommand::KXEPasteToDocumentCommand the given XML document object is empty." << endl;

  m_pDomTargetDoc = pDomTargetDoc;
  m_domSourceElement = domSourceElement;
}

KXEPasteToDocumentCommand::~KXEPasteToDocumentCommand()
{
}

void KXEPasteToDocumentCommand::execute()
{
  // Insert root element
  QDomNode newNode = m_pDomTargetDoc->importNode(m_domSourceElement, true);
  m_pDomTargetDoc->appendChild(newNode);
  m_pPart->updateNodeCreated(newNode);
}

void KXEPasteToDocumentCommand::unexecute()
{
  QDomNode removedNode = m_pDomTargetDoc->removeChild( m_pDomTargetDoc->documentElement());
  
  if ( removedNode.isNull() )
		kdError() << "KXEPasteToDocumentCommand::unexecute error removing node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(removedNode);
  }
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Pasting node from clipboard to element		//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEPasteToElementCommand::KXEPasteToElementCommand(
  KXMLEditorPart *pPart,
  QDomElement & domTargetElement,
  QDomNode &domSourceNode
  )
 : KXECommand(pPart)
{
  if ( domTargetElement.isNull() )
    kdError() << "KXEPasteCommand::KXEPasteCommand the given XML element object is empty." << endl;

  m_domTargetElement = domTargetElement;
  m_domSourceNode = domSourceNode;
}

KXEPasteToElementCommand::~KXEPasteToElementCommand()
{
}

void KXEPasteToElementCommand::execute()
{
  m_domTargetElement.appendChild(m_domSourceNode);
 	m_pPart->updateNodeCreated(m_domSourceNode);
}

void KXEPasteToElementCommand::unexecute()
{
  if ( m_domSourceNode.parentNode().removeChild( m_domSourceNode ).isNull() )
		kdError() << "KXEPasteToElementCommand::unexecute error removing the node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(m_domSourceNode);
  }
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Pasting proc.instr from clipboard to proc.instr		//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEPasteToProcInstrCommand::KXEPasteToProcInstrCommand(
  KXMLEditorPart *pPart,
  QDomProcessingInstruction &domTargetProcInstr,
  QDomProcessingInstruction &domSourceProcInstr
  )
 : KXECommand(pPart)
{
  if ( domTargetProcInstr.isNull() )
    kdError() << "KXEPasteToProcInstrCommand::KXEPasteToProcInstrCommand the given object is empty." << endl;

  if ( domSourceProcInstr.isNull() )
    kdError() << "KXEPasteToProcInstrCommand::KXEPasteToProcInstrCommand the given object is empty." << endl;
  
  m_domTargetProcInstr = domTargetProcInstr;
  m_strNewData = domSourceProcInstr.data();
}

KXEPasteToProcInstrCommand::~KXEPasteToProcInstrCommand()
{
}

void KXEPasteToProcInstrCommand::execute()
{
  // Replace contents of selected proc. instr.
  m_strOldData = m_domTargetProcInstr.data();
 	m_domTargetProcInstr.setData(m_strNewData);    
  m_pPart->updateNodeChanged(m_domTargetProcInstr);
}

void KXEPasteToProcInstrCommand::unexecute()
{
  // Rverse action.
 	m_domTargetProcInstr.setData(m_strOldData);
  m_pPart->updateNodeChanged(m_domTargetProcInstr);
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Pasting char. data from clipboard to char. data		//////////
//////////////////////////////////////////////////////////////////////////////////////////


KXEPasteToCharDataCommand::KXEPasteToCharDataCommand(
  KXMLEditorPart *pPart,
  QDomCharacterData &domTargetCharData,
  QDomCharacterData &domSourceCharData
  )
 : KXECommand(pPart)
{
  if ( domTargetCharData.isNull() )
    kdError() << "KXEPasteToCharDataCommand::KXEPasteToCharDataCommand the given object is empty." << endl;

  if ( domSourceCharData.isNull() )
    kdError() << "KXEPasteToCharDataCommand::KXEPasteToCharDataCommand the given object is empty." << endl;

  m_domTargetCharData = domTargetCharData;
  m_strNewData = domSourceCharData.data();
}

KXEPasteToCharDataCommand::~KXEPasteToCharDataCommand()
{
}

void KXEPasteToCharDataCommand::execute()
{
  // replace target contents with source
  m_strOldData = m_domTargetCharData.data();
  m_domTargetCharData.setData(m_strNewData);
 	m_pPart->updateNodeChanged(m_domTargetCharData);
}

void KXEPasteToCharDataCommand::unexecute()
{
  m_domTargetCharData.setData(m_strOldData);
 	m_pPart->updateNodeChanged(m_domTargetCharData);
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Drag & drop node                                              		//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEDragDropMoveCommand::KXEDragDropMoveCommand(
  KXMLEditorPart *pPart,
  QDomElement & domTargetElement,
  QDomNode &domSourceNode
  )
 : KXECommand(pPart)
{
  if ( domTargetElement.isNull() )
    kdError() << "KXEDragDropMoveCommand::KXEDragDropMoveCommand the given XML element object is empty." << endl;

  m_domTargetElement = domTargetElement;
  m_domSourceNode = domSourceNode;
  m_domPreviousParentNode = m_domSourceNode.parentNode();
}

KXEDragDropMoveCommand::~KXEDragDropMoveCommand()
{
}

void KXEDragDropMoveCommand::execute()
{
  // 1st, remove source node from its parent
  if( m_domPreviousParentNode.removeChild( m_domSourceNode ).isNull() )
      kdError() << "KXMLEditorPart::slotXmlElementDelete error removing the selected node." << endl;
  else
      m_pPart->updateNodeDeleted(m_domSourceNode);

  // 2nd, append moved node to new parent
  m_domTargetElement.appendChild(m_domSourceNode);
 	m_pPart->updateNodeCreated(m_domSourceNode);
}

void KXEDragDropMoveCommand::unexecute()
{
  // 1st, remove source node from its parent
  if ( m_domTargetElement.removeChild( m_domSourceNode ).isNull() )
		kdError() << "KXEPasteToElementCommand::unexecute error removing the node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(m_domSourceNode);
  }

  // 2nd, append moved node to previous parent
  m_domPreviousParentNode.appendChild(m_domSourceNode);
  m_pPart->updateNodeCreated(m_domSourceNode);
}


//////////////////////////////////////////////////////////////////////////////////////////
///////////		Deleting node 					//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEDeleteNodeCommand::KXEDeleteNodeCommand(KXMLEditorPart *pPart, QDomNode &domNode)
 : KXECommand(pPart)
{
  m_domNode = domNode;
  m_domParentNode = m_domNode.parentNode();
  m_afterNode = m_domNode.previousSibling();

  if ( m_domParentNode.isNull() )
    kdError() << "KXEDeleteNodeCommand::KXEDeleteNodeCommand selected nodes parent node is empty." << endl;
}

KXEDeleteNodeCommand::~KXEDeleteNodeCommand()
{
}

void KXEDeleteNodeCommand::execute()
{
  if ( m_domParentNode.removeChild( m_domNode ).isNull() )
		kdError() << "KXEDeleteNodeCommand::execute error removing the selected node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(m_domNode);
  }
}

void KXEDeleteNodeCommand::unexecute()
{
	if (m_afterNode.isNull())
		m_domParentNode.insertBefore(m_domNode,m_afterNode);
	else
  		m_domParentNode.insertAfter( m_domNode,m_afterNode );
  	m_pPart->updateNodeCreated(m_domNode);
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Deleting one attribute				//////////
//////////////////////////////////////////////////////////////////////////////////////////


KXEDeleteAttrCommand::KXEDeleteAttrCommand(
  KXMLEditorPart *pPart,
  QDomElement &domOwnerElement,
  QDomAttr &domAttr
  )
 : KXECommand(pPart)
{
  m_domOwnerElement = domOwnerElement;
  m_domAttr = domAttr;
}

KXEDeleteAttrCommand::~KXEDeleteAttrCommand()
{
}

void KXEDeleteAttrCommand::execute()
{
  m_domOwnerElement.removeAttributeNode(m_domAttr);
	m_pPart->updateNodeChanged(m_domOwnerElement);
}

void KXEDeleteAttrCommand::unexecute()
{
  m_domOwnerElement.setAttributeNode( m_domAttr );
	m_pPart->updateNodeChanged( m_domOwnerElement ) ;
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Deleting all attributes				//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEDeleteAllAttribCommand::KXEDeleteAllAttribCommand(
  KXMLEditorPart *pPart,
  QDomElement &domOwnerElement
  )
 : KXECommand(pPart)
{
  m_domOwnerElement = domOwnerElement;
  m_listRemovedAttributes.setAutoDelete( true ); // the list owns the objects
}

KXEDeleteAllAttribCommand::~KXEDeleteAllAttribCommand()
{
}

void KXEDeleteAllAttribCommand::execute()
{
  QDomNamedNodeMap mapAttributes = m_domOwnerElement.attributes();
  uint nAttributes = mapAttributes.count();

 if( nAttributes == 0 )
		return;

	for( uint nRow = nAttributes; nRow > 0; nRow-- )
	{
		QDomNode node = mapAttributes.item(nRow-1);
		if ( node.isAttr() )
			{ QDomAttr domAttr = node.toAttr();

        QDomAttr *pNodeCloned = new QDomAttr(domAttr.cloneNode(true).toAttr());

        m_listRemovedAttributes.append(pNodeCloned);
        m_domOwnerElement.removeAttributeNode(node.toAttr()); 
      }
		else
			kdDebug() << "KXMLEditor " << k_funcinfo << " node is not an attribute (but should be)" << node.nodeName() << endl;
	}

  m_pPart->updateNodeChanged(m_domOwnerElement);
}

void KXEDeleteAllAttribCommand::unexecute()
{
  QDomNamedNodeMap mapAttributes = m_domOwnerElement.attributes();
  uint nAttributes = m_listRemovedAttributes.count();

	if ( nAttributes == 0 )
		return;

 QDomAttr *pDomAttr;
 for ( pDomAttr = m_listRemovedAttributes.first(); pDomAttr; pDomAttr = m_listRemovedAttributes.next() )
	{
		if(!pDomAttr->namespaceURI().isEmpty())
      m_domOwnerElement.setAttribute(pDomAttr->name(), pDomAttr->value());
    else
      m_domOwnerElement.setAttributeNS(pDomAttr->namespaceURI(), pDomAttr->name(), pDomAttr->value());
  }

  m_listRemovedAttributes.clear();
   
  m_pPart->updateNodeChanged(m_domOwnerElement);
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Moving node up					//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEUpCommand::KXEUpCommand(KXMLEditorPart *pPart, QDomNode &domNode)
 : KXECommand(pPart)
{
  m_domNode = domNode;
  m_domParentNode = m_domNode.parentNode();

  if ( m_domParentNode.isNull() )
    kdError() << "KXEUpCommand::KXEUpCommand selected nodes parent node is empty." << endl;
}

KXEUpCommand::~KXEUpCommand()
{
}

void KXEUpCommand::execute()
{
  QDomNode domPrevSibling = m_domNode.previousSibling();
	if ( domPrevSibling.isNull() )
	{
		kdError() << "KXEUpCommand::execute selected node doesn't seem to have a previous sibling." << endl;
		return;
	}
	
	QDomNode domNode = m_domParentNode.removeChild( m_domNode );
	if ( domNode.isNull() )
		kdError() << "KXEUpCommand::execute can't remove child node." << endl;
	else
	{
		domNode = m_domParentNode.insertBefore( domNode, domPrevSibling );
		if ( domNode.isNull() )
			kdError() << "KXEUpCommand::execute can't insert child node." << endl;
		else
		{
			m_pPart->updateNodeMoved(domNode);
		}
  }
}

void KXEUpCommand::unexecute()
{
  QDomNode domNextSibling = m_domNode.nextSibling();
	if ( domNextSibling.isNull() )
	{
		kdError() << "KXEUpCommand::unexecute selected node doesn't seem to have a next sibling." << endl;
		return;
	}

	QDomNode domNode = m_domParentNode.removeChild( m_domNode );
	if ( domNode.isNull() )
		kdError() << "KXEUpCommand::unexecute can't remove child node." << endl;
	else
	{
		domNode = m_domParentNode.insertAfter( domNode, domNextSibling );
		if ( domNode.isNull() )
			kdError() << "KXEUpCommand::unexecute can't insert child node." << endl;
		else
		{
			m_pPart->updateNodeMoved(domNode);
		}
	}
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Moving node down				//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEDownCommand::KXEDownCommand(KXMLEditorPart *pPart, QDomNode &domNode)
 : KXECommand(pPart)
{
  m_domNode = domNode;
  m_domParentNode = m_domNode.parentNode();

  if ( m_domParentNode.isNull() )
    kdError() << "KXEDownCommand::KXEDownCommand selected nodes parent node is empty." << endl;
}

KXEDownCommand::~KXEDownCommand()
{
}

void KXEDownCommand::execute()
{
  QDomNode domNextSibling = m_domNode.nextSibling();
	if ( domNextSibling.isNull() )
	{
		kdError() << "KXEDownCommand::execute selected node doesn't seem to have a next sibling." << endl;
		return;
	}

	QDomNode domNode = m_domParentNode.removeChild( m_domNode );
	if ( domNode.isNull() )
		kdError() << "KXEDownCommand::execute can't remove child node." << endl;
	else
	{
		domNode = m_domParentNode.insertAfter( domNode, domNextSibling );
		if ( domNode.isNull() )
			kdError() << "KXEDownCommand::execute can't insert child node." << endl;
		else
		{
			m_pPart->updateNodeMoved(domNode);
		}
	}
}

void KXEDownCommand::unexecute()
{
  QDomNode domPrevSibling = m_domNode.previousSibling();
	if ( domPrevSibling.isNull() )
	{
		kdError() << "KXEDownCommand::unexecute selected node doesn't seem to have a previous sibling." << endl;
		return;
	}

	QDomNode domNode = m_domParentNode.removeChild( m_domNode );
	if ( domNode.isNull() )
		kdError() << "KXEDownCommand::unexecute can't remove child node." << endl;
	else
	{
		domNode = m_domParentNode.insertBefore( domNode, domPrevSibling );
		if ( domNode.isNull() )
			kdError() << "KXEDownCommand::unexecute can't insert child node." << endl;
		else
		{
			m_pPart->updateNodeMoved(domNode);
		}
  }
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Editing char. data properties			//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEEditCharDataCommand::KXEEditCharDataCommand(
  KXMLEditorPart *pPart,
  QDomCharacterData &domCharacterData,
  const QString strNewContents
  )
 : KXECommand(pPart)
{
  m_domCharacterData = domCharacterData;
  m_strNewContents = strNewContents;
}

KXEEditCharDataCommand::~KXEEditCharDataCommand()
{
}

void KXEEditCharDataCommand::execute()
{
  m_strOldContents = m_domCharacterData.data();
  m_domCharacterData.setData( m_strNewContents );
  m_pPart->updateNodeChanged( m_domCharacterData );
}

void KXEEditCharDataCommand::unexecute()
{
  m_domCharacterData.setData( m_strOldContents );
  m_pPart->updateNodeChanged( m_domCharacterData );
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Editing proc. instr properties			//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEEditProcInstrCommand::KXEEditProcInstrCommand(
  KXMLEditorPart *pPart,
  QDomProcessingInstruction &domProcInstr,
  const QString strNewData
  )
 : KXECommand(pPart)
{
  m_domProcInstr = domProcInstr;
  m_strNewData = strNewData;
}

KXEEditProcInstrCommand::~KXEEditProcInstrCommand()
{
}

void KXEEditProcInstrCommand::execute()
{
  m_strOldData = m_domProcInstr.data();
  m_domProcInstr.setData( m_strNewData );
  m_pPart->updateNodeChanged( m_domProcInstr );
}

void KXEEditProcInstrCommand::unexecute()
{
  m_domProcInstr.setData( m_strOldData );
  m_pPart->updateNodeChanged( m_domProcInstr );
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Editing element data properties			//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEEditElementCommand::KXEEditElementCommand(
  KXMLEditorPart *pPart,
  QDomElement &domElement,
  const QString strNewPrefix,
  const QString strNewName
  )
 : KXECommand(pPart)
{
  m_domElement = domElement;
  m_strNewPrefix = strNewPrefix;
  m_strNewName = strNewName;
}

KXEEditElementCommand::~KXEEditElementCommand()
{
}

void KXEEditElementCommand::execute()
{
  m_strOldPrefix = m_domElement.prefix();
  m_strOldName = m_domElement.tagName();
  

  if ( ! m_domElement.namespaceURI().isNull() )
		m_domElement.setPrefix( m_strNewPrefix );

	m_domElement.setTagName( m_strNewName );
  
  m_pPart->updateNodeChanged( m_domElement );
}

void KXEEditElementCommand::unexecute()
{
  if ( ! m_domElement.namespaceURI().isNull() )
		m_domElement.setPrefix( m_strOldPrefix );

	m_domElement.setTagName( m_strOldName );
  
  m_pPart->updateNodeChanged( m_domElement );
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Edit attribute value				//////////
//////////////////////////////////////////////////////////////////////////////////////////


KXEEditAttrValueCommand::KXEEditAttrValueCommand(
  KXMLEditorPart *pPart,
  const QDomAttr &domAttr,
  const QString strNewValue
  )
 : KXECommand(pPart)
{
  m_domAttr = domAttr;
  m_strNewValue = strNewValue;
}

KXEEditAttrValueCommand::~KXEEditAttrValueCommand()
{
}

void KXEEditAttrValueCommand::execute()
{
  m_strOldValue = m_domAttr.value();
  m_domAttr.setValue( m_strNewValue );
  
	m_pPart->updateNodeChanged(m_domAttr.ownerElement());
}

void KXEEditAttrValueCommand::unexecute()
{
  m_domAttr.setValue( m_strOldValue );
  
  m_pPart->updateNodeChanged(m_domAttr.ownerElement());
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Edit attribute name				//////////
//////////////////////////////////////////////////////////////////////////////////////////


KXEEditAttrNameCommand::KXEEditAttrNameCommand(
  KXMLEditorPart *pPart,
  const QDomAttr &domOldAttr,
  const QString strNewName
  )
 : KXECommand(pPart)
{
  m_strNewName = strNewName;
  m_strOldName = domOldAttr.name();
  m_strValue = domOldAttr.value();

  if(!domOldAttr.namespaceURI().isEmpty())
    m_strNamespaceURI = domOldAttr.namespaceURI();

  m_domOwnerElement = domOldAttr.ownerElement();
}

KXEEditAttrNameCommand::~KXEEditAttrNameCommand()
{
}

void KXEEditAttrNameCommand::execute()
{
  // it's not possible to change name. Must delete attribute and create new one
  if(m_strNamespaceURI.isEmpty())
  {
    m_domOwnerElement.setAttribute(m_strNewName, m_strValue);
    m_domOwnerElement.attributes().removeNamedItem(m_strOldName);
  }
  else
  {
    m_domOwnerElement.setAttributeNS( m_strNamespaceURI, m_strNewName, m_strValue);
    m_domOwnerElement.attributes().removeNamedItemNS (m_strNamespaceURI, m_strOldName);
  }

  m_pPart->updateNodeChanged(m_domOwnerElement);
}

void KXEEditAttrNameCommand::unexecute()
{
  // it's not possible to change name. Must delete attribute and create new one
  if(m_strNamespaceURI.isEmpty())
  {
    m_domOwnerElement.setAttribute(m_strOldName, m_strValue);
    m_domOwnerElement.attributes().removeNamedItem(m_strNewName);

  }
  else
  {
    m_domOwnerElement.setAttributeNS( m_strNamespaceURI, m_strOldName, m_strValue);
    m_domOwnerElement.attributes().removeNamedItemNS (m_strNamespaceURI, m_strNewName);
  }

  m_pPart->updateNodeChanged(m_domOwnerElement);
}

//////////////////////////////////////////////////////////////////////////////////////////
///////////		Editing element and its subtree as raw XML			//////////
//////////////////////////////////////////////////////////////////////////////////////////

KXEEditRawXmlCommand::KXEEditRawXmlCommand(
  KXMLEditorPart *pPart,
  QDomElement &domOldElement,
  QDomElement &domNewElement
  )
 : KXECommand(pPart)
{
  m_domOldElement = domOldElement;
  m_domNewElement = domNewElement;
  m_domParentNode = domOldElement.parentNode();
  m_afterNode = domOldElement.previousSibling();

  if ( m_domParentNode.isNull() )
    kdError() << "KXEEditRawXmlCommand::KXEEditRawXmlCommand selected nodes parent node is empty." << endl;

}

KXEEditRawXmlCommand::~KXEEditRawXmlCommand()
{
}

void KXEEditRawXmlCommand::execute()
{
  // first delete node
  if ( m_domParentNode.removeChild( m_domOldElement ).isNull() )
		kdError() << "KXEEditRawXmlCommand::execute error removing the selected node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(m_domOldElement);
  }

  // then insert new node
  if (m_afterNode.isNull())
		m_domParentNode.insertBefore(m_domNewElement, m_afterNode);
	else
  		m_domParentNode.insertAfter( m_domNewElement, m_afterNode );
  	m_pPart->updateNodeCreated(m_domNewElement);
}

void KXEEditRawXmlCommand::unexecute()
{
  // first delete node
  if ( m_domParentNode.removeChild( m_domNewElement ).isNull() )
		kdError() << "KXEEditRawXmlCommand::unexecute error removing the selected node." << endl;
	else
	{
		m_pPart->updateNodeDeleted(m_domNewElement);
  }

  // then insert new node
  if (m_afterNode.isNull())
		m_domParentNode.insertBefore(m_domOldElement, m_afterNode);
	else
  		m_domParentNode.insertAfter( m_domOldElement, m_afterNode );
  	m_pPart->updateNodeCreated(m_domOldElement);
}
