###
### $Rev: 51 $
### $Release: 0.6.0 $
### copyright(c) 2005 kuwata-lab all rights reserved.
###

require 'yaml'
require 'erb'
require 'kwalify'
require 'kwalify/util/yaml-helper'
#require 'kwalify/util/option-parser'


module Kwalify


   class CommandOptionError < KwalifyError
      def initialize(message, option, error_symbol)
         super(message)
         @option = option
         @error_symbol = error_symbol
      end
      attr_reader :option, :error_symbol
   end


   ##
   ## ex.
   ##   command = File.basename($0)
   ##   begin
   ##      main = Kwalify::Main.new(command)
   ##      s = main.execute
   ##      print s if s
   ##   rescue Kwalify::CommandOptionError => ex
   ##      $stderr.puts "ERROR: #{ex.message}"
   ##      exit 1
   ##   rescue Kwalify::KwalifyError => ex
   ##      $stderr.puts "ERROR: #{ex.message}"
   ##      exit 1
   ##   end
   ##
   class Main


      def initialize(command=nil)
         @command = command || File.basename($0)
         @options = {}
         @properties    = {}
         @template_path  = []
         $:.each do |path|
            tpath = "#{path}/kwalify/templates"
            @template_path << tpath if test(?d, tpath)
         end
      end


      def debug?
         @options[:debug]
      end


      def _inspect()
         sb = []
         sb <<    "command: #{@command}\n"
         sb <<    "options:\n"
         @options.keys.sort {|k1,k2| k1.to_s<=>k2.to_s }.each do |key|
            sb << "  - #{key}: #{@options[key]}\n"
         end
         sb <<    "properties:\n"
         @properties.keys.sort_by {|k| k.to_s}.each do |key|
            sb << "  - #{key}: #{@properties[key]}\n"
         end
         #sb <<    "template_path:\n"
         #@template_path.each do |path|
         #   sb << "  - #{path}\n"
         #end
         return sb.join
      end


      def execute(argv=ARGV)
         # parse command-line options
         filenames = _parse_argv(argv)

         # help or version
         if @options[:help] || @options[:version]
            action = @options[:action]
            s = ''
            s << _version() << "\n"           if @options[:version]
            s << _usage()                     if @options[:help] && !action
            s << _describe_properties(action) if @options[:help] && action
            return s
         end

         # validation
         if @options[:meta2]
            s = _quick_meta_validate(filenames)
         elsif @options[:meta]
            s = _meta_validate(filenames)
         elsif @options[:action]
            if !@options[:schema]
               #* key=:command_option_actionnoschema  msg="schema filename is not specified."
               raise option_error(:command_option_actionnoschema, @options[:action])
            end
            s = _perform_action(@options[:action], @options[:schema])
         elsif @options[:schema]
            if @options[:debug]
               s = _inspect_schema(@options[:schema])
            else
               s = _validate(filenames, @options[:schema])
            end
         else
            #* key=:command_option_noaction  msg="command-line option '-f' or '-m' required."
            raise option_error(:command_option_noaction, @command)
         end
         return s   # or return (s == nil || s.empty?) ? nil : s
      end


      def self.main(command, argv=ARGV)
         begin
            main = Kwalify::Main.new(command)
            s = main.execute(argv)
            print s if s
         rescue Kwalify::CommandOptionError => ex
            $stderr.puts ex.message
            exit 1
         rescue Kwalify::KwalifyError => ex
            $stderr.puts "ERROR: #{ex.message}"
            exit 1
         #rescue => ex
         #   if main.debug?
         #      raise ex
         #   else
         #      $stderr.puts ex.message
         #      exit 1
         #   end
         end
      end


      private


      def option_error(error_symbol, arg)
         msg = Kwalify.msg(error_symbol) % arg
         return CommandOptionError.new(msg, arg, error_symbol)
      end


      def _find_template(action)
         template_filename = action + '.eruby'
         unless test(?f, template_filename)
            tpath = @template_path.find { |path| test(?f, "#{path}/#{template_filename}") }
            #* key=:command_option_notemplate  msg="%s: invalid action (template not found).\n"
            raise option_error(:command_option_notemplate, action) unless tpath
            template_filename = "#{tpath}/#{action}.eruby"
         end
         return template_filename
      end


      def _apply_template(template_filename, hash)
         template = File.read(template_filename)
         trim_mode = 1
         erb = ERB.new(template, $SAFE, trim_mode)
         context = Object.new
         hash.each do |key, val|
            context.instance_variable_set("@#{key}", val)
         end
         return context.instance_eval(erb.src, template_filename)
      end


      def _describe_properties(action)
         template_filename = _find_template(action)
         s = _apply_template(template_filename, :describe=>true)
         return s
      end


      def _perform_action(action, schema_filename, describe=false)
         template_filename = _find_template(action)
         schema = _load_schema_file(schema_filename)
         rule = Rule.new(schema)
         @properties[:schema_filename] = schema_filename
         s = _apply_template(template_filename, :rule=>rule, :properties=>@properties)
         return s
      end


      def _inspect_schema(schema_filename)
         filename = schema_filename
         str = File.open(schema_filename) { |f| f.read() }
         str = YamlUtil.untabify(str) if @options[:untabify]
         schema = YAML.load(str)
         return nil if schema == nil
         validator = Kwalify::Validator.new(schema)  # error raised when schema is wrong
         s = validator._inspect()
         s << "\n" unless s[-1] == ?\n
         return s
      end


      #--
      #class QuickMetaValidator
      #   def validate(schema)
      #      errors = []
      #      begin
      #         validator = Kwalify::Validator.new(schema) # error raised when schema is wrong
      #      rescue Kwalify::SchemaError => ex
      #         errors << ex
      #      end
      #      return errors
      #   end
      #end
      #++


      def _quick_meta_validate(filenames)
         meta_validator = Object.new
         def meta_validator.validate(schema)
            errors = []
            begin
               validator = Kwalify::Validator.new(schema) # error raised when schema is wrong
            rescue Kwalify::SchemaError => ex
               errors << ex
            end
            return errors
         end
         s = _validate_files(meta_validator, filenames)
         return s
      end


      def _load_schema_file(schema_filename)
         filename = schema_filename
         str = File.read(filename)
         str = YamlHelper.untabify(str) if @options[:untabify]
         schema = YAML.load(str)
         return schema
      end


      def _meta_validate(filenames)
         meta_validator = Kwalify::MetaValidator.instance()
         s = _validate_files(meta_validator, filenames)
         return s
      end


      def _validate(filenames, schema_filename)
         schema = _load_schema_file(schema_filename)
         if schema
            validator = Kwalify::Validator.new(schema)
            s = _validate_files(validator, filenames)
         else
            #* key=:schema_empty  msg="%s: empty schema.\n"
            s = Kwalify.msg(:schema_emtpy) % filename
         end
         return s
      end


      def _validate_files(validator, filenames)
         s = ''
         filenames = [ nil ] if filenames.empty?
         filenames.each do |filename|
            if filename
               str = File.open(filename) { |f| f.read() }   # or File.read(filename) in ruby1.8
            else
               str = $stdin.read()
               filename = '(stdin)'
            end
            str = YamlHelper.untabify(str) if @options[:untabify]
            if @options[:linenum]
               parser = Kwalify::YamlParser.new(str)
               i = 0
               while parser.has_next?
                  doc = parser.parse()
                  s << _validate_document(validator, doc, filename, i, parser)
                  i += 1
               end
            else
               parser = nil
               i = 0
               YAML.load_documents(str) do |doc|
                  s << _validate_document(validator, doc, filename, i, nil)
                  i += 1
               end
            end
         end
         return s
      end


      def _validate_document(validator, doc, filename, i, parser=nil)
         s = ''
         if doc == nil
            #* key=:validation_empty  msg="%s#%d: empty.\n"
            s << kwalify.msg(:validation_empty) % [filename, i]
            return s
         end
         errors = validator.validate(doc)
         if errors == nil || errors.empty?
            #* key=:validation_valid  msg="%s#%d: valid.\n"
            s << Kwalify.msg(:validation_valid) % [filename, i] unless @options[:silent]
         else
            #* key=:validation_invalid  msg="%s#%d: INVALID\n"
            s << Kwalify.msg(:validation_invalid) % [filename, i]
            if @options[:linenum]
               #assert parser != nil
               raise unless parser != nil
               parser.set_errors_linenum(errors)
               errors.sort!
            end
            errors.each do |error|
               if @options[:emacs]
                  #assert @options[:linenum]
                  raise unless @options[:linenum]
                  s << "#{filename}:#{error.linenum}: [#{error.path}] #{error.message}\n"
               elsif @options[:linenum]
                  s << "  - (line #{error.linenum}) [#{error.path}] #{error.message}\n"
               else
                  s << "  - [#{error.path}] #{error.message}\n"
               end
            end
         end
         return s
      end


      def _usage()
         #msg = Kwalify.msg(:command_help) % [@command, @command, @command]
         msg = Kwalify.msg(:command_help)
         return msg
      end


      def _version()
         return RELEASE
      end


      def _to_value(str)
         case str
         when nil, "null", "nil"         ;   return nil
         when "true", "yes"              ;   return true
         when "false", "no"              ;   return false
         when /\A\d+\z/                  ;   return str.to_i
         when /\A\d+\.\d+\z/             ;   return str.to_f
         when /\/(.*)\//                 ;   return Regexp.new($1)
         when /\A'.*'\z/, /\A".*"\z/     ;   return eval(str)
         else                            ;   return str
         end
      end


      def _parse_argv(argv)
         option_table = {
            ?h => :help,
            ?v => :version,
            ?s => :silent,
            ?t => :untabify,
            ?m => :meta,
            ?M => :meta2,
            ?E => :emacs,
            ?l => :linenum,
            ?f => :schema,
            ?D => :debug,
            ?a => :action,
            ?I => :tpath,
         }

         errcode_table = {
            #* key=:command_option_noschema  msg="-%s: schema filename required."
            ?f => :command_option_noschema,
            #* key=:command_option_noaction  msg="-%s: action required."
            ?a => :command_option_noaction,
            #* key=:command_option_notpath   msg="-%s: template path required."
            ?I => :command_option_notpath,
         }

         while argv[0] && argv[0][0] == ?-
            optstr = argv.shift
            optstr = optstr[1, optstr.length-1]
            # property
            if optstr[0] == ?-
               unless optstr =~ /\A\-([-\w]+)(?:=(.*))?\z/
                  #* key=:command_property_invalid  msg="%s: invalid property."
                  raise option_error(:command_property_invalid, optstr)
               end
               prop_name = $1;  prop_value = $2
               key   = prop_name.gsub(/-/, '_').intern
               value = prop_value == nil ? true : _to_value(prop_value)
               @properties[key] = value
            # option
            else
               while optstr && !optstr.empty?
                  optchar = optstr[0]
                  optstr[0,1] = ""
                  unless option_table.key?(optchar)
                     #* key=:command_option_invalid  msg="-%s: invalid command option."
                     raise option_error(:command_option_invalid, optchar.chr)
                  end
                  optkey = option_table[optchar]
                  case optchar
                  when ?f, ?a, ?I
                     arg = optstr.empty? ? argv.shift : optstr
                     raise option_error(errcode_table[optchar], optchar.chr) unless arg
                     @options[optkey] = arg
                     optstr = ''
                  else
                     @options[optkey] = true
                  end
               end
            end
         end  # end of while
         #
         @options[:linenum] = true if @options[:emacs]
         @options[:help]    = true if @properties[:help]
         @options[:version] = true if @properties[:version]
         filenames = argv
         return filenames
      end


      def _domain_type?(doc)
         klass = defined?(YAML::DomainType) ? YAML::DomainType : YAML::Syck::DomainType
         return doc.is_a?(klass)
      end

   end

end
