/***************************************************************************
*   Copyright (C) 2004 by Christoph Thielecke                             *
*   crissi99@gmx.de                                                       *
*                                                                         *
*   @description This class imports a openvpn configuration file                             *
*                                                                         *
*   @author Christoph Thielecke <crissi99@gmx.de>                    *
*                                                                         *
*   This program is free software; you can redistribute it and/or modify  *
*   it under the terms of the GNU General Public License as published by  *
*   the Free Software Foundation; either version 2 of the License, or     *
*   (at your option) any later version.                                   *
*                                                                         *
*   This program is distributed in the hope that it will be useful,       *
*   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
*   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
*   GNU General Public License for more details.                          *
*                                                                         *
*   You should have received a copy of the GNU General Public License     *
*   along with this program; if not, write to the                         *
*   Free Software Foundation, Inc.,                                       *
*   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
***************************************************************************/ 
//BEGIN INCLUDES
#include <kmessagebox.h>
#include <kio/netaccess.h>
#include <ktempfile.h>
#include <klocale.h>
#include <kconfig.h>
#include <kdialogbase.h>
#include <kcombobox.h>
#include <kurlrequester.h>
#include <klineedit.h>
#include <kpassdlg.h>
#include <kstandarddirs.h>
#include <qfile.h>
#include <qurl.h>
#include <kurl.h>
#include <qtextstream.h>
#include <qcheckbox.h>
#include <string>
#include <qprocess.h>

#include <iostream>

#include "importopenvpnprofiledialog.h"
#include "utils.h"
#include "importcertificatedialog.h" 
//END INCLUDES

ImportOpenvpnProfileDialog::ImportOpenvpnProfileDialog( KVpncConfig *config, QWidget *parent, const QString& caption, QString file )
		: KDialogBase( parent, "Import_Cisco_OpenVPN_profile", true, caption,
               KDialogBase::Ok | KDialogBase::Cancel, KDialogBase::Ok, true ) {
	if ( !file.isEmpty() )
		filename = file;
	else
		filename = "";
	Pkcs12CertFile = "";
	// 	CertPath="/etc/openvpn";
	importOk = false;
	this->config = config;

	main = new ImportOpenvpnProfileDialogBase( this );
	setMainWidget( main );
	//main->setMinimumSize(main->sizeHint());

	main->FilenameUrlrequester->setFilter( "*.ovpn *.conf" );

	main->FilenameUrlrequester->setURL( filename );

}

ImportOpenvpnProfileDialog::~ImportOpenvpnProfileDialog() {
	delete main;
}

void ImportOpenvpnProfileDialog::accept() {

	filename = main->FilenameUrlrequester->url();
	if ( !filename.isEmpty() ) {
		if ( config->KvpncDebugLevel > 0 )
			config->appendLogEntry( i18n( "OpenVPN import: file: %1" ).arg( filename ), config->debug );
		canAccept();
	} else {
		config->appendLogEntry( i18n( "OpenVPN import: file name empty" ), config->error );
		KMessageBox::sorry( 0, i18n( "File name can not be empty!" ), i18n( "Empty File Name" ) );
	}
}

void ImportOpenvpnProfileDialog::canAccept() {

	QFile f( filename );
	if ( !f.exists() ) {
		KMessageBox::sorry( 0, i18n( "File not found." ), i18n( "No File" ) );

		//		emit progress( 100 );
		return ;
	}

	//BEGIN Sample
	// sample config

	/*
	#
	# Sample OpenVPN configuration file for
	# home using SSL/TLS mode and RSA certificates/keys.
	#
	# '#' or ';' may be used to delimit comments.

	#######################################################
	# PATS INFORMATION:
	# No adaptations necessary in this file, unless you
	# give different names to your certificates or you
	# place them in different folders
	######################################################


	# Use a dynamic tun device.
	# For Linux 2.2 or non-Linux OSes,
	# you may want to use an explicit
	# unit number such as "tun1".
	# OpenVPN also supports virtual
	# ethernet "tap" devices.
	dev tap

	# Our OpenVPN peer is the PATS gateway.
	remote 143.129.70.202

	float

	# In SSL/TLS key exchange, Office will
	# assume server role and Home
	# will assume client role.
	tls-client

	# pull necessary here. This option will
	# enable the server to push commands to
	# the client
	pull

	route 10.0.0.0 255.255.0.0 vpn_gateway

	# Certificate Authority file
	ca cacert.pem

	# Our certificate/public key
	cert cert.pem

	# Our private key
	key key.pem

	# OpenVPN uses UDP port 5000 by default.
	port 5000

	# Downgrade UID and GID to
	# "nobody" after initialization
	# for extra security.
	; user nobody
	; group nobody

	# If you built OpenVPN with
	# LZO compression, uncomment
	# out the following line.
	# PREFFERED
	comp-lzo

	# Uncomment this section for a more reliable detection when a system
	# loses its connection.  For example, dial-ups or laptops that
	# travel to other locations.
	# This commands will be pushed through by the server
	; ping 15
	; ping-restart 45
	; ping-timer-rem
	; persist-tun
	; persist-key

	# Verbosity level.
	# 0 -- quiet except for fatal errors.
	# 1 -- mostly quiet, but display non-fatal network errors. PREFERRED
	# 3 -- medium output, good for normal operation.
	# 9 -- verbose, good for troubleshooting
	verb 1
	*/

	/* Example 2
	client
	dev tun

	proto udp
	remote 10.7.21.1 1194

	resolv-retry infinite
	ns-cert-type server
	nobind
	user nobody
	group nogroup
	persist-key
	persist-tun

	ca certs/richard-root-ca.crt
	cert certs/richard-openvpn-notebook-richard.crt
	key certs/richard-openvpn-notebook-richard.key

	cipher AES-128-CBC
	ping 15
	ping-restart 45
	ping-timer-rem
	persist-tun
	persist-key

	comp-lzo
	verb 3
	*/ 
	//END Sample

	QFile OpenvpnConfigFile( filename );
	QTextStream stream ( &OpenvpnConfigFile );

	QString importprefix = QFileInfo ( filename ).dirPath();
	
	if ( config->KvpncDebugLevel > 2 )
		config->appendLogEntry( i18n( "OpenVPN import: import prefix: %1" ).arg( importprefix ), config->debug );

	QString certprefix = locateLocal ( "data", "kvpnc" );

	if ( OpenvpnConfigFile.open( IO_ReadOnly ) ) {
		VpnAccountData::ConnectionType ConnType = VpnAccountData::openvpn;
		acc = new VpnAccountData ( ConnType, "" );
		QString profilename = Utils(config).removeSpecialCharsForFilename(filename).section( '/', -1, -1 ).section( '.', -2, -2 );
		acc->setName( profilename ); // /home/user/openvpn/CompanyABC.conf -> CompanyABC
		acc->setDescription( i18n( "Imported from %1" ).arg( Utils(this->config).removeSpecialCharsForFilename( filename).section( '/', -1, -1 ) ) );
		acc->setLocalPort( 1194 );
		acc->setAuthWithUsernameAndPassword( false );
		acc->setRemoteNetMask( "24" );
		acc->setAuthenticationAlgorithm("MD5");

		acc->setUseTlsAuth( false );
		// 		acc->setUseConnectionStatusCheck( false );
		// 		acc->setDoReconnectAfterConnectionLost( false );

		QString line = "";
		while ( !stream.atEnd() ) {
			line = stream.readLine().simplifyWhiteSpace(); // line of text excluding '\n' and replace all white chars with one blank
			line.remove( '\"' );
			if ( !line.startsWith( "#" ) && !line.startsWith( ";" ) ) {
				int commentidx = line.find( '#' );
				if ( commentidx > -1 )
					line = line.left( commentidx - 1 );

				if ( line.startsWith( "remote " ) ) {
					acc->setGateway( line.section( ' ', 1, 1 ) ); // IP address
					if ( !line.section( ' ', 2, 2 ).isEmpty() ) {
						acc->setUseRemotePort( true );
						acc->setRemotePort ( line.section( ' ', 2, 2 ).toInt() );

						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: use userdefinied remote port: %1" ).arg( acc->getRemotePort() ), config->debug );
					}
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: gateway: %1" ).arg( acc->getGateway() ), config->debug );
				}
				else if ( line.startsWith( "dev " ) ) {
					QString Type = line.section( ' ', 1, 1 ); // tun or tap
					if ( Type == "tun" || Type == "tap" ) {
						acc->setTunnelDeviceType( Type );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: tunnel device type: %1" ).arg( acc->getTunnelDeviceType() ), config->debug );
					}
				}
				else if ( line.startsWith( "route " ) ) {
					// FIXME no fully supported yet!!!

					// FIXME made it possible to remember what type have to been set!
					QString gateway_type = "";
					bool useGateway;
					acc->setUseAdditionalNetworkRoutes( true );
					QStringList AdditionalNetworkRoutes;
					// example entry:
					// <network>/<netmask>#<gateway>

					QString network = line.section( ' ', 1, 1 ) + "/" + line.section( ' ', 2, 2 ); // ip and netmask

					QString Part3 = line.section( ' ', 3, 3 ); // possible gateway
					if ( Part3 != "vpn_gateway" && Part3 != "net_gateway" && Part3 != "remote_host" ) {
						network.append( "#" );
						network.append( Part3 ); // gateway
						gateway_type = line.section( ' ', 3, 3 );
						useGateway = true;
					} else {
						gateway_type = Part3;
					}

					AdditionalNetworkRoutes.append( network );
					acc->setAdditionalNetworkRoutes( AdditionalNetworkRoutes );

					if ( config->KvpncDebugLevel > 0 )
						if ( useGateway )
							config->appendLogEntry( i18n( "OpenVPN import: special route found: %1, type: %2" ).arg( network ).arg( gateway_type ), config->debug );
						else
							config->appendLogEntry( i18n( "OpenVPN import: special route found: %1 over %3, type: %2" ).arg( network ).arg( gateway_type ).arg( Part3 ), config->debug );
				}
				else if ( line.startsWith( "port " ) ) {
					acc->setUseLocalPort( true );
					acc->setLocalPort( line.section( ' ', 1, 1 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: local port specified: %1" ).arg( acc->getLocalPort() ), config->debug );
				}
				else if ( line.startsWith( "comp-lzo" ) ) {
					acc->setDisableLzoCompression( false );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use LZO compression" ), config->debug );
				}
				else if ( line.startsWith( "cert " ) ) {
					acc->setAuthType( VpnAccountData::cert );
					QString cert = line.section( ' ', 1 ).stripWhiteSpace();

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry( i18n( "OpenVPN import: cert file: %1" ).arg( importprefix+"/"+cert), config->debug );

					if ( QFile( QString( importprefix+"/"+cert )).exists())
					{
						// copy to ~/.kde/share/apps/kvpnc/ and rename it
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: copy %1 to %2" ).arg(QString( importprefix+"/"+cert )).arg(QString(certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert)), config->debug );
						QProcess CopyProcess( this );
						CopyProcess.addArgument("cp");
						CopyProcess.addArgument(QString( importprefix+"/"+cert ));
						CopyProcess.addArgument(QString(certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert));
						if (CopyProcess.start())
							acc->setX509Certificate( QString(certprefix+"/"+"openvpn_cert_"+profilename+"_"+cert) );
						else
						{
							config->appendLogEntry( i18n( "Could not start %1!" ).arg( "cp" ), config->error );
							acc->setX509Certificate( cert );
						}
					}
					else
					{
						// we use the given path
						acc->setX509Certificate( cert );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: certificate: %1" ).arg( acc->getX509Certificate() ), config->debug );
				}
				else if ( line.startsWith( "key " ) ) {

					QString privatekey = line.section( ' ', 1 ).stripWhiteSpace();

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry( i18n( "OpenVPN import: private key file: %1" ).arg( importprefix+"/"+privatekey ), config->debug );

					if ( QFile( QString( importprefix+"/"+privatekey )).exists())
					{
						// copy to ~/.kde/share/apps/kvpnc/ and rename it
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: copy %1 to %2" ).arg(QString( importprefix+"/"+privatekey )).arg(QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey)), config->debug );
						QProcess CopyProcess( this );
						CopyProcess.addArgument("cp");
						CopyProcess.addArgument(QString( importprefix+"/"+privatekey ));
						CopyProcess.addArgument(QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey));
						if (CopyProcess.start())
							acc->setPrivateKey( QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+privatekey) );
						else
						{
							config->appendLogEntry( i18n( "Could not start %1!" ).arg( "cp" ), config->error );
							acc->setPrivateKey( privatekey );
						}
					}
					else
					{
						// we use the given path
						acc->setPrivateKey( privatekey );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: PSK in file: %1" ).arg( acc->getPrivateKey() ), config->debug );
				}
				else if ( line.startsWith( "ca " ) ) {
					acc->setAuthType( VpnAccountData::cert );

					QString ca = line.section( ' ', 1 );

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry( i18n( "OpenVPN import: ca file: %1" ).arg( importprefix+"/"+ca ), config->debug );

					if ( QFile( QString( importprefix+"/"+ca )).exists())
					{
						// copy to ~/.kde/share/apps/kvpnc/ and rename it
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: copy %1 to %2" ).arg(QString( importprefix+"/"+ca )).arg(QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca)), config->debug );
						QProcess CopyProcess( this );
						CopyProcess.addArgument("cp");
						CopyProcess.addArgument(QString( importprefix+"/"+ca ));
						CopyProcess.addArgument(QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca));
						if (CopyProcess.start())
							acc->setCaCertificate( QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+ca) );
						else
						{
							config->appendLogEntry( i18n( "Could not start %1!" ).arg( "cp" ), config->error );
							acc->setCaCertificate( ca );
						}
					}
					else
					{
						// we use the given path
						acc->setCaCertificate( ca );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: CA certificate: %1" ).arg( acc->getCaCertificate() ), config->debug );
				}
				else if ( line.startsWith( "proto " ) ) {
					if ( line.section( ' ', 1, 1 ) == "udp" ) {
						acc->setUseUdp( true );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: use UDP" ), config->debug );
					} else if ( line.section( ' ', 1, 1 ) == "tcp-client" ) {
						acc->setUseUdp( false );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: dont use UDP" ), config->debug );
					}

				}
				else if ( line.startsWith( "cipher " ) ) {
					acc->setUseUserdefiniedCipher( true );
					acc->setUserdefiniedCipher( line.section( ' ', 1, 1 ) );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use userdefinied cipher" ), config->debug );
				}
				else if ( line.startsWith( "tls-auth " ) ) {
					acc->setUseTlsAuth( true );

				QString tlsauth = line.section( ' ', 1,1 );

					if ( config->KvpncDebugLevel > 2 )
						config->appendLogEntry( i18n( "OpenVPN import: TLS auth file: %1" ).arg( importprefix+"/"+tlsauth ), config->debug );

					if ( QFile( QString( importprefix+"/"+tlsauth )).exists())
					{
						// copy to ~/.kde/share/apps/kvpnc/ and rename it
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: copy %1 to %2" ).arg(QString( importprefix+"/"+tlsauth )).arg(QString(certprefix+"/"+"openvpn_privatekey_"+profilename+"_"+tlsauth)), config->debug );
						QProcess CopyProcess( this );
						CopyProcess.addArgument("cp");
						CopyProcess.addArgument(QString( importprefix+"/"+tlsauth ));
						CopyProcess.addArgument(QString(certprefix+"/"+"openvpn_tlsauth_"+profilename+"_"+tlsauth));
						if (CopyProcess.start())
							acc->setTlsAuthFile( QString(certprefix+"/"+"openvpn_tlsauth_"+profilename+"_"+tlsauth) );
						else
						{
							config->appendLogEntry( i18n( "Could not start %1!" ).arg( "cp" ), config->error );
							acc->setTlsAuthFile( tlsauth );
						}
					}
					else
					{
						// we use the given path
						acc->setTlsAuthFile( tlsauth );
					}

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use TLS auth" ), config->debug );
				}
				else if ( line.startsWith( "redirect-gateway" ) ) {
					acc->setUseRedirectGateway( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use redirect gateway" ), config->debug );
				}
				else if ( line.startsWith( "ns-cert-type " ) ) {
					acc->setUseNsCertType( true );

					if ( line.section( ' ', 1, 1 ) == "client" ) {
						acc->setNsCertType( "client" );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: use NS certificate type: %1" ).arg( "client" ), config->debug );
					}
					if ( line.section( ' ', 1, 1 ) == "server" ) {
						acc->setNsCertType( "server" );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: use NS certificate type: %1" ).arg( "server" ), config->debug );
					}
				}
				else if ( line.startsWith( "auth-user-pass" ) ) {
					acc->setAuthWithUsernameAndPassword( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: authenticate with username and password" ), config->debug );
				}
				else if ( line.startsWith( "http-proxy " ) ) {
					acc->setUseHttpProxy( true );
					acc->setHttpProxy( line.section( ' ', 1, 1 ) );
					acc->setHttpProxyPort( line.section( ' ', 2, 2 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use HTTP proxy: %1, Port: %2" ).arg( acc->getHttpProxy() ).arg( QString().setNum( acc->getHttpProxyPort() ) ), config->debug );
				}
				else if ( line.startsWith( "http-proxy-timeout " ) ) {
					acc->setUseHttpProxy( true );
					acc->setHttpProxyTimeout( line.section( ' ', 1, 1 ).toInt() );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: use NS certificate type: %1" ).arg( QString().setNum( acc->getHttpProxyTimeout() ) ), config->debug );
				}
				else if ( line.startsWith( "pkcs12 " ) ) {
					acc->setAuthType( VpnAccountData::cert );
					Pkcs12CertFile = line.section( ' ', 1 );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: pkcs12 file found: %1" ).arg( Pkcs12CertFile ), config->debug );
				}
				else if ( line.startsWith( "cd " ) ) {
					CertPath = line.section( ' ', 1 );

					//
					// 					if (!Pkcs12CertFile.isEmpty() && !acc->getCaCertificate().startsWith(CertPath))
					// 						Pkcs12CertFile=CertPath+"/"+Pkcs12CertFile;


					acc->setCertPath( CertPath );

					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: certificate prefix found: %1" ).arg( CertPath ), config->debug );
				}
				else if ( line.startsWith( "tls-remote" ) ) {
					if (!line.section( ' ', 1, 1 ).isEmpty())
					{
						acc->setUseTlsRemoteHost( true );
						acc->setTlsRemoteHost( line.section( ' ', 1, 1 ) );
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: use tls remote host: %1" ).arg( acc->getTlsRemoteHost() ), config->debug );
					}
					else
					{
						if ( config->KvpncDebugLevel > 0 )
							config->appendLogEntry( i18n( "OpenVPN import: tls remote host option was specified but server is empty, disabling TLS remote host." ), config->debug );
					}
				}
				else if ( line.startsWith( "float" ) ) {
					acc->setAllowIpAddressChangeOfPeer( true );
					if ( config->KvpncDebugLevel > 0 )
						config->appendLogEntry( i18n( "OpenVPN import: allow IP address change of peer (for DHCP)" ), config->debug );
				}

			}
		}

// 		if ( acc->getCertPath().isEmpty() ) {
// 			// 						acc->setCertPath("/etc/openvpn/");
// 			acc->setCertPath( locateLocal ( "data", "kvpnc" ) );
// 			CertPath = acc->getCertPath();
// 		}
// 		// 					/* we need to put the prefix to all cert paths */
// 		if ( !acc->getCaCertificate().isEmpty() && !acc->getCaCertificate().startsWith( CertPath ) ) {
// 			QString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = QString( Utils(this->GlobalConfig).removeSpecialCharsForFilename( filename).section( '/', 0, -2 ) + "/" );
// 
// 			QFile CaCertFileSrc ( prefix + acc->getCaCertificate() );
// 			QFile CaCertFileDst ( QString( CertPath + "/openvpnimport_" + acc->getCaCertificate() ) );
// 			acc->setCaCertificate( QString( CertPath + "/openvpnimport_" + acc->getCaCertificate() ) );
// 
// 			QTextStream CaCertDstStream( &CaCertFileDst );
// 
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy ca cert file %1 to %2." ).arg( CaCertFileSrc.name() ).arg( CaCertFileDst.name() ), config->debug );
// 
// 
// 			if ( CaCertFileSrc.open( IO_ReadOnly ) ) {
// 				if ( CaCertFileDst.open( IO_WriteOnly ) ) {
// 					QString CaCertFileString = QString( CaCertFileSrc.readAll() );
// 					CaCertDstStream << CaCertFileString;
// 					CaCertFileDst.close();
// 					if ( config->KvpncDebugLevel > 1 )
// 						config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 copied to %2." ).arg( CaCertFileSrc.name() ).arg( CaCertFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 could not open for writing." ).arg( CaCertFileDst.name() ), config->debug );
// 
// 					//								return;
// 				}
// 				CaCertFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: ca cert file %1 could not open for reading." ).arg( CaCertFileSrc.name() ), config->debug );
// 
// 				//							return;
// 			}
// 		}
// 		if ( !acc->getX509Certificate().isEmpty() && !acc->getX509Certificate().startsWith( CertPath ) ) {
// 			QString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = QString( Utils(this->GlobalConfig).removeSpecialCharsForFilename(filename).section( '/', 0, -2 ) + "/" );
// 
// 			QFile X509CertFileSrc ( prefix + acc->getX509Certificate() );
// 			QFile X509CertFileDst ( QString( CertPath + "/openvpnimport_" + acc->getX509Certificate() ) );
// 			acc->setX509Certificate( QString( CertPath + "/openvpnimport_" + acc->getX509Certificate() ) );
// 
// 			QTextStream X509CertDstStream( &X509CertFileDst );
// 
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy pre shared key file %1 to %2." ).arg( X509CertFileSrc.name() ).arg( X509CertFileDst.name() ), config->debug );
// 
// 
// 			if ( X509CertFileSrc.open( IO_ReadOnly ) ) {
// 				if ( X509CertFileDst.open( IO_WriteOnly ) ) {
// 					QString X509CertFileString = QString( X509CertFileSrc.readAll() );
// 					X509CertDstStream << X509CertFileString;
// 					X509CertFileDst.close();
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 copied to %2." ).arg( X509CertFileSrc.name() ).arg( X509CertFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 could not open for writing." ).arg( X509CertFileDst.name() ), config->debug );
// 
// 					//								return;
// 				}
// 				X509CertFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: X509 cert file %1 could not open for reading." ).arg( X509CertFileSrc.name() ), config->debug );
// 
// 				//							return;
// 			}
// 		}
// 		if ( !acc->getPreSharedKey().isEmpty() && !acc->getPreSharedKey().startsWith( CertPath ) ) {
// 			QString prefix = "./";
// 			if ( !filename.find( '/' ) > -1 )
// 				prefix = QString( Utils(this->GlobalConfig).removeSpecialCharsForFilename( filename).section( '/', 0, -2 ) + "/" );
// 
// 			QFile PreSharedKeyFileSrc ( QString( prefix + acc->getPreSharedKey() ) );
// 			QFile PreSharedKeyFileDst ( QString( CertPath + "/openvpnimport_" + acc->getPreSharedKey() ) );
// 			acc->setPreSharedKey( QString( CertPath + "/openvpnimport_" + acc->getPreSharedKey() ) );
// 
// 			QTextStream PreSharedKeyDstStream( &PreSharedKeyFileDst );
// 
// 			if ( config->KvpncDebugLevel > 1 )
// 				config->appendLogEntry( i18n( "OpenVPN import: try to copy pre shared key file %1 to %2." ).arg( PreSharedKeyFileSrc.name() ).arg( PreSharedKeyFileDst.name() ), config->debug );
// 
// 			if ( PreSharedKeyFileSrc.open( IO_ReadOnly ) ) {
// 				if ( PreSharedKeyFileDst.open( IO_WriteOnly ) ) {
// 					QString PreSharedKeyFileString = QString( PreSharedKeyFileSrc.readAll() );
// 					PreSharedKeyDstStream << PreSharedKeyFileString;
// 					PreSharedKeyFileDst.close();
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 copied to %2." ).arg( PreSharedKeyFileSrc.name() ).arg( PreSharedKeyFileDst.name() ), config->debug );
// 				} else {
// 					if ( config->KvpncDebugLevel > 2 )
// 						config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 could not open for writing." ).arg( PreSharedKeyFileDst.name() ), config->debug );
// 
// 					//								return;
// 				}
// 				PreSharedKeyFileDst.close();
// 			} else {
// 				if ( config->KvpncDebugLevel > 2 )
// 					config->appendLogEntry( i18n( "OpenVPN import: pre shared key file %1 could not open for reading." ).arg( PreSharedKeyFileSrc.name() ), config->debug );
// 
// 				//							return;
// 			}
// 		}

		OpenvpnConfigFile.close();
		importOk = true;
	} else
		importOk = false;

	if ( !Pkcs12CertFile.isEmpty() ) {

		QString filepath = filename.section( '/', 0, -2 ); // should give the base path

		ImportCertificateDialog dlg( this, i18n( "Import Certificate..." ).ascii(), config );

		QString p12file="";

		if (Pkcs12CertFile.find('/') < 0 ) 
			p12file = filepath + "/" + Pkcs12CertFile;
		else
			p12file = Pkcs12CertFile;

		dlg.main->ImporttypeComboBox->setCurrentItem( ImportCertificateDialog::pkcs12_openvpn );
		dlg.typeToggeled( ImportCertificateDialog::pkcs12_openvpn );
		dlg.main->ImporttypeComboBox->setEnabled( false );
		// 		dlg.main->RacoonCertificatePathUrlrequester->setEnabled(true);
		dlg.main->RacoonCertificatePathUrlrequester->setURL( "/etc/openvpn" );
		dlg.main->FreeswanGroupBox->setEnabled( true );
// 		dlg.main->FreeswanGroupBox->setCurrentText(i18n("OpenVPN"));
		dlg.main->PrivateKeyPasswordEdit->setEnabled( true );
		dlg.main->PrivateKeyAgainPasswordEdit->setEnabled( true );

		dlg.main->FilenameUrlrequester->setURL( p12file );
		dlg.main->ImportPasswordEdit->setEnabled( true );
		dlg.main->P12GroupBox->setEnabled( true );
		dlg.main->FreeswanGroupBox->setEnabled( true );
		dlg.main->FilenameUrlrequester->setFilter( "*.p12" );

		dlg.main->ImportPasswordEdit->setFocus();

		int result = dlg.exec();

		if ( result == QDialog::Accepted ) {
			if ( dlg.importSuccess ) {
				config->appendLogEntry( i18n( "OpenVPN import: import of pkcs12 certificate file %1 was successful." ).arg( Pkcs12CertFile ), config->info );
				importOk = true;

				acc->setX509Certificate( dlg.main->RacoonCertificatePathUrlrequester->url() + "/mykeys_" + ( p12file.left( p12file.length() - 4 ) ).section( '/', -1, -1 ) + ".pem" );
				acc->setPrivateKey( dlg.main->RacoonCertificatePathUrlrequester->url() + "/mykeys_" + ( p12file.left( p12file.length() - 4 ) ).section( '/', -1, -1 ) + ".pem" );
				acc->setCaCertificate( dlg.main->RacoonCertificatePathUrlrequester->url() + "/ca_" + ( p12file.left( p12file.length() - 4 ) ).section( '/', -1, -1 ) + ".pem" );
				acc->setCertPath( dlg.main->RacoonCertificatePathUrlrequester->url() );
				acc->setPrivateKeyPass( dlg.main->ImportPasswordEdit->text() );


			} else {
				config->appendLogEntry( i18n( "OpenVPN import: import of pkcs12 certificate file %1 failed!" ).arg( Pkcs12CertFile ), config->error );
				importOk = false;
			}
		} else {
			config->appendLogEntry( i18n( "OpenVPN import: import of pkcs12 certificate file was cancelled." ).arg( Pkcs12CertFile ), config->info );
			importOk = false;
		}

	}

	//std::cout << "accept" << std::endl;
	QDialog::accept();
}


#include "importopenvpnprofiledialog.moc"
