// =============================================================================
//
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviImageWindow"

#define __KVIRC_PLUGIN__

#include "kvi_filedialog.h"
#include "kvi_iconloader.h"
#include "kvi_image_view.h"
#include "kvi_image_window.h"
#include "kvi_locale.h"
#include "kvi_popupmenu.h"
#include "kvirc_plugin.h"

static QPixmap *g_pIcon;
static KviStr g_szLastBrowsedDirectory("/");

static const QString g_tbIcon("icon_image.png");
static const QString g_tbOpen("fileopen.png");
static const QString g_tbZoomIn("viewmag+.png");
static const QString g_tbZoomOut("viewmag-.png");
static const QString g_tbZoomNone("viewmag1.png");

KviImageWindow::KviImageWindow(KviFrame *lpFrm)
	: KviWindow("[Image Viewer]: no file loaded", KVI_WND_TYPE_PLUGIN, lpFrm)
{
	m_pImageView = new KviImageView(this);
	m_pToolBar   = new KviToolBar("Image Toolbar", lpFrm, this, Qt::Unmanaged, true);

	m_pToolBar->insertButton(g_tbOpen, 0, SIGNAL(clicked()), this, SLOT(openFile()), true, "Open image", "Open image");

	KviPopupMenu *m = new KviPopupMenu(this);
	m->insertItem("+ 1 %",   this, SLOT(zoomIn()));
	m->insertItem("+ 5 %",   this, SLOT(zoomIn5()));
	m->insertItem("+ 10 %",  this, SLOT(zoomIn10()));
	m->insertItem("+ 50 %",  this, SLOT(zoomIn50()));
	m->insertItem("+ 100 %", this, SLOT(zoomIn100()));
	m_pToolBar->insertButton(g_tbZoomIn, 1, SIGNAL(clicked()), this, SLOT(zoomIn10()), true, "Zoom in", "Zoom in");
	m_pToolBar->setDelayedPopup(1, m);

	m = new KviPopupMenu(this);
	m->insertItem("- 1 %",   this, SLOT(zoomOut()));
	m->insertItem("- 5 %",   this, SLOT(zoomOut5()));
	m->insertItem("- 10 %",  this, SLOT(zoomOut10()));
	m->insertItem("- 50 %",  this, SLOT(zoomOut50()));
	m->insertItem("- 100 %", this, SLOT(zoomOut100()));
	m_pToolBar->insertButton(g_tbZoomOut, 2, SIGNAL(clicked()), this, SLOT(zoomOut10()), true, "Zoom out", "Zoom out");
	m_pToolBar->setDelayedPopup(2, m);

	m_pToolBar->insertButton(g_tbZoomNone, 3, SIGNAL(clicked()), this, SLOT(noZoom()), true, "No zoom", "No zoom");
}

KviImageWindow::~KviImageWindow()
{
	// Nothing here
}

QPixmap *KviImageWindow::myIconPtr()
{
	return g_pIcon;
}

void KviImageWindow::resizeEvent(QResizeEvent *)
{
	int hght = 39; // TODO: this always returns 16: m_pToolBar->sizeHint().height();
	m_pToolBar->setGeometry(0, 0, width(), hght);
	m_pImageView->setGeometry(0, hght, width(), height() - hght);
}

bool KviImageWindow::loadImage(const char *fName)
{
	KviStr filename = fName;
	filename.stripWhiteSpace();
	if( filename.isEmpty() )
		filename = KviFileDialog::getOpenFileName(g_szLastBrowsedDirectory.ptr());
	if( filename.isEmpty() ) return true;
	if( !m_pImageView->loadImage(filename.ptr()) ) {
		g_pApp->warningBox(_i18n_("Could not load image %s"), filename.ptr());
		return false;
	}
	QFileInfo fi(filename.ptr());
	g_szLastBrowsedDirectory = fi.filePath();
	KviStr tmp(KviStr::Format, "[Image Viewer]: %s", filename.ptr());
	setWindowCaption(tmp.ptr());
	return true;
}

void KviImageWindow::openFile()
{
	loadImage("");
}

void KviImageWindow::zoomIn()       { m_pImageView->setZoom(m_pImageView->zoom() +   1); }
void KviImageWindow::zoomIn5()      { m_pImageView->setZoom(m_pImageView->zoom() +   5); }
void KviImageWindow::zoomIn10()     { m_pImageView->setZoom(m_pImageView->zoom() +  10); }
void KviImageWindow::zoomIn50()     { m_pImageView->setZoom(m_pImageView->zoom() +  50); }
void KviImageWindow::zoomIn100()    { m_pImageView->setZoom(m_pImageView->zoom() + 100); }
void KviImageWindow::zoomOut()      { m_pImageView->setZoom(m_pImageView->zoom() -   1); }
void KviImageWindow::zoomOut5()     { m_pImageView->setZoom(m_pImageView->zoom() -   5); }
void KviImageWindow::zoomOut10()    { m_pImageView->setZoom(m_pImageView->zoom() -  10); }
void KviImageWindow::zoomOut50()    { m_pImageView->setZoom(m_pImageView->zoom() -  50); }
void KviImageWindow::zoomOut100()   { m_pImageView->setZoom(m_pImageView->zoom() - 100); }

void KviImageWindow::noZoom()
{
	m_pImageView->setZoom(100);
}

bool image_plugin_command_image(KviPluginCommandStruct *cmd)
{
	KviImageWindow *wnd = new KviImageWindow(cmd->frame);
	kvirc_plugin_add_window(cmd->handle, cmd->frame,wnd);
	KviStr tmp = kvirc_plugin_param(cmd, 1);
	wnd->loadImage(tmp.ptr());
	return true;
}

bool image_plugin_init(KviPluginCommandStruct *cmd)
{
	g_pIcon = new QPixmap(KviIconLoader::loadIcon(g_tbIcon));
	g_szLastBrowsedDirectory = QDir::homeDirPath();
	kvirc_plugin_register_command(cmd->handle, "IMAGE", image_plugin_command_image);
	return true;
}

void image_plugin_cleanup()
{
	delete g_pIcon;
	g_pIcon = 0;
}

/**
	@document: doc_plugin_image.kvihelp
	@title: The image viewer plugin
		Adds a simple internal image viewer to KVIrc.<br>
		You can view images by using the command<br>
		<b>/image [filename]</b><br>
		Filename must be an absolute file path.<br>
		If no filename is specified, a "file open" dialog
		will be shown to allow you to select a file.<br>
		The image types supported are dependant on
		the underlying Qt and compilation options.<br>
		Generally, bmp, png, pnm and xpm are available.<br>
		If KVIrc has been compiled with jpeg support, that
		format will be also viewable.<br>
		You can use the viewer as the default "player"
		of images (see <a href="play.kvihelp">PLAY</a>).<br>
		In order to do so, put "IMAGE <a href="s_execfilename.kvihelp">$execFileName</a>" in the
		"Run script" field (misc options dialog, mimetypes tab)
		for the mimetypes that you want to play with the viewer.<br>
		Note:<br>
		If you unload the plugin, the /IMAGE command will no longer work.<br>
		Here comes a simple workaround "commandline field" script for this problem:<br>
		Version 1: Use the plugin if is loaded.<br>
		<b><a href="if.kvihelp">if</a>(<a href="s_pluginLoaded.kvihelp">$pluginLoaded</a>(image))image <a href="s_execfilename.kvihelp">$ExecFileName</a>; else <a href="exec.kvihelp">exec</a> -q myviewer <a href="s_execfilename.kvihelp">$ExecFileName</a></b><br>
		Workaround 2: Use the plugin always, if it is not loaded, load it.<br>
		<b><a href="if.kvihelp">if</a>(!<a href="s_pluginloaded.kvihelp">$pluginLoaded</a>(image))<a href="plugin.kvihelp">PLUGIN</a> load libkviimage.so; image <a href="s_execfilename.kvihelp">$ExecFileName</a></b><br>
*/
void image_plugin_help()
{
	kvirc_plugin_show_help_page("doc_plugin_image.kvihelp");
}

KviPlugin kvirc_plugin =
{
	"Image",
	"A simple image viewer",
	"0.1",
	"Szymon Stefanek <stefanek@tin.it>",
	"<html>"
		"Exports the <b>/IMAGE</b> command."
		"</html>",
	image_plugin_init,
	image_plugin_cleanup,
	0,
	image_plugin_help
};

#include "m_kvi_image_window.moc"
