// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptTreeViewItem"

#include <qpixmap.h>

#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_listview.h"
#include "kvi_script_listview.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_treeview_item.h"
#include "kvi_script_widget.h"

/*
	@class: treeviewitem
	@short:
		A tree view item
	@inherits:
		[class]object[/class]
	@functions:
		!fn: $setText(&lt;idx&gt;, &lt;text&gt;)
		Sets this item's column &lt;idx&gt; text to &lt;text&gt;.
		&lt;idx&gt; is a zero-based index.
		The text will be obviously visible only if
		the column &lt;idx&gt; has been added to the [class]treeview[/class]
		that this item belongs to.

		!fn: $setPixmap(&lt;idx&gt;, &lt;image path&gt;)
		Sets this item's column &lt;idx&gt; pixmap to the specified image.
		If the image cannot be loaded or the &lt;image path&gt; is an empty
		string, any previous pixmap for this column is unset.

		!fn: $text(&lt;idx&gt;)
		Returns the text in the column &lt;idx&gt; (zero based index!)

		!fn: $setOpen(&lt;boolean&gt;)
		If &lt;boolean&gt; is non 0, opens this item showing its children tree,
		otherwise closes this item.

		!fn: $isSelected()
		Returns 1 if this item is currently selected, 0 otherwise

		!fn: $firstChild()
		Returns the id of the first child treeviewitem, or an empty string if this item
		has no children items.

		!fn: $nextSibling()
		Returns the id of the next item that is a child of the parent of this item.

	@description:
		The treeviewitem object class is to be used with the
		[class]treeview[/class] class.<br> Basically, you create
		a treeview widget and then add a tree of treeviewitems to it.<br>
		The treeviewitem is able to display text and icons in the
		columns of the treeview.<br>
		To add a treeviewitem to a [class]treeview[/class] widget
		simply create the treeviewitem object with [fnc]$new[/fnc].<br>
		If you create the treeviewitem object as child of the treeview,
		it will appear as toplevel item in the tree, otherwise
		you must create it as child of another treviewitem (so it
		will be a children in the item tree too).<br>
		When a [class]treeview[/class] widget is destroyed,
		all the contained items are destroyed too.<br>
		To remove a treeviewitem from a [class]treeview[/class]
		just [cmd]destroy[/cmd] the treeviewitem object.
		You can naviage the tree of items by using the $firstChild() and $nextSibling()
		functions.<br>
	@examples:
		See the example for class [class]treeview[/class]
	@seealso:
		class [class]object[/class]
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>
*/

/**
 * TREEVIEWITEM class
 */
void KviScriptTreeViewItem::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setText",     (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_SETTEXT);
	d->addBuiltinFunction("setPixmap",   (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_SETPIXMAP);
	d->addBuiltinFunction("text",        (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_TEXT);
	d->addBuiltinFunction("setOpen",     (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_SETOPEN);
	d->addBuiltinFunction("isSelected",  (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_ISSELECTED);
	d->addBuiltinFunction("firstChild",  (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_FIRSTCHILD);
	d->addBuiltinFunction("nextSibling", (scriptObjectFunction) &KviScriptTreeViewItem::builtinFunction_NEXTSIBLING);
}

KviScriptTreeViewItem::KviScriptTreeViewItem(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptObject(cntrl, p, name, pDef)
{
	m_pItem = 0;
}

KviScriptTreeViewItem::~KviScriptTreeViewItem()
{
	if( m_pItem ) {
		KviScriptListViewItem *it = m_pItem;
		m_pItem = 0;
		delete it;
	}
}

bool KviScriptTreeViewItem::init(QPtrList<KviStr> *params)
{
	QString s1, s2, s3, s4;

	KviStr *pS = params ? params->first() : 0;
	if( pS )
		s1 = pS->ptr();
	else
		s1 = "?";

	pS = params && pS ? params->next() : 0;
	if( pS )
		s2 = pS->ptr();
	else
		s2 = QString::null;

	pS = params && pS ? params->next() : 0;
	if( pS )
		s3 = pS->ptr();
	else
		s3 = QString::null;

	pS = params && pS ? params->next() : 0;
	if( pS )
		s4 = pS->ptr();
	else
		s4 = QString::null;

	if( parent() ) {
		if( parent()->inherits("KviScriptTreeView") ) {
			m_pItem = new KviScriptListViewItem(this,
				((KviListView *) (((KviScriptWidget *) parent())->m_pWidget)), s1, s2, s3, s4
			);
		} else if(parent()->inherits("KviScriptTreeViewItem")) {
			m_pItem = new KviScriptListViewItem(this,
				((KviScriptListViewItem *) (((KviScriptTreeViewItem *) parent())->m_pItem)), s1, s2, s3, s4
			);
		}
	}

	if( !m_pItem )
		return false;
	return true;
}

int KviScriptTreeViewItem::builtinFunction_FIRSTCHILD(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviScriptListViewItem *it = (KviScriptListViewItem *) (m_pItem->firstChild());
	if( it )
		buffer.append(it->m_pScriptObjectClass->id());
	return KVI_ERROR_Success;
}

int KviScriptTreeViewItem::builtinFunction_NEXTSIBLING(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviScriptListViewItem *it = (KviScriptListViewItem *) (m_pItem->nextSibling());
	if( it )
		buffer.append(it->m_pScriptObjectClass->id());
	return KVI_ERROR_Success;
}

int KviScriptTreeViewItem::builtinFunction_SETTEXT(QPtrList<KviStr> *params, KviStr &)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk ) {
				pS = params->next();
				m_pItem->setText(idx, pS ? pS->ptr() : "");
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_SETPIXMAP(QPtrList<KviStr> *params, KviStr &)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk ) {
				pS = params->next();
				if( pS ) {
					KviStr path;
					if( g_pApp->findImage(path, pS->ptr()) ) {
						m_pItem->setPixmap(idx, QPixmap(path.ptr()));
						return KVI_ERROR_Success;
					}
				}
				m_pItem->setPixmap(idx, QPixmap());
				return KVI_ERROR_Success;
			}
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_TEXT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk && (idx >= 0) ) {
				buffer.append(m_pItem->text(idx));
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_SETOPEN(QPtrList<KviStr> *params, KviStr &)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int idx = pS->toInt(&bOk);
			if( bOk ) {
				m_pItem->setOpen(idx);
				return KVI_ERROR_Success;
			}
			return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptTreeViewItem::builtinFunction_ISSELECTED(QPtrList<KviStr> *, KviStr &buffer)
{
	buffer.append(KviStr::Format, "%d", m_pItem->isSelected() ? 0 : 1);
	return KVI_ERROR_Success;
}

#include "m_kvi_script_treeview_item.moc"
