// =============================================================================
//
//      --- kvi_completionbox_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   but is lifted almost in its entirety from kcompletionbox.cpp
//
//   Copyright (c) 2000,2001,2002 Carsten Pfeiffer <pfeiffer@kde.org>
//   Copyright (c) 2000 Stefan Schimanski <1Stein@gmx.de>
//   Copyright (c) 2000,2001 Dawit Alemayehu <adawit@kde.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qapplication.h>
#include <qevent.h>
#include <qstringlist.h>
#include <qstyle.h>

#include "kvi_completionbox_qt.h"

KviCompletionBox::KviCompletionBox(QWidget *parent, const char *name)
	: QListBox(parent, name, WType_Popup)
{
	m_parent = parent;

	setColumnMode(1);
	setLineWidth(1);
	setFrameStyle(QFrame::Box | QFrame::Plain);

	if( parent )
		setFocusProxy(parent);
	else
		setFocusPolicy(NoFocus);

	setVScrollBarMode(Auto);
	setHScrollBarMode(AlwaysOff);

	connect(this, SIGNAL(doubleClicked(QListBoxItem *)),  SLOT(slotActivated(QListBoxItem *)));
	connect(this, SIGNAL(currentChanged(QListBoxItem *)), SLOT(slotCurrentChanged()));
	connect(this, SIGNAL(clicked(QListBoxItem *)),        SLOT(slotItemClicked(QListBoxItem *)));
}

KviCompletionBox::~KviCompletionBox()
{
	m_parent = 0L;
}

QStringList KviCompletionBox::items() const
{
	QStringList list;
	for( uint i = 0; i < count(); i++ )
		list.append(text(i));
	return list;
}

void KviCompletionBox::slotActivated(QListBoxItem *item)
{
	if( !item ) return;

	hide();
	emit activated(item->text());
}

bool KviCompletionBox::eventFilter(QObject *o, QEvent *e)
{
	int type = e->type();

	if( o == m_parent ) {
		if( isVisible() ) {
			if( type == QEvent::KeyPress ) {
				QKeyEvent *ev = static_cast<QKeyEvent *> (e);
				switch( ev->key() ) {
					case Key_Down:
						down();
						ev->accept();
						return true;
					case Key_Up:
						up();
						ev->accept();
						return true;
					case Key_Prior:
						pageUp();
						ev->accept();
						return true;
					case Key_Next:
						pageDown();
						ev->accept();
						return true;
					case Key_Escape:
						cancelled();
						ev->accept();
						return true;
					case Key_Enter:
					case Key_Return:
						if( ev->state() & ShiftButton ) {
							hide();
							ev->accept();  // Consume the Enter event
							return true;
						}
						break;
					case Key_End:
						if( ev->state() & ControlButton ) {
							end();
							ev->accept();
							return true;
						}
					case Key_Home:
						if( ev->state() & ControlButton ) {
							home();
							ev->accept();
							return true;
						}
					default:
						break;
				}
			}
			else if( type == QEvent::AccelOverride ) {
				// Override any acceleartors that match
				// the key sequences we use here...
				QKeyEvent *ev = static_cast<QKeyEvent *> (e);
				switch( ev->key() ) {
					case Key_Down:
					case Key_Up:
					case Key_Prior:
					case Key_Next:
					case Key_Escape:
					case Key_Enter:
					case Key_Return:
						ev->accept();
						return true;
						break;

					case Key_Home:
					case Key_End:
						if( ev->state() & ControlButton ) {
							ev->accept();
							return true;
						}
						break;
					default:
						break;
				}
			}
			// parent loses focus or gets a click -> we hide
			else if( type == QEvent::FocusOut || type == QEvent::Resize ||
			         type == QEvent::Close    || type == QEvent::Hide   ||
			         type == QEvent::Move ) {
				hide();
			}
			else if( type == QEvent::Move )
				move(m_parent->mapToGlobal(QPoint(0, m_parent->height())));
			else if( type == QEvent::Resize )
				resize(sizeHint());
		}
	}

	// Any mouse-click on something else than "this" makes us hide
	else if( type == QEvent::MouseButtonPress ) {
		QMouseEvent *ev = static_cast<QMouseEvent *> (e);
		if( !rect().contains(ev->pos()) )
			hide();
	}

	return QListBox::eventFilter(o, e);
}


void KviCompletionBox::popup(const QPoint &p)
{
	if( count() == 0 )
		hide();
	else {
		ensureCurrentVisible();
		bool block = signalsBlocked();
		blockSignals(true);
		setCurrentItem(0);
		blockSignals(block);
		clearSelection();
		if( !isVisible() ) {
			resize(sizeHint());
			setCurrentItem(0);
			setSelected(0, true);
			move(p);
			qApp->installEventFilter(this);
			QListBox::show();
		} else if( size().height() < sizeHint().height() )
			resize(sizeHint());
	}
}

void KviCompletionBox::hide()
{
	if( m_parent )
		qApp->removeEventFilter(this);
	QListBox::hide();
}

QSize KviCompletionBox::sizeHint() const
{
	int ih = itemHeight() + 1;
	int h = QMIN(15 * ih, (int) count() * ih) + 2;
	h = QMAX(h, QListBox::minimumSizeHint().height());

	int w = QListBox::minimumSizeHint().width();
	w = QMAX(QListBox::minimumSizeHint().width(), w);
	return QSize(w, h);
}

void KviCompletionBox::down()
{
	int i = currentItem();

	if( i < (int) count() - 1 )
		setCurrentItem(i + 1);
}

void KviCompletionBox::up()
{
	if( currentItem() > 0 )
		setCurrentItem(currentItem() - 1);
}

void KviCompletionBox::pageDown()
{
	int i = currentItem() + numItemsVisible();
	i = i > (int) count() - 1 ? (int) count() - 1 : i;
	setCurrentItem(i);
}

void KviCompletionBox::pageUp()
{
	int i = currentItem() - numItemsVisible();
	i = i < 0 ? 0 : i;
	setCurrentItem(i);
}

void KviCompletionBox::home()
{
	setCurrentItem(0);
}

void KviCompletionBox::end()
{
	setCurrentItem(count() - 1);
}

void KviCompletionBox::cancelled()
{
	if( isVisible() )
		hide();
}

class KviCompletionBoxItem : public QListBoxItem
{
public:
	void reuse(const QString &text) { setText(text); }
};


void KviCompletionBox::insertItems(const QStringList &items, int index)
{
	bool block = signalsBlocked();
	blockSignals(true);
	insertStringList(items, index);
	blockSignals(block);
}

void KviCompletionBox::setItems(const QStringList &items)
{
	bool block = signalsBlocked();
	blockSignals(true);

	QListBoxItem *item = firstItem();
	if( !item )
		insertStringList(items);
	else {
		for( QStringList::ConstIterator iter = items.begin(); iter != items.end(); iter++ ) {
			if( item ) {
				((KviCompletionBoxItem *) item)->reuse(*iter);
				item = item->next();
			} else
				insertItem(new QListBoxText(*iter));
		}
		QListBoxItem *tmp = item;
		while( (item = tmp) ) {
			tmp = item->next();
			delete item;
		}
		triggerUpdate(false);
	}

	blockSignals(block);
}

void KviCompletionBox::slotCurrentChanged()
{
	// Nothing here
}

void KviCompletionBox::slotItemClicked(QListBoxItem *item)
{
	if( item ) {
		hide();
		emit activated(item->text());
	}
}

bool KviCompletionBox::activate()
{
	if( selectedItem() ) {
		slotActivated(selectedItem());
		return true;
	}
	return false;
}

#include "m_kvi_completionbox_qt.moc"
