//
//   File : libkvisystem.cpp
//   Creation date : Fri Nov 16 03:50:12 2001 GMT by Szymon Stefanek
//
//   This system is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"
#include "kvi_module.h"
#include "kvi_string.h"
#include "kvi_uparser.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_env.h"

#ifndef COMPILE_ON_WINDOWS
	#include <sys/utsname.h>
	#include <stdlib.h>
	#include <unistd.h>
#endif

#ifdef COMPILE_KDE_SUPPORT
	#include <dcopclient.h>
#endif


/*
	@doc: system.ostype
	@type:
		function
	@title:
		$system.ostype
	@short:
		Returns the type of the operating system
	@syntax:
		$system.ostype
	@description:
		Returns the current type of operating system: unix or windows.[br]
*/

static bool system_module_fnc_ostype(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	buffer.append("unix");
#endif
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.osname
	@type:
		function
	@title:
		$system.osname
	@short:
		Returns the name of the operating system
	@syntax:
		$system.osname
	@description:
		Returns the name of the operating system.[br]
*/

static bool system_module_fnc_osname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.sysname);
#endif
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.osversion
	@type:
		function
	@title:
		$system.osversion
	@short:
		Returns the version of the operating system
	@syntax:
		$system.osversion
	@description:
		Returns the version of the operating system.[br]
*/

static bool system_module_fnc_osversion(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.version);
#endif
	return true; //c->leaveStackFrame();
}


/*
	@doc: system.osrelease
	@type:
		function
	@title:
		$system.osrelease
	@short:
		Returns the release of the operating system
	@syntax:
		$system.osrelease
	@description:
		Returns the release of the operating system.[br]
*/

static bool system_module_fnc_osrelease(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.release);
#endif
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.osmachine
	@type:
		function
	@title:
		$system.osmachine
	@short:
		Returns the machine of the operating system
	@syntax:
		$system.osmachine
	@description:
		Returns the machine of the operating system.[br]
*/

static bool system_module_fnc_osmachine(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.machine);
#endif
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.osnodename
	@type:
		function
	@title:
		$system.osnodename
	@short:
		Returns the nodename of the operating system
	@syntax:
		$system.osnodename
	@description:
		Returns the nodename of the operating system.[br]
*/

static bool system_module_fnc_osnodename(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
#ifdef COMPILE_ON_WINDOWS
	buffer.append("windows");
#else
	struct utsname uts;
	if(uname(&uts) == 0)buffer.append(uts.nodename);
#endif
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.getenv
	@keyterms:
		Enviroinement variables
	@type:
		function
	@title:
		$system.getenv
	@short:
		Returns the value of an enviroinement variable
	@syntax:
		$system.getenv(<variable>)
	@description:
		Returns the value of the enviroinement <variable>.[br]
*/

static bool system_module_fnc_getenv(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
	char * b = kvi_getenv(parms->safeFirstParam());
	if(b)buffer.append(b);
	return true; //c->leaveStackFrame();
}



/*
	@doc: system.hostname
	@keyterms:
		System information
	@type:
		function
	@title:
		$system.hostname
	@short:
		Returns the hostname of the machine that KVIrc is running on
	@syntax:
		$system.hostname
	@description:
		Returns the hostname of the machine that KVIrc is running on.[br]
*/

static bool system_module_fnc_hostname(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"system_module_fnc_size");
	char hbuffer[1024];
	if(gethostname(hbuffer,1024) == 0)buffer.append(hbuffer);
	return true; //c->leaveStackFrame();
}

/*
	@doc: system.dcop
	@keyterms:
		System information
	@type:
		function
	@title:
		$system.dcop
	@short:
		Performs a DCOP call
	@syntax:
		$system.dcop(<application>,<objectid>,<function>,<parameter1>,<parameter2>,...)
	@description:
		This function allows performing simple DCOP calls without executing
		an external process. This feature is available ONLY when KDE support
		is compiled in the executable: this means that this function is absolutely
		non portable (don't use it in scripts that you're going to distribute).
		<application> is the name of the application being called, <objectid> is the
		identifier of the object called, <function> is the function to be executed
		on the remote object and <parameter1>,<parameter2>,... is the list of
		parameters to be passed. The <function> name must contain the
		trailing parenthesis and parameter specification (see examples).
		The parameters MUST be in the form "type=value"
		where "type" is the C++ type of the parameter and value
		is the string rappresentation of the parameter data. Currently
		KVIrc supports only QString,QCString,bool,int and uint data types.[br]
		The returned value is the string rappresentation of the returned
		data if the return type is known, otherwise it is the name of the data type returned.
		[br]
		If the application name is prefixed with "?" then the call is performed in "remote test"
		mode: no "remote" errors are printed and the function returns 1 if the call executed
		succesfully and 0 if the call failed. This can be used with the very first
		call to programmaticaly test if the remote application is running.
	@examples:
		[example]
			echo $system.dcop("kdesktop","KBackgroundIface","currentWallpaper(int)","int=0")
			echo $system.dcop("kdesktop","KScreensaverIface","lock()")
			# we can also ignore the return value in several ways
			%dummy = $system.dcop("kicker","kicker","showKMenu()")
			$system.dcop("kdesktop","KScreensaverIface","save()")
			$system.dcop("kicker","Panel","addBrowserButton(QString)","QString=/")
			# runtime test if a call would work (i.e. , kicker is running at all, parameters are right etc...)
			if($system.dcop("?kicker","kicker","showKMenu()"))echo "Can't make dcop calls to kicker!"
		[/example]
*/

static bool system_module_fnc_dcop(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"system_module_fnc_dcop");

	bool bTestMode = false;

	QCString szApp,szObj,szFun;
	
	szApp = parms->safeFirstParam();
	szObj = parms->safeNextParam();
	szFun = parms->safeNextParam();

	if((szApp.data()) && (szApp.length() > 1))
	{
		if(*(szApp.data()) == '?')
		{
			bTestMode = true;
			szApp.remove(0,1);
		}
	}

#ifdef COMPILE_KDE_SUPPORT

	QByteArray ba;
	QDataStream ds(ba, IO_WriteOnly);

	while(KviStr * p = parms->next())
	{
		if(p->isEmpty())
		{
			c->warning(__tr("Invalid DCOP parameter syntax"));
			return false;
		}

		KviStr tmp = *p;
		KviStr szType = tmp.leftToFirst('=',false);
		tmp.cutToFirst('=',true);
		if(szType.isEmpty())szType = "int";
		bool bOk;
		if(kvi_strEqualCI("int",szType.ptr()) || kvi_strEqualCI("long",szType.ptr()))
		{
			int iii = tmp.toInt(&bOk);
			if(!bOk)
			{
				c->warning(__tr("The specified parameter is not an integer"));
				return false;
			}
			ds << iii;
		} else if(kvi_strEqualCI("QString",szType.ptr()))
		{
			QString ddd = tmp.ptr();
			ds << ddd;
		} else if(kvi_strEqualCI("QCString",szType.ptr()))
		{
			QCString qcs = tmp.ptr();
			ds << qcs;
		} else if(kvi_strEqualCI("bool",szType.ptr()))
		{
			bool bbb = kvi_strEqualCI(tmp.ptr(),"true");
			ds << bbb;
		} else if(kvi_strEqualCI("unsigned int",szType.ptr()) || kvi_strEqualCI("uint",szType.ptr()) || kvi_strEqualCI("Q_UINT32",szType.ptr()))
		{
			unsigned int uii = tmp.toUInt(&bOk);
			if(!bOk)
			{
				c->warning(__tr("The specified parameter is not an integer"));
				return false;
			}
			ds << uii;
		} else {
			c->warning(__tr("Unsupported DCOP parameter type %s"),tmp.ptr());
			return false;
		}
	}

	QByteArray rba;
	QCString szRetType;

	bool bRet = g_pApp->dcopClient()->call(szApp,szObj,szFun,ba,szRetType,rba);

	if(!bRet)
	{
		if(bTestMode)
			buffer.append('0'); // remote failure
		else
			c->warning(__tr("DCOP call failed"));
	} else {
		if(bTestMode)
			buffer.append('1');
		else {
			QDataStream ret(rba, IO_ReadOnly);
			if(szRetType == "bool")
			{
				bool bqw;
				ret >> bqw;
				buffer.append(bqw ? '1' : '0');
			} else if(szRetType == "QString")
			{
				QString szz;
				ret >> szz;
				buffer.append(szz);
			} else if(szRetType == "QCString")
			{
				QCString sss;
				ret >> sss;
				buffer.append(sss.data());
			} else if((szRetType == "uint") || (szRetType == "unsigned int") || (szRetType == "Q_UINT32"))
			{
				unsigned int ui3;
				ret >> ui3;
				buffer.append(KviStr::Format,"%u",ui3);
			} else if((szRetType == "int") || (szRetType == "long"))
			{
				int iii;
				ret >> iii;
				buffer.append(KviStr::Format,"%d",iii);
			} else buffer.append(szRetType.data());
		}
	}
#else
	if(bTestMode)
		buffer.append('0');
	else
		c->warning(__tr("DCOP calls are available only when KDE support is compiled in"));
#endif

	return c->leaveStackFrame();
}


/*
	@doc: system.setenv
	@type:
		command
	@title:
		system.setenv
	@keyterms:
		Enviroinement variables
	@short:
		Sets an enviroinement variable
	@syntax:
		system.setenv <variable> [<value>]
	@description:
		Sets the enviroinement <variable> to the <value> string.[br]
		If <value> is not given , the <variable> is unset.[br]
	@seealso:
		[fnc]$system.getenv[/fnc]()
*/

static bool system_module_cmd_setenv(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"system_module_cmd_setenv");
	KviStr szVariable,szValue;
	if(!g_pUserParser->parseCmdSingleToken(c,szVariable))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szValue))return false;

	if(szVariable.hasData())
	{
		if(szValue.hasData())
		{
			kvi_setenv(szVariable.ptr(),szValue.ptr());
		} else {
			kvi_unsetenv(szVariable.ptr());
		}
	} else {
		c->warning(__tr("No environment variable name given"));
	}

	return c->leaveStackFrame();
}




static bool system_module_init(KviModule * m)
{
	m->registerFunction("ostype",system_module_fnc_ostype);
	m->registerFunction("osname",system_module_fnc_osname);
	m->registerFunction("osversion",system_module_fnc_osversion);
	m->registerFunction("osrelease",system_module_fnc_osrelease);
	m->registerFunction("osmachine",system_module_fnc_osmachine);
	m->registerFunction("osnodename",system_module_fnc_osnodename);
	m->registerFunction("getenv",system_module_fnc_getenv);
	m->registerFunction("hostname",system_module_fnc_hostname);
	m->registerFunction("dcop",system_module_fnc_dcop);

	m->registerCommand("setenv",system_module_cmd_setenv);

	return true;
}

static bool system_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"System",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"System informations module",
	system_module_init,
	0,
	0,
	system_module_cleanup
)
