//vim: ts=8
//   File : class_lineedit.cpp
//   Creation date : Wed 20 Sep 2000 04:34:40 CEST by Krzysztof Godlewski
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_debug.h"
#include "kvi_scriptobject.h"
#include "kvi_locale.h"

#include "class_lineedit.h"

static const char * mode_tbl[] = {
			"Normal",
			"NoEcho",
			"Password"
			  };

static const int mode_cod[] =  {
		QLineEdit::Normal,
		QLineEdit::NoEcho,
		QLineEdit::Password
			};

#define mode_num	(sizeof(mode_tbl) / sizeof(mode_tbl[0]))

static KviScriptObjectClass * g_pLineEditClass = 0;

static KviScriptObject * lineEditClassCreateInstance(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n)
{
	return new KviScriptLineEditObject(c, p, n);
}

KviScriptLineEditObject::KviScriptLineEditObject(KviScriptObjectClass * c,
	KviScriptObject * p, const char * n) : KviScriptWidgetObject(c,p,n)
{
}

KviScriptLineEditObject::~KviScriptLineEditObject()
{
}

/*
	@doc:	lineedit
	@keyterms:
		lineedit object class, line editor, input
	@title:
		lineedit class
	@type:
		class
	@short:
		Simple line editor for inputting short text
	@inherits:
		[class]object[/class]
		[class]widget[/class]
	@description:
		This is a simple, one-line editor usually used for inputting
		short text like names, phone numbers etc. It can display text 
		in three echo modes (see [classfnc]$setEchoMode[/classfnc]() 
		for details).
	@functions:
		!fn: $text()
		Returns text contained by the widget.[br]
		See also [classfnc]$setText[/classfnc]().
		!fn: $setText(<text>)
		Sets editor text to <text>.[br]
		See also [classfnc]$text[/classfnc]().
		!fn: $maxLength()
		Returns the current maximum length of the text that can be typed
		 in the editor.[br]
		See also [classfnc]$setMaxLength[/classfnc]().
		!fn: $setMaxLength(<value>)
		Sets maximum length of the text that can be typed in the 
		editor.[br]
		See also [classfnc]$maxLength[/classfnc]().
		!fn: $frame()
		Returns '1' if the editor is drawn inside a frame, which is 2 
		pixels wide, '0' otherwise.[br]
		See also [classfnc]$setFrame[/classfnc]().
		!fn: $setFrame(<bool>)
		Sets editor frame drawing to <bool>.[br]
		See also [classfnc]$frame[/classfnc]().
		!fn: $cursorPosition()
		Returns current cursor position in the editor.[br]
		See also [classfnc]$setCursorPosition[/classfnc]().
		!fn: $setCursorPosition(<index>)
		Moves the cursor to place pointed by <index>.[br]
		See also [classfnc]$cursorPosition[/classfnc]().
		!fn: $selectAll()
		Selects whole text inside the editor.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$copy[/classfnc](), 
		[classfnc]$paste[/classfnc]().
		!fn: $setSelection(<start>, <length>)
		Selects <length> characters in the editor starting at position 
		<start>.[br]
		See also [classfnc]$selectAll[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$copy[/classfnc](), 
		[classfnc]$paste[/classfnc]().
		!fn: $copy()
		Copies selected text in the editor to the clipboard.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$cut[/classfnc](), [classfnc]$paste[/classfnc]().
		!fn: $cut()
		Cuts selected text from the editor to the clipboard.[br]   
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$copy[/classfnc](), [classfnc]$paste[/classfnc]().
		!fn: $paste()
		Pastes text from the clipboard into the editor. The pasted text 
		will start at cursor position.[br]
		See also [classfnc]$setSelection[/classfnc](), 
		[classfnc]$copy[/classfnc](), [classfnc]$cut[/classfnc]().
		!fn: $clear()
		Deletes all text from the editor. Equal to calling 
		[classfnc]$setText[/classfnc]("").
		!fn $setDragEnabled(<bool>)
		With this property user can drag text in the lineedit.[br]
		!fn $setInputMask();
		Sets the validation input mask to inputMask. [br]
		Es:[br]
		%esempio->$setInputMask( "+99 99 99 99 99;_" );[br]
		%esempio->$setInputMask( "000.000.000.000;_" );[br]
		%esempio->Ip Number Mask.[br]
		%esempio->setInputMask( ">AAAAA-AAAAA-AAAAA-AAAAA-AAAAA;#" );[br]
		 The mask format understands these mask characters:[br]
						Character 	Meaning[br]
		A 	ASCII alphabetic character required. A-Z, a-z.[br]
		a 	ASCII alphabetic character permitted but not required.[br]
		N 	ASCII alphanumeric character required. A-Z, a-z, 0-9.[br]
		n 	ASCII alphanumeric character permitted but not required.[br]
		X 	Any character required.[br]
		x 	Any character permitted but not required.[br]
		9 	ASCII digit required. 0-9.[br]
		0 	ASCII digit permitted but not required.[br]
		D 	ASCII digit required. 1-9.[br]
		d 	ASCII digit permitted but not required.[br]
		# 	ASCII digit or plus/minus sign permitted but not required.[br]
		> 	All following alphabetic characters are uppercased.[br]
		< 	All following alphabetic characters are lowercased.[br]
		! 	Switch off case conversion.[br]
		\ 	Use \ to escape the special characters listed above to use them as separators.[br]
		[br]
		The mask consists of a string of mask characters and separators, optionally[br]
		followed by a semi-colon and the character used for blanks: the blank characters [br]
		are always removed from the text after editing. The default blank character is space. [br]

*/

bool KviScriptLineEditObject::init(KviCommand *)
{
	setObject(new QLineEdit(parentScriptWidget(), name()), true);
	connect(widget(),SIGNAL(returnPressed()),this,SLOT(slotreturnPressed()));
	connect(widget(),SIGNAL(lostFocus()),this,SLOT(slotlostFocus()));

	return true;
}

#define lineeditFuncReg(__nam, __func) \
        g_pLineEditClass->registerFunctionHandler(__nam, \
        (KviScriptObjectFunctionHandlerProc)( \
	KVI_PTR2MEMBER(KviScriptLineEditObject::__func)), 0, true);

void KviScriptLineEditObject::registerSelf()
{
	KviScriptObjectClass * base = g_pScriptObjectController-> \
		lookupClass("widget");
	__range_valid(base);

	g_pLineEditClass = new KviScriptObjectClass(base, "lineedit",
		lineEditClassCreateInstance, true);

	lineeditFuncReg("text", functionText);
	lineeditFuncReg("setText", functionSetText);
	lineeditFuncReg("maxLength", functionMaxLength);
	lineeditFuncReg("setMaxLength", functionSetMaxLength);
	lineeditFuncReg("frame", functionFrame);
	lineeditFuncReg("setFrame", functionSetFrame);
	lineeditFuncReg("cursorPosition", functionCursorPosition);
	lineeditFuncReg("setCursorPosition", functionSetCursorPosition);
	lineeditFuncReg("selectAll", functionSelectAll);
	lineeditFuncReg("setSelection", functionSetSelection);
	lineeditFuncReg("copy", functionCopy);
	lineeditFuncReg("cut", functionCut);
	lineeditFuncReg("paste", functionPaste);
	lineeditFuncReg("echoMode", functionEchoMode);
	lineeditFuncReg("setEchoMode", functionSetEchoMode);
	lineeditFuncReg("clear", functionClear);
	lineeditFuncReg("setDragEnabled", functionDragAndDrop);//G&N
	lineeditFuncReg("setInputMask", functionSetInputMask);//G&N
	lineeditFuncReg("returnPressedEvent", functionreturnPressedEvent);
	lineeditFuncReg("lostFocusEvent", functionlostFocusEvent);
}

void KviScriptLineEditObject::unregisterSelf()
{
	delete g_pLineEditClass;
    g_pLineEditClass = 0;
}

bool KviScriptLineEditObject::functionText(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(((QLineEdit *)widget())->text().latin1());
	return true;
}

bool KviScriptLineEditObject::functionSetText(KviCommand *, KviParameterList * p,
	KviStr &)
{
	if(widget())
	{
		if(p->count())
			((QLineEdit *)widget())->setText(p->safeFirst()->ptr());
		else
			((QLineEdit *)widget())->setText("");
	}
	return true;
}

bool KviScriptLineEditObject::functionMaxLength(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QLineEdit *)widget())-> \
			maxLength());
	else
		b.append("-1");
	return true;
}

bool KviScriptLineEditObject::functionSetMaxLength(KviCommand * c, \
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "lineedit::setMaxLength");
	if(widget())
	{
		bool bOk;
		int val = p->getInt(&bOk);
		if(bOk)
		{
			((QLineEdit *)widget())->setMaxLength(val);
			return c->leaveStackFrame();
		}
		else
			return c->error(KviError_integerParameterExpected);
	}
	return c->leaveStackFrame();
}

bool KviScriptLineEditObject::functionFrame(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(((QLineEdit *)widget())->frame() ? '1' : '0');
	return true;
}

bool KviScriptLineEditObject::functionSetFrame(KviCommand *, KviParameterList * p,
	KviStr & b)
{
	if(widget())
		((QLineEdit *)widget())->setFrame(p->getBool());
	return true;
}

bool KviScriptLineEditObject::functionCursorPosition(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(widget())
		b.append(KviStr::Format, "%d", ((QLineEdit *)widget())-> \
			cursorPosition());
	return true;
}

bool KviScriptLineEditObject::functionSetCursorPosition(KviCommand * c,
	KviParameterList * p, KviStr & b)
{
	ENTER_STACK_FRAME(c, "lineedit::setCursorPosition");
	if(widget())
	{
		bool bOk;
		int val = p->getInt(&bOk);
		if(bOk)
		{
			((QLineEdit *)widget())->setCursorPosition(val);
			return c->leaveStackFrame();
		}
		else
			return c->error(KviError_integerParameterExpected);
	}
	return c->leaveStackFrame();
}

bool KviScriptLineEditObject::functionSelectAll(KviCommand *, KviParameterList *,
	KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->selectAll();
	return true;
}

bool KviScriptLineEditObject::functionSetSelection(KviCommand * c, \
	KviParameterList * p, KviStr &)
{
	ENTER_STACK_FRAME(c, "lineedit::setSelection");
	if(!widget())
		return true;
	
	bool bOk1, bOk2;
	int start = p->getInt(&bOk1);
	int len = p->getInt(&bOk2);
	if(!bOk1 || !bOk2)
		return c->error(KviError_integerParameterExpected);

	((QLineEdit *)widget())->setSelection(start, len);
	
	return c->leaveStackFrame();
}

bool KviScriptLineEditObject::functionCopy(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->copy();
	return true;
}

bool KviScriptLineEditObject::functionCut(KviCommand *, KviParameterList *, KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->cut();
	return true;
}

bool KviScriptLineEditObject::functionPaste(KviCommand *, KviParameterList *,KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->paste();
	return true;
}

bool KviScriptLineEditObject::functionEchoMode(KviCommand *, KviParameterList *,
	KviStr & b)
{
	if(!widget())
	{
		b.append('0');
		return true;
	}

	int mode = ((QLineEdit *)widget())->echoMode();
	for(unsigned int i = 0; i < mode_num; i++)
	{
		if(mode == mode_cod[i])
		{
			b.append(mode_tbl[i]);
			return true;
		}
	}

	b.append(__tr("Hey ! You wanna know my echo mode or what ?"));
	return true;
}

bool KviScriptLineEditObject::functionSetEchoMode(KviCommand * c,KviParameterList * p,
	KviStr &)
{
	ENTER_STACK_FRAME(c, "lineedit::setEchoMode");
	if(!widget())
		return c->leaveStackFrame();

	for(unsigned int i = 0; i < mode_num; i++)
	{
		if(kvi_strEqualCI(p->safeFirst()->ptr(), mode_tbl[i]))
		{
			((QLineEdit *)widget())->setEchoMode( \
				((QLineEdit::EchoMode)mode_cod[i]));
			return c->leaveStackFrame();
		}
	}

	c->warning(__tr("Unknown echo mode"));
	return c->leaveStackFrame();
}

bool KviScriptLineEditObject::functionClear(KviCommand *, KviParameterList *,KviStr &)
{
	if(widget())
		((QLineEdit *)widget())->clear();
	return true;
}

//-| Grifisx & Noldor |-Start:
bool KviScriptLineEditObject::functionDragAndDrop(KviCommand *,KviParameterList * p,
	KviStr &)
{
	if(widget())
	{
		((QLineEdit *)widget())->setDragEnabled(p->getBool());
	}
	return true;
}
bool KviScriptLineEditObject::functionSetInputMask(KviCommand *, KviParameterList * p,
	KviStr &)
{
	if(widget())
	{
		if(p->count())
			((QLineEdit *)widget())->setInputMask(p->safeFirst()->ptr());
		else
			((QLineEdit *)widget())->setText(""); // ....this is a precaution.
	}
	return true;
}
bool KviScriptLineEditObject::functionreturnPressedEvent(KviCommand *c, KviParameterList *,	
	KviStr & b)
{

	ENTER_STACK_FRAME(c,"lineedit::returnPressedEvent");
	emitSignal("returnPressed",0,0,c);
	return c->leaveStackFrame();

}

void KviScriptLineEditObject::slotreturnPressed()
{
	callEventFunction("returnPressedEvent",0,0);

}

bool KviScriptLineEditObject::functionlostFocusEvent(KviCommand *c, KviParameterList *,	
	KviStr & b)
{

	ENTER_STACK_FRAME(c,"lineedit::lostFocusEvent");
	emitSignal("lostFocus",0,0,c);
	return c->leaveStackFrame();

}

void KviScriptLineEditObject::slotlostFocus()
{
	callEventFunction("lostFocusEvent",0,0);
}
//-G&N End.

#include "m_class_lineedit.moc"
