//=============================================================================
//
//   File : kvinotifierwindowtabs.cpp
//   Created on dom 02 gen 2005 15:30:50 by Iacopo Palazzi
//
//   This file is part of the KVIrc irc client distribution
//   Copyright (C) 2005 Iacopo Palazzi < iakko(at)siena(dot)linux(dot)it >
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "kvi_app.h"
#include "kvi_config.h"
#include "kvi_qstring.h"
#include "kvi_iconmanager.h"
#include "kvi_locale.h"
#include "kvi_options.h"
#include "kvi_window.h"

#include <qevent.h>
#include <qpainter.h>
#include <qbrush.h>
#include <qcolor.h>
#include <qfontmetrics.h>

#include "notifiermessage.h"
#include "notifierwindowtabs.h"

extern KviNotifierWindow * g_pNotifierWindow; // global declaration at libnotifier.cpp
KviNotifierWindowTabs * g_pTabs;


// --- tab object definition class

KviNotifierWindowTab::KviNotifierWindowTab(KviWindow * pWnd, QString label)
{
	m_pWnd = pWnd;
	m_label = label;
	m_pMessageList = new KviPtrList<KviNotifierMessage>;
	m_pMessageList->setAutoDelete(true);
	m_bFocused = false;
	m_pCurrentMessage = 0;
	
	KviStr buffer;
	g_pApp->getReadOnlyConfigPath(buffer,"libkvinotifier.kvc",KviApp::ConfigPlugins,true);
	KviConfig cfg(buffer.ptr(),KviConfig::Read);
	cfg.setGroup("NotifierSkin");
		m_clrHighlightedLabel = cfg.readColorEntry("HighlightedTabLablerColor",QColor(200,0,0));
		m_clrNormalLabel = cfg.readColorEntry("NormalTabLablerColor",QColor(0,0,0));
		m_clrChangedLabel = cfg.readColorEntry("ChangedTabLablerColor",QColor(0,0,100));
	
	if(pWnd)
	{
		connect(pWnd,SIGNAL(windowNameChanged()),this,SLOT(labelChanged()));
		connect(pWnd,SIGNAL(destroyed()),this,SLOT(closeMe()));
	}
}

KviNotifierWindowTab::~KviNotifierWindowTab()
{
	delete m_pMessageList;
}

void KviNotifierWindowTab::setNextMessageAsCurrent()
{
	if(!m_pCurrentMessage)return;
	m_pMessageList->findRef(m_pCurrentMessage);
	m_pCurrentMessage = m_pMessageList->next();
	if(!m_pCurrentMessage)m_pCurrentMessage = m_pMessageList->last();
}

void KviNotifierWindowTab::setPrevMessageAsCurrent()
{
	if(!m_pCurrentMessage)return;
	m_pMessageList->findRef(m_pCurrentMessage);
	m_pCurrentMessage = m_pMessageList->prev();
	if(!m_pCurrentMessage)m_pCurrentMessage = m_pMessageList->first();
}

void KviNotifierWindowTab::setLastMessageAsCurrent()
{
	m_pCurrentMessage = m_pMessageList->last();
}

void KviNotifierWindowTab::appendMessage(KviNotifierMessage * m)
{
	if(m_pCurrentMessage == m_pMessageList->last())m_pCurrentMessage = m;
	m_pMessageList->append(m);
	// avoid having too much messages floating around
	while(m_pMessageList->count() > MAX_MESSAGES_IN_WINDOW)
	{
		m = m_pMessageList->first();
		m_pMessageList->removeFirst();
		if(m == m_pCurrentMessage)
			m_pCurrentMessage = m_pMessageList->first();
	}
	if(focused())setState(Normal);
	else setState(Highlighted);
}

void KviNotifierWindowTab::labelChanged()
{
	if(!m_pWnd)return;
	m_label = m_pWnd->windowName();
	setState(Changed);
	g_pTabs->needToRedraw();
	g_pNotifierWindow->update();
}

void KviNotifierWindowTab::closeMe()
{
	g_pTabs->closeTab(this);
}

void KviNotifierWindowTab::setFocused(bool b)
{
	m_bFocused = b;
	if(m_bFocused)setState(Normal);
}

// --- end of tab object definition class


KviNotifierWindowTabs::KviNotifierWindowTabs(QRect r)
{
	g_pTabs = this;
	m_pTabFocused = 0;
	m_pWndTabFocused = 0;

	loadImages();
	initConfig();
	resize(r);
}

KviNotifierWindowTabs::~KviNotifierWindowTabs()
{
	QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
	for(tab = m_tabMap.begin(); tab != m_tabMap.end(); tab++ )
		delete tab.data();
	m_tabMap.clear();
	delete m_pFocusedFont;
	delete m_pUnfocusedFont;
}

void KviNotifierWindowTabs::initConfig()
{
	KviStr buffer;
	g_pApp->getReadOnlyConfigPath(buffer,"libkvinotifier.kvc",KviApp::ConfigPlugins,true);

	KviConfig cfg(buffer.ptr(),KviConfig::Read);

	cfg.setGroup("NotifierSkin");
	
	QString szFamily = cfg.readEntry("TextFontFocusedTab","Arial");
	m_pFocusedFont = new QFont(szFamily,cfg.readIntEntry("TextFocusedFontSize",10));
	m_pFocusedFont->setBold(true);
	szFamily = cfg.readEntry("TextFontUnfocusedTab","Arial");
	m_pUnfocusedFont = new QFont(szFamily,cfg.readIntEntry("TextUnfocusedFontSize",9));
}

void KviNotifierWindowTabs::loadImages()
{
	QPixmap * p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_dx.png")))
		m_pixDX = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_sx.png")))
		m_pixSX = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_bkg.png")))
		m_pixBKG = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_focused_sx.png")))
		m_pixSXFocused = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_focused_dx.png")))
		m_pixDXFocused = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_focused_bkg.png")))
		m_pixBKGFocused = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_unfocused_sx.png")))
		m_pixSXUnfocused = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_unfocused_dx.png")))
		m_pixDXUnfocused = *p;
	if((p = g_pIconManager->getPixmap("notifier_pix_tab_unfocused_bkg.png")))
		m_pixBKGUnfocused = *p;

	if((p = g_pIconManager->getPixmap("notifier_icon_tab_next_out.png")))
		m_pixIconTabNext_out = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_next_over.png")))
		m_pixIconTabNext_over = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_next_clicked.png")))
		m_pixIconTabNext_clicked = *p;

	m_pixIconTabNext = m_pixIconTabNext_out;
	
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_prev_out.png")))
		m_pixIconTabPrev_out = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_prev_over.png")))
		m_pixIconTabPrev_over = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_prev_clicked.png")))
		m_pixIconTabPrev_clicked = *p;

	m_pixIconTabPrev = m_pixIconTabPrev_out;

	if((p = g_pIconManager->getPixmap("notifier_icon_tab_close_off.png")))
		m_pixIconCloseTab_off = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_close_on.png")))
		m_pixIconCloseTab_on = *p;
	if((p = g_pIconManager->getPixmap("notifier_icon_tab_close_clicked.png")))
		m_pixIconCloseTab_clicked = *p;

	m_pixIconCloseTab = m_pixIconCloseTab_off;

	m_closeTabIconState = WDG_ICON_OFF;

}



void KviNotifierWindowTabs::addMessage(KviWindow * pWnd, KviNotifierMessage * message)
{
	QString sender = pWnd ? pWnd->windowName() : "----";

	// Qui servirebbe anche una lista... perche' quando si inserisce
	// un tab lui lo piazza dove gli pare nell'ordine
	// e quindi iterando i tab nuovi possono trovarsi PRIMA dei tab vecchi
	// il che' confonde un po
	KviNotifierWindowTab * tab;
	if (!m_tabMap.contains(pWnd)) {
		m_tabMap.insert(pWnd, tab = new KviNotifierWindowTab(pWnd, sender));
	} else {
		tab = m_tabMap[pWnd];
	}

	tab->appendMessage(message);
	
	if((g_pNotifierWindow->state()==Hidden) || (!m_pTabFocused))
	{
		setFocusOn(tab);
	} else {
		needToRedraw();
	}
}

void KviNotifierWindowTabs::resize(QRect r)
{
	m_rct.setX(r.x());
	m_rct.setY(r.y());
	m_rct.setHeight(r.height());
	setWidth(r.width());
	
	recalculatePositions();
}

void KviNotifierWindowTabs::setWidth(int w) {
	m_rct.setWidth(w);
	recalculatePositions();
}

void KviNotifierWindowTabs::recalculatePositions()
{
	m_rctCloseTabIcon.setX(m_rct.x()+m_rct.width()-m_pixIconCloseTab.width());
	m_rctCloseTabIcon.setY(m_rct.y());
	m_rctCloseTabIcon.setWidth(m_pixIconCloseTab.width());
	m_rctCloseTabIcon.setHeight(m_pixIconCloseTab.height());
	// The sensible area for the Close Icon
	m_rctCloseTabIconHotArea.setX(m_rctCloseTabIcon.x()+6);
	m_rctCloseTabIconHotArea.setY(m_rctCloseTabIcon.y()+3);
	m_rctCloseTabIconHotArea.setWidth(16);
	m_rctCloseTabIconHotArea.setHeight(16);
	
	m_rctTabs.setX(m_rct.x());
	m_rctTabs.setY(m_rct.y());
	m_rctTabs.setWidth(m_rctCloseTabIcon.x());
	m_rctTabs.setHeight(m_rct.height());
	
	needToRedraw();
}

void KviNotifierWindowTabs::prev()
{
	if(!m_pTabFocused)return;
	QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
	tab  = m_tabMap.find(m_pTabFocused->wnd());
	if (tab!=m_tabMap.begin()) {
		tab--;
		setFocusOn(tab.data());
	}
}

void KviNotifierWindowTabs::next()
{
	if(!m_pTabFocused)return;
	QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
	tab  = m_tabMap.find(m_pTabFocused->wnd());
	if (tab!=m_tabMap.end()) {
		tab++;
		if (tab!=m_tabMap.end()) setFocusOn(tab.data());
	}
}

void KviNotifierWindowTabs::mousePressEvent(QMouseEvent * e) {

	if(m_rctTabs.contains(e->pos()))
	{
		QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
		for (tab = m_tabMap.begin(); tab != m_tabMap.end(); tab++ ) {
			if (tab.data()->rect().contains(e->pos())) {
				setFocusOn(tab.data());
				return;
			}
		}
	}

}

void KviNotifierWindowTabs::mouseMoveEvent(QMouseEvent * e)
{
	if (m_rctCloseTabIconHotArea.contains(e->pos()))
	{
		setCloseTabIcon(WDG_ICON_ON);
	} else {
		if (closeTabIconState()!=WDG_ICON_OFF)
			setCloseTabIcon(WDG_ICON_OFF);
	}
}

void KviNotifierWindowTabs::mouseReleaseEvent(QMouseEvent * e)
{
}

void KviNotifierWindowTabs::setCloseTabIcon(int state)	{

	//if (m_writeIconState==WDG_ICON_OFF && state!=WDG_ICON_OFF)
	if (m_closeTabIconState!=state) {
		switch (state) {
			case WDG_ICON_ON: m_pixIconCloseTab = m_pixIconCloseTab_on; break;
			case WDG_ICON_OFF: m_pixIconCloseTab = m_pixIconCloseTab_off; break;
			case WDG_ICON_CLICKED: m_pixIconCloseTab = m_pixIconCloseTab_clicked; break;
		}
		needToRedraw();
		m_closeTabIconState=state;
	}
}

void KviNotifierWindowTabs::resetIcons()
{
	setCloseTabIcon(WDG_ICON_OFF);
}

void KviNotifierWindowTabs::setFocusOn(KviNotifierWindowTab * tab)
{	
	if(m_pTabFocused)m_pTabFocused->setFocused(false);
	m_pTabFocused = tab;
	if(m_pTabFocused)m_pTabFocused->setFocused();

	needToRedraw();

	g_pNotifierWindow->update();
}

void KviNotifierWindowTabs::draw(QPainter * p)
{
	if(!m_bNeedToRedraw)return;

	QFont tmpFont;
	tmpFont = p->font();

	QString str;
	
	//p->fillRect(m_rct,m_mac_bkgColor); //m_mac_bkgColor

	int w, h;
	int x = m_rct.x();
	int y = m_rct.y();
	int offset = 0;

	p->drawPixmap(x,y,m_pixSX);
	p->drawPixmap(x+m_rct.width()-m_pixDX.width(),y,m_pixDX);
	p->drawTiledPixmap(x+m_pixSX.width(),y,m_rct.width()-m_pixSX.width()-m_pixDX.width(),m_rct.height(),m_pixBKG);

	QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
	for (tab = m_tabMap.begin(); tab != m_tabMap.end(); tab++ )
	{
		//debug("drawing for '%s'", tab.data()->label().data());

		if (tab.data()->focused()) {

			p->setFont(*m_pFocusedFont);		
			str = tab.data()->label();
			QFontMetrics fm(p->font());
			w = fm.width(str)+2;
			h = fm.height();

			tab.data()->setRect(x+offset, m_rctTabs.y(), m_pixSXFocused.width() + m_pixDXFocused.width() + w, m_rctTabs.height());
			
			p->drawPixmap(x+offset,m_rctTabs.y(),m_pixSXFocused);
			p->drawTiledPixmap(x+offset+m_pixSXFocused.width(),m_rctTabs.y(),w,m_rctTabs.height(),m_pixBKGFocused);
			p->drawPixmap(x+offset+m_pixSXFocused.width()+w,m_rctTabs.y(),m_pixDXFocused);
			QPen tmpP = p->pen(); p->setPen(tab.data()->labelColor());
			p->drawText(x+offset+m_pixSXFocused.width()+1,m_rctTabs.y()+m_rctTabs.height()-NTF_TABS_FONT_BASELINE,str);
			p->setPen(tmpP);

			offset += m_pixSXFocused.width() + m_pixDXFocused.width() + w;

		} else {

			p->setFont(*m_pUnfocusedFont);		
			str = tab.data()->label();
			QFontMetrics fm(p->font());
			w = fm.width(str)+2;
			h = fm.height();

			tab.data()->setRect(x+offset, m_rctTabs.y(), m_pixSXUnfocused.width() + m_pixDXUnfocused.width() + w, m_rctTabs.height());

			p->drawPixmap(x+offset,m_rctTabs.y(),m_pixSXUnfocused);
			p->drawTiledPixmap(x+offset+m_pixSXUnfocused.width(),m_rctTabs.y(),w,m_rctTabs.height(),m_pixBKGUnfocused);
			p->drawPixmap(x+offset+m_pixSXUnfocused.width()+w,m_rctTabs.y(),m_pixDXUnfocused);
			QPen tmpP = p->pen(); p->setPen(tab.data()->labelColor());
			p->drawText(x+offset+m_pixSXUnfocused.width()+1,m_rctTabs.y()+m_rctTabs.height()-NTF_TABS_FONT_BASELINE+1,str);
			p->setPen(tmpP);
			offset += m_pixSXUnfocused.width() + m_pixDXUnfocused.width() + w;
			
		}
	}

	p->drawPixmap(m_rctCloseTabIcon.x(), m_rctCloseTabIcon.y(), m_pixIconCloseTab);

	m_bNeedToRedraw = false;
	
	p->setFont(tmpFont);
}

void KviNotifierWindowTabs::markAllMessagesAsHistoric()
{
	QMap<KviWindow *, KviNotifierWindowTab *>::Iterator tab;
	// Iterating every tab
	for (tab = m_tabMap.begin(); tab != m_tabMap.end(); tab++ )
	{
		// Setting all messages of the current tab as historic
		// tab.data()->expireMessages();
		KviPtrList<KviNotifierMessage> * tmpMessageList = tab.data()->messageList();
		for(KviNotifierMessage * m = tmpMessageList->last();m;m = tmpMessageList->prev()) {
			if(m->historic())break;
			m->setHistoric();
		}
	}
}

void KviNotifierWindowTabs::closeTab(KviWindow * pWnd)
{
	KviNotifierWindowTab * pTab = m_tabMap.find(pWnd).data();
	closeTab(pWnd, pTab);
}

void KviNotifierWindowTabs::closeTab(KviNotifierWindowTab * pTab)
{
	KviWindow * pWnd = pTab->wnd();
	closeTab(pWnd, pTab);
}

void KviNotifierWindowTabs::closeCurrentTab()
{
	// Paranoic checks...
	if (!m_pTabFocused) return; // The current tab is null? Please, kill me...
	KviNotifierWindowTab * pTab = m_pTabFocused;
	if (m_tabMap.isEmpty()) return; // Empty??? The World is ending...
	KviWindow * pWnd = pTab->wnd();
	if (!m_tabMap.contains(pWnd)) return; // The current tab is not on the map??? Call 911!!
	// End paranoic stuff, go back to sanity...
	
	closeTab(pWnd, pTab);
}

void KviNotifierWindowTabs::closeTab(KviWindow * pWnd, KviNotifierWindowTab * pTab)
{
	// Ok, we should be sure to delete the right tab...
	m_tabMap.remove(pWnd);
	delete pTab;
	
	if(m_tabMap.empty())
	{
		m_pTabFocused = 0;
		g_pNotifierWindow->showLineEdit(false);
		g_pNotifierWindow->doHide(false);
	} else {
		m_pTabFocused = m_tabMap.begin().data();
		m_pTabFocused->setFocused(true);
	}
	setFocusOn(m_pTabFocused);
}

#include "m_notifierwindowtabs.moc"
