//=============================================================================
//
//   File : listwindow.cpp
//   Creation date : Thu Oct 7 2001 13:27:55 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the lists of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#include "listwindow.h"

#include "kvi_debug.h"
#include "kvi_iconmanager.h"
#include "kvi_ircview.h"
#include "kvi_out.h"
#include "kvi_options.h"
#include "kvi_locale.h"
#include "kvi_out.h"
#include "kvi_mirccntrl.h"
#include "kvi_themedlabel.h"
#include "kvi_options.h"
#include "kvi_mirccntrl.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnection.h"
#include "kvi_qstring.h"

#include <qtimer.h>
#include <qpainter.h>
#include <qfontmetrics.h>
#include <qsplitter.h>
#include <qtooltip.h>
#include <qhbox.h>

extern KviPtrList<KviListWindow> * g_pListWindowList;

// kvi_ircview.cpp
//extern KVIRC_API const char * getColorBytes(const char *data_ptr,unsigned char *byte_1,unsigned char *byte_2);


KviChannelListViewItemData::KviChannelListViewItemData(const QString &szChan,const QString &szUsers,const QString &szTopic)
{
	m_szChan = szChan;
	m_szUsers = szUsers;
	m_szTopic = szTopic;
	m_szUsersKey = szUsers;
	//setText(0,szChan.upper());
	while(m_szUsersKey.length() < 6)m_szUsersKey.prepend("0");
}

KviChannelListViewItemData::~KviChannelListViewItemData()
{
}



KviChannelListViewItem::KviChannelListViewItem(QListView * v,KviChannelListViewItemData * pData)
: QListViewItem(v)
{
	m_pData = pData;
}


KviChannelListViewItem::~KviChannelListViewItem()
{
	delete m_pData;
}


#define KVI_LABEL_DEF_BACK 100
#define KVI_LABEL_DEF_FORE 101

void KviChannelListViewItem::paintCell(QPainter * p,const QColorGroup &cg,int col,int wdth,int align)
{
	QFontMetrics fm(p->font());

	p->fillRect(0, 0, wdth, height(), cg.brush( QColorGroup::Base ));
	const QChar * t;

	switch(col)
	{
		case 0:  t = KviQString::nullTerminatedArray(m_pData->m_szChan);   break;
		case 1:  t = KviQString::nullTerminatedArray(m_pData->m_szUsers);  break;
		default: t = KviQString::nullTerminatedArray(m_pData->m_szTopic);  break;
	}

	if(!t)return;


	bool curBold      = false;
	bool curUnderline = false;
	unsigned char curFore      = KVI_LABEL_DEF_FORE; //default fore
	unsigned char curBack      = KVI_LABEL_DEF_BACK; //default back

	int baseline = ((height() + fm.ascent() - fm.descent() + 1) >> 1);
	int curX = 2; //2 is the margin


	while(t->unicode() && (curX < wdth))
	{
		const QChar * aux = t;
		int len = 0;
		while(aux->unicode() && (aux->unicode() != KVI_TEXT_COLOR) && (aux->unicode() != KVI_TEXT_BOLD) &&
					(aux->unicode() != KVI_TEXT_UNDERLINE) && (aux->unicode() != KVI_TEXT_REVERSE) &&
					(aux->unicode() != KVI_TEXT_RESET) && len < 150) // need to put this length limit: see the BUG below
		{
			++len;
			++aux;
		}
		int www;

		if(len > 0)
		{
			QString txt(t,len);
			www = 10;
			www = fm.width(txt);
			if(curFore == KVI_LABEL_DEF_FORE)
			{
				p->setPen(cg.text());
			} else {
				if(curFore > 15)p->setPen(cg.base());
				else p->setPen(KVI_OPTION_MIRCCOLOR(curFore));
			}
			if(curBack != KVI_LABEL_DEF_BACK)
			{
				if(curBack > 15)
				{
					p->fillRect(curX,2,wdth,height() - 4,cg.text());
				} else {
					p->fillRect(curX,2,wdth,height() - 4,KVI_OPTION_MIRCCOLOR(curBack));
				}
			}
			
			// 13/12/2003: I REALLY CAN'T UNDERSTAND WHY
			// QT/XLib crashes when painting THIS string (253 chars long)
			// "#IRC6 - IPv6 version of QPlug for Netscape/Mozilla now released; http://q6stat.actiongal.com/ - Minor bugfixes in Q6Stat, possible pending release of the currently closed source Q6Master (with qw/q3a support), and coming soon, x0rfb server/viewer + ipv6"
			// probably a buffer problem.
			// Sometimes it gets just locked and other times spits out this
			// before killing the app...
			
			// X Error: BadLength (poly request too large or internal Xlib length error) 16
			//  Major opcode:  33
			//  Minor opcode:  0
			//  Resource id:  0x28007f7
			// X Error: BadRequest (invalid request code or no such operation) 1
			//  Major opcode:  179
			//  Minor opcode:  0
			//  Resource id:  0x28007f7
			
			// The interesting thing is that it crashes ONLY in this listview widget
			// the KviIrcView does not crash on this, for example...
			// maybe because it gets wrapped on several lines anyway...
			// don't know.
			// 253 unicode chars would give 506 bytes. it MIGHT be that the max request size is 512 and
			// 504 is the usable request size.
			// the crash happens also for any size bigger than 253
			
			// 17.12.2003 : update : qt 3.2.3 solves this problem

			p->drawText(curX,baseline,txt);
			
			if(curBold)p->drawText(curX+1,baseline,txt);
			if(curUnderline)p->drawLine(curX,baseline + 1,curX+wdth,baseline + 1);
		} else {
			www = 0;
		}

		switch(aux->unicode())
		{
			case KVI_TEXT_BOLD: curBold = !curBold; ++aux; break;
			case KVI_TEXT_UNDERLINE: curUnderline = !curUnderline; ++aux; break;
			case KVI_TEXT_REVERSE:
				{
					char auxBack = curBack;
					curBack = curFore;
					curFore = auxBack;
				}
				++aux;
			break;
			case KVI_TEXT_RESET:
				curFore = KVI_LABEL_DEF_FORE;
				curBack = KVI_LABEL_DEF_BACK;
				curBold = false;
				curUnderline = false;
				++aux;
			break;
			case KVI_TEXT_COLOR:
			{
				++aux;
				unsigned char fore;
				unsigned char back;
				aux = getUnicodeColorBytes(aux,&fore,&back);
				if(fore != KVI_NOCHANGE)
				{
					curFore = fore;
					if(back != KVI_NOCHANGE)curBack = back;
				} else {
					// only a CTRL+K
					curFore = KVI_LABEL_DEF_FORE;
					curBack = KVI_LABEL_DEF_BACK;
				}
			}
			break;
		}

		t = aux;
		curX += www;
	}
	if(isSelected())
	{
		p->setRasterOp(Qt::NotROP);
		p->fillRect(0,0,wdth,height(),Qt::black);
		p->setRasterOp(Qt::CopyROP);
	}
}
	
QString KviChannelListViewItem::key(int col,bool) const
{
	switch(col)
	{
		case 0:
			return m_pData->m_szChan;
		break;
		case 1:
			return m_pData->m_szUsersKey;
		break;
		case 2:
			return m_pData->m_szTopic;
		break;
	}
	QString ret;
	return ret;
}



KviListWindow::KviListWindow(KviFrame * lpFrm,KviConsole * lpConsole)
: KviWindow(KVI_WINDOW_TYPE_LIST,lpFrm,"list",lpConsole) , KviExternalServerDataParser()
{
	g_pListWindowList->append(this);

	m_pFlushTimer = 0;

	m_pItemList = new KviPtrList<KviChannelListViewItemData>;
	m_pItemList->setAutoDelete(false);

	m_pSplitter = new QSplitter(QSplitter::Horizontal,this,"splitter");
	m_pTopSplitter = new QSplitter(QSplitter::Horizontal,this,"top_splitter");
	m_pVertSplitter = new QSplitter(QSplitter::Vertical,m_pSplitter,"vsplitter");

	QHBox * box = new QHBox(m_pTopSplitter);

	m_pRequestButton = new QToolButton(box,"request_button");
	m_pRequestButton->setUsesBigPixmap(false);
	m_pRequestButton->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_LIST)));
	connect(m_pRequestButton,SIGNAL(clicked()),this,SLOT(requestList()));
	QToolTip::add(m_pRequestButton,__tr2qs("Request List"));

	m_pParamsEdit = new QLineEdit(box);
	box->setStretchFactor(m_pParamsEdit,1);
	QToolTip::add(m_pParamsEdit,__tr2qs("<center><b>/LIST command parameters:</b><br>Many servers accept special parameters that " \
						"allow you to filter the returned entries.<br>" \
						"Commonly, masked channel names (*kvirc*) are accepted as parameters, as well as strings " \
						"like <b>c&lt;n</b> or <b>c&gt;n</b> where <b>n</b> is the minimum or maximum of users on the channel.</center>"));

	m_pInfoLabel = new KviThemedLabel(m_pTopSplitter,"info_label");

	m_pListView  = new QListView(m_pVertSplitter);
	m_pListView->addColumn(__tr2qs("Channel"),150);
	m_pListView->addColumn(__tr2qs("Users"),70);
	m_pListView->addColumn(__tr2qs("Topic"),400);
	
	m_pListView->setSorting(100);
	
	connect(m_pListView,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(itemDoubleClicked(QListViewItem *)));

	m_pIrcView = new KviIrcView(m_pVertSplitter,lpFrm,this);

	m_pConsole->ircContext()->setListWindowPointer(this);

	connect(m_pConsole->context(),SIGNAL(stateChanged()),
		this,SLOT(connectionStateChange()));

	connectionStateChange();
	
}

KviListWindow::~KviListWindow()
{
	g_pListWindowList->removeRef(this);
	m_pConsole->ircContext()->setListWindowPointer(0);
	if(m_pFlushTimer)delete m_pFlushTimer;
	m_pItemList->setAutoDelete(true);
	delete m_pItemList;
}

void KviListWindow::getBaseLogFileName(KviStr &buffer)
{
	buffer.sprintf("LIST_%d",console()->ircContextId());
}


void KviListWindow::requestList()
{
	if(m_pConsole->isConnected())
	{
		KviStr parms = m_pParamsEdit->text();
		if(parms.isEmpty())m_pConsole->connection()->sendFmtData("list");
		else m_pConsole->connection()->sendFmtData("list %s",parms.ptr());
		outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr2qs("Sent list request, waiting for reply..."));
		m_pRequestButton->setEnabled(false);
	} else {
		outputNoFmt(KVI_OUT_SYSTEMERROR,__tr2qs("Cannot request list: No active connection"));
	}
}

void KviListWindow::connectionStateChange()
{
	KviIrcContext::State st = m_pConsole->context()->state();
	m_pRequestButton->setEnabled(st == KviIrcContext::Connected);
	if(st == KviIrcContext::Connected)
	{
		QString tmp;
		KviQString::sprintf(tmp,
				__tr2qs("Connected to %s (%s)"),
				m_pConsole->connection()->currentServerName().latin1(),
				m_pConsole->currentNetworkName());
		m_pInfoLabel->setText(tmp);
	} else {
		m_pInfoLabel->setText(__tr2qs("List cannot be requested: Not connected to a server"));
	}
}

QPixmap * KviListWindow::myIconPtr()
{
	return g_pIconManager->getSmallIcon(KVI_SMALLICON_LIST);
}

void KviListWindow::resizeEvent(QResizeEvent *e)
{
	int hght = m_pTopSplitter->sizeHint().height();
	m_pTopSplitter->setGeometry(0,0,width(),hght);
	m_pSplitter->setGeometry(0,hght,width(),height() - hght);
}

QSize KviListWindow::sizeHint() const
{
	return m_pSplitter->sizeHint();
}

void KviListWindow::fillCaptionBuffers()
{
	KviQString::sprintf(m_szPlainTextCaption,__tr2qs("Channel List [IRC Context %u]"),m_pConsole->ircContextId());

	KviQString::sprintf(m_szHtmlActiveCaption,
		__tr2qs("<nobr><font color=\"%s\"><b>Channel List</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextActive2).name().ascii(),
		m_pConsole->ircContextId());

	KviQString::sprintf(m_szHtmlInactiveCaption,
		__tr2qs("<nobr><font color=\"%s\"><b>Channel list</b></font> " \
			"<font color=\"%s\">[IRC Context %u]</font></nobr>"),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive).name().ascii(),
		KVI_OPTION_COLOR(KviOption_colorCaptionTextInactive2).name().ascii(),
		m_pConsole->ircContextId());
}

void KviListWindow::die()
{
	close();
}

void KviListWindow::control(int message)
{
	switch(message)
	{
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_RESET:       reset();       break;
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_STARTOFDATA: startOfList(); break;
		case EXTERNAL_SERVER_DATA_PARSER_CONTROL_ENDOFDATA:   endOfList();   break;
	}
}

void KviListWindow::reset()
{
	outputNoFmt(KVI_OUT_SYSTEMMESSAGE,__tr2qs("Reset"));
}

void KviListWindow::endOfList()
{
	if(m_pFlushTimer)
	{
		delete m_pFlushTimer;
		m_pFlushTimer = 0;
	}
	m_pRequestButton->setEnabled(true);
	flush(); // give it the last kick
}

void KviListWindow::startOfList()
{
	m_pItemList->setAutoDelete(true);
	m_pItemList->clear();
	m_pItemList->setAutoDelete(false);

	m_pListView->clear();

	m_pRequestButton->setEnabled(false);
}

void KviListWindow::processData(KviIrcMessage *msg)
{
	if(!m_pFlushTimer)
	{
		m_pFlushTimer = new QTimer(this);
		connect(m_pFlushTimer,SIGNAL(timeout()),this,SLOT(flush()));
		m_pFlushTimer->start(1000);
		m_pRequestButton->setEnabled(false);
	}

	QString sz2 = msg->connection()->decodeText(msg->safeParam(2));

	m_pItemList->append(
		new KviChannelListViewItemData(
			msg->connection()->decodeText(msg->safeParam(1)),
			msg->connection()->decodeText(msg->safeParam(2)),
			msg->connection()->decodeText(msg->safeTrailing()))
	);
       
	if(_OUTPUT_VERBOSE)
	{
		QString zzz = msg->connection()->decodeText(msg->allParams());
		output(KVI_OUT_LIST,__tr2qs("Processing list: %Q"),&zzz);
	}
}

void KviListWindow::flush()
{
	m_pListView->setUpdatesEnabled(false);
	while(KviChannelListViewItemData * d = m_pItemList->first())
	{
		(void)new KviChannelListViewItem(m_pListView,d);
		m_pItemList->removeFirst();
	}
	m_pListView->setUpdatesEnabled(true);
	m_pListView->viewport()->update();
}

void KviListWindow::itemDoubleClicked(QListViewItem *it)
{
	QString sz = ((KviChannelListViewItem *)it)->channel();
	if(sz.isEmpty())return;
	if(!connection())return;
	QCString dat = connection()->encodeText(sz);
	if(!dat.data())return;
	m_pConsole->connection()->sendFmtData("join %s",dat.data());
}


//
//#warning "Load & save properties of this kind of window"

//void KviListWindow::saveProperties()
//{
//	KviWindowProperty p;
//	p.rect = externalGeometry();
//	p.isDocked = isAttacched();
//	QValueList<int> l(m_pSplitter->sizes());
//	if(l.count() >= 1)p.splitWidth1 = *(l.at(0));
//	if(l.count() >= 2)p.splitWidth2 = *(l.at(1));
//	p.timestamp = m_pView->timestamp();
//	p.imagesVisible = m_pView->imagesVisible();
//	p.isMaximized = isAttacched() && isMaximized();
//	p.topSplitWidth1 = 0;
//	p.topSplitWidth2 = 0;
//	p.topSplitWidth3 = 0;
//	g_pOptions->m_pWinPropertiesList->setProperty(caption(),&p);
//}
//
//void KviListWindow::setProperties(KviWindowProperty *p)
//{
//	QValueList<int> l;
//	l.append(p->splitWidth1);
//	l.append(p->splitWidth2);
//	m_pVertSplitter->setSizes(l);
//	m_pIrcView->setTimestamp(p->timestamp);
//	m_pIrcView->setShowImages(p->imagesVisible);
//}

void KviListWindow::applyOptions()
{
	m_pIrcView->applyOptions();
	KviWindow::applyOptions();
}

#include "listwindow.moc"
