//
//   File : libkvifile.cpp
//   Creation date : Fri Nov  9 03:27:59 2001 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//


#include "kvi_module.h"
#include "kvi_uparser.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_command.h"
#include "kvi_malloc.h"
#include "kvi_app.h"
#include "kvi_options.h"


#include <qfileinfo.h>
#include <qfile.h>
#include <qdir.h>

/*
	@doc: file.copy
	@type:
		command
	@title:
		file.copy
	@keyterms:
		copying files
	@short:
		Makes a copy of a file
	@syntax:
		file.copy [-o] <source> <destination>
	@description:
		Makes a copy of the <source> file as <destination>.[br]
		If the [-o] switch is used , the <destination> file is overwritten , if already exists.[br]
		With no [-o] switch , this command does not overwrite files.[br]
		The destination path must be already existing: if you want to ensure this, use [fnc]$file.mkdir[/fnc] first.[br]
		The paths (<source> and <destination>) are adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.rename[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_copy(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_copy");
	KviStr szSrc,szDst;
	if(!g_pUserParser->parseCmdSingleToken(c,szSrc))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szSrc);
	kvi_adjustFilePath(szDst);

	if(kvi_fileExists(szDst.ptr()) && !c->hasSwitch('o'))
	{
		c->warning(__tr("Destination file exists: no copy made"));
		return c->leaveStackFrame();
	}

	if(!kvi_copyFile(szSrc.ptr(),szDst.ptr()))
		c->warning(__tr("Failed to copy %s to %s"),szSrc.ptr(),szDst.ptr());

	return c->leaveStackFrame();
}


/*
	@doc: file.addimagepath
	@type:
		command
	@title:
		file.addimagepath
	@short:
		Adds an image search path
	@syntax:
		file.addimagepath <path>
	@description:
		Adds <path> to the image search path list.[br]
		KVIrc will look for images files (also) in that path (when a relative image file name is used).[br]
	@seealso:
		[cmd]file.delimagepath[/cmd]
*/

static bool file_module_cmd_addimagepath(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file::addimagepath");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	QString x(szDst.ptr());

	QStringList::Iterator it = KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).find(x);
	if(it == KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).end())
	{
		KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).append(QString(szDst.ptr()));
	}

	return c->leaveStackFrame();
}

/*
	@doc: file.delimagepath
	@type:
		command
	@title:
		file.delimagepath
	@short:
		Removes an image search path
	@syntax:
		file.addimagepath <path>
	@description:
		Removes <path> from the image search path list.[br]
	@seealso:
		[cmd]file.addimagepath[/cmd]
*/

static bool file_module_cmd_delimagepath(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file::delimagepath");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	QString x(szDst.ptr());

	QStringList::Iterator it = KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).find(x);
	if(it != KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).end())
	{
		KVI_OPTION_STRINGLIST(KviOption_stringlistImageSearchPaths).remove(it);
	}

	return c->leaveStackFrame();
}

/*
	@doc: file.write
	@type:
		command
	@title:
		file.write
	@keyterms:
		writing data to files
	@short:
		Writes an ascii data string to a file
	@syntax:
		file.write [-a] <filename> <data>
	@description:
		Writes <data> (which is an ASCII string) to the file <filename>.[br]
		It does NOT append a traling LF character: if you want it you must explicitly specify it in the <data> parameter.[br]
		-a causes the command to append the <data> to the file instead of overwriting the entire file.[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.rename[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_write(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_write");
	KviStr szFile,szData;
	if(!g_pUserParser->parseCmdSingleToken(c,szFile))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szData))return false;

	kvi_adjustFilePath(szFile);

	if(!kvi_writeFile(szFile.ptr(),szData,c->hasSwitch('a')))
		c->warning(__tr("Failed to write to file %s"),szFile.ptr());

	return c->leaveStackFrame();
}


/*
	@doc: file.rename
	@type:
		command
	@title:
		file.rename
	@keyterms:
		copying files
	@short:
		Makes a copy of a file
	@syntax:
		file.rename <oldname> <newname>
	@description:
		Renames a file from <oldname> to <newname>.[br]
		This command can also rename directories.[br]
		If the <newname> file already exists , this command fails.[br]
		The paths (<source> and <destination>) are adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed.Just use UNIX style paths for them.[br]
	@seealso:
		[cmd]file.copy[/cmd], [fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_rename(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_rename");
	KviStr szSrc,szDst;
	if(!g_pUserParser->parseCmdSingleToken(c,szSrc))return false;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szSrc);
	kvi_adjustFilePath(szDst);

	if(kvi_fileExists(szDst.ptr()))
	{
		c->warning(__tr("Destination file exists: file not renamed"));
		return c->leaveStackFrame();
	}

	if(!kvi_renameFile(szSrc.ptr(),szDst.ptr()))
		c->warning(__tr("Failed to rename %s to %s"),szSrc.ptr(),szDst.ptr());

	return c->leaveStackFrame();
}

/*
	@doc: file.mkdir
	@type:
		command
	@title:
		file.mkdir
	@keyterms:
		creating directories
	@short:
		Creates a directory
	@syntax:
		file.mkdir <directory>
	@description:
		Creates the <directory>.[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_mkdir(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_mkdir");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_makeDir(szDst.ptr()))
		c->warning(__tr("Failed to make the directory %s"),szDst.ptr());

	return c->leaveStackFrame();
}


/*
	@doc: file.remove
	@type:
		command
	@title:
		file.remove
	@keyterms:
		removing files
	@short:
		Removes a file
	@syntax:
		file.remove [-q] <name>
	@description:
		Removes the file <name>.[br]
		-q suppresses any warning message (about non existing file , for example).[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
*/

static bool file_module_cmd_remove(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_remove");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_removeFile(szDst.ptr()))
	{
		if(!c->hasSwitch('q'))c->warning(__tr("Failed to remove the file %s"),szDst.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: file.rmdir
	@type:
		command
	@title:
		file.rmdir
	@keyterms:
		removing directories
	@short:
		Removes a directory
	@syntax:
		file.rmdir [-q] <name>
	@description:
		Removes the directory <name>.[br]
		The directory must be empty for this command to success.[br]
		-q suppresses any warning message (about non existing directory , for example).[br]
		The path is adjusted according to the system that KVIrc
		is running on so you don't have to bother about portability: it *should* be automatically
		guaranteed. Just use an UNIX style path.[br]
	@seealso:
		[fnc]$file.exists[/fnc]()
		[cmd]file.remove[/cmd]
*/

static bool file_module_cmd_rmdir(KviModule *m,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"file_module_cmd_rmdir");
	KviStr szDst;
	if(!g_pUserParser->parseCmdFinalPart(c,szDst))return false;

	kvi_adjustFilePath(szDst);

	if(!kvi_removeDir(szDst.ptr()))
	{
		if(!c->hasSwitch('q'))c->warning(__tr("Failed to remove the directory %s"),szDst.ptr());
	}

	return c->leaveStackFrame();
}

/*
	@doc: file.exists
	@type:
		function
	@title:
		$file.exists
	@short:
		Check if a file exists
	@syntax:
		$file.exists(<filename>)
	@description:
		Returns 1 if the file <filename> exists (this is also valid for directories!).[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_exists(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_exists");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	buffer.append(f.exists() ? "1" : "0");
	return true; //c->leaveStackFrame();
}

/*
	@doc: file.type
	@type:
		function
	@title:
		$file.type
	@short:
		Checks the type of a path
	@syntax:
		$file.type(<filename>)
	@description:
		Returns "f" if the <filename> points to a real file , "d" if <filename>
		is the name of a directory or "l" if it is a symbolic link.[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_type(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_type");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	if(f.isFile())buffer.append("f");
	else if(f.isDir())buffer.append("d");
	else if(f.isSymLink())buffer.append("l");
	return true; //c->leaveStackFrame();
}


/*
	@doc: file.ls
	@type:
		function
	@title:
		$file.ls
	@short:
		Returns a directory listing
	@syntax:
		$file.ls(<directory>,[<flags>[,<namefilter>]])
	@description:
		Returns the listing of the specified directory as an array.[br]
		The <directory> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
		<flags> may be a combination of the following characters:[br]
		d: list directories[br]
		f: list files[br]
		l: list symbolic links[br]
		r: list readable files[br]
		w: list writeable files[br]
		x: list executable files[br]
		h: list hidden files[br]
		s: list system files[br]
		n: sort files by name[br]
		t: sort files by file time[br]
		b: sort files by file size[br]
		z: put the directories first, then the files[br]
		k: invert sort order[br]
		i: case insensitive sort[br]
		If <flags> is empty then a default of 'dfrwxhs'.[br]
		If none of r,w,x is set then KVIrc sets all of them by default.[br]
		If <namefilter> is passed then it is interpreted as a wildcard string
		that must match the entries to be returned.[br]
	@example:
		[example]
			%dir[]=$file.ls(/,"d")
			[cmd]foreach[/cmd](%f,%dir[])echo %f
			%dir[]=$file.ls(/usr/include,"f","t*.h");
			[cmd]foreach[/cmd](%f,%dir[])echo %f
			%dir[]=$file.ls($file.homedir,"dfr");
			[cmd]foreach[/cmd](%f,%dir[])echo %f
		[/example]
*/

static bool file_module_fnc_ls(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.ls");

	KviStr szFile = parms->safeFirstParam();
	KviStr szFlags = parms->safeNextParam();
	KviStr szFilter = parms->safeNextParam();

	kvi_adjustFilePath(szFile);
	QDir d(szFile.ptr());
	if(d.exists())
	{
		int iFlags = 0;
		if(szFlags.isEmpty())iFlags = QDir::Dirs | QDir::Files | QDir::NoSymLinks | QDir::Readable | QDir::Writable | QDir::Executable | QDir::Hidden | QDir::System;
		else {
			if(szFlags.contains('d'))iFlags |= QDir::Dirs;
			if(szFlags.contains('f'))iFlags |= QDir::Files;
			if(!szFlags.contains('l'))iFlags |= QDir::NoSymLinks;
			if(szFlags.contains('r'))iFlags |= QDir::Readable;
			if(szFlags.contains('w'))iFlags |= QDir::Writable;
			if(szFlags.contains('x'))iFlags |= QDir::Executable;
			if(szFlags.contains('h'))iFlags |= QDir::Hidden;
			if(szFlags.contains('s'))iFlags |= QDir::System;
		}
		int iSort = 0;
		if(szFlags.isEmpty())iSort = QDir::Unsorted;
		else {
			if(szFlags.contains('n'))iSort |= QDir::Name;
			if(szFlags.contains('t'))iSort |= QDir::Time;
			if(szFlags.contains('b'))iSort |= QDir::Size;
			if(szFlags.contains('z'))iSort |= QDir::DirsFirst;
			if(szFlags.contains('k'))iSort |= QDir::Reversed;
			if(szFlags.contains('i'))iSort |= QDir::IgnoreCase;
		}

		QStringList sl;
		if(szFilter.hasData())sl = d.entryList(szFilter.ptr(),iFlags,iSort);
		else sl = d.entryList(iFlags,iSort);
		if(!sl.isEmpty())
		{
			c->beginListArrayOrDictionaryReturnIdentifier();
			int idx = 0;
			for(QStringList::Iterator it = sl.begin();it != sl.end();++it)
			{
				c->addListArrayOrDictionaryReturnValue(idx++,(*it).latin1(),buffer);
			}
		}
	} else {
		c->warning(__tr("The specified directory does not exist '%s'"),szFile.ptr());
	}

	return c->leaveStackFrame();
}


/*
	@doc: file.size
	@type:
		function
	@title:
		$file.size
	@short:
		Returns the size of a file
	@syntax:
		$file.size(<filename>)
	@description:
		Returns the size of the file pointed by <filename>.[br]
		If the file does not exist , this function returns 0.[br]
		The <filename> should be an unix-style file path and is adjusted according to the system that KVIrc is running on.[br]
*/

static bool file_module_fnc_size(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_size");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	QFileInfo f(szFile.ptr());
	buffer.append(KviStr::Format,"%u",f.size());
	return true; //c->leaveStackFrame();
}


/*
	@doc: file.fixpath
	@type:
		function
	@title:
		$file.fixpath
	@short:
		Converts file paths
	@syntax:
		$file.fixpath(<filename>)
	@description:
		Returns the <filename> adjusted to match the current operating
		system file path conventions.[br] This means that on UNIX , a path like "C:\folder\file.mp3"
		will be returned as "/folder/file.mp3" and vice-versa.[br]
		There is a minor problem with unix paths converted to the windows system: unix
		has no "drive" concept thus the unix paths do not contain a drive. KVIrc will
		always map the paths to the "default" C: drive.[br]
		This is a good reason to avoid using absolute hard-coded paths :).[br]
		Please note that you DON'T NEED to call this function on paths that you
		pass to the other file.* functions: they are adjusted automatically.[br]
	@seealso:
		[fnc]$file.ps[/fnc]
*/

static bool file_module_fnc_fixpath(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_size");
	KviStr szFile = parms->safeFirstParam();
	kvi_adjustFilePath(szFile);
	buffer.append(szFile);
	return true; //c->leaveStackFrame();
}


/*
	@doc: file.ps
	@type:
		function
	@title:
		$file.ps
	@short:
		Returns the file path separator
	@syntax:
		$file.ps
	@description:
		Returns the file path separator for the current operating system.[br]
		On windows , '\' is returned , on UNIX , '/'.[br]
	@seealso:
		[fnc]$file.fixpath[/fnc]
*/

static bool file_module_fnc_ps(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_size");
	buffer.append(KVI_PATH_SEPARATOR_CHAR);
	return true; //c->leaveStackFrame();
}



/*
	@doc: file.read
	@type:
		function
	@title:
		$file.read
	@short:
		Reads a text file
	@syntax:
		$file.read(<filename>[,<size>])
	@description:
		Reads at most <size> bytes of the file pointed by <filename>.[br]
		<size> is an upper limit but may be not reached if the real file is smaller.[br]
		The data read is returned as a string , so if the file contains binary data,
		expect strange results.[br] If <size> is not specified, then KVIrc tries to read
		the whole file up to the 1 MB limit (so if you want to read a file that is
		bigger thatn 1 MB then you MUST specify the <size>).[br]
		WARNING: always check the file size before attemting to read a whole file...
		reading a CDROM iso image may sit down your system :) (and will prolly crash while
		allocating memory , before attempting to read anything)[br]
		An empty string is returned if a serious error occures.[br]
		The <filename> is adjusted according to the system that KVIrc is running on.[br]
	@examples:
		[example]
			echo $file.read(/proc/cpuinfo)
		[/example]
	@seealso:
		[fnc]$file.readbinary[/fnc]
*/

static bool file_module_fnc_read(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.read");
	KviStr szPath = parms->safeFirstParam();
	kvi_adjustFilePath(szPath);
	QFile f(szPath.ptr());
	if(!f.open(IO_ReadOnly))
	{
		c->warning(__tr("Can't open the file \"%s\" for reading"),szPath.ptr());
		return c->leaveStackFrame();
	}

	KviStr szSize = parms->safeNextParam();

	bool bOk;

	unsigned int uSize = szSize.toUInt(&bOk);

	if(!bOk)
	{
		uSize = 1024 * 1024; // 1 meg file is the default
	}

	char * buf = (char *)kvi_malloc(sizeof(char) * (uSize + 1));

	unsigned int uReaded = 0;

	unsigned int uRetries = 0;


	while((uReaded < uSize) && (!f.atEnd()))
	{
		int readedNow = f.readBlock(buf + uReaded,uSize - uReaded);
		if(readedNow < 0)
		{
			kvi_free(buf);
			c->warning(__tr("Read error for file %s"),szPath.ptr());
			return c->leaveStackFrame();
		} else readedNow += uReaded;
		uRetries ++;
		if(uRetries > 1000)
		{
			// ops
			kvi_free(buf);
			c->warning(__tr("Read error for file %s (have been unable to read the requested size in 1000 retries)"),szPath.ptr());
			return c->leaveStackFrame();
		}
		uReaded += readedNow;
	}

	buf[uReaded] = '\0';

	buffer.append(buf);

	kvi_free(buf);

	return c->leaveStackFrame();
}


/*
	@doc: file.localdir
	@type:
		function
	@title:
		$file.localdir
	@short:
		Get the KVIrc local directory
	@syntax:
		$file.localdir(<relative_path>)
		$file.localdir
	@description:
		Returns the path to the KVIrc local data directory.[br]
		The KVIrc local data directory is always writeable and contains
		the various subdirectories that KVIrc uses internally: audio , avatars ,
		config , help , incoming , log , modules , msgcolors and pics.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
	@examples:
		[example]
			echo KVIrc looks for pictures in $file.localdir(pics)
			echo panic.png would be translated to $file.localdir(pics/panic.png)
		[/example]
*/

static bool file_module_fnc_localdir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.localdir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath;
	g_pApp->getLocalKvircDirectory(szPath,KviApp::None,szFile.hasData() ? szFile.ptr() : 0);

	kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return c->leaveStackFrame();
}


/*
	@doc: file.homedir
	@type:
		function
	@title:
		$file.homedir
	@short:
		Get the user's HOME directory
	@syntax:
		$file.homedir(<relative_path>)
		$file.homedir
	@description:
		Returns the user's HOME directory path.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
*/

static bool file_module_fnc_homedir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.homedir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath = QDir::homeDirPath();
	szPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szPath.append(szFile);

	kvi_adjustFilePath(szPath);
	buffer.append(szPath);
	return c->leaveStackFrame();
}


/*
	@doc: file.rootdir
	@type:
		function
	@title:
		$file.rootdir
	@short:
		Get the system root directory
	@syntax:
		$file.rootdir(<relative_path>)
		$file.rootdir
	@description:
		Returns the system's root directory (/ on UNIX and C:/ on Windows).[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
*/

static bool file_module_fnc_rootdir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.rootdir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath = QDir::rootDirPath();
	szPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szPath.append(szFile);

	kvi_adjustFilePath(szPath);
	buffer.append(szPath);
	return c->leaveStackFrame();
}


/*
	@doc: file.cwd
	@type:
		function
	@title:
		$file.currentdir
	@short:
		Get the current directory
	@syntax:
		$file.cwd(<relative_path>)
		$file.cwd
	@description:
		Returns the current working directory.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
*/

static bool file_module_fnc_cwd(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.cwd");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath = QDir::currentDirPath();
	szPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);
	szPath.append(szFile);

	kvi_adjustFilePath(szPath);
	buffer.append(szPath);
	return c->leaveStackFrame();
}

/*
	@doc: file.globaldir
	@type:
		function
	@title:
		$file.globaldir
	@short:
		Get the KVIrc global directory
	@syntax:
		$file.globaldir(<relative_path>)
		$file.globaldir
	@description:
		Returns the path to the KVIrc global data directory.[br]
		The KVIrc local data directory is always readable but usually not writeable and contains
		the various subdirectories that KVIrc uses internally: audio , avatars ,
		config , help , incoming , log , modules , msgcolors and pics.[br]
		If <relative_path> is passed , then it is appended at the end of the directory
		to form a complete filepath.[br]
		The path is adjusted to contain single separators suitable for the platform
		that KVIrc is atually running on (thus you not need to care about path
		separators in the <relative_path> , KVIrc will adjust them).[br]
	@examples:
		[example]
			echo KVIrc looks for pictures in $file.globaldir(pics)
			echo panic.png would be translated to $file.globaldir(pics/panic.png)
		[/example]
*/

static bool file_module_fnc_globaldir(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"$file.globaldir");
	KviStr szFile = parms->safeFirstParam();

	if(szFile.isEmpty())szFile.append(KVI_PATH_SEPARATOR_CHAR);

	KviStr szPath;
	g_pApp->getGlobalKvircDirectory(szPath,KviApp::None,szFile.ptr());

	kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return c->leaveStackFrame();
}

/*
	@doc: file.extractpath
	@type:
		function
	@title:
		$file.extractpath
	@short:
		Extract the path from a filename
	@syntax:
		$file.extractpath(<filepath>)
	@description:
		Returns the path part of the <filepath> translated to match the current
		platform filesystem conventions.[br]
		The path will NOT contain a trailing path separator.[br]
		For example, if <filepath> is /usr/arch/mp3/Carisma_SuonoDelSilenzio.mp3 then
		this function will return /usr/arch/mp3 on UNIX and C:\usr\arch\mp3 on Windows.
	@seealso:
		[fnc]$file.extractfilename[/fnc]
*/

static bool file_module_fnc_extractpath(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_size");

	KviStr szPath = parms->safeFirstParam();
	kvi_adjustFilePath(szPath);

	szPath.cutFromLast(KVI_PATH_SEPARATOR_CHAR);

	while(szPath.lastCharIs(KVI_PATH_SEPARATOR_CHAR))szPath.cutRight(1);

	//szPath.ensureLastCharIs(KVI_PATH_SEPARATOR_CHAR);

	//kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return true; //c->leaveStackFrame();
}

/*
	@doc: file.extractfilename
	@type:
		function
	@title:
		$file.extractfilename
	@short:
		Extract the filename from a file path
	@syntax:
		$file.extractpath(<filepath>)
	@description:
		Returns the filename part of the filepath translated to match the current
		platform filesystem conventions.[br]
		For example, if <filepath> is /usr/arch/mp3/Carisma_SuonoDelSilenzio.mp3 then
		this function will return Carisma_SuonoDelSilenzio.mp3
	@seealso:
		[fnc]$file.extractpath[/fnc]
*/

static bool file_module_fnc_extractfilename(KviModule *m,KviCommand *c,KviParameterList * parms,KviStr &buffer)
{
	//ENTER_STACK_FRAME(c,"file_module_fnc_size");

	KviStr szPath = parms->safeFirstParam();

	kvi_adjustFilePath(szPath);

	szPath.cutToLast(KVI_PATH_SEPARATOR_CHAR);

	//kvi_adjustFilePath(szPath);

	buffer.append(szPath);
	return true; //c->leaveStackFrame();
}

static bool file_module_init(KviModule * m)
{
	m->registerCommand("copy",file_module_cmd_copy);
	m->registerCommand("rename",file_module_cmd_rename);
	m->registerCommand("mkdir",file_module_cmd_mkdir);
	m->registerCommand("write",file_module_cmd_write);
	m->registerCommand("remove",file_module_cmd_remove);
	m->registerCommand("rmdir",file_module_cmd_rmdir);
	m->registerCommand("addimagepath",file_module_cmd_addimagepath);
	m->registerCommand("delimagepath",file_module_cmd_delimagepath);

	m->registerFunction("exists",file_module_fnc_exists);
	m->registerFunction("type",file_module_fnc_type);
	m->registerFunction("size",file_module_fnc_size);
	m->registerFunction("fixpath",file_module_fnc_fixpath);
	m->registerFunction("ps",file_module_fnc_ps);
	m->registerFunction("read",file_module_fnc_read);
	m->registerFunction("localdir",file_module_fnc_localdir);
	m->registerFunction("homedir",file_module_fnc_homedir);
	m->registerFunction("rootdir",file_module_fnc_rootdir);
	m->registerFunction("cwd",file_module_fnc_cwd);
	m->registerFunction("globaldir",file_module_fnc_globaldir);
	m->registerFunction("extractpath",file_module_fnc_extractpath);
	m->registerFunction("extractfilename",file_module_fnc_extractfilename);
	m->registerFunction("ls",file_module_fnc_ls);

	return true;
}

static bool file_module_cleanup(KviModule *m)
{
	return true;
}

KVIRC_MODULE(
	"File",                                                 // module name
	"1.0.0",                                                // module version
	"Copyright (C) 2001 Szymon Stefanek (pragma at kvirc dot net)", // author & (C)
	"Interface to the file system",
	file_module_init,
	0,
	0,
	file_module_cleanup
)
