//=============================================================================
//
//   File : kvi_up_fnc.cpp
//   Creation date : Mon Jul 03 2000 20:41:21 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_debug.h"
#include "kvi_uparser.h"
#include "kvi_command.h"
#include "kvi_app.h"
#include "kvi_error.h"
#include "kvi_frame.h"
#include "kvi_window.h"
#include "kvi_locale.h"
#include "kvi_console.h"
#include "kvi_query.h"
#include "kvi_channel.h"
#include "kvi_scriptobject.h"
#include "kvi_modulemanager.h"
#include "kvi_parameterlist.h"
#include "kvi_ircuserdb.h"
#include "kvi_avatar.h"
#include "kvi_timermanager.h"
#include "kvi_ircuserdb.h"
#include "kvi_mirccntrl.h"
#include "kvi_iconmanager.h"
#include "kvi_options.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnectionuserinfo.h"

#include <stdlib.h> // for rand()
#include <time.h>   // for time()

#include <qregexp.h>


// kvi_app.cpp
extern QAsciiDict<KviWindow> * g_pGlobalWindowDict;
extern KviTimerManager * g_pTimerManager;

void KviUserParser::initFunctionDict()
{
	m_pFunctionDict = new QAsciiDict<KviFunctionParseProc>(109);
	m_pFunctionDict->setAutoDelete(true);

	KviFunctionParseProc * p;

#define ALLOC_PARSE_PROC(__proc) \
	p = new KviFunctionParseProc; \
	p->proc = KVI_PTR2MEMBER(KviUserParser::__proc)

#define FNC_REG(__name,__proc) \
	ALLOC_PARSE_PROC(__proc); \
	m_pFunctionDict->insert(__name,p);

	FNC_REG("WINDOW",parseFnc_WINDOW);
	FNC_REG("CHANNEL",parseFnc_CHANNEL);
	FNC_REG("QUERY",parseFnc_QUERY);
	FNC_REG("CONSOLE",parseFnc_CONSOLE);
	FNC_REG("CONTEXT",parseFnc_CONTEXT);
	FNC_REG("IC",parseFnc_CONTEXT);
	FNC_REG("NEW",parseFnc_NEW);
	FNC_REG("THIS",parseFnc_THIS);
	FNC_REG("RAND",parseFnc_RAND);
	FNC_REG("UNIXTIME",parseFnc_UNIXTIME);
	FNC_REG("SELECTED",parseFnc_SELECTED);
	FNC_REG("ISMEOP",parseFnc_ISMEOP);
	FNC_REG("TARGET",parseFnc_TARGET);
	FNC_REG("OPTION",parseFnc_OPTION);
	FNC_REG("MASK",parseFnc_MASK);
	FNC_REG("AVATAR",parseFnc_AVATAR);
	FNC_REG("ISTIMER",parseFnc_ISTIMER);
	FNC_REG("AWAY",parseFnc_AWAY);
	FNC_REG("ISWELLKNOWN",parseFnc_ISWELLKNOWN);
	FNC_REG("USERNAME",parseFnc_USERNAME);
	FNC_REG("HOSTNAME",parseFnc_HOSTNAME);
	FNC_REG("SERVER",parseFnc_SERVER);
	FNC_REG("FMTLINK",parseFnc_FMTLINK);
	FNC_REG("LF",parseFnc_LF);
	FNC_REG("CR",parseFnc_CR);
	FNC_REG("ASCII",parseFnc_ASCII);
	FNC_REG("CHAR",parseFnc_CHAR);
	FNC_REG("K",parseFnc_K);
	FNC_REG("B",parseFnc_B);
	FNC_REG("U",parseFnc_U);
	FNC_REG("O",parseFnc_O);
	FNC_REG("R",parseFnc_R);
	FNC_REG("ISANYCONSOLECONNECTED",parseFnc_ISANYCONSOLECONNECTED);
	FNC_REG("ME",parseFnc_ME);
	FNC_REG("TIME",parseFnc_TIME);
	FNC_REG("DATE",parseFnc_DATE);
	FNC_REG("BASE64TOTEXT",parseFnc_BASE64TOASCII);
	FNC_REG("HEXTOASCII",parseFnc_HEXTOASCII);
	FNC_REG("ACTIVE",parseFnc_ACTIVE);
	FNC_REG("ISMEVOICE", parseFnc_ISMEVOICE);
	FNC_REG("ISMEHALFOP", parseFnc_ISMEHALFOP);
	FNC_REG("ISMEUSEROP", parseFnc_ISMEUSEROP);
	FNC_REG("CLASSDEFINED",parseFnc_CLASSDEFINED);
	FNC_REG("FEATURES",parseFnc_FEATURES);
	FNC_REG("VERSION",parseFnc_VERSION);
	FNC_REG("SPLIT",parseFnc_SPLIT);
	FNC_REG("ICON",parseFnc_ICON);
	FNC_REG("ICONNAME",parseFnc_ICONNAME);
	FNC_REG("SW",parseFnc_SW);
	FNC_REG("ISMAINWINDOWMINIMIZED",parseFnc_ISMAINWINDOWMINIMIZED);
	FNC_REG("MSGTYPE",parseFnc_MSGTYPE);
	FNC_REG("ARRAY",parseFnc_ARRAY);
	FNC_REG("TR",parseFnc_TR);
}

// FIXME: $eval ???

/*
	@doc: script_localization
	@type:
		generic
	@title:
		Localization of scripts
	@short:
		Explains how to add translation capabilities to your scripts
	@body:
		[p]
		[big]Introduction[/big]
		[p]
		Adding the translated versions of the strings adds a great
		value to your scripts. The process of translating a part of
		software is called localization. KVIrc offers some commands
		and functions for this purpose and this document explains
		briefly how to use them.
		[/p]
		[p]
		[big]The big picture[/big]
		[p]
		All of the strings in your script are written in a "primary language".
		The most common "primary language" is english, but theoretically
		it can be any language of your choice. 
		[/p]



		[p]
		[big]Putting it all together[/big]
		[p]

*/

bool KviUserParser::parseFnc_TR(KviCommand *c,KviParameterList *params,KviStr &buffer)
{
	/*
		@doc: tr
		@type:
			function
		@title:
			$tr
		@short:
			Translates an english string to the current language
		@syntax:
			$tr(<default_language_string>[,<catalogue>])
		@description:
			This function searches for the translation of <default_language_string>
			in the specified translation <catalogue> or in the main
			KVIrc translation file if <catalogue> is omitted (or not found).[br]
			If no translation is found then <english_string> is returned.[br]
			<default_language_string> is a string in your script default
			language (which should probably be english since it is the
			most common language spoken by the translators...).[br]
			You can load a translation catalogue by the means of the [cmd]trload[/cmd]
			command.[br]
			For more informations see the documentation about [doc:script_localization]script localization[/doc].
		@examples:
			[cmd]trload[/cmd] myscript
			[cmd]echo[/cmd] $tr("Hello World!")
		@seealso:
			[cmd]trload[/cmd], [cmd]trunload[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$tr");

	KviStr * pStr = params->first();
	if(pStr)
	{
		KviStr * pCat = params->next();
		QString translation = pCat ? __tr2qs_ctx_no_xgettext(pStr->ptr(),pCat->ptr()) : __tr2qs_no_xgettext(pStr->ptr());
		if(!translation.isEmpty())
			buffer.append(translation);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_BASE64TOASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: base64toAscii
		@type:
			function
		@title:
			$base64ToAscii
		@short:
			Returns a decoded base64 string
		@syntax:
			$bast64ToAscii(<base_64_encoded_string>)
		@description:
		@examples:
	*/

	ENTER_STACK_FRAME(c,"$base64toascii");
	char * buf;

	KviStr * pStr = params->first();
	if(pStr)
	{
		int len = pStr->base64ToBuffer(&buf,true);
		if(len > 0)
		{
			buffer.append(buf,len);
			KviStr::freeBuffer(buf);
		}
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_HEXTOASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: hextoAscii
		@type:
			function
		@title:
			$hexToAscii
		@short:
			Returns a decoded hex string
		@syntax:
			$hexToAscii(<hex_encoded_string>)
		@description:
			Decodes the <hex_encoded_string> to its ASCII representation.
		@examples:
			[cmd]echo[/cmd] $hexToAscii(6B76697263)
	*/

	ENTER_STACK_FRAME(c,"$hextoascii");
	char * buf;

	KviStr * pStr = params->first();
	if(pStr)
	{
		int len = pStr->hexToBuffer(&buf,true);
		if(len > 0)
		{
			buffer.append(buf,len);
			KviStr::freeBuffer(buf);
		}
	}
	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_ICON(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: icon
		@type:
			function
		@title:
			$icon
		@short:
			Returns the ID of an icon
		@syntax:
			$icon(<iconname>)
		@description:
			Returns the ID of the icon <iconname>.
			You should always use this function where an <image_id> is required
			and you want to use an internal icon.
			See [fnc]$iconname[/fnc] for a list of the icon names supported by kvirc.[br]
		@examples:
			[example]
				[cmd]echo[/cmd] $icon(linux)
			[/example]
		@seealso:
			[fnc]$iconname[/fnc]
	*/

	ENTER_STACK_FRAME(c,"$icon");

	KviStr * pStr = params->first();
	if(pStr)
	{
		buffer.append(KviStr::Format,"%d",g_pIconManager->getSmallIconIdFromName(QString(pStr->ptr())));
		return c->leaveStackFrame();
	}
	buffer.append("0");
	return c->leaveStackFrame();
}



bool KviUserParser::parseFnc_ICONNAME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: iconname
		@type:
			function
		@title:
			$iconname
		@short:
			Returns the name of an icon
		@syntax:
			$iconname(<iconid>)
		@description:
			Returns the name of a builtin icon given an <iconid>.
			The <iconid> is returned by the function [fnc]$icon[/fnc].
			If the <iconid> is not valid, an empty name is returned.[br]
			The following code will list all the available icon names:[br]
			[example]
				%i = 0
				do {
					%name = $iconname(%i)
					echo The icon by ID %i is named %name
					%i++
				} while(%name != "")
			[/example]
		@examples:
			[example]
				[cmd]echo[/cmd] $iconid(24)
				[cmd]echo[/cmd] $iconid([fnc]$icon[/fnc](linux))
			[/example]
		@seealso:
			[fnc]$iconname[/fnc]
	*/

	ENTER_STACK_FRAME(c,"$iconname");

	KviStr * pStr = params->first();
	if(pStr)
	{
		bool bOk;
		int idx = pStr->toInt(&bOk);
		if(bOk)
		{
			if(idx >= 0 && idx < KVI_NUM_SMALL_ICONS)
			{
				buffer.append(g_pIconManager->getSmallIconName(idx));
				return c->leaveStackFrame();
			}
		}
	}
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_MSGTYPE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: msgtype
		@type:
			function
		@title:
			$msgtype
		@short:
			Returns the id of a message type color set
		@syntax:
			$msgtype(<message type color set name>)
		@description:
			Returns the id of a message type color set used by the [cmd]echo[/cmd] command.
			You can take a look at the options dialog section related to the
			message output to see the list of available message type names.
		@examples:
			[example]
				[cmd]echo[/cmd] -i=$msgtype(Highlight) Highlighted text!
			[/example]
		@seealso:
			[cmd]echo[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$msgtype");

	KviStr * pStr = params->first();
	if(pStr)
	{
		for(int i=0;i< KVI_NUM_MSGTYPE_OPTIONS;i++)
		{
			if(kvi_strEqualCI(pStr->ptr(),(g_msgtypeOptionsTable[i].name + 7)))
			{
				buffer.append(KviStr::Format,"%d",i);
				return c->leaveStackFrame();
			}
		}
	}
	buffer.append("0");
	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_TARGET(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: target
		@type:
			function
		@title:
			$target
		@short:
			Returns the target of the current window
		@syntax:
			$target
			$target(<window id>)
		@description:
			The form with the <window id> parameter returns the target
			of the channel,query or dcc that has the specified ID.
			The form without parameters returns the target of the current window,
			thus it is equivalent to calling $target([fnc]$window[/fnc]).
			For channel windows the target is the channel name,
			for query windows it is the list of the "queried" users, for the
			dcc windows it is the remote end of the connection.
			The other windows have an empty target.
		@examples:
			[example]
			[cmd]echo[/cmd] $target
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$console[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$target");

	KviStr * pStr = params->first();
	if(pStr)
	{
		KviWindow * wnd = g_pApp->findWindow(pStr->ptr());
		if(wnd)buffer.append(wnd->target());
		else c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
	} else buffer.append(c->window()->target());

	return c->leaveStackFrame();
}



bool KviUserParser::parseFnc_CHANNEL(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: channel
		@type:
			function
		@title:
			$channel
		@short:
			Retrieves the window ID of a specified channel
		@syntax:
			$channel[(<channel name>[,<irc context id>])]
		@description:
			Returns the [b]window ID[/b] of channel matching the
			<channel name> and bound to the connection specified by
			<irc context id>[br]
			If no window matches the specified name or connection, an invalid
			window ID is returned (0).[br]
			If no <irc context id> is specified, this function looks for
			the channel in the current connection context (if any).[br]
			If no <channel name> is specified, this function returns the current
			channel window ID, if executed in a channel, else 0.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$query[/fnc],
			[fnc]$console[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$channel");

	KviWindow *wnd = 0;

	KviStr * pName = params->first();
	KviStr * pCntx = params->next();

	if(pName)
	{
		if(pCntx)
		{
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			KviConsole * cons = g_pApp->findConsole(ircContextId);
			if(!cons)c->warning(__tr2qs("No such IRC context (%u)"),ircContextId);
			else {
				if(cons->connection())
					wnd = cons->connection()->findChannel(pName->ptr());
				else
					wnd = 0;
			}
		} else {
			if(c->window()->connection())wnd = c->window()->connection()->findChannel(pName->ptr());
			else if(!c->window()->console())c->warnNoIrcContext();
		}
	} else if(c->window()->type() == KVI_WINDOW_TYPE_CHANNEL)wnd = c->window();

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_CONSOLE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: console
		@type:
			function
		@title:
			$console
		@short:
			Retrieves the window ID of a specified console
		@syntax:
			$console[(<irc context id>)]
		@description:
			Returns the [b]window ID[/b] of the console bound
			to the IRC context specified by <irc context id>.
			If no window matches the specified IRC context, an invalid
			window ID is returned (0).[br]
			If no <irc context id> is specified, this function looks for
			the console in the current IRC context (if any).[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$console");

	KviWindow *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		wnd = g_pApp->findConsole(ircContextId);
	} else {
		if(c->window()->console())wnd = c->window()->console();
		else c->warnNoIrcContext();
	}

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_ACTIVE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: active
		@type:
			function
		@title:
			$active
		@short:
			Retrieves the window ID of the active window
		@syntax:
			$active[(<irc context id>)]
		@description:
			Returns the [b]window ID[/b] of the active window 
			bound to the IRC context specified by <irc context id>.
			If no window matches the specified IRC context, and invalid
			window ID is returned (0).[br]
			If no <irc context id> is specified, this function returns
			the active window of the current IRC context (if any).[br]
			If the current window is not bound to any IRC context,
			its own ID is returned.
			If the <irc context id> is the special word 'all' then
			the application active window is returned (the window
			that currently has the input focus). Note that in this
			case the returned window may also belong to another IRC
			context or be not bound to any IRC context at all.
			In some extreme cases you may even get a window that
			has no output widget and thus has its output redirected.
			Using the "global" active window should be used only
			for communicating something REALLY urgent (and maybe
			unrelated to a specific IRC connection) to the user.
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
	*/

	ENTER_STACK_FRAME(c,"$active");

	KviWindow *wnd = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		bool bOk;
		unsigned int ircContextId = pCntx->toUInt(&bOk);
		if(!bOk)
		{
			if(kvi_strEqualCI(pCntx->ptr(),"all"))wnd = g_pActiveWindow;
			else return c->error(KviError_invalidIrcContextId,pCntx->ptr());
		} else {
			wnd = g_pApp->findConsole(ircContextId);
			if(wnd)wnd = ((KviConsole *)wnd)->activeWindow();
		}
	} else {
		if(c->window()->console())
		{
			wnd = c->window()->console()->activeWindow();
		} else {
			wnd = c->window();
		}
	}

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveStackFrame();
}



bool KviUserParser::parseFnc_QUERY(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: query
		@type:
			function
		@title:
			$query
		@short:
			Retrieves the window ID of a specified query
		@syntax:
			$query[(<target>[,<irc context id>])]
		@description:
			Returns the [b]window ID[/b] of the query that has <target>
			in the list of targets and is bound to the connection specified by
			<irc context id>[br]
			If no window matches the specified target or context, and invalid
			window ID is returned (0).[br]
			If no <irc context id> is specified, this function looks for
			the query in the current connection context (if any).[br]
			If no <target> is specified, this function returns the current
			query window ID, if executed in a query, else 0.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[fnc]$console[/fnc]
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$query");

	KviWindow *wnd = 0;

	KviStr * pName = params->first();
	KviStr * pCntx = params->next();

	if(pName)
	{
		if(pCntx)
		{
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			KviConsole * cons = g_pApp->findConsole(ircContextId);
			if(!cons)c->warning(__tr2qs("No such IRC context (%u)"),ircContextId);
			else {
				if(cons->connection())
					wnd = cons->connection()->findQuery(pName->ptr());
				else
					wnd = 0;
			}
		} else {
			if(c->window()->connection())wnd = c->window()->connection()->findQuery(pName->ptr());
			else if(!c->window()->console())c->warnNoIrcContext();
		}
	} else if(c->window()->type() == KVI_WINDOW_TYPE_QUERY)wnd = c->window();

	buffer.append(wnd ? wnd->id() : "0");

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_CONTEXT(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: context
		@type:
			function
		@title:
			$context
		@short:
			Retrieves the ID of the specified IRC context
		@syntax:
			$context[(<server>,<nickname>)]
		@description:
			Returns the [b]IRC context ID[/b] of the IRC context that uses
			the specified <server> and <nickname>.[br] This function can
			find only connected IRC contexts.
			If no context matches the server and nickname, and invalid
			[b]IRC context ID[/b] is returned (0).[br]
			If <server> is an empty string, the first context that matches
			the specified nickname is returned. If <nickname> is an empty string
			the first context that uses the specified server is returned.
			If both parameters are missing this function returns the
			id of the current IRC context, or '0' if the
			window in that this call is executed is not bound to any IRC context.
			Please note that in this last case you may find an [b]IRC context[/b]
			that is 'not connected'.
			This can only happen if the current window is a console that is
			in "idle" state, with no connection established yet.[br]
			It is a good idea to take a look at the
			[doc:window_naming_conventions]window naming conventions[/doc].
			This identifier is equivalent to [fnc]$ic[/fnc].[br]
		@examples:
			[example]
			[/example]
		@seealso:
	*/

	/*
		@doc: ic
		@type:
			function
		@title:
			$ic
		@short:
			Retrieves the ID of the specified IRC context
		@syntax:
			$ic[(<server>,<nickname>)]
		@description:
			This identifier is equivalent to [fnc]$context[/fnc].
			See the related documentation.[br]
		@examples:
			[example]
			[/example]
		@seealso:
	*/


	ENTER_STACK_FRAME(c,"$context");

	KviStr * pServ = params->first();
	KviStr * pNick = params->next();

	if(!pServ)pServ = &m_szEmptyString;
	if(!pNick)pNick = &m_szEmptyString;

	KviConsole * cons = 0;

	if(pServ->hasData() || pNick->hasData())
	{
		// some parameters passed
		cons = g_pApp->findConsole(*pServ,*pNick);
	} else {
		// both are empty
		cons = c->window()->console();
		//if(!cons)c->warnNoIrcContext();
	}

	if(cons)buffer.append(KviStr::Format,"%u",cons->ircContextId());
	else buffer.append('0');

	return c->leaveStackFrame();
}

// FIXME: #warning "$window should accept the type as optional parameter"
// FIXME: #warning "$type"

bool KviUserParser::parseFnc_WINDOW(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: window
		@type:
			function
		@title:
			$window
		@short:
			Retrieves the ID of a specified window
		@syntax:
			$window[(<caption text>)]
		@description:
			Returns the [b]window ID[/b] of the first window that
			has the specified <caption text>.[br]
			If no window matches the specified <caption text>, and invalid
			window ID is returned (0).[br]
			If no <caption text> is specified, this function returns the id
			of the current window.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			[fnc]$channel[/fnc],
			[fnc]$query[/fnc],
			[fnc]$console[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$window");

	KviWindow *wnd = c->window();

	KviStr * pStr = params->first();

	if(pStr)
	{
		wnd = g_pApp->findWindowByCaption(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with caption '%s' not found, returning 0"),pStr->ptr());
			buffer.append('0');
			return c->leaveStackFrame();
		}
	}

	buffer.append(wnd->id());

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_NEW(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: new
		@type:
			function
		@title:
			$new
		@short:
			Creates a new object
		@syntax:
			$new(<class>,[<parent_id> [,<name>[,<param>[,<param>[...]]]])
		@description:
			Creates a new instance of the object <class> with
			the parent object <parent_id> and the specified <name>.[br]
			<name> and <parent_id> are optional: if not specified, <name>
			is assumed to be an empty string and <parent_id> default to 0 (parentless object).[br]
			Please see the [doc:objects]objects documentation[/doc] for more information.[br]
		@examples:
			[example]
				%myobj = $new(widget,0,pippo)
			[/example]
		@seealso:
			[doc:objects]Objects documentation[/doc], [cmd]delete[/cmd]

	*/

	ENTER_STACK_FRAME(c,"$new");

	KviStr * pszClass  = params->first();
	if(!pszClass)return c->error(KviError_missingObjectClassName);

	KviStr * pszParent = params->next();
	KviScriptObject * pParent = 0;
	if(pszParent)
	{
		if(!kvi_strEqualCS(pszParent->ptr(),"0"))
		{
			pParent = g_pScriptObjectController->lookupObject(pszParent->ptr());
			if(!pParent)return c->error(KviError_noSuchObject,"%s",pszParent->ptr());
		}
	}

	KviStr * pszName   = pszParent ? params->next() : 0;

	KviScriptObjectClass * pClass = g_pScriptObjectController->lookupClass(pszClass->ptr());

	if(!pClass)return c->error(KviError_noSuchObjectClass,"%s",pszClass->ptr());

	KviScriptObject * ob = pClass->allocateInstance(pParent,pszName ? pszName->ptr() : "",c,params);

	buffer.append(ob ? ob->id() : "0");
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_THIS(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: this
		@type:
			function
		@title:
			$this
		@short:
			Retrieves the ID of the current object
		@syntax:
			$this
		@description:
			Returns the ID of the current object or ('0') if there is
			none. This function has a "quick" version with syntax:
			[b]$$[/b][br]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

	/*
		@doc: $
		@type:
			function
		@title:
			$$
		@short:
			Retrieves the ID of the current object
		@syntax:
			$$
		@description:
			Returns the ID of the current object or ('0') if there is
			none. This function has equivalent to [fnc]$this[/fnc]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

// FIXME: #warning "BETTER DOCS FOR THIS"

	ENTER_STACK_FRAME(c,"$this");

	if(c->thisPointer())buffer.append(c->thisPointer()->id());
	else buffer.append('0');

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_CLASSDEFINED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: classdefined
		@type:
			function
		@title:
			$classdefined
		@short:
			Checks if a class is defined
		@syntax:
			$classdefined(<class_name>)
		@description:
			Returns 1 if the class <class_name> is defined, else 0.[br]
		@examples:
			[example]
			[/example]
		@seealso:
			
	*/

	ENTER_STACK_FRAME(c,"$classdefined");

	KviStr szClass = params->safeFirstParam();

	buffer.append(g_pScriptObjectController->lookupClass(szClass.ptr()) ? '1' : '0');

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_RAND(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: rand
		@type:
			function
		@title:
			$rand
		@short:
			Generates a random number
		@syntax:
			$rand(<max>)
		@description:
			Returns a random integer number from 0 to max inclusive.
			You can repeat sequences of random numbers by calling [cmd]srand[/cmd]
			with the same seed value. If [cmd]srand[/cmd] has not been called
			$rand is automatically seeded with value of 1.
			If no <max> is specified, this function returns an integer between
			0 and RAND_MAX that is system dependant.
		@examples:
			[example]
			[/example]
		@seealso:
			[cmd]srand[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$rand");

	bool bOk;
	int max = params->safeFirst()->toInt(&bOk);
	if(!bOk)max = RAND_MAX;
	buffer.append(KviStr::Format,"%d",rand() % max);
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_UNIXTIME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: unixtime
		@type:
			function
		@title:
			$unixTime
		@short:
			Returns the current UNIX time
		@syntax:
			$unixtime
		@description:
			Returns the time since the Epoch (00:00:00 UTC, January 1, 1970),
			measured in seconds.  
		@seealso:
			[fnc]$time[/fnc]
		@examples:
			[example]
			[/example]
	*/

	buffer.append(KviStr::Format,"%u",time(0));
	return true;
}

bool KviUserParser::parseFnc_TIME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: time
		@type:
			function
		@title:
			$time
		@short:
			Returns a formatted date/time string
		@syntax:
			$time(<unixtime>)
			$time
		@description:
			Returns the string representation of <unixtime> or
			of the current time if <unixtime> is not given.[br]  
		@examples:
			[example]
			[/example]
	*/

	ENTER_STACK_FRAME(c,"$time");

	KviStr * pStr = params->safeFirst();

	time_t t;
	if(pStr->hasData())
	{
		bool bOk;
		t = (time_t)pStr->toUInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr2qs("The specified UNIX time is not valid (%s)"),pStr->ptr());
			return c->leaveStackFrame();
		}
	} else {
		t = time(0);
	}

	buffer.append(ctime(&t));
	if(buffer.lastCharIs('\n'))buffer.cutRight(1);
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_DATE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: date
		@type:
			function
		@title:
			$date
		@short:
			Returns a date/time string using a specified format
		@syntax:
			$date(<format>[,<unixtime>])
		@description:
			Returns the string representation of <unixtime> or
			of the current time if <unixtime> is not given,
			based on <format>.[br]
			Valid codes for the format string are those accepted
			by the strftime() C function and UNIX 'date'.[br]  
		@examples:
			[example]
				[cmd]echo[/cmd] $date(\%d-\%m-\%Y)
			[/example]
	*/

	ENTER_STACK_FRAME(c,"$date");
	char buf[256];

	KviStr * pStr = params->safeNext();

	time_t t;
	if(pStr->hasData())
	{
		bool bOk;
		t = (time_t)pStr->toUInt(&bOk);
		if(!bOk)
		{
			c->warning(__tr2qs("The specified UNIX time is not valid (%s)"),pStr->ptr());
			return c->leaveStackFrame();
		}
	} else {
		t = time(0);
	}

	if(strftime(buf,255,params->safeFirst()->ptr(),localtime(&t)) > 0)
	{
		buffer.append(buf);
		if(buffer.lastCharIs('\n'))buffer.cutRight(1);
	} else
		c->warning(__tr2qs("The specified time format is not valid (%s)"),params->safeFirst()->ptr());
	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_SELECTED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
// FIXME: #warning "THIS HAS TO WORK FOR QUERIES TOO!"
	/*
		@doc: selected
		@type:
			function
		@title:
			$selected
		@short:
			Returns the list of selected nicknames in the channel
		@syntax:
			$selected
			$selected(<window id>)
		@description:
			The form with the <window id> parameter returns an array of the selected
			nicknames in the channel designated by <window id>.
			The form without parameters returns an array of the selected nicknames
			in the current window (assuming that it is a channel),
			thus it is equivalent to calling $selected([fnc]$window[/fnc])
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
			In a non-array/dictionary context it returns the selected nicknames as a comma separated list.
		@examples:
			[example]
				[cmd]echo[/cmd] $selected
				[cmd]foreach[/cmd](%i,$selected)[cmd]echo[/cmd] %i
			[/example]
		@seealso:
			[fnc]$window[/fnc],
			[fnc]$channel[/fnc],
			[doc:window_naming_conventions]Window naming conventions[/doc]
	*/

	ENTER_STACK_FRAME(c,"$selected");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveStackFrame();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("The specified window is not a channel"));
		return c->leaveStackFrame();
	}

	c->beginListArrayOrDictionaryReturnIdentifier();
	int id = 0;

	for(QString * s = ((KviChannel *)wnd)->firstSelectedNickname();s;s = ((KviChannel *)wnd)->nextSelectedNickname())
	{
		KviStr tmp(*s);
		c->addListArrayOrDictionaryReturnValue(id++,tmp,buffer);
	}

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_ARRAY(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: array
		@type:
			function
		@title:
			$array
		@short:
			Explicitly creates an array
		@syntax:
			$array(<item>,<item>,<item>,....);
		@description:
			Returns an array with the specified items. The items are indexed starting from 0.
			This is just an explicit way of creating an array with a defined set of items,
			useful for increasing readability. 
		@examples:
			%x[] = $array(1,2,3);
			[cmd]foreach[/cmd](%y,%x[])
			{
				[cmd]echo[/cmd] %y;
			}
		@seealso:
	*/

	ENTER_STACK_FRAME(c,"$array");

	KviStr * pStr;
	pStr = params->first();
	
	c->beginListArrayOrDictionaryReturnIdentifier();
	
	int id = 0;

	for(KviStr * s = params->first();s;s = params->next())
	{
		c->addListArrayOrDictionaryReturnValue(id++,*s,buffer);
	}

	return c->leaveStackFrame();
}



bool KviUserParser::parseFnc_ISMEOP(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: ismeop
		@type:
			function
		@title:
			$isMeOp
		@short:
			Returns 1 if the current user has operator status
		@syntax:
			$ismeop
			$ismeop(<window id>)
		@description:
			NOTE: This function is DEPRECATED: use [fnc]$chan.ismeop[/fnc]()
		@seealso:
	*/
	ENTER_STACK_FRAME(c,"$ismeop");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveStackFrame();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("The specified window is not a channel"));
		return c->leaveStackFrame();
	}
		
	buffer.append(((KviChannel *)wnd)->isMeOp() ? '1' : '0');

	return c->leaveStackFrame();
}



bool KviUserParser::parseFnc_ISMEVOICE(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    /*
        @doc: ismevoice
        @type:
            function
        @title:
            $isMeVoice
        @short:
            Returns 1 if the current user has voice status
        @syntax:
            $ismevoice
            $ismevoice(<window id>)
        @description:
            The form with the <window id> parameter returns '1' if the current
            user has voice status on channel designated by <window id>.
            The form without parameters is equivalent to $ismevoice([fnc]$window[/fnc])
        @seealso:
			[fnc]$isvoice[/fnc],
			[fnc]$isop[/fnc],
			[fnc]$ismeop[/fnc],
			[fnc]$ishalfop[/fnc],
			[fnc]$ismehalfop[/fnc],
			[fnc]$isuserop[/fnc],
			[fnc]$ismeuserop[/fnc],
            [fnc]$window[/fnc],
            [fnc]$channel[/fnc],
            [doc:window_naming_conventions]Window naming conventions[/doc]

    */
	ENTER_STACK_FRAME(c,"$ismevoice");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
			return c->leaveStackFrame();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("The specified window is not a channel"));
		return c->leaveStackFrame();
	}
	buffer.append(((KviChannel *)wnd)->isMeVoice() ? '1' : '0');

	return c->leaveStackFrame();
}
	
bool KviUserParser::parseFnc_ISMEHALFOP(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    /*
        @doc: ismehalfop
        @type:
            function
        @title:
            $isMeHalfOp
        @short:
            Returns 1 if the current user has halfop status
        @syntax:
            $ismehalfop
            $ismehalfop(<window id>)
        @description:
            The form with the <window id> parameter returns '1' if the current
            user has half-operator status on channel designated by <window id>.
            The form without parameters is equivalent to $ismehalfop([fnc]$window[/fnc])
        @seealso:
			[fnc]$ismevoice[/fnc],
            [fnc]$isvoice[/fnc],
            [fnc]$isop[/fnc],
            [fnc]$ismeop[/fnc],
            [fnc]$ishalfop[/fnc],
	    [fnc]$isuserop[/fnc],
            [fnc]$ismeuserop[/fnc],
            [fnc]$window[/fnc],
            [fnc]$channel[/fnc],
            [doc:window_naming_conventions]Window naming conventions[/doc]

    */
	ENTER_STACK_FRAME(c,"$ismehalfop");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
            return c->leaveStackFrame();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("The specified window is not a channel"));
		return c->leaveStackFrame();
	}

	buffer.append(((KviChannel *)wnd)->isMeHalfOp() ? '1' : '0');

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_ISMEUSEROP(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    /*
        @doc: ismeuserop
        @type:
            function
        @title:
            $isMeUserOp
        @short:
            Returns 1 if the current user has userop status
        @syntax:
            $ismeuserop
            $ismeuserop(<window id>)
        @description:
            The form with the <window id> parameter returns '1' if the current
            user has user-operator status on channel designated by <window id>.
            The form without parameters is equivalent to $ismeuserop([fnc]$window[/fnc])
        @seealso:
			[fnc]$ismevoice[/fnc],
            [fnc]$isvoice[/fnc],
            [fnc]$isop[/fnc],
            [fnc]$ismeop[/fnc],
            [fnc]$ishalfop[/fnc],
            [fnc]$ismehalfop[/fnc],
            [fnc]$isuserop[/fnc],
            [fnc]$window[/fnc],
            [fnc]$channel[/fnc],
            [doc:window_naming_conventions]Window naming conventions[/doc]

    */
	ENTER_STACK_FRAME(c,"$ismeuserop");

	KviStr * pStr = params->first();
	KviWindow * wnd = c->window();
	if(pStr)
	{
		wnd = g_pApp->findWindow(pStr->ptr());
		if(!wnd)
		{
			c->warning(__tr2qs("Window with ID '%s' not found, returning empty string"),pStr->ptr());
            return c->leaveStackFrame();
		}
	}

	if(wnd->type() != KVI_WINDOW_TYPE_CHANNEL)
	{
		c->warning(__tr2qs("The specified window is not a channel"));
		return c->leaveStackFrame();
	}

	buffer.append(((KviChannel *)wnd)->isMeUserOp() ? '1' : '0');

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_ISTIMER(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: istimer
		@type:
			function
		@title:
			$isTimer
		@short:
			Checks for a timer existence
		@syntax:
			$istimer(<name>)
		@description:
			Returns 1 if the timer named <name> is actually running, else 0
		@seealso:
			[cmd]timer[/cmd],[cmd]killtimer[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$istimer");

	KviStr * pStr = params->first();
	if(pStr)buffer.append(g_pTimerManager->isTimer(pStr->ptr()) ? '1' : '0');
	else return c->error(KviError_notEnoughParameters,__tr("Missing timer name"));

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_OPTION(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: option
		@type:
			function
		@title:
			$option
		@short:
			Returns the value of an option
		@syntax:
			$option(<optionName>)
		@description:
			Returns the current value of the internal option named <optionName>.
			See the [cmd]option[/cmd] command documentation for more info about options.
		@examples:
			[example]
				[cmd]echo[/cmd] $option(fontIrcView)
			[/example]
		@seealso:
			[cmd]option[/cmd],
	*/

	ENTER_STACK_FRAME(c,"$option");

	KviStr * pStr = params->first();
	if(pStr)
	{
		QString tmp;
		if(g_pApp->getOptionString(pStr->ptr(),tmp))buffer.append(tmp);
		else c->warning(__tr2qs("No option named '%s'"),pStr->ptr());
	} else c->warning(__tr2qs("No option name specified"));

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_MASK(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: mask
		@type:
			function
		@title:
			$mask
		@short:
			Retrieves the host mask of a user
		@syntax:
			$mask[(<nickname>[,<mask_type>])]
		@description:
			Returns the specified type of mask for the user with <nickname>.[br]
			If the host or username are not known, the mask may contain less information
			than requested.[br]
			If the <nickname> is not given it is assumed to be the current nickname.[br]
			If <mask_type> is not given or is invalid, it is assumed to be 0.[br]
			Available mask types:[br]
			0 : nick!user@machine.host.top  (nick!user@XXX.XXX.XXX.XXX) (default)[br]
			1 : nick!user@*.abc.host.top        (nick!user@XXX.XXX.XXX.*)[br]
			2 : nick!user@*[br]
			3 : nick!*@machine.host.top     (nick!user@XXX.XXX.XXX.XXX)[br]
			4 : nick!*@*.abc.host.top           (nick!user@XXX.XXX.XXX.*)[br]
			5 : nick!*@*[br]
			6 : *!user@machine.host.top     (*!user@XXX.XXX.XXX.XX)[br]
			7 : *!user@*.abc.host.top           (*!user@XXX.XXX.XXX.*)[br]
			8 : *!user@*[br]
			9 : *!*@machine.host.top        (*!*@XXX.XXX.XXX.XXX)[br]
			10: *!*@*.abc.host.top              (*!*@XXX.XXX.XXX.*)[br]
			11: nick!*user@machine.host.top (nick!*user@XXX.XXX.XXX.XXX)[br]
			12: nick!*user@*.abc.host.top       (nick!*user@XXX.XXX.XXX.*)[br]
			13: nick!*user@*[br]
			14: *!*user@machine.host.top    (*!*user@XXX.XXX.XXX.XXX)[br]
			15: *!*user@*.abc.host.top          (*!*user@XXX.XXX.XXX.*)[br]
			16: *!*user@*[br]
			17: nick!~user@*.host.top       (nick!~user@XXX.XXX.*)[br]
			18: nick!*@*.host.top          (nick!*@XXX.XXX.*)[br]
			19: *!~user@*.host.top          (*!~user@XXX.XXX.*)[br]
			20: nick!*user@*.host.top          (nick!*user@XXX.XXX.*)[br]
			21: *!*user@*.host.top          (*!user@*XXX.XXX.*)[br]
			22: nick!~user@*.host.top       (nick!~user@XXX.XXX.*)[br]
			23: nick!*@*.host.top          (nick!*@XXX.XXX.*)[br]
			24: *!~user@*.host.top          (*!~user@XXX.XXX.*)[br]
			25: nick!*user@*.host.top          (nick!*user@XXX.XXX.*)[br]
			26: *!*user@*.host.top          (*!user@*XXX.XXX.*)[br]
			If some data is missing, these types may change:[br]
			For example, if the hostname is missing, the mask type 3 or 4 may be reduced to type 5.[br]
			If the user with <nickname> is not found in the current IRC context user database,
			an empty string is returned.[br]
			The masks 22-26 are the smart versions of the masks 17-21 that try take care of masked ip addresses
			in the form xxx.xxx.INVALID-TOP-MASK. If a masked ip address is found then
			the XXX.XXX.* or XXX.* host mask is returned instead of the (wrong) *.INVALID-TOP-MASK
		@examples:
		@seealso:
	*/

	ENTER_STACK_FRAME(c,"$mask");

	KviStr * nk = params->safeFirst();
	KviStr * mt = params->safeNext();
	bool bOk;
	int maskType = mt->toInt(&bOk);
	if((!bOk) || (maskType < 0) || (maskType > 26))maskType = 0;

	if(!(c->window()->console()))return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();
	KviIrcUserDataBase * db = c->window()->connection()->userDataBase();

	const char * nick = nk->hasData() ? nk->ptr() : c->window()->connection()->currentNickName().latin1();

	KviIrcUserEntry * e = db->find(nick);
	if(e)
	{
		KviIrcMask u;
		u.setNick(nick);
		u.setUsername(e->user());
		u.setHost(e->host());

		QString tmp;
		u.mask(tmp,(KviIrcMask::MaskType)maskType);

		buffer.append(tmp);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_AVATAR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: avatar
		@type:
			function
		@title:
			$avatar
		@short:
			Obsolete
		@syntax:
			$avatar[([<nickname>])]
		@description:
			This function is obsolete and has been replaced by [fnc]$avatar.name[/fnc]
			and [fnc]$avatar.path[/fnc]. It is provided only for backward compatibility
			and will be removed in a future version.
	*/

	ENTER_STACK_FRAME(c,"$avatar");

	KviStr * nk = params->safeFirst();

	if(!(c->window()->console()))return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviIrcUserDataBase * db = c->window()->connection()->userDataBase();
	if(!db)return c->notConnectedToServer();

	const char * nick = nk->hasData() ? nk->ptr() : c->window()->connection()->currentNickName().latin1();

	KviIrcUserEntry * e = db->find(nick);
	if(e)
	{
		KviAvatar * a = e->avatar();
		if(a)buffer.append(a->localPath());
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_AWAY(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: away
		@type:
			function
		@title:
			$away
		@short:
			Returns 1 if the current user is away
		@syntax:
			$away
		@description:
			Returns 1 if the current user is away, else 0.
			If the current IRC context is not connected at all, this function prints
			a warning and returns 0 anyway.
	*/

	ENTER_STACK_FRAME(c,"$away");

	bool bAway = false;

	if(!c->window()->console())c->warnNoIrcContext();
	else {
		if(!c->window()->console()->isConnected())c->warnNotConnectedToServer();
		else bAway = c->window()->connection()->userInfo()->isAway();
	}

	buffer.append(bAway ? '1' : '0');

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_ISANYCONSOLECONNECTED(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
	   @doc: isanyconsoleconnected
	   @type:
	   	   function
	   @title:
	       $isAnyConsoleConnected
	   @short:
	       Returns 1 if there is any console connected
	   @syntax:
	       $isAnyConsoleConnected()
	   @description:
	       Returns 1 if there is at least one console connected to a server. [br]If that's not the case,
		   returns 0.[br]
	*/
	
	ENTER_STACK_FRAME(c,"$isanyconsoleconnected");

	if (g_pApp->isAnyConsoleConnected()) buffer.append('1');
	else buffer.append('0');

	return c->leaveStackFrame();
}
	

bool KviUserParser::parseFnc_ISWELLKNOWN(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: iswellknown
		@type:
			function
		@title:
			$isWellKnown
		@short:
			Returns 1 if the specified user is well known
		@syntax:
			$isWellKnown(<nickname>)
		@description:
			Returns 1 if KVIrc has the basic user information about the specified <nickname>.[br]
			The basic information include the username and hostname.[br]
			This is almost always true if the user is on a channel with you or
			you have an open query with him.[br]
			If $isWellKnown returns 0, [fnc]$username[/fnc] and [fnc]$hostname[/fnc]
			will return empty strings.[br]
			In this case you must use [cmd]awhois[/cmd] to obtain the user basic information.[br]
	*/

	ENTER_STACK_FRAME(c,"$isWellKnown");

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviIrcUserEntry * e = c->window()->connection()->userDataBase()->find(params->safeFirstParam());
	bool bGotIt = false;
	if(e)
	{
		bGotIt = (e->hasHost() && e->hasUser());
	}

	buffer.append(bGotIt ? '1' : '0');

	return c->leaveStackFrame();
}

bool KviUserParser::parseFnc_USERNAME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: username
		@type:
			function
		@title:
			$username
		@short:
			Returns the username of the specified user
		@syntax:
			$username(<nickname>)
		@description:
			Returns the username of the specified IRC user IF it is known.[br]
			The username is known if [fnc]$isWellKnown[/fnc] returns 1.[br]
			The username is generally known if the user is on a channel with you
			or has an open query with you.[br]
			Detailed explaination:[br]
			KVIrc has an internal database of users that are currently
			visible by *this client*: this includes users on open channels
			and queries.[br] The other IRC users are NOT in the database:
			this means that KVIrc knows NOTHING about them and can't return
			any information immediately. In this case this function will return
			an EMPTY string.[br]
			If a user is in the database, at least his nickname is known.[br]
			The username and hostname are known only if the server provides that information
			spontaneously or after a KVIrc request.[br]
			KVIrc requests user information for all the users in open queries
			and channels. This information takes some time to be retrieved,
			in this interval of time KVIrc knows only the user's nickname.
			This function will return the string "*" in this case.[br]
		@seealso:
			[fnc]$iswellknown[/fnc], [$fnc]$hostname[/fnc], [cmd]awhois[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$username");

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviIrcUserEntry * e = c->window()->connection()->userDataBase()->find(params->safeFirstParam());

	if(e)buffer.append(e->user());

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_HOSTNAME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: hostname
		@type:
			function
		@title:
			$hostname
		@short:
			Returns the hostname of the specified user
		@syntax:
			$hostname(<nickname>)
		@description:
			Returns the hostname of the specified IRC user IF it is known.[br]
			The hostname is known if [fnc]$isWellKnown[/fnc] returns 1.[br]
			The hostname is generally known if the user is on a channel with you
			or has an open query with you.[br]
			Detailed explaination:[br]
			KVIrc has an internal database of users that are currently
			visible by *this client*: this includes users on open channels
			and queries.[br] The other IRC users are NOT in the database:
			this means that KVIrc knows NOTHING about them and can't return
			any information immediately. In this case this function will return
			an EMPTY string.[br]
			If a user is in the database, at least his nickname is known.[br]
			The username and hostname are known only if the server provides that information
			spontaneously or after a KVIrc request.[br]
			KVIrc requests user information for all the users in open queries
			and channels. This information takes some time to be retrieved,
			in this interval of time KVIrc knows only the user's nickname.
			This function will return the string "*" in this case.[br]
		@seealso:
			[fnc]$iswellknown[/fnc], [$fnc]$username[/fnc], [cmd]awhois[/cmd]
	*/

	ENTER_STACK_FRAME(c,"$hostname");

	if(!c->window()->console())return c->noIrcContext();
	if(!c->window()->console()->isConnected())return c->notConnectedToServer();

	KviIrcUserEntry * e = c->window()->connection()->userDataBase()->find(params->safeFirstParam());

	if(e)buffer.append(e->host());

	return c->leaveStackFrame();
}


bool KviUserParser::parseFnc_SERVER(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: server
		@type:
			function
		@title:
			$server
		@short:
			Returns the current server name (if any)
		@syntax:
			$server[(irc_context_id)]
		@description:
			Returns the current server name of the specified IRC context.[br]
			If no <irc_context_id> is specified, the current IRC context is used.[br]
			If <irc_context_id> is the special word 'all', this function will return server
			names of all IRC contexts.[br]
			If you are not connected to a server, this function will return an empty string.[br]
			If the current window does not belong to any IRC context and no irc_context_id
			is specified, this function prints a warning and also returns an empty string.[br]
	*/

	ENTER_STACK_FRAME(c,"$server");

	KviConsole *cns = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		if(kvi_strEqualCI(pCntx->ptr(),"all"))
		{
			c->beginListArrayOrDictionaryReturnIdentifier();
			int idx = 0;
			QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

			while(KviWindow * wnd = it.current())
			{
				if((wnd->type() == KVI_WINDOW_TYPE_CONSOLE) && ((KviConsole *)wnd)->isConnected())
					c->addListArrayOrDictionaryReturnValue(idx++,((KviConsole *)wnd)->connection()->currentServerName().latin1(),buffer);
				++it;
			}
		} else {
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			cns = g_pApp->findConsole(ircContextId);
		}
	} else {
		if(c->window()->console())cns = c->window()->console();
		else c->warnNoIrcContext();
	}

	if(cns)
	{
		if(cns->isConnected())buffer.append(cns->connection()->currentServerName());
	}

	return c->leaveStackFrame();

}


bool KviUserParser::parseFnc_ME(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: me
		@type:
			function
		@title:
			$me
		@short:
			Returns the current nickname
		@syntax:
			$me[(irc_context_id)]
		@description:
			Returns the current nickname used in the specified IRC context.[br]
			If no <irc_context_id> is specified, the current IRC context is used.[br]
			If <irc_context_id> is the special word 'all', this function will return the
			nicknames on all IRC contexts.[br]
			If you are not connected to a server, this function will return an empty string.[br]
			If the current window is a DCC chat and no irc_context is specified,
			the local nickname is returned.[br]
			If the current window does not belong to any IRC context, no irc_context_id
			is specified, and the current window is not a DCC chat, a warning is printed.[br]
	*/

	ENTER_STACK_FRAME(c,"$me");

	KviConsole *cns = 0;

	KviStr * pCntx = params->first();

	if(pCntx)
	{
		if(kvi_strEqualCI(pCntx->ptr(),"all"))
		{
			c->beginListArrayOrDictionaryReturnIdentifier();
			int idx = 0;
			QAsciiDictIterator<KviWindow> it(*g_pGlobalWindowDict);

			while(KviWindow * wnd = it.current())
			{
				if((wnd->type() == KVI_WINDOW_TYPE_CONSOLE) && ((KviConsole *)wnd)->isConnected())
					c->addListArrayOrDictionaryReturnValue(idx++,((KviConsole *)wnd)->connection()->currentNickName().latin1(),buffer);
				++it;
			}
		} else {
			bool bOk;
			unsigned int ircContextId = pCntx->toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidIrcContextId,pCntx->ptr());
			cns = g_pApp->findConsole(ircContextId);
		}
	} else {
		if(c->window()->console())cns = c->window()->console();
		else {
			if(c->window()->type() == KVI_WINDOW_TYPE_DCCCHAT)
				buffer.append(c->window()->localNick()); // DCC CHAT!
			else
				c->warning(__tr2qs("This window has no associated IRC context and is not a DCC chat"));
		}
	}

	if(cns)
	{
		if(cns->isConnected())buffer.append(cns->connection()->currentNickName());
	}

	return c->leaveStackFrame();

}

bool KviUserParser::parseFnc_FMTLINK(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: fmtsocket
		@type:
			function
		@title:
			$fmtsocket
		@short:
			Returns a formatted socket buffer
		@syntax:
			$fmtsocket(<socket_text>,<double_click_command>[,<tooltip_text>])
		@description:
			Returns a socket formatted for the [cmd]echo[/cmd] command.[br]
			If you pass the returned string to the echo command, the string will be displayed
			as a socket and will be highlighted when the user moves the mouse over it.[br]
			If the user will leave the mouse for a few seconds over the socket, the <tooltip_text>
			will be displayed in a small tooltip window. If <tooltip_text> is not given,
			then no tooltip will be shown.[br]
			The <double_click_command> will be executed when the user will double click on the socket.[br]
			Please remember that if <double_click_command> contains identifiers
			that must be evaluated at double-click time, you MUST escape them in the $fmtsocket() call
			to prevent the evaluation.[br]
			You might also take a look at [doc:escape_sequences]the escape sequences documentation[/doc]
			to learn more about how the sockets are implemented and how to create more powerful sockets (add
			right and middle button actions, use predefined kvirc sockets etc...)
		@seealso:
			[doc:escape_sequences]the escape sequences documentation[/doc]
	*/

	ENTER_STACK_FRAME(c,"$fmtsocket");

	KviStr * pLinkText = params->safeFirst();
	KviStr * pCommand  = params->safeNext();
	KviStr * pTooltip  = params->safeNext();

	if(pLinkText->isEmpty())
	{
		c->warning(__tr2qs("No socket text specified"));
		return c->leaveStackFrame();
	}

	if(pCommand->isEmpty())
	{
		c->warning(__tr2qs("No command specified"));
		return c->leaveStackFrame();
	}

	KviStr szCmd(KviStr::Format,"[!dbl]%s",pCommand->ptr());
	if(pTooltip->hasData())szCmd.append(KviStr::Format,"[!txt]%s",pTooltip->ptr());

	buffer.append(KviStr::Format,"\r!%s\r%s\r",szCmd.ptr(),pLinkText->ptr());

	return c->leaveStackFrame();

}

bool KviUserParser::parseFnc_CR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: cr
		@type:
			function
		@title:
			$cr
		@short:
			Returns a carriage return character
		@syntax:
			$cr
		@description:
			Returns a carriage return character
		@seealso:
			[fnc]$lf[/fnc],[fnc]$ascii[/fnc],[fnc]$char[/fnc]
	*/

	buffer.append("\r");
	return true;
}


bool KviUserParser::parseFnc_LF(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: lf
		@type:
			function
		@title:
			$lf
		@short:
			Returns a line feed character
		@syntax:
			$lf
		@description:
			Returns a line feed character
		@seealso:
			[fnc]$cr[/fnc],[fnc]$ascii[/fnc],[fnc]$char[/fnc]
	*/

	buffer.append("\n");
	return true;
}

bool KviUserParser::parseFnc_CHAR(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: char
		@type:
			function
		@title:
			$char
		@short:
			Returns a character specified by ascii code
		@syntax:
			$char(<ascii_code>)
		@description:
			Returns a character corresponding to the ASCII code <ascii_code>.[br]
			This function can not return NUL character (ASCII 0). Basically
			you should never need it: if you do, drop me a mail.[br]
			If the <ascii_code> is not a valid ASCII code (or is 0), this function returns
			an empty string.[br]
		@seealso:
			[fnc]$cr[/fnc],[fnc]$lf[/fnc],[fnc]$char[/fnc]
	*/

	KviStr * pCode = params->safeFirst();
	bool bOk;
	int iCode = pCode->toInt(&bOk) % 256;
	if(bOk && (iCode != 0))buffer.append(KviStr::Format,"%c",iCode);
	return true;
}

bool KviUserParser::parseFnc_ASCII(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: ascii
		@type:
			function
		@title:
			$ascii
		@short:
			Returns the ASCII code of a character
		@syntax:
			$char(<char>)
		@description:
			Returns the ASCII code corresponding to the given character.[br]
		@seealso:
			[fnc]$cr[/fnc],[fnc]$lf[/fnc],[fnc]$char[/fnc]
	*/

	KviStr * pChar = params->safeFirst();
	if(pChar->hasData())
	{
		unsigned char c = (unsigned char) *(pChar->ptr());
		buffer.append(KviStr::Format,"%u",c);
	}
	return true;
}

bool KviUserParser::parseFnc_K(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: k
		@type:
			function
		@title:
			$k
		@short:
			Returns the COLOR mIRC control character
		@syntax:
			$k(<foreground>[,<background>])
			$k
		@description:
			Returns the COLOR mIRC control character (CTRL+K).[br]
			If <foreground> and <background> are passed, a standard mIRC
			color escape is returned.[br]
		@seealso:
			[fnc]$b[/fnc]
	*/

	KviStr * pCode1 = params->safeFirst();
	KviStr * pCode2 = params->safeNext();
	buffer.append(KVI_TEXT_COLOR);
	if(pCode1->hasData())
	{
		buffer.append(pCode1->ptr());
		if(pCode2->hasData())
		{
			buffer.append(',');
			buffer.append(pCode2->ptr());
		}
	}
	return true;
}

bool KviUserParser::parseFnc_B(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: b
		@type:
			function
		@title:
			$b
		@short:
			Returns the BOLD mIRC control character
		@syntax:
			$b
		@description:
			Returns the BOLD mIRC control character (CTRL+B).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$u[/fnc]
	*/

	buffer.append(KVI_TEXT_BOLD);
	return true;
}

bool KviUserParser::parseFnc_U(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: u
		@type:
			function
		@title:
			$u
		@short:
			Returns the UNDERLINE mIRC control character
		@syntax:
			$u
		@description:
			Returns the UNDERLINE mIRC control character (CTRL+U).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$r[/fnc],[fnc]$o[/fnc]
	*/

	buffer.append(KVI_TEXT_UNDERLINE);
	return true;
}

bool KviUserParser::parseFnc_R(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: r
		@type:
			function
		@title:
			$r
		@short:
			Returns the REVERSE mIRC control character
		@syntax:
			$u
		@description:
			Returns the REVERSE mIRC control character (CTRL+R).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$u[/fnc],[fnc]$o[/fnc]
	*/

	buffer.append(KVI_TEXT_REVERSE);
	return true;
}

bool KviUserParser::parseFnc_O(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: o
		@type:
			function
		@title:
			$o
		@short:
			Returns the RESET mIRC control character
		@syntax:
			$o
		@description:
			Returns the RESET mIRC control character (CTRL+O).[br]
		@seealso:
			[fnc]$k[/fnc],[fnc]$b[/fnc],[fnc]$u[/fnc],[fnc]$r[/fnc]
	*/

	buffer.append(KVI_TEXT_RESET);
	return true;
}

bool KviUserParser::parseFnc_SW(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: sw
		@type:
			function
		@title:
			$sw
		@short:
			Returns the value of a switch for an alias
		@syntax:
			$sw(<letter>)
		@description:
			This function is valid and useful only in aliases.
			It allows an alias to handle switches just like any other
			KVIrc command. If a switch in the form -<letter> was
			passed to the current alias then $sw(<letter>) 
			returns 1 (true). If a switch in the form -<letter>=<value>
			was passed to the current alias then <value> is returned.
			If the switch was not present at all then this function
			returns an empty string (that evaluates to false in an expression).
			A warning is printed if this function is used non-alias code.
		@examples:
			alias(test){
				if($sw(a))echo "Switch -a was passed"
				%x = $sw(x);
				if(%x)echo "Switch -x=%x was passed"
			}
			test -a
			test -x
			test -a -x
			test -a -x=test
			test -a=10 -x=test
	*/

	KviPtrList<KviStr> * l = c->aliasSwitchList();
	if(!l)
	{
		c->warning(__tr2qs("$sw should be called only in aliases"));
		return true;
	}
	KviStr * pSep = params->safeFirst();
	char letter = *(pSep->ptr());
	if(!letter)
	{
		// doh ?
		return true;
	}
	
	for(KviStr * s=l->first();s;s=l->next())
	{
		if(tolower(*(s->ptr())) == tolower(letter))
		{
			if(s->len() < 2)
			{
				// valueless
				buffer.append('1');
				return true;
			}
			// with a value
			KviStr tmp = *s;
			tmp.cutLeft(2);
			buffer.append(tmp);
			return true;
		}
	}

	return true;
}


bool KviUserParser::parseFnc_SPLIT(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: split
		@type:
			function
		@title:
			$split
		@short:
			Splits a string to an array
		@syntax:
			$split(<separator>,<string>[,<flags>])
		@description:
			Splits the <string> by <separator> and returns an array of substrings.[br]
			<flags> may be a combination of the characters s w and r.[br]
			If s is specified, <separator> matching is case sensitive, otherwise is case insensitive.[br]
			If w is specified, <separator> is treated as a wildcard-type regular expression
			(with * and ? wildcars).[br]
			If r is specified, <separator> is treated as a extended-type regular expression
			(with character classes, special escapes etc..).[br]
			If both w and r are specified w takes precedence.[br]
			If none of w and r are specified <separator> is treated as a simple string to be matched.[br]
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
		@examples:
			[example]
				[comment]# Split the fields[/comment]
				%test[] = $split(!,"Field0!Field1!Field2!Field3!!Field5")
				echo %test[]
				%i = 0
				[cmd]while[/cmd](%i < %test[]#)
				{
					[cmd]echo[/cmd] "Field %i: %test[%i]"
					%i++;
				}
			[/example]
			Regexp splitting:
			[example]
				%Test[] = $split("[ ]*[0-9][0-9]*-","AllOfThem: 1-Balboy 2-Pragma 3-Iakkolo 4-Crocodile",r)
				echo %Test[]
				%Test[] = $split("Y*H","hihiYeaHhohohoyeahYepYEAHhi",sw)
				echo %Test[]
			[/example]
			If used in "non-array" context it returns just a comma separated list of substrings:[br]
			[example]
				[cmd]echo[/cmd] $split("[ ]*","Condense spaces and change &nbsp; &nbsp; all &nbsp; &nbsp; &nbsp; it in commas",r)
			[/example]
	*/

	KviStr * pSep = params->safeFirst();
	KviStr * pDat = params->safeNext();
	KviStr * pFla = params->safeNext();

	c->beginListArrayOrDictionaryReturnIdentifier();

	if(pSep->isEmpty())
	{
		c->addListArrayOrDictionaryReturnValue(0,pDat->ptr(),buffer);
		return true;
	}

	bool bWild = pFla->contains('w');
	bool bContainsR = pFla->contains('r');
	bool bCaseSensitive = pFla->contains('s');

	int id = 0;
	char * begin = pDat->ptr();

	if(bContainsR || bWild)
	{
		QRegExp re(pSep->ptr(),bCaseSensitive,bWild);
		QString szSearch = pDat->ptr();
		
		__range_valid(szSearch.length() == pDat->len());
		int iMatch = 0;
		int iOldMatch;
#if QT_VERSION >= 300
		while((iMatch != -1) && *begin)
		{
			iOldMatch = iMatch;
			iMatch = re.search(szSearch,iMatch);
			if(iMatch != -1)
			{
				int len = re.matchedLength();
				if((len == 0) && (iOldMatch == iMatch))iMatch++; // safety measure for empty string matching

				char * ptr = pDat->ptr() + iMatch;
				char tmp = *ptr;
				*ptr = '\0';
				c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
				*ptr = tmp;

				id++;
				iMatch += len;
				begin = pDat->ptr() + iMatch;
			}
		}
#else
		int len;
		while((iMatch != -1) && *begin)
		{
			iOldMatch = iMatch;
			iMatch = re.match(szSearch,iMatch,&len,false);
			if(iMatch != -1)
			{
				if((len == 0) && (iOldMatch == iMatch))iMatch++; // safety measure for empty string matching
				char * ptr = pDat->ptr() + iMatch;
				char tmp = *ptr;
				*ptr = '\0';
				c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
				*ptr = tmp;

				id++;
				iMatch += len;
				begin = pDat->ptr() + iMatch;
			}
		}
#endif
		if(*begin)c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
		return true;
	}

	char chr = *(pSep->ptr());
	char * p = pDat->ptr();

	if(pSep->len() == 1)
	{
		if(bCaseSensitive)
		{
			while(*p)
			{
				if(*p == chr)
				{
					*p = '\0';
					c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
					*p = chr;
					id++;
					p++;
					begin = p;
				} else p++;
			}
		} else {
			char lowchr = tolower(chr);
			while(*p)
			{
				if(tolower(*p) == lowchr)
				{
					*p = '\0';
					c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
					*p = chr;
					id++;
					p++;
					begin = p;
				} else p++;
			}
		}
	} else {
		int len = pSep->len();
		while(*p)
		{
			if(bCaseSensitive)
			{
				if(*p == chr)
				{
					if(kvi_strEqualCSN(p,pSep->ptr(),len))
					{
						*p = '\0';
						c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
						*p = chr;
						id++;
						p += len;
						begin = p;
					} else p++;
				} else p++;
			} else {
				char lowchr = tolower(chr);
				if(tolower(*p) == lowchr)
				{
					if(kvi_strEqualCIN(p,pSep->ptr(),len))
					{
						*p = '\0';
						c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
						*p = chr;
						id++;
						p += len;
						begin = p;
					} else p++;
				} else p++;
			}
		}
	}
	if(begin != p)c->addListArrayOrDictionaryReturnValue(id,begin,buffer);
	return true;
}


bool KviUserParser::parseFnc_FEATURES(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
	/*
		@doc: features
		@type:
			function
		@title:
			$features
		@short:
			Returns the features that KVIrc supports
		@syntax:
			$features
			$features(<test_feature>)
		@description:
			The parameterless form returns an array of feature descripton strings that this KVIrc executable supports.[br]
			This function is useful when some part of your script depends on
			an optional KVIrc feature (like SSL support or IPV6 support).[br]
			The returned value may be assigned to a dictionary too: it will be used to simulate an array.[br]
			The form with the [test_feature] parameter returns 1 if and only if [test_feature] is available.[br]
		@examples:
			[example]
			%myfeats[] = $features
			echo %myfeats[]
			%i = %myfeats[]#
			[cmd]while[/cmd](%i > 0)
			{
				[cmd]echo[/cmd] "Supporting feature %myfeats[%i]"
				%i--;
			}
			[/example]
			You can test for a specific feature in the following way:
			[example]
			[cmd]if[/cmd]($features("SSL"))[cmd]echo[/cmd] "Yes! SSL is available";
			[/example]
			If used in "non-array" context it returns just a comma separated list of entries:[br]
			[example]
			[cmd]echo[/cmd] $features
			[/example]
		@seealso:
			[fnc]$version[/fnc]
	*/

	KviStr * p = params->safeFirst();
	bool bHasParam = p->hasData();

	static const char * feature_array[]=
	{
		"IRC",
#ifdef COMPILE_IPV6_SUPPORT
		"IPv6",
#endif
#ifdef COMPILE_CRYPT_SUPPORT
		"Crypt",
#endif
#ifdef COMPILE_SSL_SUPPORT
		"SSL",
#endif
#ifdef COMPILE_LOCALE_STUFF
		"Locale",
#endif
#ifdef COMPILE_GET_INTERFACE_ADDRESS
		"IfAddr",
#endif
#ifndef COMPILE_NO_IPC
		"IPC",
#endif
#ifdef COMPILE_KDE_SUPPORT
		"KDE",
#endif
#ifdef COMPILE_OSS_SUPPORT
		"OSS",
#endif
#ifdef COMPILE_ARTS_SUPPORT
		"ARTS",
#endif
#ifdef COMPILE_ESD_SUPPORT
		"ESD",
#endif
#ifdef COMPILE_AUDIOFILE_SUPPORT
		"Audiofile",
#endif
#ifdef COMPILE_PSEUDO_TRANSPARENCY
		"Transparency",
#endif
#ifdef COMPILE_SPLASH_SCREEN
		"Splash",
#endif
#ifdef COMPILE_ix86_ASM
		"ix86-ASM",
#endif
#ifdef COMPILE_SCRIPTTOOLBAR
		"ScriptToolBar",
#endif // COMPILE_SCRIPTTOOLBAR
#if QT_VERSION >= 300
		"Qt3",
#endif
		0
	};

	if(bHasParam)
	{
		for(int i=0;feature_array[i];i++)
		{
			if(kvi_strEqualCI(feature_array[i],p->ptr()))
			{
				buffer.append('1');
				return true;
			}
		}
		buffer.append('0');
	} else {
		c->beginListArrayOrDictionaryReturnIdentifier();
		int id = 0;
		for(int i=0;feature_array[i];i++)
		{
			c->addListArrayOrDictionaryReturnValue(id++,feature_array[i],buffer);
		}
	}

	return true;
}

bool KviUserParser::parseFnc_VERSION(KviCommand *c,KviParameterList * params,KviStr &buffer)
{
    	/*
		@doc: version
		@type:
			function
		@title:
			$version
		@short:
			Returns informations about the version of KVIrc.[br]
		@syntax:
			$version
			$version(<type>)
		@description:
			Returns informations about the version of the currently running KVIrc.[br]
			Type can be one of:[br]
			[ul]
			[li]v: return the current numeric version[/li]
			[li]r: return the release name[/li]
			[/ul]
			If <type> is omitted then v is assumed.[br]
		@examples:
			[example]echo $version $version(r)[/example]
		@seealso:
			[fnc]$features[/fnc]
	*/

	const char * cc = params->safeFirstParam();
	if(!cc)buffer.append(VERSION);
	else switch(*cc)
	{
		case 'r': buffer.append(KVI_RELEASE_NAME); break;
		default: buffer.append(VERSION); break;
	}
	return true;
}

bool KviUserParser::parseFnc_ISMAINWINDOWMINIMIZED(KviCommand *c, KviParameterList *params, KviStr &buffer)
{
		/*
		@doc: isMainWindowMinimized
		@type:
			function
		@title:
			$isMainWindowMinimized
		@short:
			Checks if main KVirc window is minimized
		@syntax:
			$isMainWindowMinimized
		@description:
			Returns 1 if main KVirc window is minimized, 0 otherwise.
		*/

	KviFrame *frm = c->window()->frame();

	if(frm->isMinimized() || !frm->isActiveWindow())
		buffer.append('1');
	else
		buffer.append('0');

	return true;
}
