//
//   File : kvi_exprtree.cpp
//   Creation date : Sat Feb 27 1999 19:31:10 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
#define __KVIRC__
//#define _KVI_DEBUG_CHECK_RANGE_
#include "kvi_debug.h"

#include "kvi_uparser.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_command.h"

#include "kvi_exprtree.h"

// hack for pow()
long my_funky_pow(long x,long y)
{
	long total = 1;
	for(int i=0;i<y;i++)total = total*x;
	return total;
}

KviExprTree::KviExprTree(NodeType t,long v)
{
	m_type   = t;
	m_left   = 0;
	m_right  = 0;
	m_value  = v;
	m_parent = 0;
}

KviExprTree::KviExprTree(const QString &str,long lVal)
{
	m_type = NumericOperand;
	m_left = 0;
	m_right = 0;
	m_value = lVal;
	m_parent = 0;
	m_string = str;
}

KviExprTree::KviExprTree(const QString &str)
{
	m_string = str;
	m_type = StringOperand;
	m_left = 0;
	m_right = 0;
	m_value = 0;
	m_parent = 0;
}

KviExprTree::~KviExprTree()
{
	if(m_left)delete m_left;
	if(m_right)delete m_right;
}

void KviExprTree::setLeft(KviExprTree * node)
{
	__range_valid(node);
	m_left = node;
	node->m_parent = this;
}

void KviExprTree::setRight(KviExprTree * node)
{
	__range_valid(node);
	m_right = node;
	node->m_parent = this;
}

long KviExprTree::calculate(KviCommand *c)
{
	if(m_type == NumericOperand)return m_value;
	if(m_type == StringOperand)return (m_string.length());

	if(c->hasError())return -1;

	__range_valid(m_left);
	if(m_type == UnaryOperator)
	{
		switch(m_value){
			case OP_NOT:    return ! m_left->calculate(c); break;
			case OP_NEGATE: return - m_left->calculate(c); break;
			case OP_BITNOT: return ~ m_left->calculate(c); break;
			default:
				__range_invalid(true); //error
			break;
		}
	}

	__range_valid(m_right);
	if((m_value < 10) || (m_value > 17))
	{
		// Numeric operators...
		switch(m_value){
			case OP_BITAND:
				return (m_left->calculate(c) & m_right->calculate(c));
			break;
			case OP_BITOR:
				return (m_left->calculate(c) | m_right->calculate(c));
			break;
			case OP_SHR:
				return (m_left->calculate(c) >> m_right->calculate(c));
			break;
			case OP_SHL:
				return (m_left->calculate(c) << m_right->calculate(c));
			break;
			case OP_POWER:
				return (long)my_funky_pow(m_left->calculate(c),m_right->calculate(c));
			break;
			case OP_MUL:
				return (m_left->calculate(c) * m_right->calculate(c));
			break;
			case OP_DIV: {
				long temp = m_right->calculate(c);
				if(temp == 0){
					c->error(KviError_divisionByZero);
					return -1;
				}
				return (m_left->calculate(c) / temp);
			}
			break;
			case OP_MOD:{
				long temp = m_right->calculate(c);
				if(temp == 0){
					c->error(KviError_divisionByZero);
					return -1;
				}
				return (m_left->calculate(c) % temp);
			}
			break;
			case OP_ADD:
				return (m_left->calculate(c) + m_right->calculate(c));
			break;
			case OP_SUB:
				return (m_left->calculate(c) - m_right->calculate(c));
			break;
			case OP_AND:
				return m_left->calculate(c) ? m_right->calculate(c) : 0;
			break;
			case OP_OR:
				return m_left->calculate(c) ? 1 : m_right->calculate(c);
			break;
			default:
				__range_invalid(true); //tragedy :)
			break;
		}
	}

	if((m_left->m_type == StringOperand) && (m_right->m_type == StringOperand))
	{
		// String operator for string operands
		switch(m_value){
			case OP_GT:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)>0);
			break;
			case OP_LT:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)<0);
			break;
			case OP_EQ:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)==0);
			break;
			case OP_CSEQ:
				return (KviQString::cmpCS(m_left->m_string,m_right->m_string)==0);
			break;
			case OP_GE:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)>=0);
			break;
			case OP_LE:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)<=0);
			break;
			case OP_NE:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)!=0);
			break;
			case OP_CSNE:
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)!=0);
			break;
			default:
				__range_invalid(true);
			break;
		}
	}

	// Numeric operator for numeric/string operands
	switch(m_value)
	{
		case OP_GT:
			return (m_left->calculate(c) > m_right->calculate(c));
		break;
		case OP_LT:
			return (m_left->calculate(c) < m_right->calculate(c));
		break;
		case OP_EQ:
			if((m_left->m_type == StringOperand) || (m_right->m_type == StringOperand))
			{
				// at least one string
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string)==0);
			} else {
				// both numeric
				return (m_left->calculate(c) == m_right->calculate(c));
			}
		break;
		case OP_CSEQ:
			if((m_left->m_type == StringOperand) || (m_right->m_type == StringOperand))
			{
				// at least one string
				return (KviQString::cmpCS(m_left->m_string,m_right->m_string)==0);
			} else {
				// both numeric
				return (m_left->calculate(c) == m_right->calculate(c));
			}
		break;
		case OP_GE:
			return (m_left->calculate(c) >= m_right->calculate(c));
		break;
		case OP_LE:
			return (m_left->calculate(c) <= m_right->calculate(c));
		break;
		case OP_NE:
			if((m_left->m_type == StringOperand) || (m_right->m_type == StringOperand))
			{
				// at least one string
				return (KviQString::cmpCI(m_left->m_string,m_right->m_string) != 0);
			} else {
				// both numeric
				return (m_left->calculate(c) != m_right->calculate(c));
			}
		break;
		case OP_CSNE:
			if((m_left->m_type == StringOperand) || (m_right->m_type == StringOperand))
			{
				// at least one string
				return (KviQString::cmpCS(m_left->m_string,m_right->m_string) != 0);
			} else {
				// both numeric
				return (m_left->calculate(c) != m_right->calculate(c));
			}
		break;
		default:
			__range_invalid(true);
		break;
	}

	return -1; //Newer here...
}

//void KviExprTree::dump()
//{
//	if(m_left)m_left->dump();
//
//	switch(m_type){
//		case BinaryOperator:
//			debug("Node : Binary operator %d",m_value);
//		break;
//		case UnaryOperator:
//			debug("Node : Unary operator %d",m_value);
//		break;
//		default:
//			debug("Node : Operand %d",m_value);	
//		break;
//	}
//
//	if(m_right)m_right->dump();
//}

KviExprTree * KviExprTree::parentWithPrecedenceLowerThan(long operValue)
{
	__range_valid(m_type == BinaryOperator); //?
	if(operator_precedence(m_value) > operator_precedence(operValue))return this;
	if(m_parent == 0)return 0;
	return m_parent->parentWithPrecedenceLowerThan(operValue);
}

long KviExprTree::firstBinaryOperator()
{
	if(m_type == BinaryOperator)return m_value;
	__range_valid(m_type == UnaryOperator);
	return m_left->firstBinaryOperator();
}
