//=============================================================================
//
//   File : kvi_topicw.cpp
//   Creation date : Fri Aug 4 2000 12:09:21 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2005 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#include "kvi_topicw.h"
#include "kvi_options.h"
#include "kvi_mirccntrl.h"
#include "kvi_locale.h"
#include "kvi_defaults.h"
#include "kvi_settings.h"
#include "kvi_iconmanager.h"
#include "kvi_texticonmanager.h"
#include "kvi_app.h"
#include "kvi_colorwin.h"
#include "kvi_texticonwin.h"
#include "kvi_window.h"

extern KviTextIconWindow * g_pTextIconWindow;
extern KviColorWindow * g_pColorWindow;


#include <qpainter.h>
#include <qfontmetrics.h>
#include <qtooltip.h>
#include <qcombobox.h>
#include <qlineedit.h>
#include <qregexp.h>
#include <qclipboard.h>
#include <qpopupmenu.h>


// FIXME: #warning "The combo should disappear when it looses focus!...(how to do it ?)"

#ifdef COMPILE_PSEUDO_TRANSPARENCY
	extern QPixmap * g_pShadedChildGlobalDesktopBackground;

#endif


extern QStringList  * g_pRecentTopicList;

KviTopicWidget::KviTopicWidget(QWidget * par,const char * name)
: QFrame(par,name)

{
	setFrameStyle(QFrame::Sunken | QFrame::StyledPanel);
	setFont(KVI_OPTION_FONT(KviOption_fontLabel));
	m_pComboBox = 0;
	m_pAccept = 0;
	m_pDiscard = 0;
	m_pContextPopup = 0;
//	m_pIconButton = new QToolButton(this);
	setBackgroundMode(QWidget::NoBackground);
	reset();
}

KviTopicWidget::~KviTopicWidget()
{
}

void KviTopicWidget::reset()
{
	QToolTip::remove(this);
	m_szTopic = __tr2qs("Unknown");
	QToolTip::add(this,__tr2qs("No topic message has been received from the server yet"));
	m_szSetAt = "";
	m_szSetBy = "";
	update();
}

void KviTopicWidget::applyOptions()
{
	setFont(KVI_OPTION_FONT(KviOption_fontLabel));
	if(m_pComboBox)m_pComboBox->setFont(KVI_OPTION_FONT(KviOption_fontLabel));
	resizeEvent(0);
}

#define KVI_LABEL_DEF_BACK 100
#define KVI_LABEL_DEF_FORE 101

void KviTopicWidget::drawContents(QPainter *p)
{
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	if(g_pShadedChildGlobalDesktopBackground)
	{
		QPoint pnt = mapToGlobal(contentsRect().topLeft());
		p->drawTiledPixmap(contentsRect(),*g_pShadedChildGlobalDesktopBackground,pnt);
	} else {
#endif
		if(KVI_OPTION_PIXMAP(KviOption_pixmapLabelBackground).pixmap())
		{
			p->drawTiledPixmap(contentsRect(),*(KVI_OPTION_PIXMAP(KviOption_pixmapLabelBackground).pixmap()));
		} else {
			p->fillRect(contentsRect(),KVI_OPTION_COLOR(KviOption_colorLabelBackground));
		}
#ifdef COMPILE_PSEUDO_TRANSPARENCY
	}
#endif

	QFontMetrics fm(p->font());

	bool curBold      = false;
	bool curUnderline = false;
	unsigned char curFore      = KVI_LABEL_DEF_FORE; //default fore
	unsigned char curBack      = KVI_LABEL_DEF_BACK; //default back
	int baseline = ((height() + fm.ascent() - fm.descent() + 1) >> 1);

	int curX = contentsRect().x() + 2; //2 is the margin

	unsigned int idx = 0;

	while((idx < m_szTopic.length()) && (curX < width()))
	{
		unsigned short c = m_szTopic[(int)idx].unicode();

		unsigned int start = idx;

		while((idx < m_szTopic.length()) &&
				(c != KVI_TEXT_COLOR) &&
				(c != KVI_TEXT_BOLD) &&
				(c != KVI_TEXT_UNDERLINE) &&
				(c != KVI_TEXT_REVERSE) &&
				(c != KVI_TEXT_RESET) &&
				(c != KVI_TEXT_ICON)
			)
		{
			idx++;
			c = m_szTopic[(int)idx].unicode();
		}

		int len = idx - start;
		int wdth;

		if(len > 0)
		{
			QString szText = m_szTopic.mid(start,len);

			wdth = fm.width(szText);
	
			if(curFore == KVI_LABEL_DEF_FORE)
			{
				p->setPen(KVI_OPTION_COLOR(KviOption_colorLabelForeground));
			} else {
				if(curFore > 16)p->setPen(KVI_OPTION_COLOR(KviOption_colorLabelBackground));
				else p->setPen(KVI_OPTION_MIRCCOLOR(curFore));
			}
		
			if(curBack != KVI_LABEL_DEF_BACK)
			{
				if(curBack > 16)
				{
					p->fillRect(curX,contentsRect().y() + 2,wdth,contentsRect().height() - 4,
						KVI_OPTION_COLOR(KviOption_colorLabelForeground));
				} else {
					p->fillRect(curX,contentsRect().y() + 2,wdth,contentsRect().height() - 4,
						KVI_OPTION_MIRCCOLOR(curBack));
				}
			}

			p->drawText(curX,baseline,szText,0,len);
	
			if(curBold)p->drawText(curX+1,baseline,szText,0,len);
			if(curUnderline)
			{
				p->drawLine(curX,baseline + 1,curX+wdth,baseline + 1);
			}
		} else {
			wdth = 0;
		}


		curX += wdth;

		switch(c)
		{
			case KVI_TEXT_BOLD: curBold = !curBold; ++idx; break;
			case KVI_TEXT_UNDERLINE: curUnderline = !curUnderline; ++idx; break;
			case KVI_TEXT_REVERSE:
				{
					char auxBack = curBack;
					curBack = curFore;
					curFore = auxBack;
				}
				++idx;
			break;
			case KVI_TEXT_RESET:
				curFore = KVI_LABEL_DEF_FORE;
				curBack = KVI_LABEL_DEF_BACK;
				curBold = false;
				curUnderline = false;
				++idx;
			break;
			case KVI_TEXT_COLOR:
			{
				++idx;
				unsigned char fore;
				unsigned char back;
				idx = getUnicodeColorBytes(m_szTopic,idx,&fore,&back);
				if(fore != KVI_NOCHANGE)
				{
					curFore = fore;
					if(back != KVI_NOCHANGE)curBack = back;
				} else {
					// only a CTRL+K
					curBack = KVI_LABEL_DEF_BACK;
					curFore = KVI_LABEL_DEF_FORE;
				}
			}
			break;
			case KVI_TEXT_ICON:
			{
				++idx;

				unsigned int icoStart = idx;
				while((idx < m_szTopic.length()) && (m_szTopic[(int)idx].unicode() > 32))idx++;

				KviStr lookupString = m_szTopic.mid(icoStart,idx - icoStart);

				int icon_id = g_pTextIconManager->lookupTextIcon(lookupString.ptr());
				if(icon_id >= 0)
				{
					QPixmap * pigzmap = g_pIconManager->getSmallIcon(icon_id);
					p->drawPixmap(curX,(baseline + 2) - pigzmap->height(),*(pigzmap));
					curX += pigzmap->width();
				} else {
					idx = icoStart;
				}
			}
			break;
		}
	}
}

void KviTopicWidget::setTopic(const QString & topic)
{
	m_szTopic = topic;
	bool bFound = false;
	for(QStringList::Iterator it=g_pRecentTopicList->begin();it != g_pRecentTopicList->end(); ++it)
	{
		if(*it == m_szTopic)
		{
			bFound = true;
			break;
		}
	}
	if(!bFound && (!m_szTopic.isEmpty()))
	{
		if(g_pRecentTopicList->count() >= KVI_RECENT_TOPIC_ENTRIES)
		{
			g_pRecentTopicList->remove(g_pRecentTopicList->begin());
		}
		g_pRecentTopicList->append(m_szTopic);
	}
	updateToolTip();
	update();
}

void KviTopicWidget::setTopicSetBy(const QString & setBy)
{
	m_szSetBy = setBy;
	updateToolTip();
}

void KviTopicWidget::setTopicSetAt(const QString & setAt)
{
	m_szSetAt = setAt;
	updateToolTip();
}

void KviTopicWidget::updateToolTip()
{
	QToolTip::remove(this);

	QString txt = "<html>" \
			"<body>" \
				"<table width=\"100%\">";

	if(!m_szTopic.isEmpty())
	{
		txt +=          "<tr><td bgcolor=\"#303030\">" \
							"<font color=\"#FFFFFF\"><b>";
		txt += __tr2qs("Channel topic:");
		txt +=              "</b></font>" \
						"</td></tr>";

		txt +=          "<tr><td><center><br>";

		QString tmp = KviMircCntrl::stripControlBytes(m_szTopic);
		tmp.replace(QChar('&'),"&amp;");
		tmp.replace(QChar('<'),"&lt;");
		tmp.replace(QChar('>'),"&gt;");
		
		txt += tmp;
		txt +=          "</center><br><br></td></tr>";
		
		if(!m_szSetBy.isEmpty())
		{
			txt +=      "<tr><td bgcolor=\"#D0D0D0\">";
			txt +=       __tr2qs("Set by") + " <b>" + m_szSetBy + "</b>";
			txt +=      "</td></tr>";

			if(!m_szSetAt.isEmpty())
			{
				txt +=      "<tr><td bgcolor=\"#D0D0D0\">";
				txt +=       __tr2qs("Set on") + " <b>" + m_szSetAt + "</b>";
				txt +=      "</td></tr>";
			}
		}

		txt +=          "<tr><td>";
		txt += __tr2qs("Double-click to edit...");
		txt +=          "</td></tr>";

	} else {
		txt +=          "<tr><td>";
		txt += __tr2qs("No topic is set");
		txt +=          "</td></tr>";
		txt +=          "<tr><td>";
		txt += __tr2qs("Double-click to set...");
		txt +=          "</td></tr>";
	}

	txt += "</table>" \
		"</body>" \
	"<html>";

	QToolTip::add(this,txt);
}

QSize KviTopicWidget::sizeHint() const
{
	QFontMetrics fm(font());
	int hght = fm.lineSpacing() + (frameWidth() << 1) + 4;
	int baseline = ((hght + fm.ascent() - fm.descent() + 1) >> 1);
	if(baseline < 16)hght += (16 - baseline);
	return QSize(width(),hght);
}

void KviTopicWidget::mouseDoubleClickEvent(QMouseEvent *)
{
	if(isEnabled() && (m_pComboBox == 0))
	{
		m_pComboBox = new QComboBox(true,this);
		m_pComboBox->setFont(font());
		m_pComboBox->setGeometry(0,0,width() - (height() << 1),height());
		if(m_pComboBox->lineEdit())
			m_pComboBox->lineEdit()->installEventFilter(this);
		m_pAccept = new QPushButton(this);
		m_pAccept->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_ACCEPT)));
		m_pAccept->setGeometry(width() - (height() << 1),0,height(),height());
		m_pAccept->show();
		QToolTip::add(m_pAccept,__tr2qs("Commit Changes"));
		connect(m_pAccept,SIGNAL(clicked()),this,SLOT(acceptClicked()));
		m_pDiscard = new QPushButton(this);
		m_pDiscard->setPixmap(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_DISCARD)));
		m_pDiscard->setGeometry(width() - height(),0,height(),height());
		QToolTip::add(m_pDiscard,__tr2qs("Discard Changes"));
		m_pDiscard->show();
		connect(m_pDiscard,SIGNAL(clicked()),this,SLOT(discardClicked()));
		m_pComboBox->insertStringList(*g_pRecentTopicList);
		m_pComboBox->lineEdit()->setText(m_szTopic);
		m_pComboBox->lineEdit()->selectAll();
		m_pComboBox->show();
		m_pComboBox->setFocus();
	}
}

void KviTopicWidget::mousePressEvent(QMouseEvent * e)
{
	if(!(e->button() & RightButton))return;
	if(!m_pContextPopup)
	{
		m_pContextPopup = new QPopupMenu(this);
		connect(m_pContextPopup,SIGNAL(aboutToShow()),this,SLOT(contextPopupAboutToShow()));
	}
	m_pContextPopup->popup(mapToGlobal(e->pos()));
}

void KviTopicWidget::contextPopupAboutToShow()
{
	if(!m_pContextPopup)return; // hm ?
	m_pContextPopup->clear();
	m_pContextPopup->insertItem(*(g_pIconManager->getSmallIcon(KVI_SMALLICON_COPY)),__tr2qs("Copy to clipboard"),this,SLOT(copy()));
}

void KviTopicWidget::copy()
{
	QClipboard * c = QApplication::clipboard();
	if(!c)return;
	if(c->supportsSelection())c->setText(m_szTopic,QClipboard::Selection);
	c->setText(m_szTopic,QClipboard::Clipboard);
}

bool KviTopicWidget::eventFilter(QObject *o,QEvent *e)
{
	if(e->type() == QEvent::KeyPress)
	{
		return handleKeyPressEvent((QKeyEvent *)e);
	}
	return false;
}

bool KviTopicWidget::handleKeyPressEvent(QKeyEvent * e)
{
	if(!m_pComboBox->lineEdit())return false;
	if(!m_pComboBox->lineEdit()->isVisible())return false;

	if(!(e->state() & ControlButton))
	{
		switch(e->key())
		{
			case Qt::Key_Enter:
			case Qt::Key_Return:
				acceptClicked();
				return true;
				break;
			case Qt::Key_Escape:
				deactivate();
				return true;
				break;
			default:
				return false;
				break;
		}
	}

	// control is pressed
	switch(e->key())
	{
		case Qt::Key_K:
			{
				insertChar(KVI_TEXT_COLOR);
				int xPos = 24;
				if(xPos > 24)xPos-=24;
				if(!g_pColorWindow)g_pColorWindow = new KviColorWindow();
				if(xPos+g_pColorWindow->width() > width())xPos = width()-(g_pColorWindow->width()+2);
				g_pColorWindow->move(mapToGlobal(QPoint(xPos,-35)));
				g_pColorWindow->popup(m_pComboBox->lineEdit());
				return true;
			}
			break;
		case Qt::Key_B:
			insertChar(KVI_TEXT_BOLD);
			return true;
			break;
		case Qt::Key_O:
			insertChar(KVI_TEXT_RESET);
			return true;
			break;
		case Qt::Key_U:
			insertChar(KVI_TEXT_UNDERLINE);
			return true;
			break;
		case Qt::Key_R:
			insertChar(KVI_TEXT_REVERSE);
			return true;
			break;
		case Qt::Key_I:
			{
				insertChar(KVI_TEXT_ICON); // THE NEXT WORD IS AN ICON NAME
				int xPos = 24;
				if(xPos > 24)xPos-=24;
				if(!g_pTextIconWindow)g_pTextIconWindow = new KviTextIconWindow();
				if(xPos+g_pTextIconWindow->width() > width())xPos = width()-(g_pTextIconWindow->width()+2);
				g_pTextIconWindow->move(mapToGlobal(QPoint(xPos,-KVI_TEXTICON_WIN_HEIGHT)));
				g_pTextIconWindow->popup(m_pComboBox->lineEdit()); 
				return true;
			}
			break;
		default:
			return false;
			break;
	}

}

void KviTopicWidget::keyPressEvent(QKeyEvent * e)
{
	if(handleKeyPressEvent(e))
	{
		e->accept();
		return;
	}
}

void KviTopicWidget::resizeEvent(QResizeEvent *e)
{
	if(e)QFrame::resizeEvent(e);
	if(m_pComboBox)
	{
		m_pComboBox->setGeometry(0,0,width() - (height() << 1),height());
		m_pAccept->setGeometry(width() - (height() << 1),0,height(),height());
		m_pDiscard->setGeometry(width() - height(),0,height(),height());
	}
}

void KviTopicWidget::deactivate()
{
	if(m_pComboBox)
	{
		delete m_pComboBox;
		m_pComboBox = 0;
		delete m_pAccept;
		m_pAccept = 0;
		delete m_pDiscard;
		m_pDiscard = 0;
	}
	
	// try to find a KviWindow parent and give it the focus
	
	QObject * w = parent();
	while(w)
	{
		if(w->inherits("KviWindow"))
		{
			((KviWindow *)w)->setFocus();
			return;
		}
		w = w->parent();
	}

	// no KviWindow on the path
	w = parent();
	if(w)
	{
		if(w->inherits("QWidget"))
			((QWidget *)w)->setFocus();
	}
}

void KviTopicWidget::discardClicked()
{
	deactivate();
}

void KviTopicWidget::acceptClicked()
{
	QString tmp = m_pComboBox->currentText();
	if(tmp != m_szTopic)emit topicSelected(tmp);
	deactivate();
}

void KviTopicWidget::insertChar(QChar c)
{
	insertText(QString(c));
}

void KviTopicWidget::insertText(const QString &c)
{
	if(c.isEmpty())return;
	QLineEdit * e = m_pComboBox->lineEdit();
	if(!e)return;
	QString tmp = e->text();
	tmp.insert(e->cursorPosition(),c);
	e->setText(tmp);
	e->setCursorPosition(e->cursorPosition() + c.length());
}


#include "kvi_topicw.moc"
