//=============================================================================
//
//   File : kvi_kvs_parser_specialcommands.cpp
//   Creation date : Thu 06 Now 2003 14.14 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__


#include "kvi_kvs_parser.h"

#include "kvi_kvs_treenode.h"

#include "kvi_kvs_report.h"
#include "kvi_kvs_kernel.h"

#include "kvi_kvs_parser_macros.h"

#include "kvi_locale.h"

#include "kvi_cmdformatter.h"


KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandBreak()
{
	/*
		@doc: break
		@type:
			command
		@title:
			break
		@syntax:
			break
		@short:
			Interrupts an iteration loop
		@description:
			Interrupts an iteration loop like [cmd]while[/cmd].[br]
			This command always jumps out of a single code block.[br]
			If called outside an iteration loop , will act just like [cmd]halt[/cmd]
			has been called but has no additional semantics for events.[br]
	*/
	const QChar * pBegin = KVSP_curCharPointer; // FIXME: this is not accurate at all : it may be even the end of the cmd
	skipSpaces();
	if(!KVSP_curCharIsEndOfCommand)
	{
		warning(KVSP_curCharPointer,__tr2qs("Trailing garbage at the end of the break command: ignored"));
	}
	
	while(!KVSP_curCharIsEndOfCommand)KVSP_skipChar;
	if(!KVSP_curCharIsEndOfBuffer)KVSP_skipChar;
	return new KviKvsTreeNodeSpecialCommandBreak(pBegin);
}

KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandGlobal()
{
	/*
		@doc: global
		@type:
			command
		@title:
			global
		@syntax:
			global <variable_list>
		@keyterms:
			explicitly declaring global variables
		@short:
			Explicitly declares global variables
		@description:
			Declares a list of global variables.
			Once a variable has been declared as global
			it refers to the global kvirc instance for the scope of the script.
			Global variables are shared between scripts and keep their
			value until they are explicitly unset or kvirc quits.
			Please note that the commands typed in from the commandline
			already assume that all the variables are global: this command
			has no effect in that case.[br]
			The versions of KVIrc prior to 3.0.0 recognized global variables
			by the initial uppercase letter. This method is incoherent with the
			rest of the language and is deprecated (even if it is still supported
			for backward compatibility).
		@examples:
			global %a,%b,%c;
	*/
	
	while(KVSP_curCharUnicode == '%')
	{
		KVSP_skipChar;
		const QChar * pBegin = KVSP_curCharPointer;

	
		while((KVSP_curCharIsLetterOrNumber) || (KVSP_curCharUnicode == '_'))KVSP_skipChar;

		QString szIdentifier(pBegin,KVSP_curCharPointer - pBegin);

		if(!m_pGlobals)
		{
			m_pGlobals = new QDict<QString>(17,false);
			m_pGlobals->setAutoDelete(true);
		}
		m_pGlobals->replace(szIdentifier,new QString());
		
		skipSpaces();
		
		if(KVSP_curCharUnicode == ',')
		{
			KVSP_skipChar;
			skipSpaces();
		}
	}

	if(!KVSP_curCharIsEndOfCommand)
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'global' command needs a variable list"));
		error(KVSP_curCharPointer,__tr2qs("Found character %q (unicode %x) where a variable was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
		return 0;
	}

	if(!KVSP_curCharIsEndOfBuffer)KVSP_skipChar;

	return 0;
}


KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandWhile()
{
	/*
		@doc: while
		@type:
			command
		@title:
			while
		@syntax:
			while (<condition>) <command>
		@keyterms:
			iteration commands, flow control commands
		@short:
			Iteration command
		@description:
			Executes <command> while the <condition> evaluates
			to true (non zero result).[br]
			<command> may be either a single command or a block of commands.[br]
			It can contain the [cmd]break[/cmd] command: in that case the
			execution of the <command> will be immediately interrupted and the control
			transferred to the command following this while block.[br]
			It is valid for <command> to be an empty command terminated with a ';'.
			<condition> is an expression as the ones evaluated by [doc:expressioneval]$(*)[/doc]
			with the following extensions:[br]
			If <condition> is a string, its length is evaluated: in this way a non-empty string
			causes the <condition> to be true, an empty string causes it to be false.[br]
			If <condition> is an array, its size is evaluated: in this way a non-empty array
			causes the <condition> to be true, an empty array causes it to be false.[br]
			If <condition> is a hash, the number of its entries is evaluated: in this way a non-empty hash
			causes the <condition> to be true, an empty hash causes it to be false.[br]
		@examples:
			[example]
			%i = 0;
			while(%i < 100)%i++
			while(%i > 0)
			{
				%i -= 10
				if(%i < 20)break;
			}
			echo %i
			[/example]
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The while command needs an expression enclosed in parentheses"));
		error(KVSP_curCharPointer,__tr2qs("Found character %q (unicode %x) where an open parenthesis was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
		return 0;
	}

	const QChar * pBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	KviKvsTreeNodeExpression * e = parseExpression(')');
	if(!e)
	{
		// must be an error
		return 0;
	}

	skipSpacesAndNewlines();

	if(KVSP_curCharUnicode == 0)
	{
		warning(pBegin,__tr2qs("The last while command in the buffer has no conditional instructions: it's senseless"));
		warning(KVSP_curCharPointer,__tr2qs("Unexpected end of script while looking for the instruction block of the while command"));
	}

	KviKvsTreeNodeInstruction * i = parseInstruction();
	if(!i)
	{
		if(error())
		{
			delete e;
			return 0;
		}
	} // else , just an empty instruction

	return new KviKvsTreeNodeSpecialCommandWhile(pBegin,e,i);
}

KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandDo()
{
	/*
		@doc: do
		@type:
			command
		@title:
			do
		@syntax:
			do <command> while (<condition>)
		@keyterms:
			iteration commands, flow control commands
		@short:
			Iteration command
		@description:
			Executes <command> once then evaluates the <condition>.
			If <condition> evaluates to true (non zero result) then repeats the execution again.[br]
			<command> may be either a single command or a block of commands.[br]
			It can contain the [cmd]break[/cmd] command: in that case the
			execution of the <command> will be immediately interrupted and the control
			transferred to the command following this while block.[br]
			It is valid for <command> to be an empty command terminated with a ';'.
			<condition> is an expression as the ones evaluated by [doc:expressioneval]$(*)[/doc]
			with the following extensions:[br]
			If <condition> is a string, its length is evaluated: in this way a non-empty string
			causes the <condition> to be true, an empty string causes it to be false.[br]
			If <condition> is an array, its size is evaluated: in this way a non-empty array
			causes the <condition> to be true, an empty array causes it to be false.[br]
			If <condition> is a hash, the number of its entries is evaluated: in this way a non-empty hash
			causes the <condition> to be true, an empty hash causes it to be false.[br]
					@examples:
			[example]
			%i = 0;
			do %i++; while(%i < 100);
			echo "After first execution:  %i";
			%i = 10
			do {
				echo "Executed!";
				%i++;
			} while(%i < 1)
			echo "After second execution:  %i";
			[/example]
		@seealso:
			[cmd]while[/cmd]
	*/

	const QChar * pBegin = KVSP_curCharPointer;

	KviKvsTreeNodeInstruction * i = parseInstruction();
	if(!i)
	{
		if(error())return 0;
	}

	skipSpacesAndNewlines();

	static const unsigned short while_chars[10] = { 'W','w','H','h','I','i','L','l','E','e' };

	for(int j=0;j<10;j++)
	{
		if(KVSP_curCharUnicode != while_chars[j])
		{
			j++;
			if(KVSP_curCharUnicode != while_chars[j])
			{
				if(KVSP_curCharIsEndOfBuffer)
					error(KVSP_curCharPointer,__tr2qs("Unexpected end of command after the 'do' command block: expected 'while' keywoard"));
				else
					error(KVSP_curCharPointer,__tr2qs("Found character %q (unicode %x) where a 'while' keyword was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
				if(i)delete i;
				return 0;
			}
		} else j++;
		KVSP_skipChar;
	}
	
	skipSpacesAndNewlines();

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'while' block of the 'do' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"do");
		if(i)delete i;
		return 0;
	}

	KVSP_skipChar;

	KviKvsTreeNodeExpression * e = parseExpression(')');
	if(!e)
	{
		// must be an error
		if(i)delete i;
		return 0;
	}

	skipSpaces();

	if(!KVSP_curCharIsEndOfCommand)
	{
		warning(KVSP_curCharPointer,__tr2qs("Garbage string after the expression in 'do' command: ignored"));
		while(!KVSP_curCharIsEndOfCommand)KVSP_skipChar;
	}
	
	if(!KVSP_curCharIsEndOfBuffer)KVSP_skipChar;

	return new KviKvsTreeNodeSpecialCommandDo(pBegin,e,i);
}




KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandIf()
{
	/*
		@doc: if
		@type:
			command
		@title:
			if
		@syntax:
			if (<condition>) <command1> [else <command2>]
		@keyterms:
			conditional commands, flow control commands
		@short:
			Flow control command
		@description:
			Executes <command1> if the <condition> evaluates
			to true (non zero result).
			If the "else part" is given <command2> is executed
			if the <condition> evaluates to false (result == '0')
			<condition> is an expression as the ones evaluated by [doc:expressioneval]$(*)[/doc]
			with the following extensions:[br]
			If <condition> is a string, its length is evaluated: in this way a non-empty string
			causes the <condition> to be true, an empty string causes it to be false.[br]
			If <condition> is an array, its size is evaluated: in this way a non-empty array
			causes the <condition> to be true, an empty array causes it to be false.[br]
			If <condition> is a hash, the number of its entries is evaluated: in this way a non-empty hash
			causes the <condition> to be true, an empty hash causes it to be false.[br]
		@examples:
			if(%a != 10)[cmd]echo[/cmd] \%a was != 10
			else [cmd]echo[/cmd] \%a was 10!
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'if' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"if");
		return 0;
	}

	const QChar * pBegin = KVSP_curCharPointer;

	KVSP_skipChar;


	KviKvsTreeNodeExpression * e = parseExpression(')');
	if(!e)
	{
		// must be an error
		return 0;
	}

	skipSpacesAndNewlines();

	if(KVSP_curCharUnicode == 0)
	{
		warning(pBegin,__tr2qs("The last if command in the buffer has no conditional instructions: it's senseless"));
		warning(KVSP_curCharPointer,__tr2qs("Unexpected end of script while looking for the instruction block of the if command"));
	}

	KviKvsTreeNodeInstruction * i = parseInstruction();
	if(!i)
	{
		if(error())
		{
			delete e;
			return 0;
		}
	} // else , just an empty instruction

	skipSpacesAndNewlines();

	const QChar * pElse = KVSP_curCharPointer;

	if((KVSP_curCharUnicode != 'e') && (KVSP_curCharUnicode != 'E'))
		return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,0);
	KVSP_skipChar;
	if((KVSP_curCharUnicode != 'l') && (KVSP_curCharUnicode != 'L'))
	{
		KVSP_setCurCharPointer(pElse);
		return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,0);
	}
	KVSP_skipChar;
	if((KVSP_curCharUnicode != 's') && (KVSP_curCharUnicode != 'S'))
	{
		KVSP_setCurCharPointer(pElse);
		return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,0);
	}
	KVSP_skipChar;
	if((KVSP_curCharUnicode != 'e') && (KVSP_curCharUnicode != 'E'))
	{
		KVSP_setCurCharPointer(pElse);
		return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,0);
	}
	KVSP_skipChar;
	if(KVSP_curCharIsLetterOrNumber)
	{
		KVSP_setCurCharPointer(pElse);
		return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,0);
	}

	skipSpacesAndNewlines();

	KviKvsTreeNodeInstruction * i2 = parseInstruction();
	if(!i2)
	{
		if(error())
		{
			delete e;
			if(i)delete i;
			return 0;
		}
	} // else , just an empty instruction

	return new KviKvsTreeNodeSpecialCommandIf(pBegin,e,i,i2);
}

bool KviKvsParser::skipToEndOfForControlBlock()
{
	bool bInString = false;
	int iParLevel = 0;

	for(;;)
	{
		switch(KVSP_curCharUnicode)
		{
			case '"':
				bInString = !bInString;
				KVSP_skipChar;
			break;
			case '(':
				if(!bInString)iParLevel++;
				KVSP_skipChar;
			break;
			case ')':
				if(!bInString)
				{
					if(iParLevel == 0)return true;
					else iParLevel--;
				}
				KVSP_skipChar;
			break;
			case 0:
				error(KVSP_curCharPointer,__tr2qs("Unexpected end of buffer while looking for the closing ')' in the 'for' command"));
				return false;
			break;
			//case '\n':
				// that's ok.. it may have a parenthesis on the next line
				//KVSP_skipChar;
			//break;
			default:
				KVSP_skipChar;
			break;
		}
	}
	// not reached
	KVSP_ASSERT(false);
	return false;
}

KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandFor()
{
	/*
		@doc: for
		@type:
			command
		@title:
			for
		@syntax:
			for (<initialization>;<condition>;<update>) <command>
		@keyterms:
			iterational control commands
		@short:
			Iteration control command
		@description:
			Executes <initialization> once then runs the following iteration loop:
			if <condition> evaluates to true then <command> is executed followed
			by the <update> command. The iteration is repeated until <condition> evaluates to false.[br]
			<condition> is an expression as the ones evaluated by [doc:expressioneval]$(*)[/doc]
			with the following extensions:[br]
			If <condition> is a string, its length is evaluated: in this way a non-empty string
			causes the <condition> to be true, an empty string causes it to be false.[br]
			If <condition> is an array, its size is evaluated: in this way a non-empty array
			causes the <condition> to be true, an empty array causes it to be false.[br]
			If <condition> is a hash, the number of its entries is evaluated: in this way a non-empty hash
			causes the <condition> to be true, an empty hash causes it to be false.[br]
		@examples:
			for(%a = 0;%a < 100;%a++)echo %a
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'for' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"for");
		return 0;
	}

	const QChar * pForBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	skipSpaces();

	KviKvsTreeNodeInstruction * i1 = parseInstruction();
	if(!i1)
	{
		if(error())return 0;
	} // else just empty instruction

	skipSpaces();

	KviKvsTreeNodeExpression * e = parseExpression(';');
	if(!e)
	{
		if(error())
		{
			if(i1)delete i1;
			return 0;
		}
	} // else just empty expression : assume true

	skipSpaces();

	// skip to the first non matching ')' that is not in a string

	const QChar * pBegin = KVSP_curCharPointer;

	if(!skipToEndOfForControlBlock())
	{
		if(error()) // <-- that's always true
		{
			if(i1)delete i1;
			if(e)delete e;
			return 0;
		}
	}

	
	// HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK
	// KVSP_curCharPointer is const!
	// we shouldn't be able to modify it
	QChar cSave = *(KVSP_curCharPointer);

	QChar * pHack = (QChar *)KVSP_curCharPointer;
	*pHack = QChar('\n');

	KVSP_curCharPointer = pBegin;

	KviKvsTreeNodeInstruction * i2 = parseInstruction();
	*pHack = cSave;

	KVSP_setCurCharPointer(pHack);
	// EOF HACK EOF HACK EOF HACK EOF HACK EOF HACK EOF HACK EOF HACK


	if(!i2)
	{
		if(error())
		{
			if(i1)delete i1;
			if(e)delete e;
			return 0;
		}
	} // else just empty instruction

	skipSpaces();
	
	if(KVSP_curCharUnicode != ')')
	{
		error(KVSP_curCharPointer,__tr2qs("Found char %q (unicode %x) while looking for the terminating ')' in 'for' command"),KVSP_curCharPointer,KVSP_curCharUnicode);
		if(i1)delete i1;
		if(e)delete e;
		if(i2)delete i2;
		return 0;
	}

	KVSP_skipChar;

	skipSpacesAndNewlines();

	KviKvsTreeNodeInstruction * loop = parseInstruction();
	if(!loop)
	{
		if(error())
		{
			if(i1)delete i1;
			if(e)delete e;
			if(i2)delete i2;
			return 0;
		}
		
		if((!i1) && (!e) && (!i2))
		{
			error(pForBegin,__tr2qs("Empty infinite 'for' loop: fix the script"));
			if(i1)delete i1;
			if(e)delete e;
			if(i2)delete i2;
			return 0;
		}
	} // else just an empty instruction
	
	return new KviKvsTreeNodeSpecialCommandFor(pForBegin,i1,e,i2,loop);
}



KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandForeach()
{
	/*
		@doc: foreach
		@type:
			command
		@title:
			foreach
		@syntax:
			foreach(<variable>,[<item>[,<item>[,<item>[...]]]) <command>
		@keyterms:
			iteration commands, flow control commands
		@short:
			Iteration command
		@description:
			Executed <command> while assigning to <variable> each <item>.[br]
			<item> may be a constant , a variable , an array , a dictionary or a function returning
			either a constant string an array reference or a dictionary reference.[br]
			If <item> is an array , a dictionary or a function that returns a dictionary or array reference
			the iteration is done through all the dictionary/array items.[br]
			Please note that the iteration order of dictionary items is undefined.[br]
			You can always break from the loop by using the [cmd]break[/cmd] command.[br]
		@examples:
			[example]
				foreach(%i,1,2,3,4,5,6,7,8,9)[cmd]echo[/cmd] %i
				foreach(%chan,[fnc]$window.list[/fnc](channel))[cmd]me[/cmd] -r=%chan This is a test!
				[comment]# This will work too, and will do the same job[/comment]
				%windows[] = [fnc]$window.list[/fnc](channel)
				foreach(%chan,%windows[])[cmd]me[/cmd] -r=%chan This is a test!
				[comment]# And this too[/comment]
				%windows[] = [fnc]$window.list[/fnc](channel)
				foreach(%key,[fnc]$keys[/fnc](%windows[]))[cmd]me[/cmd] -r=%windows[%key] This is a test!
				[comment]# Another interesting example[/comment]
				[cmd]alias[/cmd](test){ [cmd]return[/cmd] [fnc]$hash[/fnc](1,a,2,b,3,c,4,d); };
				foreach(%x,[fnc]$keys[/fnc]($test)){ [cmd]echo[/cmd] %x, $test{%x}; }
			[/example]
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'foreach' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"foreach");
		return 0;
	}

	const QChar * pForeachBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	skipSpaces();


	if((KVSP_curCharUnicode != '%') && (KVSP_curCharUnicode != '$'))
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'foreach' command expects a writeable iteration variable as first parameter"));
		error(KVSP_curCharPointer,__tr2qs("Found character '%q' (unicode %x) where '%' or '$' was expected: see /help foreach for the command syntax"),KVSP_curCharPointer,KVSP_curCharUnicode);
		return 0;
	}

	KviKvsTreeNodeData * d = parsePercentOrDollar();
	if(!d)return 0;
	
	if(d->isFunctionCall() || d->isReadOnly())
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'foreach' command expects a writeable iteration variable as first parameter"));
		if(d->isFunctionCall())
			error(KVSP_curCharPointer,__tr2qs("Unexpected function call as 'foreach' iteration variable"));
		else
			error(KVSP_curCharPointer,__tr2qs("Unexpected read-only variable as 'foreach' iteration variable"));
		delete d;
		return 0;
	}

	skipSpaces();
	if(KVSP_curCharUnicode != ',')
	{
		if(KVSP_curCharUnicode == ')')
		{
			error(KVSP_curCharPointer,__tr2qs("Unexpected end of 'foreach' parameters: at least one iteration data argument must be given"));
			delete d;
			return 0;
		}
		warning(KVSP_curCharPointer,__tr2qs("The 'foreach' command expects a comma separated list of iteration data items after the first parameter"));
		errorBadChar(KVSP_curCharPointer,',',"foreach");
		return 0;
	}

	KviKvsTreeNodeDataList * l = parseCommaSeparatedParameterList();
	if(!l)return 0;

	skipSpacesAndNewlines();

	const QChar * pLoopBegin = KVSP_curCharPointer;

	KviKvsTreeNodeInstruction * loop = parseInstruction();
	if(!loop)
	{
		if(error())return 0;
		error(pLoopBegin,__tr2qs("Empty 'foreach' execution block: fix the script"));
		delete d;
		delete l;
		return 0;
	}

	return new KviKvsTreeNodeSpecialCommandForeach(pForeachBegin,d,l,loop);
}


KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandSwitch()
{
	/*
		@doc: switch
		@type:
			command
		@title:
			switch
		@syntax:
			switch(<expression>)
			{
				case(<value>)[:]<command>
				[break]
				case(<value>)[:]<command>
				[break]
				....
				match(<wildcard_expression>)[:]<command>
				[break]
				....
				regexp(<regular_expression>)[:]<command>
				[break]
				....
				case(<value>)[:]<command>
				[break]
				....
				default[:]<command>
				[break]
			}
		@short:
			Another flow control command
		@description:
			The switch command is based on the standard C 'switch' keyword.
			It executes conditionally groups of commands choosen from
			a larger set of command groups.[br]
			First <expression> is evaluated (<expression> is any arithmetic or string expression).[br]
			Then the 'match','regexp','case' and 'default' labels are evaluated sequentially
			in the order of appearance.[br]
			[b]case(<value>)[:]<command>[/b][br]
			The <value> is evaluated and is compared against the result of <expression>.
			The comparison is case insensitive (if the values are strings).[br]
			If <value> is equal to <expression> then <command> is executed.
			<command> is a single instruction or an instruction block enclosed in braces.
			If <command> contains a [cmd]break[/cmd] statement inside or if [cmd]break[/cmd]
			is specified just after the <command> then the execution of the switch is terminated
			otherwise the nex label is evaluated.[br]
			[b]match(<value>)[:]<command>[/b][br]
			The <value> is expected to be a wildcard expression (containing '*' and '?' wildcards)
			that is matched against <expression>.[br]
			If there is a match (a complete case insensitive match!) then the related <command>
			is executed. [cmd]brea[/cmd] is treated just like in the case label.[br]
			[b]regexp(<value>)[:]<command>[/b][br]
			The <value> is expected to be a complete standard regular expression
			that is matched agains <expression>.[br]
			If there is a match (a complete case insensitive match!) then the related <command>
			is executed. [cmd]brea[/cmd] is treated just like in the case label.[br]
			[b]default[:]<command>[/b][br]
			The default label is executed unconditionally (unless there was a previous label
			that terminated the execution with break).[br]
		@examples:
			[comment]# Try to change the 1 below to 2 or 3 to see the results[/comment]
			%tmp = 1
			switch(%tmp)
			{
				case(1):
					echo \%tmp was 1!
				break;
				case(2)
					echo \%tmp was 2!
				break;
				default:
					echo \%tmp was not 1 nor 2: it was %tmp!
				break;
			}
			[comment]# A complexier example: change the 1 in 2 or 3[/comment]
			%tmp = 1
			switch(%tmp)
			{
				case(1):
					echo \%tmp was 1!
				case(2)
					echo \%tmp was 2!
				break;
				default:
					echo \%tmp was either 1 or something different from 2 (%tmp)
				break;
			}
			[comment]# An example with strings[/comment]
			%tmp = "This is a test"
			%tmp2 = "This is not a test"
			switch(%tmp)
			{
				case(%tmp2)
					echo \%tmp == \%tmp2
					break;
				case(%tmp)
				{
					# do not break here
					echo "Yeah.. it's stupid.. \%tmp == \%tmp :D"
				}
				match("*TEST"):
					echo "Matched *TEST"
				regexp("[a-zA-Z ]*test"):
					echo "Matched [a-zA-Z ]*text"
				regexp("[a-zA-Z ]*not[a-zA-Z ]*"):
					echo "Matched [a-zA-Z ]*not[a-zA-Z ]*"
				default:
					echo This is executed anyway (unless some break was called)
				break;
			}
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'switch' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"switch");
		return 0;
	}

	const QChar * pBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	KviKvsTreeNodeExpression * e = parseExpression(')');
	if(!e)
	{
		// must be an error
		return 0;
	}

	skipSpacesAndNewlines();

	if(KVSP_curCharUnicode != '{')
	{
		errorBadChar(KVSP_curCharPointer,'{',"switch");
		delete e;
		return 0;
	}

	KVSP_skipChar;

	KviKvsTreeNodeSpecialCommandSwitch * pSwitch = new KviKvsTreeNodeSpecialCommandSwitch(pBegin,e);

	skipSpacesAndNewlines();

	KviKvsTreeNodeSpecialCommandSwitchLabel * pLabel = 0;

	while(KVSP_curCharUnicode != '}')
	{
		// look for a 'case','match','default' or 'regexpr' label

		const QChar * pLabelBegin = KVSP_curCharPointer;
		while(KVSP_curCharIsLetter)KVSP_skipChar;

		if(KVSP_curCharIsEndOfBuffer)
		{
			error(KVSP_curCharPointer,__tr2qs("Unexpected end of buffer in switch condition block"));
			delete pSwitch;
			return 0;
		}

		if(KVSP_curCharPointer == pLabelBegin)
		{
			error(KVSP_curCharPointer,__tr2qs("Found character %q (unicode %x) where a 'case','match','regexp','default' or 'break' label was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
			delete pSwitch;
			return 0;
		}

		QString szLabel(pLabelBegin,KVSP_curCharPointer - pLabelBegin);
		QString szLabelLow = szLabel.lower();

		bool bNeedParam = true;

		if(szLabelLow == "case")
		{
			pLabel = new KviKvsTreeNodeSpecialCommandSwitchLabelCase(pLabelBegin);
		} else if(szLabelLow == "match")
		{
			pLabel = new KviKvsTreeNodeSpecialCommandSwitchLabelMatch(pLabelBegin);
		} else if(szLabelLow == "regexp")
		{
			pLabel = new KviKvsTreeNodeSpecialCommandSwitchLabelRegexp(pLabelBegin);
		} else if(szLabelLow == "default")
		{
			pLabel = new KviKvsTreeNodeSpecialCommandSwitchLabelDefault(pLabelBegin);
			bNeedParam = false;
		} else if(szLabelLow == "break")
		{
			if(pLabel)
			{
				pLabel->setTerminatingBreak(true);
				skipSpaces();
				if(KVSP_curCharUnicode == ';')KVSP_skipChar;
				skipSpacesAndNewlines();
				continue;
			} else {
				error(pLabelBegin,__tr2qs("Found 'break' label where a 'case','match','regexp' or 'default' label was expected"));
				delete pSwitch;
				return 0;
			}
		} else {
			error(pLabelBegin,__tr2qs("Found token '%Q' where a 'case','match','regexp','default' or 'break' label was expected"),&szLabel);
			delete pSwitch;
			return 0;
		}

		if(bNeedParam)
		{
			skipSpaces();
			if(KVSP_curCharUnicode != '(')
			{
				errorBadChar(KVSP_curCharPointer,'(',"switch");
				delete pSwitch;
				delete pLabel;
				return 0;
			}
			KVSP_skipChar;

			KviKvsTreeNodeData * pParameter = parseSingleParameterInParenthesis();
			if(!pParameter)
			{
				delete pSwitch;
				delete pLabel;
				return 0;
			}

			pLabel->setParameter(pParameter);
		}

		skipSpaces();
		if(KVSP_curCharUnicode == ':')KVSP_skipChar;
		skipSpacesAndNewlines();

		KviKvsTreeNodeInstruction * pInstruction = parseInstruction();
		if(!pInstruction)
		{
			// may be an empty instruction
			if(error())
			{
				delete pSwitch;
				delete pLabel;
				return 0;
			}
		}

		pLabel->setInstruction(pInstruction);
		pSwitch->addLabel(pLabel);

		skipSpacesAndNewlines();
	}

	KVSP_skipChar;

	if(pSwitch->isEmpty())
	{
		error(pBegin,__tr2qs("Senseless empty switch command: fix the script"));
		delete pSwitch;
		return 0;
	}

	return pSwitch;
}

KviKvsTreeNodeSpecialCommandDefpopupLabelPopup * KviKvsParser::parseSpecialCommandDefpopupLabelPopup()
{
	if(KVSP_curCharUnicode != '{')
	{
		errorBadChar(KVSP_curCharPointer,'{',"defpopup");
		return 0;
	}

	KviKvsTreeNodeSpecialCommandDefpopupLabelPopup * pPopup = new KviKvsTreeNodeSpecialCommandDefpopupLabelPopup(KVSP_curCharPointer);

	KVSP_skipChar;
	
	skipSpacesAndNewlines();

	while(KVSP_curCharUnicode != '}')
	{
		// look for 'label', 'prologue', 'epilogue', 'popup', 'item', 'separator' or 'extpopup' label
		const QChar * pLabelBegin = KVSP_curCharPointer;
		while(KVSP_curCharIsLetter)KVSP_skipChar;

		if(KVSP_curCharIsEndOfBuffer)
		{
			error(KVSP_curCharPointer,__tr2qs("Unexpected end of buffer in defpopup block"));
			delete pPopup;
			return 0;
		}

		if(KVSP_curCharPointer == pLabelBegin)
		{
			error(KVSP_curCharPointer,__tr2qs("Found character %q (unicode %x) where a 'prologue','separator','label','popup','item','extpopup' or 'epilogue' label was expected"),KVSP_curCharPointer,KVSP_curCharUnicode);
			delete pPopup;
			return 0;
		}

		QString szLabel(pLabelBegin,KVSP_curCharPointer - pLabelBegin);
		QString szLabelLow = szLabel.lower();

		KviPtrList<QString> * pParameters = 0;
		
		QString szCondition;

		if((szLabelLow == "prologue") || (szLabelLow == "epilogue"))
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
			bool bPrologue = (szLabelLow == "prologue");
			skipSpacesAndNewlines();
			const QChar * pBegin = KVSP_curCharPointer;
			KviKvsTreeNodeInstruction * pInstruction = parseInstruction();
			if(!pInstruction)
			{
				// may be an empty instruction
				if(!error())
				{
					// senseless empty instruction
					if(bPrologue)
						error(pLabelBegin,__tr2qs("Senseless empty prologue block: fix the script"));
					else
						error(pLabelBegin,__tr2qs("Senseless empty epilogue block: fix the script"));
				} // else the error has been already triggered
				delete pPopup;
				return 0;
			}
			delete pInstruction;
			int iLen = KVSP_curCharPointer - pBegin;
			if(iLen > 0)
			{
				QString szInstruction(pBegin,KVSP_curCharPointer - pBegin);
				KviCommandFormatter::bufferFromBlock(szInstruction);
				if(bPrologue)
					pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelPrologue(pLabelBegin,szInstruction));
				else
					pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelEpilogue(pLabelBegin,szInstruction));
			} else {
				error(KVSP_curCharPointer,__tr2qs("defpopup: internal error"));
				return 0;
			}
		} else if(szLabelLow == "separator")
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
#define EXTRACT_POPUP_LABEL_CONDITION \
			skipSpacesAndNewlines(); \
			if(KVSP_curCharUnicode == '(') \
			{ \
				const QChar * pBegin = KVSP_curCharPointer; \
				KVSP_skipChar; \
				KviKvsTreeNodeExpression * pExpression = parseExpression(')'); \
				if(!pExpression) \
				{ \
					if(pParameters)delete pParameters; \
					delete pPopup; \
					return 0; \
				} \
				int cLen = (KVSP_curCharPointer - pBegin) - 2; \
				if(cLen > 0) \
				{ \
					szCondition.setUnicode(pBegin + 1,cLen); \
				} \
				delete pExpression; \
				skipSpacesAndNewlines(); \
			}

			EXTRACT_POPUP_LABEL_CONDITION
			if(KVSP_curCharUnicode == ';')KVSP_skipChar;
			pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelSeparator(pLabelBegin,szCondition));

		} else if(szLabelLow == "label")
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
#define EXTRACT_POPUP_LABEL_PARAMETERS \
			skipSpacesAndNewlines(); \
			if(KVSP_curCharUnicode != '(') \
			{ \
				errorBadChar(KVSP_curCharPointer,'(',"defpopup"); \
				delete pPopup; \
				return 0; \
			} \
			pParameters = parseCommaSeparatedParameterListNoTree(); \
			if(!pParameters)return 0;

			EXTRACT_POPUP_LABEL_PARAMETERS
			EXTRACT_POPUP_LABEL_CONDITION

			QString * pText = pParameters->first();
			if(!pText)
			{
				error(pLabelBegin,__tr2qs("Unexpected empty <text> field in label parameters. See /help defpopup for the syntax"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			QString * pIcon = pParameters->next();
			if(KVSP_curCharUnicode == ';')KVSP_skipChar;
			pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelLabel(pLabelBegin,szCondition,*pText,pIcon ? *pIcon : QString::null));
			delete pParameters;
		} else if(szLabelLow == "popup")
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
			EXTRACT_POPUP_LABEL_PARAMETERS
			EXTRACT_POPUP_LABEL_CONDITION

			QString * pText = pParameters->first();
			if(!pText)
			{
				error(pLabelBegin,__tr2qs("Unexpected empty <text> field in extpopup parameters. See /help defpopup for the syntax"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			QString * pIcon = pParameters->next();

			KviKvsTreeNodeSpecialCommandDefpopupLabelPopup * pSubPopup = parseSpecialCommandDefpopupLabelPopup();
			if(!pSubPopup)
			{
				delete pParameters;
				delete pPopup;
				return 0;
			}
			
			pSubPopup->setCondition(szCondition);
			pSubPopup->setText(*pText);
			if(pIcon)pSubPopup->setIcon(*pIcon);
			pPopup->addLabel(pSubPopup);
			delete pParameters;
		} else if(szLabelLow == "item")
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
			EXTRACT_POPUP_LABEL_PARAMETERS
			EXTRACT_POPUP_LABEL_CONDITION

			QString * pText = pParameters->first();
			if(!pText)
			{
				error(pLabelBegin,__tr2qs("Unexpected empty <text> field in extpopup parameters. See /help defpopup for the syntax"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			QString * pIcon = pParameters->next();

			const QChar * pBegin = KVSP_curCharPointer;
			KviKvsTreeNodeInstruction * pInstruction = parseInstruction();
			if(!pInstruction)
			{
				// senseless empty instruction ?
				error(pLabelBegin,__tr2qs("Senseless empty instruction for popup item: fix the script"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			delete pInstruction;
			int iLen = KVSP_curCharPointer - pBegin;
			if(iLen > 0)
			{
				QString szInstruction(pBegin,KVSP_curCharPointer - pBegin);
				KviCommandFormatter::bufferFromBlock(szInstruction);
				pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelItem(pLabelBegin,szCondition,*pText,pIcon ? *pIcon : QString::null,szInstruction));
			} else {
				error(KVSP_curCharPointer,__tr2qs("defpopup: internal error"));
				delete pParameters;
				return 0;
			}
			delete pParameters;
		} else if(szLabelLow == "extpopup")
		{
			/////////////////////////////////////////////////////////////////////////////////////////////////
			EXTRACT_POPUP_LABEL_PARAMETERS
			EXTRACT_POPUP_LABEL_CONDITION

			QString * pText = pParameters->first();
			if(!pText)
			{
				error(pLabelBegin,__tr2qs("Unexpected empty <text> field in extpopup parameters. See /help defpopup for the syntax"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			QString * pName = pParameters->next();
			if(!pName)
			{
				error(pLabelBegin,__tr2qs("Unexpected empty <name> field in extpopup parameters. See /help defpopup for the syntax"));
				delete pParameters;
				delete pPopup;
				return 0;
			}
			QString * pIcon = pParameters->next();
			if(KVSP_curCharUnicode == ';')KVSP_skipChar;
			pPopup->addLabel(new KviKvsTreeNodeSpecialCommandDefpopupLabelExtpopup(pLabelBegin,szCondition,*pText,pIcon ? *pIcon : QString::null,*pName));
			delete pParameters;
		} else {
			/////////////////////////////////////////////////////////////////////////////////////////////////
			error(pLabelBegin,__tr2qs("Found token '%Q' where a 'prologue','separator','label','popup','item','extpopup' or 'epilogue' label was expected"),&szLabel);
			delete pPopup;
			return 0;
		}

		skipSpacesAndNewlines();
	}

	KVSP_skipChar;
	return pPopup;
}


KviKvsTreeNodeSpecialCommand * KviKvsParser::parseSpecialCommandDefpopup()
{
	// FIXME: This SHOULD be renamed to popup.create (NOT popup.define!)
	//        and internally aliased to defpopup as backward compat
	//        There should be then also popup.destroy etc..

	/*
		@doc: defpopup
		@type:
			command
		@title:
			defpopup
		@syntax:
			defpopup [-m] (<popup_name>)
			{
				prologue <prologue_command>

				epilogue <epilogue_command>

				label(<text>)[(<expression>)][;]

				item(<text>[,<icon>])[(<expression>)]<command>

				popup(<text>[,<icon>])[(<expression>)]
				{
					<popup body>
				}

				extpopup(<text>,<name>[,<icon>])[(<expression>)][;]

				separator[(<expression>)][;]
				...
			}
		@short:
			Defines a popup menu
		@switches:
			!sw: -m
			Merges the new popup contents with an older popup version
		@description:
			Defines the popup menu <popup_name>. If the -m switch is NOT used
			the previous contents of the popups are cleared, otherwise are preserved.[br]
			The popup is generated 'on the fly' when the [cmd]popup[/cmd] command
			is called.[br]
			The 'item' keyword adds a menu item with visible <text> ,
			the optional <icon> and <command> as code to be executed when the item
			is clicked. <text> is a string that is evaluated at [cmd]popup[/cmd]
			call time and may contain identifiers and variables. If <expression>
			is given, it is evaluated at [cmd]popup[/cmd] call time and if the result
			is 0, the item is not shown in the physical popup.[br]
			The 'popup' keyword adds a submenu with visible <text> , the optional
			<icon> and a popup body that follows exactly the same syntax
			as the defpopup body. The <expression> has the same meaning as with the
			'item' keyword.[br]
			The 'extpopup' keyword adds a submenu with visible <text> , the optional
			icon and a popup body that is defined by the popup menu <name>. This
			basically allows to nest popup menus and define their parts separately.
			<icon> and <expression> have the same meaning as with the 'item' keyword.[br]
			The 'separator' keyword adds a straight line between items (separator).[br]
			The 'label' keywork adds a descriptive label that acts like a separator.[br]
			The 'prologue' keyword adds a <prologue_command> to be executed
			just before the popup is filled at [cmd]popup[/cmd] command call.[br]
			The 'epilogue' keyword adds an <epilogue_command> to be executed
			just after the popup has been filled at [cmd]popup[/cmd] command call.[br]
			There can be multiple prologue and epilogue commands: their execution order
			is undefined.[br]
			<icon> is always an [doc:image_id]image identifier[/doc].[br]
			Please note that using this command inside the prologue , epilogue
			or item code of the modified popup menu is forbidden.
			In other words: self modification of popup menus is NOT allowed.[br]
			To remove a popup menu use this command with an empty body:[br]
			[example]
				defpopup(test){}
			[/example]
			This will remove the popup 'test' and free its memory.
			Popups have a special kind of local variables that have an extended lifetime:
			these are called "extended scope variables" and are described in the [doc:data_structures]Data structures documentation[/doc].[br]
			The syntax for these variables is:[br]
			[b]%:<variable name>[/b][br]
			These variables are visible during all the "visible lifetime" of the popup:
			from the [cmd]popup[/cmd] command call to the moment in that the user selects an item
			and the corresponding code is executed (substantially from a [cmd]popup[/cmd] call to the next one).[br]
			This allows you to pre-calculate data and conditions in the porologue of the popup
			and then use it in the item handlers or item conditions.[br]
		@seealso:
			[cmd]popup[/cmd]
		@examples:
	*/

	if(KVSP_curCharUnicode != '(')
	{
		warning(KVSP_curCharPointer,__tr2qs("The 'defpopup' command needs an expression enclosed in parentheses"));
		errorBadChar(KVSP_curCharPointer,'(',"defpopup");
		return 0;
	}

	const QChar * pBegin = KVSP_curCharPointer;

	KVSP_skipChar;

	KviKvsTreeNodeData * pPopupName = parseSingleParameterInParenthesis();
	if(!pPopupName)return 0;

	skipSpacesAndNewlines();

	KviKvsTreeNodeSpecialCommandDefpopupLabelPopup * pMainPopup = parseSpecialCommandDefpopupLabelPopup();
	if(!pMainPopup)
	{
		delete pPopupName;
		return 0;
	}

	return new KviKvsTreeNodeSpecialCommandDefpopup(pBegin,pPopupName,pMainPopup);
}

