//=============================================================================
//
//   File : kvi_kvs_parser_expression.cpp
//   Creation date : Mon 6 Oct 2003 01.31 CEST by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2003 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__


#include "kvi_kvs_parser.h"

#include "kvi_kvs_treenode.h"

#include "kvi_kvs_report.h"
#include "kvi_kvs_kernel.h"

#include "kvi_kvs_parser_macros.h"

#include "kvi_locale.h"


//#warning "FIXME: expression eval doc!"

	/*
		@doc: expressioneval
		@type:
			language
		@title:
			Expression evaluation identifier
		@syntax:
			$(<expression>)
		@keyterms:
			expressions
		@short:
			Expression evaluation identifier
		@description:
			Evaluates <expression> and returns its result.[br]
			If <expression> is a single string, array or hash, it is returned unmodified.[br]
			In any other case the expression evaluation returns a numeric value, either real or integer.[br]
			The expressions are really close to the C ones and have some minor extensions.[br]
			The supported operators are +,-,*,/,|,&,^,||,&&,^^,>>,<<,<,>,<=,>=,==,!= and <> (synonim for !=).[br]
			The following table describes their meaning.[br]
			[table]
				[tr][td][b]Operator[/b][/td][td][b]Description[/b][/td][/tr]
				[tr][td]a + b[/td][td]Arithmetic sum: valid only for numeric operands[/td][/tr]
				[tr][td]a - b[/td][td]Arithmetic subtraction: valid only for numeric operands[/td][/tr]
				[tr][td]a / b[/td][td]Arithmetic division: valid only for numeric operands[/td][/tr]
				[tr][td]a * b[/td][td]Arithmetic multiplication: valid only for numeric operands[/td][/tr]
				[tr][td]a % b[/td][td]Arithmetic modulus: valid only for numeric operands[/td][/tr]
				[tr][td]a || b[/td][td]Logical or: valid only for boolean operands[/td][/tr]
				[tr][td]a && b[/td][td]Logical and: valid only for boolean operands[/td][/tr]
				[tr][td]a ^^ b[/td][td]Logical xor: valid only for boolean operands[/td][/tr]
				[tr][td]a >> b[/td][td]Bitwise shift right: valid only for integer operands[/td][/tr]
				[tr][td]a << b[/td][td]Bitwise shift left: valid only for integer operands[/td][/tr]
				[tr][td]a | b[/td][td]Bitwise or: valid only for integer operands[/td][/tr]
				[tr][td]a & b[/td][td]Bitwise and: valid only for integer operands[/td][/tr]
				[tr][td]a ^ b[/td][td]Bitwise xor: valid only for integer operands[/td][/tr]
				[tr][td]a > b[/td][td]Greater than: valid for numeric or string operands. Case sensitive[/td][/tr]
				[tr][td]a < b[/td][td]Lower than: valid for numeric or string operands. Case sensitive[/td][/tr]
				[tr][td]a >= b[/td][td]Greater or equal to: valid for numeric or string operands. Case sensitive[/td][/tr]
				[tr][td]a <= b[/td][td]Lower or equal to: valid for numeric or string operands. Case sensitive[/td][/tr]
				[tr][td]a != b[/td][td]Not equal to: valid for numeric or string operands. Case sensitive[/td][/tr]
				[tr][td]a == b[/td][td]Equal to: valid for numeric or string operands. Case sensitive[/td][/tr]
			[/table]
			The expressions can contain integer, real or string constants and variable operands.[br]
			The integer constants can be also specified as hexadecimal numbers by prefixing them by '0x'.[br]
			The string constants should be enclosed in quotes.[br]
		@examples:
			[example]
				echo $(10 + 5 * 100)
				echo $(10 / 3)
				echo $(10 / 3.0)
				echo $(10.0 + 5 * 100)
				echo $(145 & 2)
				echo $("hello" > "ciao")
				echo $(10 == "10")
				%a = 100
				%b = 50.3
				%c = "test"
				echo $(%a + %b)
				echo $("%a%b" + 1)
				echo $(%a + %b > %c)
				echo $(-(10 + 20) * 3)
				echo $(1 ^ 2)
				echo $(1 ^ 1)
				echo $(0xffff == 65535)
				...
			[/example]
	*/


KviKvsTreeNodeExpressionBinaryOperator * KviKvsParser::parseExpressionBinaryOperator()
{
	switch(KVSP_curCharUnicode)
	{
		case '=':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '=')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeExpressionBinaryOperatorEqualTo(KVSP_curCharPointer);
			} else {
				error(KVSP_curCharPointer,__tr2qs("Unknown binary operator '=%q': did you mean '==' ?"),KVSP_curCharPointer);
			}
		break;
		case '!':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '=')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeExpressionBinaryOperatorNotEqualTo(KVSP_curCharPointer);
			}
		break;
		case '+':
			KVSP_skipChar;
			return new KviKvsTreeNodeExpressionBinaryOperatorSum(KVSP_curCharPointer);
		break;
		case '-':
			KVSP_skipChar;
			return new KviKvsTreeNodeExpressionBinaryOperatorSubtraction(KVSP_curCharPointer);
		break;
		case '/':
			KVSP_skipChar;
			return new KviKvsTreeNodeExpressionBinaryOperatorDivision(KVSP_curCharPointer);
		break;
		case '%':
			KVSP_skipChar;
			return new KviKvsTreeNodeExpressionBinaryOperatorModulus(KVSP_curCharPointer);
		break;
		case '*':
			KVSP_skipChar;
			return new KviKvsTreeNodeExpressionBinaryOperatorMultiplication(KVSP_curCharPointer);
		break;
		case '&':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '&')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeExpressionBinaryOperatorAnd(KVSP_curCharPointer);
			}
			return new KviKvsTreeNodeExpressionBinaryOperatorBitwiseAnd(KVSP_curCharPointer);
		break;
		case '|':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '|')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeExpressionBinaryOperatorOr(KVSP_curCharPointer);
			}
			return new KviKvsTreeNodeExpressionBinaryOperatorBitwiseOr(KVSP_curCharPointer);
		break;
		case '^':
			KVSP_skipChar;
			if(KVSP_curCharUnicode == '^')
			{
				KVSP_skipChar;
				return new KviKvsTreeNodeExpressionBinaryOperatorXor(KVSP_curCharPointer);
			}
			return new KviKvsTreeNodeExpressionBinaryOperatorBitwiseXor(KVSP_curCharPointer);
		break;
		case '>':
			KVSP_skipChar;
			switch(KVSP_curCharUnicode)
			{
				case '>':
					KVSP_skipChar;
					return new KviKvsTreeNodeExpressionBinaryOperatorShiftRight(KVSP_curCharPointer);
				break;
				case '=':
					KVSP_skipChar;
					return new KviKvsTreeNodeExpressionBinaryOperatorGreaterOrEqualTo(KVSP_curCharPointer);
				break;
				default:
					return new KviKvsTreeNodeExpressionBinaryOperatorGreaterThan(KVSP_curCharPointer);
				break;
			}
		break;
		case '<':
			KVSP_skipChar;
			switch(KVSP_curCharUnicode)
			{
				case '>':
					KVSP_skipChar;
					return new KviKvsTreeNodeExpressionBinaryOperatorNotEqualTo(KVSP_curCharPointer);
				break;
				case '<':
					KVSP_skipChar;
					return new KviKvsTreeNodeExpressionBinaryOperatorShiftLeft(KVSP_curCharPointer);
				break;
				case '=':
					KVSP_skipChar;
					return new KviKvsTreeNodeExpressionBinaryOperatorLowerOrEqualTo(KVSP_curCharPointer);
				break;
				default:
					return new KviKvsTreeNodeExpressionBinaryOperatorLowerThan(KVSP_curCharPointer);
				break;
			}
		break;
	}

	error(KVSP_curCharPointer,__tr2qs("Unknown binary operator '%q'"),KVSP_curCharPointer);
	return 0;
}

KviKvsTreeNodeExpression * KviKvsParser::parseExpressionOperand(char terminator)
{
	switch(KVSP_curCharUnicode)
	{
		case 0:
		case '\n':
			error(KVSP_curCharPointer,__tr2qs("Unexpected end of script in expression"));
			return 0;
		break;
		case '"':
		{
			// a string
			KviKvsTreeNodeData * d = parseStringParameter();
			if(!d)return 0;
			return new KviKvsTreeNodeExpressionVariableOperand(d->location(),d);
		}
		break;
		case '%':
		case '$':
		{
			KviKvsTreeNodeData * d = parseParameterPercentOrDollar();
			if(!d)return 0;
			return new KviKvsTreeNodeExpressionVariableOperand(d->location(),d);
		}
		break;
		case '(':
			KVSP_skipChar;
			skipSpaces();
			return parseExpression(')'); // sub expression
		break;
		case '-':
		{
			KVSP_skipChar;
			skipSpaces();
			KviKvsTreeNodeExpression * d = parseExpressionOperand(terminator);
			if(!d)return 0;
			return new KviKvsTreeNodeExpressionUnaryOperatorNegate(d->location(),d);
		}
		break;
		case '!':
		{
			KVSP_skipChar;
			skipSpaces();
			KviKvsTreeNodeExpression * d = parseExpressionOperand(terminator);
			if(!d)return 0;
			return new KviKvsTreeNodeExpressionUnaryOperatorLogicalNot(d->location(),d);
		}
		break;
		case '~':
		{
			KVSP_skipChar;
			skipSpaces();
			KviKvsTreeNodeExpression * d = parseExpressionOperand(terminator);
			if(!d)return 0;
			return new KviKvsTreeNodeExpressionUnaryOperatorBitwiseNot(d->location(),d);
		}
		break;
		default:
			// literal ?
			if(KVSP_curCharIsLetterOrNumber || (KVSP_curCharUnicode == '.'))
			{
				const QChar * pBegin = KVSP_curCharPointer;
				while(KVSP_curCharIsLetterOrNumber || (KVSP_curCharUnicode == '.'))KVSP_skipChar;
				QString tmp(pBegin,KVSP_curCharPointer - pBegin);
				bool bOk;
				int iVal = tmp.toInt(&bOk);
				if(bOk)return new KviKvsTreeNodeExpressionConstantOperand(pBegin,new KviKvsVariant(iVal));
				if(pBegin->unicode() == '0')
				{
					if(tmp.length() > 2)
					{
						if((tmp[1] == 'x') || (tmp[1] == 'X'))
						{
							// hexadecimal constant ?
							QString hex = tmp.right(tmp.length() - 2);
							iVal = hex.toInt(&bOk,16);
							if(bOk)return new KviKvsTreeNodeExpressionConstantOperand(pBegin,new KviKvsVariant(iVal));
						}
					}
				}
				double dVal = tmp.toDouble(&bOk);
				if(bOk)return new KviKvsTreeNodeExpressionConstantOperand(pBegin,new KviKvsVariant(new double(dVal)));
				return new KviKvsTreeNodeExpressionConstantOperand(pBegin,new KviKvsVariant(new QString(tmp)));
			} else {
				error(KVSP_curCharPointer,__tr2qs("Unexpected character %q (unicode %h) in expression. If it meant to be a string use the quotes."),KVSP_curCharPointer,KVSP_curCharUnicode);
				return 0;
			}
		break;
	}

	// not reached
	KVSP_ASSERT(false);
	return 0;
}


KviKvsTreeNodeExpression * KviKvsParser::parseExpression(char terminator)
{
	// we're inside the expression now
	skipSpaces();

	if(KVSP_curCharUnicode == terminator)
	{
		// empty expression
		// constant 0 ?
		KVSP_skipChar;
		return new KviKvsTreeNodeExpressionConstantOperand(KVSP_curCharPointer,new KviKvsVariant(0));
	}

	KviKvsTreeNodeExpression * left = parseExpressionOperand(terminator);
	if(!left)return 0;

	skipSpaces();

	if(KVSP_curCharUnicode == terminator)
	{
		KVSP_skipChar;
		return left;
	}

	// not a terminator... must be an operator (or an error , eventually)

	KviKvsTreeNodeExpression * curTopOperator = parseExpressionBinaryOperator();
	if(!curTopOperator)
	{
		delete left;
		return 0; // error
	}

	curTopOperator->setLeft(left);

	// ok.. parse the right side
	


	// Now curTopOperator has the left subtree (one node) set
	// and it points to the TOP (=ROOT) node
	// Evaluate the rest
	
	KviKvsTreeNodeExpression * operand;
	KviKvsTreeNodeExpression * incompleteOperator = curTopOperator;
	KviKvsTreeNodeExpression * auxOperator;

	for(;;)
	{
		skipSpaces();
	
		operand = parseExpressionOperand(terminator);
		if(!operand)
		{
			delete curTopOperator;
			return 0;
		}
		
		skipSpaces();
		
		if(KVSP_curCharUnicode == terminator)
		{
			KVSP_skipChar;
			incompleteOperator->setRight(operand);
			return curTopOperator;
		}

		auxOperator = parseExpressionBinaryOperator();
		if(!auxOperator)
		{
			delete curTopOperator;
			delete operand;
			return 0;
		}

		//now compare operators...
		if(incompleteOperator->precedence() >= auxOperator->precedence())
		{
			incompleteOperator->setRight(auxOperator);
			auxOperator->setLeft(operand);
		} else {
			// precedence of operators...
			incompleteOperator->setRight(operand); //right tree complete
			// go up until we find an operator with lower precedence than auxOperator (>=)
			KviKvsTreeNodeExpression * tempOperator = incompleteOperator->parentWithPrecedenceLowerThan(auxOperator->precedence());
			if(tempOperator == 0)
			{
				auxOperator->setLeft(curTopOperator);
				curTopOperator = auxOperator;
			} else {
				KVSP_ASSERT(tempOperator->right());
				auxOperator->setLeft(tempOperator->right());
				tempOperator->setRight(auxOperator);
			}
		}
		incompleteOperator = auxOperator;
		KVSP_ASSERT(incompleteOperator->right() == 0);
	}

	KVSP_ASSERT(false);
	return 0; //newer here

/*
	
	
	KviKvsTreeNodeExpression * right = parseExpression(terminator);
	if(!right)
	{
		delete op;
		return 0;
	}

	// left * a + b
	
	//      *
	// left      +    
	//         a    b

*/
/*
	// now.. the left side is a single operand for sure
	// the right side might be a single operand or a sequence of operations
	if(right->isOperator())
	{
		// if the operator has lower precedence than op then 
		if(right->precedence() < op->precedence())
		{
			right->attachHighPrecedenceOperator(op);
			return right;
		}
	}
*/
/*
	// a single operand or a greater precedence operator
	op->setRight(right);

	return op;
	*/
}

