#ifndef _KVI_HTTP_H_
#define _KVI_HTTP_H_
//
//   File : kvi_http.h
//   Creation date : Sat Aug 17 13:43:31 2002 GMT by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 2002 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 33, Boston, MA 02111-1307, USA.
//

#include "kvi_settings.h"
#include "kvi_heapobject.h"
#include "kvi_string.h"
#include "kvi_thread.h"
#include "kvi_sockettype.h"
#include "kvi_databuffer.h"

#include "kvi_url.h"

#include <qobject.h>
#include <qasciidict.h>
#include <qfile.h>
#include <qstringlist.h>

class KviDns;
class KviHttpRequestThread;

class KVILIB_API KviHttpRequest : public QObject, public KviHeapObject
{
	Q_OBJECT
public:
	enum ProcessingType { HeadersOnly, WholeFile, Blocks, Lines, StoreToFile };
	enum ExistingFileAction { Overwrite, RenameIncoming, RenameExisting, Resume };
public:
	KviHttpRequest();
	~KviHttpRequest();
protected:
	// data
	KviUrl                 m_url;
	QString                m_szFileName;
	ProcessingType         m_eProcessingType;
	ExistingFileAction     m_eExistingFileAction;
	void                 * m_pPrivateData;
	unsigned int           m_uMaxContentLength;
	unsigned int           m_uContentOffset;

	// status
	KviStr                 m_szLastError;
	unsigned int           m_uTotalSize;
	unsigned int           m_uReceivedSize;

	// internal status
	KviStr                 m_szIp;
	KviDns               * m_pDns;
	KviHttpRequestThread * m_pThread;
	KviDataBuffer        * m_pBuffer;
	bool                   m_bHeaderProcessed;
	QFile                * m_pFile;
protected:
	bool startDnsLookup();
	virtual bool event(QEvent *e);
	void processData(KviDataBuffer * data);
	bool processHeader(KviStr &szHeader);
	bool openFile();
	void emitLines();

	void resetStatus();
	void resetData();
	void resetInternalStatus();
protected slots:
	void dnsLookupDone(KviDns *d);
	void haveServerIp();
public:
	const KviUrl & url(){ return m_url; };
	ProcessingType processingType(){ return m_eProcessingType; };
	ExistingFileAction existingFileAction(){ return m_eExistingFileAction; };
	const QString &fileName(){ return m_szFileName; };
	void * privateData(){ return m_pPrivateData; };
	unsigned int maxContentLength(){ return m_uMaxContentLength; };
	unsigned int contentOffset(){ return m_uContentOffset; };
	unsigned int totalSize(){ return m_uTotalSize; };
	unsigned int receivedSize(){ return m_uReceivedSize; };

	void reset();

	void setUrl(const KviUrl &u){ m_url = u; };
	void setProcessingType(ProcessingType t){ m_eProcessingType = t; };
	void setExistingFileAction(ExistingFileAction a){ m_eExistingFileAction = a; };
	void setFileName(const QString &szFileName){ m_szFileName = szFileName; };
	void setPrivateData(void * ptr){ m_pPrivateData = ptr; };
	void setMaxContentLength(int uMaxContentLength){ m_uMaxContentLength = uMaxContentLength; }; //0 means unlimited
	// this will work regardless of ExistingFileAction : even if the file doesn't exist
	void setContentOffset(int uContentOffset){ m_uContentOffset= uContentOffset; };

	bool start();

	// this is a shortcut for reset()+setUrl()+setProcessingType()+setFileName()+start()
	bool get(const KviUrl &u,ProcessingType p = WholeFile,const char * szFileName = 0);

	const KviStr & lastError(){ return m_szLastError; };

	void abort();
signals:
	void resolvingHost(const char * hostname);
	void contactingHost(const char * ipandport);
	void connectionEstabilished();
	void receivedResponse(const char * response);

	void terminated(bool bSuccess);


	void status(const char * message);
	void data(const KviStr &data);
	void binaryData(const KviDataBuffer &data);
	void header(QAsciiDict<KviStr> * hdr);
	void requestSent(const QStringList &request);
};


class KviHttpRequestThread : public KviSensitiveThread
{
	friend class KviHttpRequest;
public:
	enum RequestMethod { Get , Head };
protected:
	KviHttpRequestThread(KviHttpRequest * r,
		const char * szHost,
		const char * szIp,
		unsigned short uPort,
		const char * szPath,
		unsigned int uContentOffset,
		RequestMethod m);

public:
	~KviHttpRequestThread();
protected:
	KviHttpRequest * m_pRequest;

	KviStr           m_szHost;
	KviStr           m_szIp;
	KviStr           m_szPath;
	unsigned int     m_uContentOffset;
	RequestMethod    m_eRequestMethod;

	unsigned short   m_uPort;
	kvi_socket_t     m_sock;
protected:
	int selectForReadStep();
	bool selectForRead(int iTimeoutInSecs);
	bool readData();
	bool sendBuffer(const char * buffer,int bufLen,int iTimeoutInSecs);
	bool failure(const char * error = 0);
	bool selectForWrite(int iTimeoutInSecs);
	bool connectToRemoteHost();
	bool processInternalEvents();
	virtual void run();
};


#endif //_KVI_HTTP_H_
