// =============================================================================
//
//      --- kvi_simple_editor.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//   Copyright (C) 2003 Robin Verduijn <robin@debian.org>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qclipboard.h>
#include <qcheckbox.h>
#include <qcursor.h>
#include <qfile.h>
#include <qregexp.h>

#include "kvi_accel.h"
#include "kvi_application.h"
#include "kvi_editor_syntaxhighlighter.h"
#include "kvi_filedialog.h"
#include "kvi_lineedit.h"
#include "kvi_messagebox.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"
#include "kvi_simple_editor.h"
#include "kvi_simple_findwidget.h"

KviSimpleEditor::KviSimpleEditor(QWidget *parent)
	: QTextEdit(parent)
{
	m_lastFindWidgetPosition = QPoint(0, 0);
	m_bRecordingKeystrokes   = false;
	m_szFileName             = "";

	setFocusPolicy(StrongFocus);
	setFrameStyle(QFrame::Panel | QFrame::Sunken);
	setTextFormat(PlainText);
	setWordWrap(NoWrap);

	m_pKeystrokes = new QPtrList<KviSimpleEditorKeystroke>;
	m_pKeystrokes->setAutoDelete(true);

	m_pContextPopup = new KviPopupMenu();
	m_pFindPopup    = new KviPopupMenu();

	m_pFindWidget = new KviSimpleFindWidget(this);
	m_pFindWidget->hide();
	m_pFindWidget->setPalette(parent->palette());

	m_pHighlighter = new KviEditorSyntaxHighlighter(this);
	installEventFilter(this);
}

KviSimpleEditor::~KviSimpleEditor()
{
	closeFile(); // Cancel does nothing here

	delete m_pHighlighter;  m_pHighlighter  = 0;
	delete m_pFindWidget;   m_pFindWidget   = 0;
	delete m_pKeystrokes;   m_pKeystrokes   = 0;
	delete m_pContextPopup; m_pContextPopup = 0;
	delete m_pFindPopup;    m_pFindPopup    = 0;
}

bool KviSimpleEditor::closeFile()
{
	if( !isModified() ) return true; // Safe to close

	QString msg;
	msg.sprintf(
		"The file %s has been modified\nDo you wish to save your changes?",
		m_szFileName.isEmpty() ? "Untitled" : m_szFileName.ascii()
	);
	int ret = KviMessageBox::warningYesNoCancel(msg, "Warning", this);
	if( ret == KviMessageBox::Yes ) {
		if( !saveFile() )
			return closeFile(); // Cancelled after this; repeat please
		return true; // Saved
	} else if( ret == KviMessageBox::No ) {
		setModified(false);
		return true;
	}
	return false; // Cancelled
}

void KviSimpleEditor::loadFile()
{
	loadFile(KviFileDialog::getOpenFileName());
}

bool KviSimpleEditor::loadFile(const QString &filename)
{
	QFile f(filename);
	if( !f.open(IO_ReadOnly) ) return false;

	QCString buffer(f.size() + 1);
	int readLen = f.readBlock(buffer.data(), f.size());
	*(buffer.data() + readLen) = '\0';
	f.close();
	setText("");
	KviEditorSyntaxHighlighter::ColorMode mode = KviEditorSyntaxHighlighter::Normal;
	if(      filename.endsWith(".c",    false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".cpp",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".cxx",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".h",    false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".hpp",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".htm",  false) ) mode = KviEditorSyntaxHighlighter::HTML;
	else if( filename.endsWith(".html", false) ) mode = KviEditorSyntaxHighlighter::HTML;
	else if( filename.endsWith(".hxx",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".moc",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".s",    false) ) mode = KviEditorSyntaxHighlighter::CPP;
	else if( filename.endsWith(".sgml", false) ) mode = KviEditorSyntaxHighlighter::HTML;
	else if( filename.endsWith(".xml",  false) ) mode = KviEditorSyntaxHighlighter::HTML;
	else if( filename.endsWith(".xpm",  false) ) mode = KviEditorSyntaxHighlighter::CPP;
	m_pHighlighter->setMode(mode);
	setText(buffer);
	m_szFileName = filename;
	setModified(false);
	emit fileNameChanged(this, m_szFileName);
	return true;
}

bool KviSimpleEditor::saveFile()
{
	if( m_szFileName.isEmpty() )
		return saveFileAs();
	return saveFile(m_szFileName);
}

bool KviSimpleEditor::saveFile(const QString &filename)
{
	QFile f(filename);
	if( !f.open(IO_WriteOnly) ) {
		KviMessageBox::error("Cannot open file for writing!\nSave failed", "Warning", this);
		return false;
	}
	int lastProg = -1;
	int i        =  0;
	int progress =  0;
	emit saveProgress(m_szFileName, 0);
	if( paragraphs() == 0 ) {
		emit saveProgress(m_szFileName, 100);
	} else {
		for( i = 0; i < paragraphs(); i++ ) {
			QString line = text(i);
			// Huh?
			if( (f.writeBlock(line, line.length()) != (int) line.length()) || (f.writeBlock("\n", 1) != 1) ) {
				debug("Error writing to file:\n%s", line.ascii());
				i++;
			}
			progress = (i * 100) / paragraphs();
			if( progress != lastProg ) {
				emit saveProgress(m_szFileName, progress);
				lastProg = progress;
			}
		}
	}
	f.close();
	if( m_szFileName != filename ) {
		m_szFileName = filename;
		emit fileNameChanged(this, m_szFileName);
	}
	setModified(false);
	emit saved(m_szFileName);
	return true;
}

bool KviSimpleEditor::saveFileAs()
{
	QString newName = KviFileDialog::getSaveFileName(m_szFileName, QString::null, 0);
	if( newName.isEmpty() ) return false; // Aborted

	QFileInfo fi(newName);
	if( fi.exists() ) {
		QString msg;
		msg.sprintf("The file %s already exists!\nDo you wish to overwrite it?", newName.ascii());
		int result = KviMessageBox::warningYesNo(msg, "Warning", this);
		if( result == KviMessageBox::No )
			return false; // Aborted
	}
	return saveFile(newName);
}

QString KviSimpleEditor::fileName()
{
	return m_szFileName;
}

void KviSimpleEditor::toggleFindWidget()
{
	if( m_pFindWidget->isVisible() ) {
		m_lastFindWidgetPosition = m_pFindWidget->pos();
		m_pFindWidget->hide();
		setFocus();
	} else {
		recalculateFindWidget();
		m_pFindWidget->move(m_lastFindWidgetPosition);
		m_pFindWidget->show();
		m_pFindWidget->m_pFindStringEdit->setFocus();
	}
}

void KviSimpleEditor::switchMode()
{
	m_pHighlighter->switchMode();
}

void KviSimpleEditor::indent()
{
	if( hasSelectedText() )
		removeSelection();

	int para, index;
	getCursorPosition(&para, &index);
	insertAt("\t", para, 0);

	setModified(true);
}

void KviSimpleEditor::unindent()
{
	if( hasSelectedText() )
		removeSelection();

	int para, index;
	getCursorPosition(&para, &index);
	QString line = text(para);
	if( line.isEmpty() ) return;

	if( line[0] == '\t' ) {
		line = line.right(line.length() - 1);
		removeParagraph(para);
		insertParagraph(line, para);
	}

	setModified(true);
}

void KviSimpleEditor::commentOut(bool bAlternative)
{
	if( m_pHighlighter->mode() != KviEditorSyntaxHighlighter::CPP )
		return;

	if( hasSelectedText() )
		removeSelection();

	int para, index;
	getCursorPosition(&para, &index);
	if( bAlternative ) {
		QString line = text(para);
		line.prepend("/*");
		line.append("*/");
		removeParagraph(para);
		insertParagraph(line, para);
	} else
		insertAt("//", para, 0);
	setModified(true);
}

void KviSimpleEditor::removeComment()
{
	if( m_pHighlighter->mode() != KviEditorSyntaxHighlighter::CPP )
		return;

	if( hasSelectedText() )
		removeSelection();

	int para, index;
	getCursorPosition(&para, &index);
	QString line = text(para);
	if( line.startsWith("//") ) {
		line = line.right(line.length() - 2);
		removeParagraph(para);
		insertParagraph(line, para);
		setModified(true);
	} else if( line.startsWith("/*") ) {
		if( line.endsWith("*/") && (line.length() > 3) ) {
			line = line.mid(2, line.length() - 4);
			removeParagraph(para);
			insertParagraph(line, para);
			setModified(true);
		}
	}
}

void KviSimpleEditor::findNext()
{
	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No text to find", "Find Next", this);
		return;
	}

	// Start searching from the character to the right of this word
	int para, index, x, y;
	getCursorPosition(&x, &y);
	moveCursor(MoveForward, false);
	getCursorPosition(&para, &index);
	setCursorPosition(x, y);
	if( find(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked(), false, true, &para, &index) ) {
		emit selectionChanged();
		return;
	}
	if( (x == 0) && (y == 0) ) return;

	// End reached
	if( KviMessageBox::questionYesNo(
		"No matches found!\nContinue from the beginning?",
		"Find Next", this) == KviMessageBox::Yes
	) {
		para  = 0;
		index = 0;
		if( find(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked(), false, true, &para, &index) )
			emit selectionChanged();
	}
}

void KviSimpleEditor::findPrev()
{
	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No text to find", "Find Previous", this);
		return;
	}

	// Start searching from the character to the right of this word
	int para, index, x, y;
	getCursorPosition(&x, &y);
	moveCursor(MoveBackward, false);
	getCursorPosition(&para, &index);
	setCursorPosition(x, y);
	if( (x != para) || (y != index) ) {
		if( find(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked(), false, false, &para, &index) )
			emit selectionChanged();
			return;
	}

	// End reached
	if( KviMessageBox::questionYesNo(
		"No matches found!\nContinue from the end?",
		"Find Previous", this) == KviMessageBox::Yes
	) {
		moveCursor(MoveEnd, false);
		getCursorPosition(&para, &index);
		if( find(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked(), false, false, &para, &index) )
			emit selectionChanged();
	}
}

void KviSimpleEditor::findNextRegexp()
{
	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No regular expression to find", "Find Next Regexp", this);
		return;
	}

	int para, index;
	getCursorPosition(&para, &index);
	QString line = text(para);
	QRegExp rx(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
	for( ;; ) {
		if( index < (int) line.length() ) {
			index = rx.search(line, index);
			if( index != -1 ) {
				// Found an occurrence
				setSelection(para, index, para, index + rx.matchedLength());
				emit selectionChanged();
				return;
			}
		}
		if( para >= (paragraphs() - 1) ) {
			// End reached
			if( KviMessageBox::questionYesNo(
				"No matches found!\nContinue from the beginning?",
				"Find Next Regexp", this) == KviMessageBox::No
			) {
				return;
			}
			para = 0;
		} else para++;
		index = 0;
		line = text(para);
	}
}

void KviSimpleEditor::findPrevRegexp()
{
	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No regular expression to find", "Find Previous Regexp", this);
		return;
	}

	int para, index;
	getCursorPosition(&para, &index);
	QString line = text(para);
	line = line.left(index);
	QRegExp rx(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
	for( ;; ) {
		if( index < (int) line.length() ) {
			index = rx.search(line, index);
			if( index != -1 ) {
				// Found an occurrence
				setSelection(para, index, para, index + rx.matchedLength());
				emit selectionChanged();
				return;
			}
		}
		if( para <= 0 ) {
			// End reached
			if( KviMessageBox::questionYesNo(
				"No matches found!\nContinue from the end?",
				"Find Previous Regexp", this) == KviMessageBox::No
			) {
				return;
			}
			para = paragraphs() - 1;
		} else para--;
		index = 0;
		line = text(para);
	}
}

void KviSimpleEditor::replace()
{
	if( !hasSelectedText() ) {
		KviMessageBox::sorry("No text selected", "Replace", this);
		return;
	}
	QString toReplace = m_pFindWidget->m_pReplaceStringEdit->text();
	if( toReplace.isNull() )
		toReplace = "";
	int paraFrom, indexFrom, paraTo, indexTo;
	getSelection(&paraFrom, &indexFrom, &paraTo, &indexTo);
	removeSelection();
	QString line = text(paraFrom);
	line = line.left(indexFrom);
	line.append(toReplace);
	for( int i = paraFrom; i < paraTo; i++ )
		removeParagraph(i);
	line.append(text(paraTo).right(text(paraTo).length() - indexTo));
	removeParagraph(paraTo);
	insertParagraph(line, paraTo);
	setCursorPosition(paraFrom, indexFrom);
	setFocus();
}

void KviSimpleEditor::replaceAndFindNext()
{
	replace();
	findNext();
}

void KviSimpleEditor::replaceAndFindNextRegexp()
{
	replace();
	findNextRegexp();
}

void KviSimpleEditor::replaceAllInSelection()
{
	if( !hasSelectedText() ) {
		KviMessageBox::sorry("No selection to search in", "Replace in Selection", this);
		return;
	}

	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No text to find", "Replace in Selection", this);
		return;
	}
	QString toReplace = m_pFindWidget->m_pReplaceStringEdit->text();
	if( toReplace.isNull() )
		toReplace = "";
	m_pFindWidget->hide();

	int paraFrom, indexFrom, paraTo, indexTo;
	getSelection(&paraFrom, &indexFrom, &paraTo, &indexTo);
	removeSelection();

	int count = 0;
	int index = indexFrom;
	for( int para = paraFrom; para <= paraTo; ) {
		QString line = text(para);
#if 0
		debug("line = \"%s\"", text(para).ascii());
#endif
		index = line.find(toFind, index, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());
		if( (para == paraTo) && (index + (int) toFind.length() > indexTo) )
			break; // Ran past the selection
		if( index != -1 ) {
			// Found an occurrence
			line  = line.replace(index, toFind.length(), toReplace);
			index = index + toReplace.length();
			count++;
			removeParagraph(para);
			insertParagraph(line, para);
			if( (para == paraTo) )
				indexTo = indexTo - toFind.length() + toReplace.length();
		} else {
			para++;
			index = 0;
		}
	}
	m_pFindWidget->show();
	setSelection(paraFrom, indexFrom, paraTo, indexTo);
	emit selectionChanged();

	QString msg;
	msg.sprintf("Replaced %d occurrences", count);
	emit textMessage(this, msg);
	setFocus();
}

void KviSimpleEditor::replaceAllInSelectionRegexp()
{
	if( !hasSelectedText() ) {
		KviMessageBox::sorry("No selection to search in", "Replace in Selection (regexp)", this);
		return;
	}

	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	if( toFind.isEmpty() ) {
		KviMessageBox::sorry("No regular expression to find", "Replace in Selection (regexp)", this);
		return;
	}

	QString toReplace = m_pFindWidget->m_pReplaceStringEdit->text();
	if( toReplace.isNull() )
		toReplace = "";
	m_pFindWidget->hide();

	int paraFrom, indexFrom, paraTo, indexTo;
	getSelection(&paraFrom, &indexFrom, &paraTo, &indexTo);
	removeSelection();
	QRegExp rx(toFind, m_pFindWidget->m_pCaseSensitiveCheckBox->isChecked());

	int count = 0;
	int index = indexFrom;
	for( int para = paraFrom; para <= paraTo; ) {
		QString line = text(para);
		index = rx.search(line, index);
		if( (para == paraTo) && (index + (int) rx.matchedLength() > indexTo) )
			break; // Ran past the selection
		if( index != -1 ) {
			// Found an occurrence
			line  = line.replace(index, rx.matchedLength(), toReplace);
			index = index + toReplace.length();
			count++;
			removeParagraph(para);
			insertParagraph(line, para);
			if( (para == paraTo) )
				indexTo = indexTo - rx.matchedLength() + toReplace.length();
		} else {
			para++;
			index = 0;
		}
	}
	m_pFindWidget->show();
	setSelection(paraFrom, indexFrom, paraTo, indexTo);
	emit selectionChanged();

	QString msg;
	msg.sprintf("Replaced %d occurrences", count);
	emit textMessage(this, msg);
	setFocus();
}

void KviSimpleEditor::replaceAll()
{
	m_pFindWidget->hide();
	if( KviMessageBox::warningYesNo(
		"This may be an irreversible action!\nReplace all matches in the document?",
		"Replace All", this) == KviMessageBox::No
	) {
		m_pFindWidget->show();
		return;
	}

	setSelection(0, 0, paragraphs() - 1, text(paragraphs() - 1).length());
	replaceAllInSelection();
	removeSelection();
	emit selectionChanged();
}

void KviSimpleEditor::replaceAllRegexp()
{
	m_pFindWidget->hide();
	if( KviMessageBox::warningYesNo(
		"This may be an irreversible action!\nReplace all matches in the document?",
		"Replace All (regexp)", this) == KviMessageBox::No
	) {
		m_pFindWidget->show();
		return;
	}

	setSelection(0, 0, paragraphs() - 1, text(paragraphs() - 1).length());
	replaceAllInSelectionRegexp();
	removeSelection();
	emit selectionChanged();
}

void KviSimpleEditor::recordKeystrokes()
{
	if( m_bRecordingKeystrokes ) {
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}
	m_pKeystrokes->clear();
	m_bRecordingKeystrokes = true;
	emit recordingKeystrokes(true);
}

void KviSimpleEditor::replayKeystrokes()
{
	if( m_bRecordingKeystrokes ) {
		m_bRecordingKeystrokes = false;
		emit recordingKeystrokes(false);
		return;
	}

	for( KviSimpleEditorKeystroke *k = m_pKeystrokes->first(); k; k = m_pKeystrokes->next() ) {
		QKeyEvent ev(QEvent::KeyPress, k->key, k->ascii, k->state);
		keyPressEvent(&ev);
	}
}

bool KviSimpleEditor::eventFilter(QObject *o, QEvent *e)
{
	switch( e->type() ) {
		case QEvent::MouseButtonPress: {
			QMouseEvent *ev = (QMouseEvent *) e;
			mousePressEvent(ev);
			if( ev->isAccepted() )
				return true;
			break;
		}
		case QEvent::KeyPress: {
			QKeyEvent *ev = (QKeyEvent *) e;
			keyPressEvent(ev);
			if( ev->isAccepted() )
				return true;
			break;
		}
		default:
			break;
	}
	return QTextEdit::eventFilter(o, e);
}

void KviSimpleEditor::keyPressEvent(QKeyEvent *e)
{
	if( m_bRecordingKeystrokes ) {
		if( !(((e->key() == Qt::Key_T) || (e->key() == Qt::Key_R)) && (e->state() & ControlButton)) ) {
			KviSimpleEditorKeystroke *k = new KviSimpleEditorKeystroke();
			k->ascii = e->ascii();
			k->key   = e->key();
			k->state = e->state();
			m_pKeystrokes->append(k);
		}
	}

	if( e->state() & ControlButton ) {
		bool bShiftOn = e->state() & ShiftButton;
		switch( e->key() ) {
			case Qt::Key_A:     saveFileAs();                  e->accept(); return; break;
			case Qt::Key_F:     toggleFindWidget();            e->accept(); return; break;
			case Qt::Key_G:     commentOut(bShiftOn);          e->accept(); return; break;
			case Qt::Key_I:     indent();                      e->accept(); return; break;
			case Qt::Key_M:     switchMode();                  e->accept(); return; break;
			case Qt::Key_O:     loadFile();                    e->accept(); return; break;
			case Qt::Key_P:     removeComment();               e->accept(); return; break;
			case Qt::Key_R:     recordKeystrokes();            e->accept(); return; break;
			case Qt::Key_S:     saveFile();                    e->accept(); return; break;
			case Qt::Key_T:     replayKeystrokes();            e->accept(); return; break;
			case Qt::Key_U:     unindent();                    e->accept(); return; break;
		}
	}

	if( m_pFindWidget->isVisible() ) {
		switch( e->key() ) {
			case Qt::Key_Escape:
				toggleFindWidget();
				e->accept();
				return;
				break;
			case Qt::Key_Tab:
				m_pFindWidget->focusNextPrevChild(true);
				e->accept();
				return;
				break;
			case Qt::Key_Return: // Fall-through
			case Qt::Key_Enter:
				findNext();
				e->accept();
				return;
				break;
		}
	}
	QTextEdit::keyPressEvent(e);
}

void KviSimpleEditor::mousePressEvent(QMouseEvent *e)
{
	if( e->button() & RightButton ) {
		contextPopup();
		e->accept();
	} else QTextEdit::mousePressEvent(e);
}

void KviSimpleEditor::resizeEvent(QResizeEvent *ev)
{
	bool b = m_pFindWidget->isVisible();
	if( b ) toggleFindWidget();

	QTextEdit::resizeEvent(ev);
	recalculateFindWidget();

	if( b ) toggleFindWidget();
}

void KviSimpleEditor::recalculateFindWidget()
{
	int x = width() - m_pFindWidget->width();
	if( verticalScrollBar()->isVisible() )
		x -= verticalScrollBar()->width();
	m_lastFindWidgetPosition = QPoint(x, 0);
}

void KviSimpleEditor::contextPopup()
{
	m_pContextPopup->clear();
	m_pContextPopup->insertTearOffHandle();

	int id = m_pContextPopup->insertItem("&Undo", this, SLOT(undo()), KviAccel::stringToKey("Ctrl+Z"));
	if( !isUndoAvailable() )
		m_pContextPopup->setItemEnabled(id, false);
	id = m_pContextPopup->insertItem("&Redo", this, SLOT(redo()), KviAccel::stringToKey("Ctrl+Y"));
	if( !isRedoAvailable() )
		m_pContextPopup->setItemEnabled(id, false);
	m_pContextPopup->insertSeparator();

	id = m_pContextPopup->insertItem("&Cut", this, SLOT(cut()), KviAccel::stringToKey("Ctrl+X"));
	if( !hasSelectedText() )
		m_pContextPopup->setItemEnabled(id, false);
	id = m_pContextPopup->insertItem("&Copy", this, SLOT(copy()), KviAccel::stringToKey("Ctrl+C"));
	if( !hasSelectedText() )
		m_pContextPopup->setItemEnabled(id, false);
	id = m_pContextPopup->insertItem("&Paste", this, SLOT(paste()), KviAccel::stringToKey("Ctrl+V"));
	QString t = KviApplication::clipboard()->text();
	if( t.isEmpty() )
		m_pContextPopup->setItemEnabled(id, false);

	m_pContextPopup->insertSeparator();
	m_pFindPopup->clear();
	m_pFindPopup->insertTearOffHandle();

	m_pFindPopup->insertItem(m_pFindWidget->isVisible() ? "Hide &Find Widget" : "Show &Find Widget",
		this, SLOT(toggleFindWidget()), KviAccel::stringToKey("Ctrl+F")
	);
	m_pFindPopup->insertSeparator();

	QString toFind = m_pFindWidget->m_pFindStringEdit->text();
	bool bHasToFind = !toFind.isEmpty();
	id = m_pFindPopup->insertItem("Find &Next", this, SLOT(findNext()), KviAccel::stringToKey("Ctrl+N"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem("Find &Previous", this, SLOT(findPrev()), KviAccel::stringToKey("Ctrl+P"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem("Find Next &Regexp", this, SLOT(findNextRegexp()), KviAccel::stringToKey("Ctrl+B"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem("Find Previous Reg&exp", this, SLOT(findPrevRegexp()), KviAccel::stringToKey("Ctrl+G"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	m_pFindPopup->insertSeparator();

	id = m_pFindPopup->insertItem("Rep&lace", this, SLOT(replace()), KviAccel::stringToKey("Ctrl+J"));
	if( !hasSelectedText() )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem("Replace &All", this, SLOT(replaceAll()), KviAccel::stringToKey("Ctrl+K"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem("Replace All (Re&gexp)", this, SLOT(replaceAllRegexp()), KviAccel::stringToKey("Ctrl+L"));
	if( !bHasToFind )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem(
		"Replace All in &Selection", this, SLOT(replaceAllInSelection()), KviAccel::stringToKey("Ctrl+E")
	);
	if( !(bHasToFind && hasSelectedText()) )
		m_pFindPopup->setItemEnabled(id, false);
	id = m_pFindPopup->insertItem(
		"Replace All &in Selection (regexp)", this, SLOT(replaceAllInSelectionRegexp()), KviAccel::stringToKey("Ctrl+D")
	);
	if( !(bHasToFind && hasSelectedText()) )
		m_pFindPopup->setItemEnabled(id, false);
	m_pContextPopup->insertItem("&Find", m_pFindPopup);
	m_pContextPopup->insertSeparator();

	id = m_pContextPopup->insertItem("&Save", this, SLOT(saveFile()), KviAccel::stringToKey("Ctrl+S"));
	if( !isModified() )
		m_pContextPopup->setItemEnabled(id, false);
	m_pContextPopup->insertItem("Save &As...", this, SLOT(saveFileAs()), KviAccel::stringToKey("Ctrl+A"));
	m_pContextPopup->insertSeparator();

	m_pContextPopup->insertItem(m_bRecordingKeystrokes ? "Stop &Recording Keystrokes" : "&Record Keystrokes",
		this, SLOT(recordKeystrokes()), KviAccel::stringToKey("Ctrl+R")
	);
	id = m_pContextPopup->insertItem("Replay Keys&trokes", this, SLOT(replayKeystrokes()), KviAccel::stringToKey("Ctrl+T"));
	if( m_bRecordingKeystrokes || (m_pKeystrokes->count() == 0) )
		m_pContextPopup->setItemEnabled(id, false);
	m_pContextPopup->insertSeparator();

	id = m_pContextPopup->insertItem("C&lear", this, SLOT(clear()));
	id = m_pContextPopup->insertItem("S&elect All", this, SLOT(selectAll()));

	m_pContextPopup->popup(QCursor::pos());
}

void KviSimpleEditor::clear()
{
	if( !text().isEmpty() )
		setModified(true);
	QTextEdit::clear();
}

#include "m_kvi_simple_editor.moc"
