// =============================================================================
//
//      --- kvi_biff_systray.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Authors:
//       Krzysztof Godlewski    <kristoff@poczta.wprost.pl>
//       Szymon Stefanek        <stefanek@tin.it>
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviBiffSysTray"

#include <qcursor.h>
#include <qobjectlist.h>
#include <qpainter.h>
#include <qtimer.h>

#define __KVIRC_PLUGIN__

#include "kvi_biff.h"
#include "kvi_biff_mailbox.h"
#include "kvi_biff_message.h"
#include "kvi_biff_systray.h"
#include "kvi_locale.h"
#include "kvi_popupmenu.h"
#include "kvirc_plugin.h"
#include "kvi_systray_icondisplay.h"

extern KviBiff *g_pBiff;

KviBiffSysTray::KviBiffSysTray(KviSysTray *parent, KviFrame *frm, const char *tooltip)
	: KviSysTrayWidget(parent, tooltip),
	  m_pFrm(frm),
	  m_popup(new KviPopupMenu()), m_mBoxPopup(new KviPopupMenu()),
	 m_pTextTimer(new QTimer(this))
{
	g_pBiff->registerSysTrayWidget(this);
	//
	// Create the popup menu:
	connect(m_mBoxPopup, SIGNAL(aboutToShow()), SLOT(fillMailboxPopup()));

	m_popup->insertItem(__tr("Check mail in"), m_mBoxPopup);
	m_popup->insertItem(__tr("Check all mailboxes"), g_pBiff, SLOT(slotCheckAll()));
	m_popup->insertItem(__tr("Configure mailboxes"), g_pBiff, SLOT(slotConfigureMailboxes()));
	m_popup->insertItem(__tr("Configure plugin"),    g_pBiff, SLOT(slotConfig()));
	m_popup->insertSeparator();
	// TODO: Sometimes the app freezes after doing this
	m_popup->insertItem(__tr("Stop checking mail"), g_pBiff, SLOT(stopCheckingMail()));
	m_popup->insertItem(__tr("Hide this widget"), this, SLOT(slotHide()));

	m_textToShow.sprintf(BIFF_NULL_STRING);
}

KviBiffSysTray::~KviBiffSysTray()
{
	g_pBiff->unregisterSysTrayWidget(this);
}

QString KviBiffSysTray::icon()
{
	return g_pBiff->currentWidgetIcon();
}

void KviBiffSysTray::paintEvent(QPaintEvent *e)
{
	if( !isVisible() ) return;

	QPainter pa(m_pSysTray->getMemoryBuffer());

	paintDefaultBackground(&pa);
	pa.setPen(m_pSysTray->getForeground());

	KviStr s, s2 = BIFF_NULL_STRING;
	if( m_showingText && !kvi_strEqualCS(m_textToShow.ptr(), BIFF_NULL_STRING) ) {
	    if( m_textToShow.contains('\n') ) {
	        s  = m_textToShow.left(m_textToShow.findFirstIdx('\n'));
	        s2 = m_textToShow.middle(m_textToShow.findFirstIdx('\n') + 1, 100);
	    } else s = m_textToShow;
	} else {
	    s.sprintf("*** The Biff Plugin ***");
	    s2.sprintf("Press right mouse button for menu.");
	}

	if( height() > 32 ) { // Big mode
	    g_pOptions->m_fntSysTray.setPointSize(12);
	    pa.setFont(g_pOptions->m_fntSysTray);

	    pa.drawText(4, height() - 24, s.ptr(), s.len());
	    if( !kvi_strEqualCS(s2.ptr(), BIFF_NULL_STRING) ) {
	        pa.drawText(4, height() - 8, s2.ptr(), s2.len());
	    }
	} else { // Small mode
	    if( !kvi_strEqualCS(s2.ptr(), BIFF_NULL_STRING) )
	        s.append(" " + s2);
	    g_pOptions->m_fntSysTray.setPointSize(12);
	    pa.setFont(g_pOptions->m_fntSysTray);
	    pa.drawText(4, height() - 6, s.ptr(), s.len());
	}

	paintDefaultFrame(&pa);
	bitBlt(this, 0, 0, m_pSysTray->getMemoryBuffer(), 0, 0, width(), height());
}

void KviBiffSysTray::mousePressEvent(QMouseEvent *e)
{
	switch( e->button() ) {
	    case QMouseEvent::RightButton:
	        m_popup->exec(QCursor::pos());
	        break;
	    default:
	        break;
	}
}

void KviBiffSysTray::slotHide()
{
	m_pSysTray->removeWidget(this, true);
}

void KviBiffSysTray::showText(const char *text, uint time, bool p)
{
	m_textToShow = text;
	m_showingText = true;
	repaint(false);
	if( m_pTextTimer->isActive() )
	    m_pTextTimer->stop();
	if( !p ) { // Message not permanent
	    m_pTextTimer->start(time);
	    connect(m_pTextTimer, SIGNAL(timeout()), SLOT(showOrginalText()));
	}

	// Find this systray's icon display, to force paint event.
	QObjectList *list = m_pSysTray->queryList("KviSysTrayIconDisplay", 0, false, true);
	QObjectListIt it(*list);
	KviSysTrayIconDisplay *id = (KviSysTrayIconDisplay *) it.current();
	if( id ) {
	    id->setIcon(icon());
	    id->repaint(true);
	} else {
		// Do nothing if icon not found
	}
	delete list;
}

void KviBiffSysTray::showOrginalText()
{
	m_showingText = false;
	m_textToShow.sprintf(BIFF_NULL_STRING);
	repaint(false);
}

void KviBiffSysTray::fillMailboxPopup()
{
	m_mBoxPopup->clear();
	for( KviBiffMailbox *m = g_pBiff->mailboxList()->first(); m; m = g_pBiff->mailboxList()->next() ) {
	    KviStr s;
	    s.sprintf("%s@%s", m->username(), m->hostname());
		// The popup-id of inserted item will be correct mailbox's index in the list
		// (m_pMailboxList in g_pBiff)
	    int id = g_pBiff->mailboxList()->find(m);
	    m_mBoxPopup->insertItem(s.ptr(), g_pBiff, SLOT(slotCheckMailIn(int)), 0, id);
	}
}

#include "m_kvi_biff_systray.moc"
