// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptSpinBox"

#include <qspinbox.h>

#include "kvi_error.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_spinbox.h"

/*
	@class: spinbox
	@short:
		A spinbox widget, sometimes called 'little arrows widget' :)
	@inherits:
		[class]widget[/class]<br>
		[class]object[/object]
	@functions:
		!fn: $setValue(&lt;new_val&gt;)
		Sets the current value of the spinbox to &lt;new_val&gt;.<br>
		Returns nothing.

		!fn: $value()
		Returns the current raw value of the spinbox (ie: including
		suffix).<br>

		!fn: $setMinValue()
		Sets the minimal value of the spinbox.<br>
		Returns nothing.

		!fn: $setMaxValue()
		Sets the maximal value of the spinbox.<br>
		Returns nothing.

		!fn: $setStep(&lt;new_val&gt;)
		Sets the spinbox value increment/decrement every time an arrow
		button is pressed.<br>
		Returns nothing.

		!fn: $setWrapping(&lt;state&gt;)
		Sets wrapping to &lt;state&gt; which is a boolean value:<br>
		If &lt;state&gt; is 0 wrapping is disabled, if it has any other
		value, wrapping is enabled.<br>
		Enabling wrapping will allow to step from the highest value and
		vice-versa. By default, wrapping is turned off.

		!fn: $wrapping()
		Returns 0 if wrapping is disabled, 1 otherwise.

		!fn: $setSuffix(&lt;new_val&gt;)
		Sets the text to be displayed after the spinbox value.<br>
		Returns nothing.

		!fn: $suffix()
		Returns current spinbox suffix.

		!fn: $cleanValue()
		Returns spinbox value stripped from suffix.

	@description:
		The spinbox is a widget sometimes called little arrows widget or
		up-down widget.<br>
		It allows to pick a numeric value either by typing it directly
		to the spinbox, or clicking up/down buttons.<br>

	@examples:
		The simplest way to use a spinbox is:
		<example>
		%s = $new(spinbox, $root, my_spinbox, 1, 24, 1)
		# the spinbox now can show values ranged between 10-100.
		# Every time user clicks an arrow button value will be<br>
		# increased/decreased by 1.<br>
		# Now you ask the user to set the number of hours he usually<br>
		# sleeps :)<br>
		%s->$setSuffix(" hours");
		%var = %s->$value();
		echo "Usually you sleep for %var.";
		# That is it :)
		</example>
		Or, you can do it the "hard way":<br>
		<example>
		%s = $new(spinbox, $root, my_spinbox, 1, 24, 1)
		%s->$setSuffix(" hours");
		%var1 = %s->$cleanValue();
		%var2 = %s->$suffix();
		echo "Usually you sleep for %var1%var2.";
		# This will produce same result as previous example.
		</example>

	@seealso:
		class [class]object[/class], <br>
		class [class]widget[/class], <br>
		<a href="syntax_objects.kvihelp">Objects documentation</a><br>

*/

/**
 * SPINBOX class
 */
void KviScriptSpinBox::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("setValue",    (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETVALUE);
	d->addBuiltinFunction("value",       (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_VALUE);
	d->addBuiltinFunction("setMinValue", (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETMINVALUE);
	d->addBuiltinFunction("setMaxValue", (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETMAXVALUE);
	d->addBuiltinFunction("wrapping",    (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_WRAPPING);
	d->addBuiltinFunction("setWrapping", (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETWRAPPING);
	d->addBuiltinFunction("setStep",     (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETSTEP);
	d->addBuiltinFunction("setSuffix",   (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SETSUFFIX);
	d->addBuiltinFunction("suffix",      (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_SUFFIX);
	d->addBuiltinFunction("cleanValue",  (scriptObjectFunction) &KviScriptSpinBox::builtinFunction_CLEANVALUE);
}

KviScriptSpinBox::KviScriptSpinBox(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: KviScriptWidget(cntrl, p, name, pDef)
{
	// Nothing here
}

KviScriptSpinBox::~KviScriptSpinBox()
{
	// Nothing here
}

bool KviScriptSpinBox::init(QPtrList<KviStr> *params)
{
	if( parent() ) {
		if( parent()->inherits("KviScriptWidget") ) {
			m_pWidget = new QSpinBox(((KviScriptWidget *) parent())-> m_pWidget, name());
		}
	}

	if( !m_pWidget )
		m_pWidget = new QSpinBox(0, name());
	m_bAutoDestroyControlledWidget = true;
	m_pWidget->installEventFilter(this);
	connect(m_pWidget, SIGNAL(destroyed()), this, SLOT(widgetDestroyed()));
	if( params ) {
		KviStr *pS  = params->first();
		KviStr *pS2 = params->next();
		KviStr *pS3 = params->next();
		int val;
		bool bOk;
		if( pS ) {
			val = pS->toInt(&bOk);
			if( !bOk )
				return false;
			((QSpinBox *) m_pWidget)->setMinValue(val);
		}

		if( pS2 ) {
			val = pS2->toInt(&bOk);
			if( !bOk )
				return false;
			((QSpinBox *) m_pWidget)->setMaxValue(val);
		}
		if( pS3 )
		{
			val = pS3->toInt(&bOk);
			if( !bOk )
				return false;
			((QSpinBox *) m_pWidget)->setLineStep(val);
		}
	}
	return true;
}

int KviScriptSpinBox::builtinFunction_SETVALUE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int val;
			val = pS->toInt(&bOk);
			if( !bOk )
				return KVI_ERROR_InvalidParameter;
			((QSpinBox *) m_pWidget)->setValue(val);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_VALUE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			buffer.append(((QSpinBox *) m_pWidget)->text());
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETMINVALUE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int val = pS->toInt(&bOk);
			if( bOk ) {
				((QSpinBox *) m_pWidget)->setMinValue(val);
				return KVI_ERROR_Success;
			}
			else return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETMAXVALUE(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int val = pS->toInt(&bOk);
			if( bOk ) {
				((QSpinBox *) m_pWidget)->setMaxValue(val);
				return KVI_ERROR_Success;
			} else return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_WRAPPING(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(((QSpinBox *) m_pWidget)->wrapping() ? '1' : '0');
	return KVI_ERROR_Success;
}

int KviScriptSpinBox::builtinFunction_SETWRAPPING(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		bool bEn = true;
		KviStr *pS = params->first();
		if( pS ) {
			int tmp = pS->toInt();
			if( tmp == 0 )
				bEn = false;
		}
		((QSpinBox *) m_pWidget)->setWrapping(bEn);
		return KVI_ERROR_Success;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETSTEP(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			bool bOk;
			int val = pS->toInt(&bOk);
			if( bOk ) {
				((QSpinBox *) m_pWidget)->setLineStep(val);
				return KVI_ERROR_Success;
			} else return KVI_ERROR_InvalidParameter;
		}
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SETSUFFIX(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			((QSpinBox *) m_pWidget)->setSuffix(pS->ptr());
			return KVI_ERROR_Success;
		} else return KVI_ERROR_InvalidParameter;
	}
	return KVI_ERROR_MissingParameter;
}

int KviScriptSpinBox::builtinFunction_SUFFIX(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(((QSpinBox *) m_pWidget)->suffix().latin1());
	return KVI_ERROR_Success;
}

int KviScriptSpinBox::builtinFunction_CLEANVALUE(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(((QSpinBox *) m_pWidget)->cleanText().latin1());
	return KVI_ERROR_Success;
}

#include "m_kvi_script_spinbox.moc"
