// =============================================================================
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviScriptObject"

#include <qtimer.h>

#include "kvi_console.h"
#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_mirccntrl.h"
#include "kvi_script_object.h"
#include "kvi_script_objectclassdefinition.h"
#include "kvi_script_objectcontroller.h"
#include "kvi_userparser.h"
#include "kvi_variablecache.h"
#include "kvi_window.h"

/*
	@class: object
	@short:
		The base class of all the objects.
	@inherits:
		itself only
	@functions:
		!fn: $name()
		Returns the name of this object.

		!fn: $class()
		Returns the class name of this object.

		!fn: $parent()
		Returns the ID of the parent object or 0 if this is a toplevel object.

		!fn: $inheritedClasses()
		Returns a comma separated list of inherited classes.

		!fn: $inherits(&lt;class_name&gt;)
		Returns '1' if this object inherits &lt;class_name&gt;.
		An instance of a class always inherits the class itself (eg. object inherits object).

		!fn: $children()
		Returns a comma separated list of children ID, or nothing
		if this object has no children.

		!fn: $findChild(&lt;child_name&gt;)
		Searches for a direct child with a specified name and returns its ID
		if found, 0 otherwise.<br>
		If more than one child has the same name, only the first one is returned.

		!fn: $findDirectChild(&lt;child_name&gt;)
		Searches for a direct child with a specified name and returns its ID
		if found, 0 otherwise.<br>
		This function descends only one level in the children tree,
		and is really faster than $findChild().<br>
		If more than one child has the same name, only the first one is returned.

		!fn: $emit(&lt;signal_name&gt;[,parameters])
		Emits a specified signal, passing the (optional) parameters.<br>
		See the <a href="syntax_objects.kvihelp#signals">signals and slots documentation</a>.<br>

	@events:
		!ev: OnDestroy()
		Triggered just before the object destruction.

	@description:
		The base type of all the objects.<br>
*/
KviScriptObject::KviScriptObject(
	KviScriptObjectController *cntrl, KviScriptObject *p, const char *name, KviScriptObjectClassDefinition *pDef)
	: QObject(p, name)
{
	m_pClassDefinition  = pDef;
	m_pController       = cntrl;
	m_bDying            = false;
	m_bInDelayedDestroy = false;
	m_pChildList        = new QPtrList<KviScriptObject>;
	m_pChildList->setAutoDelete(true);
	m_pInheritanceList  = new QPtrList<KviStr>;
	m_pInheritanceList->setAutoDelete(true);
	m_pVarCache         = new KviVariableCache();
	m_pEventDict        = new QAsciiDict<KviStr>(5, false);
	m_pEventDict->setAutoDelete(true);
	m_pController->getUniqueId(m_szId);
	m_bNeedRecreate     = false;
	m_pSlotDict         = new QAsciiDict<KviScriptObjectSlotList>(5, false);
	m_pSlotDict->setAutoDelete(true);
	m_pSignalList       = new QPtrList<KviScriptObjectSignal>;
	m_pSignalList->setAutoDelete(true);
	KviScriptObjectClassDefinition *def = pDef;
	while( def ) {
		m_pInheritanceList->append(new KviStr(def->getClass()));
		def = def->inheritedClass();
	}
	if( p )
		p->addChild(this);
}

KviScriptObject::~KviScriptObject()
{
	KviStr parms;
	triggerEvent("OnDestroy", parms);
	KviStr buffer;
	callFunction("destructor", 0, buffer);

	m_bDying = true;
	// Remove the children one by one
	m_pChildList->setAutoDelete(false);
	while( m_pChildList->first() ) {
		KviScriptObject *tmp = m_pChildList->first();
		delete tmp; // Trigger onDestroy inside -- "this" is still valid there! Tricky :)
		m_pChildList->removeRef(tmp);
	}
	delete m_pChildList;
	m_pChildList = 0;

	if( parentObject() )
		parentObject()->removeChild(this, false);
	else {
		__range_valid(m_pController->m_pTopLevelObject == this);
		m_pController->m_pTopLevelObject = 0;
	}
	delete m_pInheritanceList; m_pInheritanceList = 0;
	delete m_pEventDict;       m_pEventDict       = 0;
	delete m_pVarCache;        m_pVarCache        = 0;

	// Disconnect all signals
	for( KviScriptObjectSignal *s = m_pSignalList->first(); s; s = m_pSignalList->next() ) {
		s->pObject->unregisterSlot(s->szSignalName.ptr(), s->szSlotName.ptr(), this);
	}
	// Notify all slots that we are dying; disconnect all the slots
	QAsciiDictIterator<KviScriptObjectSlotList> it(*m_pSlotDict);
	for( KviScriptObjectSlotList *l = it.current(); l; ++it ) {
		for( KviScriptObjectSlot *sl = l->first(); sl; sl = l->next() ) {
			sl->pObject->unregisterSignal(it.currentKey(), sl->szFunction.ptr(), this);
		}
	}
	delete m_pSlotDict;   m_pSlotDict   = 0;
	delete m_pSignalList; m_pSignalList = 0;

	if( m_bNeedRecreate ) {
		__range_invalid(parentObject());
		m_pController->createTopLevelObject();
	}
}

void KviScriptObject::registerSlot(const char *signalName, const char *slotName, KviScriptObject *object)
{
	KviScriptObjectSlot *sl = new KviScriptObjectSlot();
	sl->pObject    = object;
	sl->szFunction = slotName;
	KviScriptObjectSlotList *l = m_pSlotDict->find(signalName);
	if( !l ) {
		l = new KviScriptObjectSlotList();
		l->setAutoDelete(true);
		m_pSlotDict->insert(signalName, l);
	}
	l->append(sl);
}

void KviScriptObject::unregisterSlot(const char *signalName, const char *slotName, KviScriptObject *object)
{
	KviScriptObjectSlotList *l = m_pSlotDict->find(signalName);
	if( l ) {
		for( KviScriptObjectSlot *sl = l->first(); sl; sl = l->next() ) {
			if( sl->pObject == object ) {
				if( kvi_strEqualCI(sl->szFunction.ptr(), slotName) ) {
					l->removeRef(sl); // Auto-deleting
					if( l->isEmpty() )
						m_pSlotDict->remove(signalName);
					return;
				}
			}
		}
	}
	debug("No such slot (%s) for signal (%s)", slotName, signalName);
}

void KviScriptObject::registerSignal(const char *signalName, const char *slotName, KviScriptObject *object)
{
	KviScriptObjectSignal *s = new KviScriptObjectSignal();
	s->szSlotName   = slotName;
	s->szSignalName = signalName;
	s->pObject      = object;
	m_pSignalList->append(s);
}

void KviScriptObject::unregisterSignal(const char *signalName, const char *slotName, KviScriptObject *object)
{
	for( KviScriptObjectSignal *s = m_pSignalList->first(); s; s = m_pSignalList->next() ) {
		if( s->pObject == object ) {
			if( kvi_strEqualCI(s->szSlotName.ptr(), slotName) ) {
				if( kvi_strEqualCI(s->szSignalName.ptr(), signalName) ) {
					m_pSignalList->removeRef(s);
					return;
				}
			}
		}
	}
	debug("No such signal (%s) for slot (%s)", signalName, slotName);
}

void KviScriptObject::connectSignalSlot(
	KviScriptObject *src, const char *signalName, KviScriptObject *dst, const char *slotName)
{
	src->registerSlot(signalName, slotName, dst);
	dst->registerSignal(signalName, slotName, src);
}

void KviScriptObject::disconnectSignalSlot(
	KviScriptObject *src, const char *signalName, KviScriptObject *dst, const char *slotName)
{
	src->unregisterSlot(signalName, slotName, dst);
	dst->unregisterSignal(signalName, slotName, src);
}

KviScriptObjectController *KviScriptObject::controller()
{
	return m_pController;
}

void KviScriptObject::die(bool bDelayed)
{
	if( m_bInDelayedDestroy ) {
		debug("Oops... object is already being destroyed");
		return;
	}
	if( bDelayed )
		dieOutOfThisEventStep();
	else
		delete this;
}

void KviScriptObject::dieAndRecreate(bool bDelayed)
{
	m_bNeedRecreate = true;
	die(bDelayed);
}

void KviScriptObject::dieOutOfThisEventStep()
{
	if( m_bInDelayedDestroy ) {
		debug("Oops... object is already being destroyed");
		return;
	}
	m_bInDelayedDestroy = true;
	QTimer::singleShot(0, this, SLOT(delayedDie()));
}

void KviScriptObject::delayedDie()
{
	delete this;
}

bool KviScriptObject::inDelayedDestroy()
{
	return m_bInDelayedDestroy;
}

KviVariableCache *KviScriptObject::varCache()
{
	return m_pVarCache;
}

KviScriptObject *KviScriptObject::parentObject()
{
	return (KviScriptObject *) parent();
}

QPtrList<KviScriptObject> *KviScriptObject::childrenList()
{
	return m_pChildList;
}

void KviScriptObject::triggerEvent(const char *evName, const KviStr &parms)
{
	KviStr *pEvBuf = m_pEventDict->find(evName);
	if( pEvBuf )
		m_pController->m_pUserParser->triggerObjectEvent(this, evName, pEvBuf->ptr(), parms);
}

void KviScriptObject::initializeClassDefinition(KviScriptObjectClassDefinition *d)
{
	d->addBuiltinFunction("name",             &KviScriptObject::builtinFunction_NAME);
	d->addBuiltinFunction("class",            &KviScriptObject::builtinFunction_CLASS);
	d->addBuiltinFunction("parent",           &KviScriptObject::builtinFunction_PARENT);
	d->addBuiltinFunction("inherits",         &KviScriptObject::builtinFunction_INHERITS);
	d->addBuiltinFunction("inheritedClasses", &KviScriptObject::builtinFunction_INHERITEDCLASSES);
	d->addBuiltinFunction("children",         &KviScriptObject::builtinFunction_CHILDREN);
	d->addBuiltinFunction("findDirectChild",  &KviScriptObject::builtinFunction_FINDDIRECTCHILD);
	d->addBuiltinFunction("findChild",        &KviScriptObject::builtinFunction_FINDCHILD);
	d->addBuiltinFunction("emit",             &KviScriptObject::builtinFunction_EMIT);
}

bool KviScriptObject::emitSignal(const char *sigName, QPtrList<KviStr> *params, KviStr &buffer)
{
	if( *sigName == '\0' )
		return false;
	KviScriptObjectSlotList *slotsL = m_pSlotDict->find(sigName);
	if( slotsL ) {
		// Trick; copy the list of slots and use that one since
		// the slots may be unregistered inside their handlers
		KviScriptObjectSlotList l(*slotsL); // Shallow copy
		l.setAutoDelete(false);
		int idx = 0;
		// Loop through all the slots
		for( KviScriptObjectSlot *sl = l.first(); sl; sl = l.next() ) {
			// Ensure that it has not been unregistered yet
			if( slotsL->findRef(sl) != -1 ) {
				// OK; slot not deleted yet
				if( idx > 0 )
					buffer.append(',');
				idx++;
				// Copy the param list: it will be deleted inside callFunction()
				QPtrList<KviStr> *tmp = new QPtrList<KviStr>;
				tmp->setAutoDelete(true);
				if( params ) {
					for( KviStr *s = params->first(); s; s = params->next() )
						tmp->append(new KviStr(*s));
				}
				// Call it
				m_pController->m_pUserParser->setLastSignalSender(id());
				int iRet = sl->pObject->callFunction(sl->szFunction.ptr(), tmp, buffer);
				// And eventually print the error
				if( iRet != KVI_ERROR_Success ) {
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,
						"[parser]: error in SLOT call: %s", kvi_getErrorString(iRet)
					);
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,
						"[parser]: caller object: %s class: %s name: %s", id(), getClass(), getName()
					);
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,
						"[parser]: caller signal: %s", sigName
					);
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,
						"[parser]: receiver object: %s class: %s name: %s",
						sl->pObject->id(), sl->pObject->getClass(), sl->pObject->getName()
					);
					m_pController->m_pUserParser->m_pConsole->output(KVI_OUT_INTERNAL,
						"[parser]: receiver slot: %s", sl->szFunction.ptr()
					);
				}
				delete tmp;
			}
		}
	}
	return true;
}

int KviScriptObject::builtinFunction_EMIT(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( params ) {
		KviStr *pS = params->first();
		if( pS ) {
			// Trick; remove the first parameter
			bool bAutoDelete = params->autoDelete();
			params->setAutoDelete(false);
			params->removeFirst();
			params->setAutoDelete(bAutoDelete);
			// Emit the signal
			emitSignal(pS->ptr(), params, buffer);
			// Reinsert it for deletion
			params->insert(0, pS);
			return KVI_ERROR_Success;
		}
	}
	return KVI_ERROR_MissingSignalName;
}

int KviScriptObject::builtinFunction_NAME(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(name());
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_CLASS(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(getClass());
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_PARENT(QPtrList<KviStr> *params, KviStr &buffer)
{
	buffer.append(parentObject() ? parentObject()->id() : "0");
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_CHILDREN(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool bFirst = true;
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( !bFirst ) {
			bFirst = false;
			buffer.append(',');
		} else bFirst = false;
		buffer.append(o->id());
	}
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_INHERITEDCLASSES(QPtrList<KviStr> *params, KviStr &buffer)
{
	bool bFirst = true;
	for( KviStr *s = m_pInheritanceList->first(); s; s = m_pInheritanceList->next() ) {
		if( !bFirst ) {
			bFirst = false;
			buffer.append(',');
		} else bFirst = false;
		buffer.append(s->ptr());
	}
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_FINDDIRECTCHILD(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviStr *pStr = params->first();
	if( !pStr )
		return KVI_ERROR_MissingParameter;
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( kvi_strEqualCI(o->getName(), pStr->ptr()) ) {
			buffer.append(o->id());
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_FINDCHILD(QPtrList<KviStr> *params, KviStr &buffer)
{
	KviStr *pStr = params->first();
	if( !pStr )
		return KVI_ERROR_MissingParameter;
	KviScriptObject *o = findObjectByName(pStr->ptr());
	if( o )
		buffer.append(o->id());
	else
		buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::builtinFunction_INHERITS(QPtrList<KviStr> *params, KviStr &buffer)
{
	if( !params )
		return KVI_ERROR_MissingParameter;
	KviStr *pStr = params->first();
	if( !pStr )
		return KVI_ERROR_MissingParameter;
	for( KviStr *s = m_pInheritanceList->first(); s; s = m_pInheritanceList->next() ) {
		if( kvi_strEqualCI(s->ptr(), pStr->ptr()) ) {
			buffer.append('1');
			return KVI_ERROR_Success;
		}
	}
	buffer.append('0');
	return KVI_ERROR_Success;
}

int KviScriptObject::callFunction(const char *fncName, QPtrList<KviStr> *params, KviStr &buffer, const char *classOverride)
{
	// This will delete the parameters
	KviScriptObjectClassDefinition *d = classOverride
		? m_pClassDefinition->lookupParentClassDefinition(classOverride)
		: m_pClassDefinition;
	if( !d ) {
		if( params )
			delete params;
		return KVI_ERROR_InvalidClassOverride;
	}
	KviScriptObjectFunctionStruct *f = d->lookupFunction(fncName);
	if( !f ) {
		if( params )
			delete params;
		return KVI_ERROR_NoSuchObjectFunction;
	}
	if( f->fncHandler ) {
		int retVal = (this->*(f->fncHandler))(params, buffer);
		if( params )
			delete params;
		return retVal;
	} else {
		// The parameters will be deleted inside callObjectFunction()
		bool bRet = m_pController->m_pUserParser->callObjectFunction(this, fncName, f->szBuffer.ptr(), params, buffer);
		return bRet ? KVI_ERROR_Success : KVI_ERROR_ObjectFunctionFailure;
	}
}

void KviScriptObject::addChild(KviScriptObject *o)
{
	childrenList()->append(o);
}

void KviScriptObject::dump(KviWindow *wnd, const char *margin)
{
	KviStr tmp(KviStr::Format,
		__tr("%s%cObject %s, class %s, name %s"), margin ? margin : "", KVI_TEXT_BOLD, id(), getClass(), getName()
	);
	wnd->output(KVI_OUT_KVIRC, tmp.ptr());
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		KviStr mrgn = margin;
		mrgn.append("    ");
		o->dump(wnd, mrgn.ptr());
	}
}

void KviScriptObject::removeChild(KviScriptObject *o, bool bDelete)
{
	if( m_bDying ) return; // All children will be removed by delete

	if( !bDelete )
		m_pChildList->setAutoDelete(false);
	if( !(m_pChildList->removeRef(o)) )
		debug("Oops... no such child: %s", o->id());
	if( !bDelete )
		m_pChildList->setAutoDelete(true);
}

KviScriptObject *KviScriptObject::childByName(const char *szName)
{
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( kvi_strEqualCI(o->getName(), szName) )
			return o;
	}
	return 0;
}

KviScriptObject *KviScriptObject::findObjectByName(const char *szName)
{
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( kvi_strEqualCI(o->getName(), szName) )
			return o;
		o = o->findObjectByName(szName);
		if( o )
			return o;
	}
	return 0;
}

KviScriptObject *KviScriptObject::findObjectByClass(const char *szClass)
{
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( kvi_strEqualCI(o->getClass(), szClass) )
			return o;
		o = o->findObjectByClass(szClass);
		if( o ) return o;
	}
	return 0;
}

KviScriptObject *KviScriptObject::findObjectById(const char *szId)
{
	for( KviScriptObject *o = m_pChildList->first(); o; o = m_pChildList->next() ) {
		if( kvi_strEqualCI(o->id(), szId) )
			return o;
		o = o->findObjectById(szId);
		if( o ) return o;
	}
	return 0;
}

void KviScriptObject::removeEventHandler(const char *evName)
{
	m_pEventDict->remove(evName);
}

void KviScriptObject::setEventHandler(const char *evName, const char *evBuffer)
{
	m_pEventDict->replace(evName, new KviStr(evBuffer));
}

bool KviScriptObject::hasEventHandler(const char *evName)
{
	return (m_pEventDict->find(evName) != 0);
}

const char *KviScriptObject::getName()
{
	return name();
}

const char *KviScriptObject::getClass()
{
	return m_pClassDefinition->getClass();
}

const char *KviScriptObject::id()
{
	return m_szId.ptr();
}

KviScriptObjectClassDefinition *KviScriptObject::classDefinition()
{
	return m_pClassDefinition;
}

#include "m_kvi_script_object.moc"
