// =============================================================================
//
//      --- kvi_options_mimetypes.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviOptionsMimeTypes"

#include <qlayout.h>

#include "kvi_app.h"
#include "kvi_listview.h"
#include "kvi_locale.h"
#include "kvi_mimemanager.h"
#include "kvi_options.h"
#include "kvi_options_mimetypes.h"
#include "kvi_pushbutton.h"
#include "kvi_string.h"

/*
	@document: doc_mimetypes.kvihelp
	@title: The mime types in KVIrc
		The mime types are the standard way of recognizing the "meaning" of the
		data contained in a file.<br>
		KVIrc contains a really simpified implementation of this "protocol".<br>
		The mimetypes are used by the <a href="play.kvihelp">PLAY</a> command,
		by the <a href="doc_ctcp_multimedia.kvihelp">CTCP MULTIMEDIA</a> handling
		protocol and by the directory browser.<br>
		Each mime type has 7 fields:<br>
		<b>Name</b><br>
		No special meaning : just a human readable name for the mimetype,
		such as "Shell script", "Text file" or "Jpeg Image".<br>
		In fact this could be the real "RFC" mime name...<br>
		<b>File mask</b><br>
		This is a "standard" file mask that may contain '*' wildcards:<br>
		*.cpp, *.rpm, *.mp3, Makefile*, *.so* ...<br>
		<b>Magic bytes</b><br>
		This is a regular expression (grep-like) that describes the first bytes
		of the file.<br>
		KVIrc will read few bytes from the beginning of the file and check
		if these match the given "Magic bytes".<br>
		For example : ^\x7f\x45\x4c\x46 recognizes the first four bytes of
		the ELF executable file format:<br>
		^ matches start of input, \x7f describes the character 7f (hex),
		\x45 describes the character 'E' (45 hex), \x4c the character 'L'
		(4c hex) and \x46 the character 'F'.<br>
		^MThd describes the first four bytes of the standard MIDI (*.mid)
		file format (that correspond literally to the characters M, T, h and d).<br>
		If you leave this field empty the matching will be done only on the file mask basis.<br>
		<b>Run script</b><br>
		This is a KVIrc script that is executed every time that "running the file of that type"
		is needed; for example when you click on a recognized file in the directory browser.<br>
		The identifier <a href="s_execfilename.kvihelp">$ExecFileName</a> will contain
		the complete path and filename of the file being executed.<br>
		The identifier <a href="s_isremoteexec.kvihelp">$IsRemoteExec</a> will contain
		1 if the execution was triggered by a remote user via <a href="doc_ctcp_multimedia.kvihelp">CTCP MULTIMEDIA</a>
		protocol, 0 otherwise. (See below)<br>
		For example, for the *.txt (text file) mimetype you can specify the following run script:<br>
		<example>
			<a href="run.kvihelp">run</a> xedit <a href="s_execfilename.kvihelp">$ExecFileName</a>
		</example>
		A more complex example; for the *.sh (shell script) mimetype use:<br>
		<example>
			<a href="if.kvihelp">if</a>(<a href="s_isremoteexec.kvihelp">$isRemoteExec</a>)<a href="run.kvihelp">run</a> xedit <a href="s_execfilename.kvihelp">$ExecFileName</a>; <a href="if.kvihelp">else</a> <a href="run.kvihelp"> /bin/bash <a href="s_execfilename.kvihelp">$ExecFileName</a>
		</example>
		This executes the shell script only if the exec is requested from the local user,
		otherwise the script is opened with a text editor (that is relatively safe).<br>
		In this dialog the script myst fit in a single line of text.<br>
		If you want to write more complex routines you can make use of aliases.<br>
		Note:<br>
		You can use variables to change your "player" applications at runtime.<br>
		<example>
			<a href="run.kvihelp">run</a> %TextEditor <a href="s_execfilename.kvihelp">$ExecFileName</a>
		</example>
		<b>Save path</b><br>
		This path is used to save the matching files when being autoaccepted (DCC).<br>
		It is also the first directory that KVIrc looks in when searching for
		a file of this kind.<br>
		If you leave this field empty, the default save path will be used.<br>
		<b>Remote exec safe</b>
		This flag indicates if the files of that type may be executed
		by remote user via the <a href="doc_ctcp_multimedia.kvihelp">CTCP MULTIMEDIA</a> protocol.<br>
		<b>Icon path</b>
		This is a filename of an image that the directory browser will use as icon for this file type.<br>
		If the filename does not contain a complete path, KVIrc will look for the image
		in a standard set of directories (Global and Local pics directory, $HOME dir etc...).<br>
		<br><br>
		KVIrc orders the mimetypes internally in a way that the mimetypes with
		less * wildcards (in file masks) are "tried" first.<br>
		The mime that has less wildcards, comes first (README);<br>
		then longer fileMasks (*.tar.gz);<br>
		then shorter ones  (*.gz) (*);<br>
		the mimes with a magic go before similar mimes without.<br>
		So for example, you can have two different mime types for
		README.txt and *.txt.<br>
		README.txt will be tried first and recognized as "Readme file"
		otherwise the *.txt (Text file) mime will match.<br>
*/

/*
	@quickhelp: KviOptionsMimeTypes
	@widget: Mime types table
		Here you can edit the types and attributes of files recognized by KVIrc.<br>
		The meaning of each field of the table is described in detail in the <a href="doc_mimetypes.kvihelp">Mime types documentation</a>.<br>
*/
KviOptionsMimeTypes::KviOptionsMimeTypes(QWidget *parent)
	: KviOptionsWidget(parent, KVI_GENERAL_OPTIONS_WIDGET_ID_MIMETYPES)
{
	QGridLayout *g = new QGridLayout(
		this, 2, 2, KVI_GENERAL_OPTIONS_GRIDLAYOUT_BORDER, KVI_GENERAL_OPTIONS_GRIDLAYOUT_SPACE
	);

	m_pMimeView = new KviListView(this, "mimetypes", true);
	m_pMimeView->setAutoResize(false);
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("Name")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("File Mask")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("Magic Bytes")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("\"Run\" Script")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("Save Path")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("Remote Exec")));
	m_pMimeView->addColumn(_CHAR_2_QSTRING(__tr("Icon Path")));
	m_pMimeView->setSorting(-1);
	g->addMultiCellWidget(m_pMimeView, 0, 0, 0, 1);

	KviPushButton *pb = new KviPushButton(_CHAR_2_QSTRING(__tr("&Add MIME Type")), this);
	connect(pb, SIGNAL(clicked()), this, SLOT(addNewMime()));
	g->addWidget(pb, 1, 0);

	pb = new KviPushButton(_CHAR_2_QSTRING(__tr("&Remove MIME Type")), this);
	connect(pb, SIGNAL(clicked()), this, SLOT(deleteSelectedMimes()));
	g->addWidget(pb, 1, 1);

	KviListViewItem *it;
	KviMimeType *m;
	for(m = g_pOptions->m_pMimeManager->m_pMimeList->first(); m; m = g_pOptions->m_pMimeManager->m_pMimeList->next() ) {
		it = new KviListViewItem(m_pMimeView,
			_CHAR_2_QSTRING(m->mimeName.ptr()),
			_CHAR_2_QSTRING(m->fileMask.ptr()),
			_CHAR_2_QSTRING(m->magicBytes.ptr()),
			_CHAR_2_QSTRING(m->commandline.ptr()),
			_CHAR_2_QSTRING(m->savePath.ptr()),
			m->remoteExecSafe ? _CHAR_2_QSTRING("true") : _CHAR_2_QSTRING("false"),
			_CHAR_2_QSTRING(m->iconPath.ptr())
		);
	}

	g->setRowStretch(0, 1);
}

KviOptionsMimeTypes::~KviOptionsMimeTypes()
{
	// Nothing here
}

void KviOptionsMimeTypes::addNewMime()
{
	(void) new KviListViewItem(m_pMimeView,
		_CHAR_2_QSTRING(__tr("__Unnamed_")),
		_CHAR_2_QSTRING("*"),
		_CHAR_2_QSTRING(""),
		_CHAR_2_QSTRING(""),
		_CHAR_2_QSTRING(""),
		_CHAR_2_QSTRING("false"),
		_CHAR_2_QSTRING("")
	);
}

void KviOptionsMimeTypes::deleteSelectedMimes()
{
	KviListViewItem *it = m_pMimeView->firstChild();
	while( it ) {
		KviListViewItem *tmp = it;
		it = it->nextSibling();
		if( tmp->isSelected() )
			delete tmp;
	}
}

void KviOptionsMimeTypes::commitChanges()
{
	while( g_pOptions->m_pMimeManager->m_pMimeList->first() )
		g_pOptions->m_pMimeManager->m_pMimeList->removeFirst();

	KviListViewItem *it = m_pMimeView->firstChild();
	while( it ) {
		KviMimeType *m = new KviMimeType;
		m->mimeName    = it->text(0);
		m->fileMask    = it->text(1);
		m->magicBytes  = it->text(2);
		m->commandline = it->text(3);
		m->savePath    = it->text(4);
		m->mimeName.stripWhiteSpace();
		m->fileMask.stripWhiteSpace();
		m->magicBytes.stripWhiteSpace();
		m->commandline.stripWhiteSpace();
		m->savePath.stripWhiteSpace();
		KviStr tmp = it->text(5);
		m->remoteExecSafe = (kvi_strEqualCI("true", tmp.ptr()) || kvi_strEqualCI("1", tmp.ptr()));
		tmp = it->text(6);
		tmp.stripWhiteSpace();
		if( tmp.hasData() ) {
			m->iconPath = tmp.ptr();
			if( !g_pApp->getFileIcon(tmp.ptr()) )
				debug("Could not load icon file %s", tmp.ptr());
		}
		if( m->mimeName.hasData() && m->fileMask.hasData() )
			g_pOptions->m_pMimeManager->inSort(m);
		else
			delete m; // Incomplete... useless
		it = it->nextSibling();
	}
	g_pApp->refreshMimeTypeIcons();
	g_pApp->updateAllDirectoryBrowsers();
}

#include "m_kvi_options_mimetypes.moc"
