// =============================================================================
//
//      --- kvi_messagecolorview.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviMessageColorView"

#include <qfontmetrics.h>
#include <qpainter.h>
#include <qscrollbar.h>

#include "kvi_app.h"
#include "kvi_config.h"
#include "kvi_fileutils.h"
#include "kvi_locale.h"
#include "kvi_messagecolorview.h"
#include "kvi_messagetypes.h"
#include "kvi_mirccntrl.h"
#include "kvi_options.h"
#include "kvi_popupmenu.h"

// Declared in kvi_app.cpp and managed by KviApp class
extern QPixmap *g_pixViewOut[KVI_OUT_NUM_IMAGES];
extern KviApp  *g_pApp;

KviMessageColorView::KviMessageColorView(QWidget *par)
	: QGridView(par, "message_color_view")
{
	setNumRows(KVI_OUT_NUM_IMAGES + 2);
	setNumCols(1);

	for( int i = 0; i < 16; i++ ) {
		m_pColorPixmap[i] = new QPixmap(32, 16);
		m_pColorPixmap[i]->fill(*(g_pOptions->m_pMircColor[i]));
	}
	m_selectedRow = -1;
	m_pFm = new QFontMetrics(g_pOptions->m_fntView);

	m_pForePopup = new KviPopupMenu();
	for( int i = 0; i < 16; i++ )
		m_pForePopup->insertItem(*(m_pColorPixmap[i]), i);
	connect(m_pForePopup, SIGNAL(activated(int)), this, SLOT(forePopupActivated(int)));
	m_pBackPopup = new KviPopupMenu();
	for( int i = 0; i < 16; i++ )
		m_pBackPopup->insertItem(*(m_pColorPixmap[i]), i);
	connect(m_pBackPopup, SIGNAL(activated(int)), this, SLOT(backPopupActivated(int)));
	m_pBackPopup->insertItem(__tr("Transparent"), 16);
	m_pPopup = new KviPopupMenu();
	m_pPopup->insertItem(__tr("Foreground"), m_pForePopup, 1000);
	m_pPopup->insertItem(__tr("Background"), m_pBackPopup, 1001);

	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		m_clrBack[i] = g_pOptions->m_cViewOutTypeBack[i];
		m_clrFore[i] = g_pOptions->m_cViewOutTypeFore[i];
	}

	m_clrBack[KVI_OUT_NUM_IMAGES] = g_pOptions->m_cViewOutSeleBack;
	m_clrFore[KVI_OUT_NUM_IMAGES] = g_pOptions->m_cViewOutSeleFore;
	m_clrBack[KVI_OUT_NUM_IMAGES + 1] = KVI_TRANSPARENT;
	m_clrFore[KVI_OUT_NUM_IMAGES + 1] = g_pOptions->m_cViewOutUrlFore;

	int hght = m_pFm->lineSpacing() + 4;
	setCellHeight(hght > 18 ? hght : 18);
	setCellWidth(200);
	setFrameStyle(QFrame::Sunken | QFrame::Panel);
	setMinimumHeight(150);
	setHScrollBarMode(AlwaysOff);
	setVScrollBarMode(Auto);
}

KviMessageColorView::~KviMessageColorView()
{
	delete m_pPopup;     m_pPopup     = 0;
	delete m_pForePopup; m_pForePopup = 0;
	delete m_pBackPopup; m_pBackPopup = 0;
	for( int i = 0; i < 16; i++ ) {
		delete m_pColorPixmap[i];
		m_pColorPixmap[i] = 0;
	}
	delete m_pFm;
	m_pFm = 0;
}

void KviMessageColorView::commit()
{
	for( int i = 0; i < KVI_OUT_NUM_IMAGES; i++ ) {
		g_pOptions->m_cViewOutTypeBack[i] = m_clrBack[i];
		g_pOptions->m_cViewOutTypeFore[i] = m_clrFore[i];
	}

	g_pOptions->m_cViewOutSeleBack = (m_clrBack[KVI_OUT_NUM_IMAGES] != KVI_TRANSPARENT)
		? m_clrBack[KVI_OUT_NUM_IMAGES]
		: KVI_BLACK;
	g_pOptions->m_cViewOutSeleFore = m_clrFore[KVI_OUT_NUM_IMAGES];
	g_pOptions->m_cViewOutUrlFore  = m_clrFore[KVI_OUT_NUM_IMAGES + 1];
}

void KviMessageColorView::forePopupActivated(int id)
{
	if( (m_selectedRow > -1) && (m_selectedRow < (KVI_OUT_NUM_IMAGES + 2)) && (id >= 0) && (id < 16) ) {
		m_clrFore[m_selectedRow] = id;
		updateCell(m_selectedRow, 0);
	}
}

void KviMessageColorView::backPopupActivated(int id)
{
	if( (m_selectedRow > -1) && (m_selectedRow < (KVI_OUT_NUM_IMAGES + 2)) && (id >= 0) && (id <= 16) ) {
		if( id < 16 )
			m_clrBack[m_selectedRow] = id;
		else
			m_clrBack[m_selectedRow] = KVI_TRANSPARENT;
		updateCell(m_selectedRow, 0);
	}
}

bool KviMessageColorView::eventFilter(QObject *o, QEvent *e)
{
	if( e->type() != QEvent::MouseButtonPress )
		return QGridView::eventFilter(o, e);

	QMouseEvent *ev = (QMouseEvent *) e;
	if( (m_selectedRow > -1) && (m_selectedRow < (KVI_OUT_NUM_IMAGES + 2)) ) {
		int i = m_selectedRow;
		m_selectedRow = -1;
		updateCell(i, 0);
	}
	m_selectedRow = rowAt(ev->pos().y());
	if( (m_selectedRow > -1) && (m_selectedRow < (KVI_OUT_NUM_IMAGES + 2)) ) {
		m_pPopup->setItemEnabled(1001, m_selectedRow < (KVI_OUT_NUM_IMAGES + 1));
		m_pBackPopup->setItemEnabled(16, m_selectedRow < KVI_OUT_NUM_IMAGES);
		m_pPopup->popup(mapToGlobal(ev->pos()));
		updateCell(m_selectedRow, 0);
	}
	return true;
}

void KviMessageColorView::paintCell(QPainter *p, int row, int col)
{
	if( g_pOptions->m_pixViewBack->isNull() )
		p->fillRect(0, 0, cellWidth(), cellHeight(), g_pOptions->m_clrViewBack);
	else
		p->drawTiledPixmap(0, 0, cellWidth(), cellHeight(), *(g_pOptions->m_pixViewBack));

	if( row < KVI_OUT_NUM_IMAGES )
		p->drawPixmap(1, 1, *(g_pixViewOut[row]));

	if( row == m_selectedRow ) {
		p->setPen(Qt::red);
		p->drawRect(0, 0, cellWidth() - 1, cellHeight() - 1);
	}

	if( m_clrBack[row] < 16 )
		p->fillRect(24, 1, cellWidth(), cellHeight() - 2, *(g_pOptions->m_pMircColor[m_clrBack[row]]));
	p->setPen(*(g_pOptions->m_pMircColor[m_clrFore[row]]));
	p->setFont(g_pOptions->m_fntView);

	p->drawText(26, cellHeight() - m_pFm->descent(), kvi_messagetypes[row]);
}

void KviMessageColorView::resizeEvent(QResizeEvent *e)
{
	QGridView::resizeEvent(e);
	setCellWidth(width() - verticalScrollBar()->width() - 1);
}

void KviMessageColorView::load()
{
	KviStr basePath;
	g_pApp->getLocalKVIrcDirectory(basePath);
	KviStr fName = kvi_askForOpenFileName(basePath.ptr(), "*.kviclr");
	if( fName.hasData() ) {
		KviConfig cfg(fName.ptr());
		cfg.setGroup("MessageColors");

		for( int j = 0; j < (KVI_OUT_NUM_IMAGES + 2); j++ ) {
			KviStr szEntry(KviStr::Format, "TypeBackground%d", j);
			m_clrBack[j] = cfg.readUCharEntry(szEntry.ptr(), KVI_TRANSPARENT);
			szEntry.sprintf("TypeForeground%d", j);
			m_clrFore[j] = cfg.readUCharEntry(szEntry.ptr(), KVI_BLACK);
			if( m_clrFore[j] > 15 )
				m_clrFore[j] = KVI_BLACK;
		}
	}
	update();
}

void KviMessageColorView::save()
{
	KviStr basePath;
	g_pApp->getLocalKVIrcDirectory(basePath, KviApp::MsgColors);
	KviStr fName = kvi_askForSaveFileName(basePath.ptr());
	if( fName.hasData() ) {
		KviConfig cfg(fName.ptr());
		cfg.setGroup("MessageColors");

		for( int j = 0; j < (KVI_OUT_NUM_IMAGES + 2); j++ ) {
			KviStr szEntry(KviStr::Format, "TypeBackground%d", j);
			cfg.writeEntry(szEntry.ptr(), m_clrBack[j]);
			szEntry.sprintf("TypeForeground%d", j);
			cfg.writeEntry(szEntry.ptr(), m_clrFore[j]);
		}
	}
}

#include "m_kvi_messagecolorview.moc"
