// =============================================================================
//
//      --- kvi_serverparser_ctcp.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviServerParserCTCP"

#include "kvi_settings.h"
#if defined(HAVE_SYS_UTSNAME_H) && defined(HAVE_UNAME)
	#include <sys/utsname.h>
#endif

#include <stdlib.h>
#include <time.h>

#include "kvi_app.h"
#include "kvi_buildnumber.h"
#include "kvi_channel.h"
#include "kvi_console.h"
#include "kvi_dcc_manager.h"
#include "kvi_event.h"
#include "kvi_frame.h"
#include "kvi_irc_socket.h"
#include "kvi_irc_user.h"
#include "kvi_locale.h"
#include "kvi_mimemanager.h"
#include "kvi_options.h"
#include "kvi_query.h"
#include "kvi_serverparser.h"
#include "kvi_userlistbox.h"
#include "kvi_userparser.h"

// TODO: Show only one flood warning for each 5 floods received

#ifdef BUILD_NUMBER
	#undef KVI_CTCP_VERSION_REPLY
	#define KVI_CTCP_VERSION_REPLY "KVIrc " VERSION " " KVI_RELEASE_NAME ": build " BUILD_NUMBER " " BUILD_DATE
#endif

// Declared in kvi_app.cpp
extern KviEventManager *g_pEventManager;

KviCtcpCommandEntry KviServerParser::ctcpCmdTable[] =
{
	// Type,    len, causesFlood?, requestHandler,                       replyHandler
	{ "PING"      ,  4, 1, &KviServerParser::handleCTCPPingRequest      , &KviServerParser::handleCTCPPingReply       },
	{ "DCC"       ,  3, 1, &KviServerParser::handleCTCPDccRequest       , &KviServerParser::handleCTCPDccReply        },
	{ "VERSION"   ,  7, 1, &KviServerParser::handleCTCPVersionRequest   , &KviServerParser::handleCTCPVersionReply    },
	{ "TIME"      ,  4, 1, &KviServerParser::handleCTCPTimeRequest      , &KviServerParser::handleCTCPTimeReply       },
	{ "USERINFO"  ,  8, 1, &KviServerParser::handleCTCPUserinfoRequest  , &KviServerParser::handleCTCPUserinfoReply   },
	{ "CLIENTINFO", 10, 1, &KviServerParser::handleCTCPClientinfoRequest, &KviServerParser::handleCTCPClientinfoReply },
	{ "INFO"      ,  4, 1, &KviServerParser::handleCTCPClientinfoRequest, &KviServerParser::handleCTCPClientinfoReply },
	{ "SOURCE"    ,  6, 1, &KviServerParser::handleCTCPSourceRequest    , &KviServerParser::handleCTCPSourceReply     },
	{ "FINGER"    ,  6, 1, &KviServerParser::handleCTCPFingerRequest    , &KviServerParser::handleCTCPFingerReply     },
	{ "ACTION"    ,  6, 0, &KviServerParser::handleCTCPActionRequest    , &KviServerParser::handleCTCPActionReply     },
	{ "ECHO"      ,  4, 0, &KviServerParser::handleCTCPPageRequest      , &KviServerParser::handleCTCPPageReply       },
	{ "PAGE"      ,  4, 0, &KviServerParser::handleCTCPPageRequest      , &KviServerParser::handleCTCPPageReply       },
	{ "SOUND"     ,  5, 1, &KviServerParser::handleCTCPMultimediaRequest, &KviServerParser::handleCTCPMultimediaReply },
	{ "ERRMSG"    ,  6, 0, &KviServerParser::handleCTCPErrmsgRequest    , &KviServerParser::handleCTCPErrmsgReply     },
	{ "ERROR"     ,  5, 0, &KviServerParser::handleCTCPErrmsgRequest    , &KviServerParser::handleCTCPErrmsgReply     },
	{ "MULTIMEDIA", 10, 1, &KviServerParser::handleCTCPMultimediaRequest, &KviServerParser::handleCTCPMultimediaReply },
	{ "MM"        ,  2, 1, &KviServerParser::handleCTCPMultimediaRequest, &KviServerParser::handleCTCPMultimediaReply },
	{ 0           ,  0, 0, 0                                            , 0                                           }
};

// TODO: CTCP multimedia? (media), audio? sed?
void KviServerParser::parseCTCPPrivmsg(KviIrcUser &source, const char *target, const char *data)
{
	// The 0x01 chars should be already stripped.
	if( *data == 0x01 )
		data++;
	// Get the CTCP Type
	KviStr type;
	data = kvi_extractToken(type, data, ' ');
	type.toUpper();

	// Now try to find it in our table
	int iType = -1;          // Index in the table of recognized CTCP commands
	bool causesFlood = true; // Unknown CTCPs cause flood

	for( int i = 0; ctcpCmdTable[i].ctcpName && (iType == -1); i++ ) {
		// msg->szCommand has already been transformed toUpper()
		if( kvi_strEqualCSN(type.ptr(), ctcpCmdTable[i].ctcpName, ctcpCmdTable[i].ctcpLen) ) {
			iType       = i;
			causesFlood = (ctcpCmdTable[i].bCausesFlood == 1);
		}
	}

	if( g_pOptions->m_bUseAntiCtcpFlood && causesFlood ) {
		// Can cause flood; requires a reply
		// One CTCP more...
		m_iCtcpCount++;
		if( g_pOptions->m_iMaxCtcpCount > 0 ) {
			// At least one CTCP allowed
			if( m_iCtcpTimerId == 0 ) {
				// Start the timer that will reset the CTCP count
				if( g_pOptions->m_iMaxCtcpTime < 1 )
					g_pOptions->m_iMaxCtcpTime = 1; // Need at least one second
				m_iCtcpTimerId = startTimer(g_pOptions->m_iMaxCtcpTime * 1000);
			} // else: the timer is already running
		}
		// Check limit overruns
		if( m_iCtcpCount > g_pOptions->m_iMaxCtcpCount ) {
			// Flood!
			KviWindow *pOut = g_pOptions->m_bCtcpFloodWarningsToConsole ? m_pConsole : m_pFrm->activeWindow();
			if( !doCTCPFloodEvent(type.ptr(), source, target, pOut, data) )
				return;
			pOut->output(KVI_OUT_FLOOD,
				__tr("CTCP flood detected: ignoring last CTCP request from %s [%s@%s] to %s: %s %s"),
				source.nick(), source.username(), source.host(), target, type.ptr(), data
			);
			return;
		}
	}

	if( iType > -1 )
		(this->*(ctcpCmdTable[iType].ctcpProc))(source, target, data);
	else {
		KviWindow *pOut = g_pOptions->m_bCtcpRequestsToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( !doCTCPEvent(type.ptr(), source, target, !(g_pOptions->m_bReplyErrmsgOnUnknownCtcp), pOut, data) )
			return;
		pOut->output(KVI_OUT_CTCPERROR,
			__tr("Unrecognized CTCP request from %s [%s@%s] to %s: %s %s"),
			source.nick(), source.username(), source.host(), target, type.ptr(), data
		);

		if( g_pOptions->m_bReplyErrmsgOnUnknownCtcp ) {
			KviStr tmp(KviStr::Format,
				"Unknown CTCP type %s: use CTCP CLIENTINFO for a list of available CTCP types", type.ptr()
			);
			replyCTCP(source.nick(), "ERRMSG", tmp.ptr());
		}
	}
}

bool KviServerParser::doCTCPFloodEvent(
	const char *ctcpName, KviIrcUser &source, const char *target, KviWindow *pOut, const char *parms)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnCTCPFlood) ) {
		if( parms ) {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s %s", source.nick(), source.username(), source.host(), target, ctcpName, parms
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCPFlood, pOut, tmp) )
				return false; // Stop the output
		} else {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s", source.nick(), source.username(), source.host(), target, ctcpName
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCPFlood, pOut, tmp) )
				return false; // Stop the output
		}
	}
	return true;
}

bool KviServerParser::showCTCPRequest(const char *ctcpName, KviIrcUser &source, const char *target, bool bIgnore)
{
	if( isMe(target) ) {
		KviWindow *pOut = g_pOptions->m_bCtcpRequestsToConsole ? ((KviWindow *) m_pConsole) : m_pFrm->activeWindow();
		if( !doCTCPEvent(ctcpName, source, target, bIgnore, m_pConsole) )
			return false;
		pOut->output(KVI_OUT_CTCPREQUEST,
			__tr("CTCP %s request from %s [%s@%s]: %s"),
			ctcpName, source.nick(), source.username(), source.host(), bIgnore ? __tr("ignored") : __tr("replied"));
	} else {
		KviChannel *chan = m_pFrm->findChannel(target);
		if( chan ) {
			KviWindow *pOut = g_pOptions->m_bChannelCtcpRequestsToConsole
				? ((KviWindow *) m_pConsole)
				: ((KviWindow *) chan);
			if( !doCTCPEvent(ctcpName, source, target, bIgnore, chan) )
				return false;
			pOut->output(KVI_OUT_CTCPREQUEST,
				__tr("Channel CTCP %s request from %s [%s@%s]: %s"),
				ctcpName, source.nick(), source.username(), source.host(), bIgnore ? __tr("ignored") : __tr("replied")
			);
		} else {
			// Oops, desync with the server.
			KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? ((KviWindow *) m_pConsole) : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: CTCP %s from %s [%s@%s] to %s: %s"),
				ctcpName, source.nick(), source.username(), source.host(), target,
				bIgnore ? __tr("ignored") : __tr("replied")
			);
		}
	}
	return true;
}

bool KviServerParser::doCTCPEvent(
	const char *ctcpName, KviIrcUser &source, const char *target, bool bIgnore, KviWindow *pOut, const char *parms)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnCTCP) ) {
		if( parms ) {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s %c %s",
				source.nick(), source.username(), source.host(), target, ctcpName, (bIgnore ? '1' : '0'), parms
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCP, pOut, tmp) )
				return false; // Stop the output
		} else {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s %c",
				source.nick(), source.username(), source.host(), target, ctcpName, (bIgnore ? '1' : '0')
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCP, pOut, tmp) )
				return false; // Stop the output
		}
	}
	return true;
}

void KviServerParser::handleCTCPPingRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( isMe(target) ) {
		KviWindow *pOut = g_pOptions->m_bCtcpRequestsToConsole ? ((KviWindow *) m_pConsole) : m_pFrm->activeWindow();
		if( !doCTCPEvent("PING", source, target, g_pOptions->m_bIgnoreCtcpPingRequests, m_pConsole, data) )
			return;
		pOut->output(KVI_OUT_CTCPREQUEST,
			__tr("CTCP PING request from %s [%s@%s] (PING %s): %s"),
			source.nick(), source.username(), source.host(), data,
			g_pOptions->m_bIgnoreCtcpPingRequests ? __tr("ignored") : __tr("replied")
		);
	} else {
		KviChannel *chan = m_pFrm->findChannel(target);
		if( chan ) {
			KviWindow *pOut = g_pOptions->m_bChannelCtcpRequestsToConsole
				? ((KviWindow *) m_pConsole)
				: ((KviWindow *) chan);
			if( !doCTCPEvent("PING", source, target, g_pOptions->m_bIgnoreCtcpPingRequests, m_pConsole, data) )
				return;
			pOut->output(KVI_OUT_CTCPREQUEST,
				__tr("Channel CTCP PING request from %s [%s@%s] (PING %s): %s"),
				source.nick(), source.username(), source.host(), data,
				g_pOptions->m_bIgnoreCtcpPingRequests ? __tr("ignored") : __tr("replied")
			);
		} else {
			// Oops, desync with the server.
			KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: CTCP PING from %s [%s@%s] to %s (PING %s): %s"),
				source.nick(), source.username(), source.host(), target, data,
				g_pOptions->m_bIgnoreCtcpPingRequests ? __tr("ignored") : __tr("replied"));
		}
	}
	if( !g_pOptions->m_bIgnoreCtcpPingRequests )
		replyCTCP(source.nick(), "PING", data);
}

void KviServerParser::handleCTCPVersionRequest(KviIrcUser &source, const char *target, const char *)
{
	if( !showCTCPRequest("VERSION", source, target, g_pOptions->m_bIgnoreCtcpVersionRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpVersionRequests ) {
		KviStr tmp(KVI_CTCP_VERSION_REPLY);
		if( g_pOptions->m_szCtcpVersionAppendix.hasData() ) {
			tmp.append(", ");
			tmp.append(g_pOptions->m_szCtcpVersionAppendix);
		}
		replyCTCP(source.nick(), "VERSION", tmp.ptr());
	}
}

void KviServerParser::handleCTCPTimeRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( !showCTCPRequest("TIME", source, target, g_pOptions->m_bIgnoreCtcpTimeRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpTimeRequests ) {
		KviStr tmp;
		QDateTime tm(QDateTime::currentDateTime());
		tmp = tm.toString();
		replyCTCP(source.nick(), "TIME", tmp.ptr());
	}
}

void KviServerParser::handleCTCPUserinfoRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( !showCTCPRequest("USERINFO", source, target, g_pOptions->m_bIgnoreCtcpUserinfoRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpUserinfoRequests ) {
		KviStr tmp(KviStr::Format,
			"IRC: %s (%s)", m_pFrm->m_global.szCurrentUserName.ptr(), m_pFrm->m_global.szCurrentRealName.ptr()
		);
		if( g_pOptions->m_bAppendRealUsernameToUserinfoReply ) {
			char *user = getenv("USER");
			if( !user ) user = getenv("USERNAME");
			if( !user ) user = getenv("LOGNAME");
			if( user ) {
				tmp.append(", LOGNAME: ");
				tmp.append(user);
			}
		}
		if( g_pOptions->m_szCtcpUserinfoAppendix.hasData() ) {
			tmp.append(", ");
			tmp.append(g_pOptions->m_szCtcpUserinfoAppendix);
		}
		replyCTCP(source.nick(), "USERINFO", tmp.ptr());
	}
}

void KviServerParser::handleCTCPClientinfoRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( !showCTCPRequest("CLIENTINFO", source, target, g_pOptions->m_bIgnoreCtcpClientinfoRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpClientinfoRequests ) {
		KviStr tmp(KVI_CTCP_CLIENTINFO_REPLY);
		tmp.append(
			", SUPPORTS: ACTION, CLIENTINFO, DCC [CHAT, RESUME, SEND, VOICE], ECHO, ERRMSG, "\
			"FINGER, PAGE, PING, SOUND [MULTIMEDIA, MM], SOURCE, TIME, USERINFO, VERSION"
		);

		// TODO: Append CPU Clock speed?

#if defined(HAVE_SYS_UTSNAME_H) && defined(HAVE_UNAME)
		if( g_pOptions->m_bAppendUnameToClientinfoReply ) {
			struct utsname u;
			if( uname(&u) == 0 ) { // nodename domainname version
				KviStr tmp2(KviStr::Format, ", SYSTEM: %s %s %s", u.machine, u.sysname, u.release);
				tmp.append(tmp2);
			}
		}
#endif
		if( g_pOptions->m_szCtcpClientinfoAppendix.hasData() ) {
			tmp.append(", ");
			tmp.append(g_pOptions->m_szCtcpClientinfoAppendix);
		}
		replyCTCP(source.nick(), "CLIENTINFO", tmp.ptr());
	}
}

void KviServerParser::handleCTCPSourceRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( !showCTCPRequest("SOURCE", source, target, g_pOptions->m_bIgnoreCtcpSourceRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpSourceRequests ) {
		KviStr tmp(KVI_CTCP_SOURCE_REPLY);
		if( g_pOptions->m_szCtcpSourceAppendix.hasData() ) {
			tmp.append(", ");
			tmp.append(g_pOptions->m_szCtcpSourceAppendix);
		}
		replyCTCP(source.nick(), "SOURCE", tmp.ptr());
	}
}

void KviServerParser::handleCTCPFingerRequest(KviIrcUser &source, const char *target, const char *data)
{
	if( !showCTCPRequest("FINGER", source, target, g_pOptions->m_bIgnoreCtcpFingerRequests) )
		return;
	if( !g_pOptions->m_bIgnoreCtcpFingerRequests ) {
		replyCTCP(source.nick(), "FINGER", g_pOptions->m_szCtcpFingerReply.ptr());
	}
}

void KviServerParser::handleCTCPActionRequest(KviIrcUser &source, const char *target, const char *data)
{
	handleCTCPActionReply(source, target, data); // Just treat as a reply
}

void KviServerParser::handleCTCPPageRequest(KviIrcUser &source, const char *target, const char *data)
{
	handleCTCPPageReply(source, target, data); // Just treat as a reply
}

void KviServerParser::handleCTCPErrmsgRequest(KviIrcUser &source, const char *target, const char *data)
{
	handleCTCPErrmsgReply(source, target, data); // Just treat as a reply
}

void KviServerParser::handleCTCPMultimediaRequest(KviIrcUser &source, const char *target, const char *data)
{
	handleCTCPMultimediaReply(source, target, data);
}

void KviServerParser::handleCTCPDccRequest(KviIrcUser &source, const char *target, const char *data)
{
	KviDccRequest *dcc = new KviDccRequest();

	dcc->szOriginalRequest = data;
	// Extract the DCC type (1st parameter)
	data = kvi_extractToken(dcc->szType, data, ' ');
	// Extract the type dependant parameter (filenames...)
	if( *data == '"' ) {
		data++;
		data = kvi_extractToken(dcc->szParam, data, '\"');
		while( (*data == ' ') || (*data == '\t') )
			data++;
	} else data = kvi_extractToken(dcc->szParam, data, ' ');
	// Extract the address
	data = kvi_extractToken(dcc->szAddress, data, ' ');
	// Extract the port
	data = kvi_extractToken(dcc->szPort, data, ' ');
	// Extract the last parameter
	data = kvi_extractToken(dcc->szLast, data, ' ');
	// The rest is garbage
	dcc->szGarbage = data;

	// Check for unrecoverable errors
	KviStr szError;
	if( dcc->szType.isEmpty() )
		szError = __tr("The DCC type is empty");
	if( dcc->szParam.isEmpty() )
		szError = __tr("The DCC type dependant parameter is empty");
	if( !isMe(target) )
		szError = __tr("The DCC target is not me!");
	if( dcc->szAddress.isEmpty() || (!dcc->szAddress.isUnsignedNum()) )
		szError = __tr("The 3rd parameter is not an unsigned number");
	if( dcc->szPort.isEmpty() || (!dcc->szPort.isUnsignedNum()) )
		szError = __tr("The 4th parameter is not an unsigned number");

	if( szError.hasData() ) {
		KviWindow *pOut = g_pOptions->m_bCtcpRequestsToConsole ? m_pConsole : m_pFrm->activeWindow();
		pOut->output(KVI_OUT_CTCPERROR,
			__tr("Malformed CTCP DCC request from %s [%s@%s] to %s: [%s]: %s"),
			source.nick(), source.username(), source.host(), target, dcc->szOriginalRequest.ptr(), szError.ptr()
		);
		delete dcc;
		return;
	}
	dcc->uPort = dcc->szPort.toUShort();
	// The string address is a number that represents the IP
	// address of the remote host read in NETWORK BYTE ORDER.

	// Now I get that number here in the form of string
	// and convert it to an unsigned long in LOCAL HOST BYTE ORDER!
	// So htonl is needed.
	dcc->uAddress = htonl(dcc->szAddress.toULong());

	m_pFrm->m_pDccManager->handleDccRequest(source, dcc);
	delete dcc;
}

void KviServerParser::replyCTCP(const char *to, const char *type, const char *msg)
{
	m_pSocket->sendFmtData("NOTICE %s :%c%s %s%c", to, 0x01, type, msg, 0x01);
}

void KviServerParser::parseCTCPNotice(KviIrcUser &source, const char *target, const char *data)
{
	// The 0x01 chars should be already stripped.
	if( *data == 0x01 )
		data++;
	// Get the CTCP Type
	KviStr type;
	data = kvi_extractToken(type, data, ' ');
	type.toUpper();

	for( int i = 0; ctcpCmdTable[i].ctcpName; i++ ) {
		// msg->szCommand has already been transformed toUpper()
		if( kvi_strEqualCSN(type.ptr(), ctcpCmdTable[i].ctcpName, ctcpCmdTable[i].ctcpLen) ) {
			(this->*(ctcpCmdTable[i].ctcpReplyProc))(source, target, data);
			return;
		}
	}

	KviWindow *pOut = g_pOptions->m_bCtcpRepliesToConsole ? m_pConsole : m_pFrm->activeWindow();
	if( !doCTCPReplyEvent(type.ptr(), source, target, pOut, data) )
		return; // No output
	pOut->output(KVI_OUT_CTCPERROR,
		__tr("Unrecognized CTCP reply from %s [%s@%s] to %s: %s %s"),
		source.nick(), source.username(), source.host(), target, type.ptr(), data
	);
}

void KviServerParser::showCTCPReply(const char *ctcpName, KviIrcUser &source, const char *target, const char *data)
{
	if( isMe(target) ) {
		KviWindow *pOut = g_pOptions->m_bCtcpRepliesToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( !doCTCPReplyEvent(ctcpName, source, target, m_pConsole, data) )
			return; // No output
		pOut->output(KVI_OUT_CTCPREPLY,
			__tr("CTCP %s reply from %s [%s@%s]: %s"),
			ctcpName, source.nick(), source.username(), source.host(), data
		);
	} else {
		// Oops, someone is having fun here
		KviWindow *pOut = g_pOptions->m_bCtcpRepliesToConsole ? m_pConsole : m_pFrm->activeWindow();
		if( !doCTCPReplyEvent(ctcpName, source, target, m_pConsole, data) )
			return; // No output
		pOut->output(KVI_OUT_CTCPREPLY,
			__tr("CTCP %s reply from %s [%s@%s] to %s: %s"),
			ctcpName, source.nick(), source.username(), source.host(), target, data
		);
	}
}

bool KviServerParser::doCTCPReplyEvent(
	const char *ctcpName, KviIrcUser &source, const char *target, KviWindow *pOut, const char *data)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnCTCPReply) ) {
		if( data ) {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s %s", source.nick(), source.username(), source.host(), target, ctcpName, data
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCPReply, pOut, tmp) )
				return false; // Stop the output
		} else {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %s", source.nick(), source.username(), source.host(), target, ctcpName
			);
			if( m_pUserParser->callEvent(KviEvent_OnCTCPReply, pOut, tmp) )
				return false; // Stop the output
		}
	}
	return true;
}

void KviServerParser::handleCTCPPingReply(KviIrcUser &source, const char *target, const char *data)
{
	KviStr tmp = data;
	bool bOk;
	unsigned int uTime = tmp.toUInt(&bOk);
	KviStr timeDiff;
	if( !bOk )
		timeDiff = __tr("Malformed PING reply");
	else {
		int iDiff = time(0) - uTime;
		timeDiff.setNum(iDiff);
		timeDiff.append(__tr(" seconds"));
	}
	showCTCPReply("PING", source, target, timeDiff.ptr());
}

void KviServerParser::handleCTCPVersionReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("VERSION", source, target, data);
}

void KviServerParser::handleCTCPTimeReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("TIME", source, target, data);
}

void KviServerParser::handleCTCPUserinfoReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("USERINFO", source, target, data);
}

void KviServerParser::handleCTCPClientinfoReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("CLIENTINFO", source, target, data);
}

void KviServerParser::handleCTCPSourceReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("SOURCE", source, target, data);
}

void KviServerParser::handleCTCPFingerReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("FINGER", source, target, data);
}

void KviServerParser::handleCTCPPageReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("PAGE", source, target, data);
}

void KviServerParser::handleCTCPErrmsgReply(KviIrcUser &source, const char *target, const char *data)
{
	showCTCPReply("ERRMSG", source, target, data);
}

bool KviServerParser::doActionEvent(
	KviIrcUser &source, const char *target, KviWindow *pOut, char create_query, const char *data)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnAction) ) {
		if( data ) {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %c %s", source.nick(), source.username(), source.host(), target, create_query, data
			);
			if( m_pUserParser->callEvent(KviEvent_OnAction, pOut, tmp) )
				return false; // Stop the output
		} else {
			KviStr tmp(KviStr::Format,
				"%s %s@%s %s %c", source.nick(), source.username(), source.host(), target, create_query
			);
			if( m_pUserParser->callEvent(KviEvent_OnAction, pOut, tmp) )
				return false; // Stop the output
		}
	}
	return true;
}

void KviServerParser::handleCTCPActionReply(KviIrcUser &source, const char *target, const char *data)
{
	if( isMe(target) ) {
		KviQuery *query = m_pFrm->findQuery(source.nick());
		if( query ) {
			if( !doActionEvent(source, target, query, '0', data) )
				return;
			query->output(KVI_OUT_ACTION,
				"\r!%s\r%s\r %s", g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), data
			);
		} else {
			if( g_pOptions->m_bCreateQueryOnPrivmsg ) {
				if( !doActionEvent(source, target, m_pConsole, '1', data) )
					return;
				query = m_pFrm->createQuery(source.nick(), true);
				query->output(KVI_OUT_INTERNAL,
					__tr("Private conversation requested by %s [%s@%s]"), source.nick(), source.username(), source.host()
				);
				query->output(KVI_OUT_ACTION,
					"\r!%s\r%s\r %s", g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), data
				);
			} else {
				if( !doActionEvent(source, target, m_pConsole, '0', data) )
					return;
				m_pFrm->activeWindow()->output(KVI_OUT_ACTION,
					__tr("%s!%s@%s %s"), source.nick(), source.username(), source.host(), data
				);
			}
		}
	} else {
		KviChannel *chan = m_pFrm->findChannel(target);
		if( chan ) {
			if( !doActionEvent(source, target, chan, '0', data) )
				return;
			chan->output(KVI_OUT_ACTION,
				"\r!%s\r%s\r %s", g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), data
			);
		} else {
			// Oops, desync with the server.
			KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: CTCP ACTION from %s [%s@%s] to %s: %s"),
				source.nick(), source.username(), source.host(), target, data
			);
		}
	}
}

void KviServerParser::handleCTCPDccReply(KviIrcUser &source, const char *target, const char *data)
{
	handleCTCPDccRequest(source, target, data); // Treat as a request
}

bool KviServerParser::doMultimediaEvent(
	KviIrcUser &source, const char *target, KviWindow *pOut, int create_query, const char *filename)
{
	if( g_pEventManager->eventEnabled(KviEvent_OnCtcpMultimedia) ) {
		KviStr tmp(KviStr::Format,
			"%s %s@%s %s %c %s", source.nick(), source.username(), source.host(), target, create_query, filename
		);
		if( m_pUserParser->callEvent(KviEvent_OnCtcpMultimedia, pOut, tmp) )
			return false; // Stop the output
	}
	return true;
}

/*
	@document: doc_ctcp_multimedia.kvihelp
	@title: The CTCP MULTIMEDIA extension
		CTCP MULTIMEDIA is the natural extension of the CTCP SOUND "protocol".<br>
		CTCP SOUND is intended for triggering "sound" events in remote clients.<br>
		MULTIMEDIA is intended for triggering any multimedia event.<br>
		<docsubtitle>Semantics</docsubtitle>
		The requesting client sends a CTCP message with the following syntax.<br><br>
		<b>MULTIMEDIA &lt;filename&gt;</b><br><br>
		The &lt;filename&gt; should not be empty, and should not contain a leading path.<br>
		<i>KVIrc strips any leading path and ignores empty filenames</i><br>
		The receiving client looks for the &lt;filename&gt; in the local filesystem.<br>
		<i>KVIrc looks for the match in a set of directories (see <a href="play.kvihelp">play</a>).</i><br>
		If the file can be found, the receiving client should find an appropriate player
		for the file type.<br>
		The algorithm for finding the player is left "undefined, " but usually
		it will be based on the mime-type matching.<br>
		<i>KVIrc tries to find a match for the &lt;filename&gt; in the mimetypes
		defined by the user in the "misc options dialog".</i><br>
		The file is then "played" on the recipient side.<br>
		If the file cannot be found, the CTCP should be ignored.<br>
		<i>KVIrc allows "requesting" the file by sending a special private message to
		the "source client".<br>
		The format of the request is "PRIVMSG &lt;source nickname&gt; :!&lt;source nickname&gt; &lt;filename&gt;".<br>
		This could be also handled by a specific CTCP message later...</i><br>
		<docsubtitle>Security issues</docsubtitle>
		The security issue handling is left to the receiving client implementation.<br>
		"Player" applications that could damage the recipient's system should
		not be executed.<br>
		For example, executing a "/bin/bash somename.sh" should be avoided,
		since the shell script could contain dangerous commands.<br>
		The CTCP MULTIMEDIA offering "somename.sh" should be either handled by another "secure" player application
		(such as xedit, for example) or simply ignored.<br>
		<i>KVIrc MIME types have a user-configurable flag that indicate whether the command
		used to handle a specific file type is secure or not. The execution "commandline"
		is a KVIrc script that can handle more security issues. When the execution is
		triggered by the remote side, the <a href="s_isremoteexec.kvihelp">$IsRemoteExec</a> variable
		will be set to 1.</i><br>
*/
void KviServerParser::handleCTCPMultimediaReply(KviIrcUser &source, const char *target, const char *data)
{
	const char *eventData  = data ? data : "";
	const char *outputData = data ? data : __tr("[nothing]");
	if( isMe(target) ) {
		KviQuery *query = m_pFrm->findQuery(source.nick());
		if( query ) {
			if( !doMultimediaEvent(source, target, query, '0', eventData) )
				return;
			query->output(KVI_OUT_MULTIMEDIA,
				"\r!%s\r%s\r plays %s", g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), outputData
			);
		} else {
			if( g_pOptions->m_bCreateQueryOnPrivmsg ) {
				if( !doMultimediaEvent(source, target, m_pConsole, '1', eventData) )
					return;
				query = m_pFrm->createQuery(source.nick(), true);
				query->output(KVI_OUT_INTERNAL,
					__tr("Private conversation requested by %s [%s@%s]"), source.nick(), source.username(), source.host()
				);
				query->output(KVI_OUT_MULTIMEDIA,
					__tr("\r!%s\r%s\r plays %s"),
					g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), outputData
				);
			} else {
				if( !doMultimediaEvent(source, target, m_pConsole, '0', eventData) )
					return;
				m_pFrm->activeWindow()->output(KVI_OUT_MULTIMEDIA,
					__tr("%s!%s@%s plays %s"), source.nick(), source.username(), source.host(), outputData
				);
			}
		}
	} else {
		KviChannel *chan = m_pFrm->findChannel(target);
		if( chan ) {
			if( !doMultimediaEvent(source, target, chan, '0', eventData) )
				return;
			chan->output(KVI_OUT_MULTIMEDIA,
				"\r!%s\r%s\r plays %s", g_pOptions->m_szPrivmsgFormatNickLinkCommand.ptr(), source.nick(), outputData
			);
		} else {
			// Oops, desync with the server.
			KviWindow *pOut = g_pOptions->m_bDesyncMsgsToConsole ? m_pConsole : m_pFrm->activeWindow();
			pOut->output(KVI_OUT_DESYNC,
				__tr("Local desync: CTCP MULTIMEDIA from %s [%s@%s] to %s: %s"),
				source.nick(), source.username(), source.host(), target, eventData
			);
		}
	}

	KviStr file = data;
	// First cut any leading path
	int idx = file.findLastIdx('/');
	if( idx >= 0 )
		file.cutLeft(idx + 1);
	file.stripWhiteSpace();
	if( file.isEmpty() )
		return;

	if( g_pOptions->m_bListenToCtcpMultimedia ) {
		KviStr fileWithPath;
		if( !g_pApp->findUserFile(fileWithPath, file.ptr()) ) {
			// The file was not found
			if( g_pOptions->m_bPlayDefaultFileWhenFileNotFound ) {
				if( g_pOptions->m_szCtcpMultimediaDefaultFileName.hasData() ) {
					if( !g_pApp->findUserFile(fileWithPath, g_pOptions->m_szCtcpMultimediaDefaultFileName.ptr()) ) {
						fileWithPath = "";
						g_pOptions->m_bPlayDefaultFileWhenFileNotFound = false;
					}
				} else g_pOptions->m_bPlayDefaultFileWhenFileNotFound = false;
			}
			if( g_pOptions->m_bSendMultimediaRequestWhenFileNotFound ) {
				m_pFrm->m_pSocket->sendFmtData("PRIVMSG %s :!%s %s", source.nick(), source.nick(), file.ptr());
				if( g_pOptions->m_bBeVerbose ) {
					m_pConsole->output(KVI_OUT_INTERNAL,
						__tr("Sent PRIVMSG %s: !%s %s"), source.nick(), source.nick(), file.ptr()
					);
				}
			}
		}
		if( fileWithPath.isEmpty() )
			return;

		// The file was found
		KviMimeType *m = g_pOptions->m_pMimeManager->findMatch(fileWithPath.ptr());
		if( m ) {
			// OK... recognized mime type
			if( m->commandline.hasData() ) {
				// OK... have a command to play it
				if( m->remoteExecSafe ) {
					// Seems that we can execute it safely
					g_pApp->executeFileWithCommand(m->commandline.ptr(), fileWithPath.ptr(), m_pFrm, true);
				} else {
					// Unsafe command, just warn
					m_pConsole->output(KVI_OUT_ERROR,
						__tr("CTCP MULTIMEDIA: cannot play file %s: the remote triggering of the commandline "\
						"(%s) is potentially dangerous"), fileWithPath.ptr(), m->commandline.ptr()
					);
				}
			} else {
				// No commandline at all
				m_pConsole->output(KVI_OUT_ERROR,
					__tr("CTCP MULTIMEDIA: cannot play file %s: no commandline specified"), fileWithPath.ptr()
				);
			}
		} else {
			// Cannot play... no appropriate player found
			m_pConsole->output(KVI_OUT_ERROR,
				__tr("CTCP MULTIMEDIA: cannot play file %s: no matching MIME type"), fileWithPath.ptr()
			);
		}
	}
}
