// =============================================================================
//
//      --- kvi_processcontroller_qt.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviProcessController"

#include <errno.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <unistd.h>

#include <qsocketnotifier.h>

#include "kvi_locale.h"
#include "kvi_process.h"
#include "kvi_processcontroller_qt.h"

/**
 *  Simple versions of the KProcess and KProcessController classes
 *  original code by (C) Christian Czezatke
 *  e9025461@student.tuwien.ac.at
 *  Really good work :)
 */

/**
 * KviProcessController class.<br>
 * Handles signals from dying children.
 */
KviProcessController *g_pProcessController = 0;

void theSigCHLDHandler(int)
{
	// A signal from a child was received (SIGPIPE or SIGCHILD)
	int status;
	pid_t this_pid;
	int saved_errno = errno;
	// Since waitpid and write change errno, we have to save it and restore it.
	// (Richard Stevens, Advanced programming in the Unix Environment)
	this_pid = waitpid(-1, &status, WNOHANG);
	// J6t: g_pProcessController might be already destroyed
	if( (this_pid != -1) && (g_pProcessController != 0) ) {
		::write(g_pProcessController->m_fd[1], &this_pid, sizeof(this_pid));
		::write(g_pProcessController->m_fd[1], &status,   sizeof(status));
	}
	errno = saved_errno;
}

KviProcessController::KviProcessController()
{
	// We keep a list of processes currently alive.
	m_pProcessList = new QPtrList<KviProcess>;
	m_pProcessList->setAutoDelete(false);      // Never delete the processes from here
	// Need a pipe to communicate from the signal handler
	if( pipe(m_fd) < 0 ) printf(strerror(errno));
	if( fcntl(m_fd[0], F_SETFL, O_NONBLOCK) == -1 ) printf(strerror(errno));
	// And a notifier for that
	m_pNotifier = new QSocketNotifier(m_fd[0], QSocketNotifier::Read);
	m_pNotifier->setEnabled(true);
	QObject::connect(m_pNotifier, SIGNAL(activated(int)), this, SLOT(slotDoHousekeeping(int)));
	// Now we set the handler
	struct sigaction act;
	act.sa_handler = &theSigCHLDHandler;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = SA_NOCLDSTOP;
	// CC: take care of SunOS which automatically restarts interrupted system
	// calls (and thus does not have SA_RESTART)
#ifdef SA_RESTART
	act.sa_flags |= SA_RESTART;
#endif
	sigaction(SIGCHLD, &act, 0L);
}

void KviProcessController::slotDoHousekeeping(int)
{
	// A process has exited, and data was written to the pipe
	// in the handler... here we remove dead processes from the list
	KviProcess *proc;
	int bytes_read;
	pid_t pid;
	int status;
	bytes_read  = ::read(m_fd[0], &pid,    sizeof(pid_t));
	bytes_read += ::read(m_fd[0], &status, sizeof(int));
	if( bytes_read != sizeof(int) + sizeof(pid_t) ) {
		if( bytes_read < 0 ) {
			debug(__tr("Error: could not read info from signal handler!"));
			debug(__tr("Error %d"), errno);
			if( (errno == EINTR) || (errno == EINPROGRESS) || (errno == EAGAIN) ) {
				fprintf(stderr, __tr("Retrying"));
				slotDoHousekeeping(0);
				return;
			}
		} else {
			debug(__tr("Error: could not read info from signal handler!"));
			debug(__tr("Read %d bytes instead of %d+%d"), bytes_read, sizeof(int), sizeof(pid_t));
		}
	}
	proc = m_pProcessList->first();
	// Cleanup
	while( proc ) {
		if( proc->pid() == pid ) {
			// Process has exited, so emit the respective events
			proc->processHasExited(status);
			return; // Only ONE process has exited
		}
		proc = m_pProcessList->next();
	}
}

KviProcessController::~KviProcessController()
{
	struct sigaction act;

	// Turn off notification for processes that have exited
	act.sa_handler = SIG_IGN;
	sigemptyset(&(act.sa_mask));
	sigaddset(&(act.sa_mask), SIGCHLD);
	act.sa_flags = 0;
	sigaction(SIGCHLD, &act, 0L);

	close(m_fd[0]);
	close(m_fd[1]);
	if( m_pProcessList ) {
		delete m_pProcessList;
		m_pProcessList = 0;
	}
	if( m_pNotifier    ) {
		delete m_pNotifier;
		m_pNotifier = 0;
	}
}

void KviProcessController::addProcess(KviProcess *proc)
{
	m_pProcessList->append(proc);
}

void KviProcessController::removeProcess(KviProcess *proc)
{
	m_pProcessList->removeRef(proc);
}

#include "m_kvi_processcontroller_qt.moc"
