// =============================================================================
//
//      --- kvi_mdi_manager.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//   Opaque move & resize code by Falk Brettschneider (gigafalk@geocities.com)
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviMDIManager"

#define _KVI_MDI_MANAGER_CPP_

#include "kvi_settings.h"
#ifdef HAVE_MATH_H
	#include <math.h>
#endif
#include <unistd.h>

#include <qpainter.h>

#include "kvi_debug.h"
#include "kvi_defines.h"
#include "kvi_locale.h"
#include "kvi_mdi_caption.h"
#include "kvi_mdi_child.h"
#include "kvi_mdi_manager.h"
#include "kvi_popupmenu.h"
#include "kvi_xutils.h"

#include <X11/Xlib.h> // For XGrabPointer, GrabModeAsync, etc.

KviMdiManager::KviMdiManager(QWidget *parent, const char *name)
	: QFrame(parent, name)
{
	setFrameStyle(QFrame::Panel | QFrame::Sunken);
	// Basic config
	m_captionFont              = QFont(KVI_DEFAULT_FONT, 12);
	QFontMetrics fm(m_captionFont);
	m_captionFontLineSpacing   = fm.lineSpacing();
	m_captionActiveBackColor   = QColor(  0,   0,   0);
	m_captionActiveForeColor   = QColor( 80, 255,   0);
	m_captionInactiveBackColor = QColor( 80,  80,  80);
	m_captionInactiveForeColor = QColor(200, 200, 200);
	m_bOpaqueMove              = true;
	m_bOpaqueResize            = true;
	// Z order stack
	m_pZ                       = new QPtrList<KviMdiChild>;
	m_pZ->setAutoDelete(true);
	setFocusPolicy(ClickFocus);

	m_pWindowMenu = new KviPopupMenu(this, "window_list_popup");
	connect(m_pWindowMenu, SIGNAL(activated(int)), this, SLOT(menuActivated(int)));
	fillWindowMenu();
}

void KviMdiManager::setMdiCaptionFont(const QFont &fnt)
{
	m_captionFont = fnt;
	QFontMetrics fm(m_captionFont);
	m_captionFontLineSpacing = fm.lineSpacing();
}

void KviMdiManager::setMdiCaptionActiveForeColor(const QColor &clr)
	{ m_captionActiveForeColor = clr; }
void KviMdiManager::setMdiCaptionActiveBackColor(const QColor &clr)
	{ m_captionActiveBackColor = clr; }
void KviMdiManager::setMdiCaptionInactiveForeColor(const QColor &clr)
	{ m_captionInactiveForeColor = clr; }
void KviMdiManager::setMdiCaptionInactiveBackColor(const QColor &clr)
	{ m_captionInactiveBackColor = clr; }

// ============ ~KviMdiManager ============
KviMdiManager::~KviMdiManager()
{
	// This will destroy all the widgets inside.
	if( m_pZ ) {
		delete m_pZ;
		m_pZ = 0;
	}
}

// ============ manageChild ============
void KviMdiManager::manageChild(KviMdiChild *lpC, bool bShow, bool bCascade)
{
	__range_valid(lpC);
	KviMdiChild *top = topChild();
	if( bShow )
		m_pZ->append(lpC); // Visible -> first in the Z order
	else
		m_pZ->insert(0, lpC); // Hidden -> last in the Z order
	if( bCascade ) lpC->move(getCascadePoint(m_pZ->count() - 1));
	if( bShow ) {
		if( top ) { // Maximize if needed
			if( top->state() == KviMdiChild::Maximized ) {
				top->setState(KviMdiChild::Normal, false);
				lpC->setState(KviMdiChild::Maximized, false);
			}
		} else {
			// Default to maximized windows
			lpC->setState(KviMdiChild::Maximized, false);
		}
		lpC->show();
		lpC->raise();
		lpC->setFocus();
	}
	fillWindowMenu();
}

// ============= focusInEvent ===============
void KviMdiManager::focusInEvent(QFocusEvent *)
{
	focusTopChild();
}

// ============ destroyChild ============
void KviMdiManager::destroyChild(KviMdiChild *lpC, bool bFocusTopChild)
{
	bool bWasMaximized = (lpC->state() == KviMdiChild::Maximized);
	disconnect(lpC);
	lpC->blockSignals(true);
#ifdef _KVI_DEBUG_CHECK_RANGE_
	// Report invalid results in a debug session
	__range_valid(m_pZ->removeRef(lpC));
#else
	m_pZ->removeRef(lpC);
#endif
	if( bWasMaximized ) {
		KviMdiChild *c = topChild();
		if( c )
			c->setState(KviMdiChild::Maximized, false);
		else {
			// SDI state change
			enterSDIMode(false);
		}
	}
	if( bFocusTopChild ) focusTopChild();
	fillWindowMenu();
}

// ============= setTopChild ============
void KviMdiManager::setTopChild(KviMdiChild *lpC, bool bSetFocus)
{
	__range_valid(lpC);
	__range_valid(lpC->isVisible());
	if( m_pZ->last() != lpC ) {
		m_pZ->setAutoDelete(false);
#ifdef _KVI_DEBUG_CHECK_RANGE_
		// Report invalid results in a debug session
		__range_valid(m_pZ->removeRef(lpC));
#else
		m_pZ->removeRef(lpC);
#endif
		// Disable the labels of all the other children
		for( KviMdiChild *pC = m_pZ->first(); pC; pC = m_pZ->next() ) {
			pC->m_pCaption->setActive(false);
		}
		KviMdiChild *pMaximizedChild = m_pZ->last();
		if( pMaximizedChild->m_state != KviMdiChild::Maximized ) pMaximizedChild = 0;
		m_pZ->setAutoDelete(true);
		m_pZ->append(lpC);
		if( pMaximizedChild ) lpC->setState(KviMdiChild::Maximized, false); // Do not animate the change
		lpC->raise();
		if( pMaximizedChild ) {
			pMaximizedChild->setState(KviMdiChild::Normal, false);
		}
		if( bSetFocus ) {
			if( !lpC->hasFocus() ) lpC->setFocus();
		}
		fillWindowMenu();
	}
}

KviPopupMenu *KviMdiManager::getWindowMenu()
{
	return m_pWindowMenu;
}

KviMdiChild *KviMdiManager::topChild()
{
	return m_pZ->last();
}

// ========== animate ============
void KviMdiManager::animate(QRect &start, QRect &finish)
{
	// Animation of movement from rect start to rect end
	QRect r;
	int times = 10;
	int cX = start.x();
	int cY = start.y();
	int cW = start.width();
	int cH = start.height();
	XGrabServer(qt_xdisplay());
	for( int i = 0; i <= times; i++ ) {
		if( cW < 4 ) cW = 4; // Prevent negative widths later...
		if( cH < 4 ) cH = 4;
		r = QRect(cX, cY, cW, cH);
		QPainter pa(this);
		kvi_drawDragRectangle(&pa, r, 1);
		XFlush(qt_xdisplay());
		XSync(qt_xdisplay(), false);
		// Lose some time...
#ifdef HAVE_USLEEP
		usleep(100);
#endif
		kvi_drawDragRectangle(&pa, r, 1);
		cX += (finish.x()      - start.x())      / times;
		cY += (finish.y()      - start.y())      / times;
		cW += (finish.width()  - start.width())  / times;
		cH += (finish.height() - start.height()) / times;
	}
	XUngrabServer(qt_xdisplay());
}

// ============== resizeEvent ================
void KviMdiManager::resizeEvent(QResizeEvent *e)
{
	// If we have a maximized child at the top, adjust its size
	KviMdiChild *lpC = m_pZ->last();
	if( lpC ) {
		if( lpC->m_state == KviMdiChild::Maximized )
			lpC->resize(width() + KVI_MDI_CHILD_DOUBLE_MAXIMIZED_HIDDEN_EDGE, height() + KVI_MDI_CHILD_BORDER + lpC->m_pCaption->heightHint() + KVI_MDI_CHILD_MAXIMIZED_HIDDEN_EDGE + 3);
	}
}

// =============== mousePressEvent =============
void KviMdiManager::mousePressEvent(QMouseEvent *e)
{
	// Popup the window menu
	if( e->button() & RightButton )
		m_pWindowMenu->popup(mapToGlobal(e->pos()));
}

// =============== getCascadePoint ============
QPoint KviMdiManager::getCascadePoint(int indexOfWindow)
{
	QPoint pnt(0, 0);
	if( indexOfWindow == 0 ) return pnt;
	KviMdiChild *lpC = m_pZ->first();
	int step = (lpC ? lpC->m_pCaption->heightHint() + KVI_MDI_CHILD_BORDER : 20);
	int availableHeight = height() - (lpC ? lpC->minimumSize().height() : KVI_MDI_CHILD_MIN_HEIGHT);
	int availableWidth  =  width() - (lpC ? lpC->minimumSize().width()  : KVI_MDI_CHILD_MIN_WIDTH);
	int ax = 0;
	int ay = 0;
	for( int i = 0; i < indexOfWindow; i++ ) {
		ax += step;
		ay += step;
		if( ax > availableWidth  ) ax = 0;
		if( ay > availableHeight ) ay = 0;
	}
	pnt.setX(ax);
	pnt.setY(ay);
	return pnt;
}

// =============== fillWindowMenu ===============
void KviMdiManager::fillWindowMenu()
{
	m_pWindowMenu->clear();
	m_pWindowMenu->insertTearOffHandle();
	int i = 100;
	for( KviMdiChild *lpC = m_pZ->first(); lpC; lpC = m_pZ->next() ) {
		KviStr szItem;
		szItem.setNum(((uint) i) - 99);
		szItem += ". ";
		if( lpC->m_state == KviMdiChild::Minimized ) {
			szItem += "( ";
			szItem += lpC->caption();
			szItem += " )";
		} else szItem += lpC->caption();
		m_pWindowMenu->insertItem(szItem.ptr(), i);
		m_pWindowMenu->setItemChecked(i, ((uint) i) == (m_pZ->count() + 99)); // Active item :)
		i++;
	}
	m_pWindowMenu->insertSeparator();
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Cascade Windows")),            this, SLOT(cascadeWindows()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Cascade &Maximized")),          this, SLOT(cascadeMaximized()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Expand &Vertical")),            this, SLOT(expandVertical()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("Expand &Horizontal")),          this, SLOT(expandHorizontal()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Anodine's Tile")),             this, SLOT(tileAnodine()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("&Pragma's Tile (Vertical)")),   this, SLOT(tilePragmaVertical()));
	m_pWindowMenu->insertItem(_CHAR_2_QSTRING(__tr("P&ragma's Tile (Horizontal)")), this, SLOT(tilePragmaHorizontal()));
}

// ================ menuActivated ===============
void KviMdiManager::menuActivated(int id)
{
	if( id < 100 ) return;
	id -= 100;
	__range_valid(((uint) id) < m_pZ->count());
	KviMdiChild *lpC = m_pZ->at(id);
	if( !lpC ) return;
	if( lpC->m_state == KviMdiChild::Minimized ) lpC->minimizePressed();
	if( lpC == m_pZ->last() ) return;
	setTopChild(lpC, true);
}

// ================ childMinimized ===============
void KviMdiManager::childMinimized(KviMdiChild *lpC, bool bWasMaximized)
{
	__range_valid(lpC);
	if( m_pZ->findRef(lpC) == -1 ) return;
	if( m_pZ->count() > 1 ) {
		m_pZ->setAutoDelete(false);
#ifdef _KVI_DEBUG_CHECK_RANGE_
		// Report invalid results in a debug session
		__range_valid(m_pZ->removeRef(lpC));
#else
		m_pZ->removeRef(lpC);
#endif
		m_pZ->setAutoDelete(true);
		m_pZ->insert(0, lpC);
		if( bWasMaximized ) {
			// Need to maximize the top child
			lpC = m_pZ->last();
			if( !lpC ) return; // ??
			if( lpC->m_state == KviMdiChild::Minimized ) {
				if( bWasMaximized ) emit enterSDIMode(false);
				return;
			}
			lpC->setState(KviMdiChild::Maximized, false); // Do not animate the change
		}
		focusTopChild();
	} else {
		// Unique window minimized... it will not lose the focus!!
		setFocus(); // Remove focus from the child
		if( bWasMaximized ) emit enterSDIMode(false);
	}
	fillWindowMenu();
}

void KviMdiManager::childMaximized(KviMdiChild *lpC, bool bWasMinimized)
{
	if( lpC == m_pZ->last() ) emit enterSDIMode(true);
}

void KviMdiManager::childRestored(KviMdiChild *lpC, bool bWasMaximized)
{
	if( (lpC == m_pZ->last()) && bWasMaximized ) emit enterSDIMode(false);
}

// ============= focusTopChild ===============
void KviMdiManager::focusTopChild()
{
	KviMdiChild *lpC = m_pZ->last();
	if( !lpC ) return;
	if( lpC->m_state == KviMdiChild::Minimized ) return;
	// Disable the labels of all the other children
	for( KviMdiChild *pC = m_pZ->first(); pC; pC = m_pZ->next() ) {
		if( pC != lpC ) pC->m_pCaption->setActive(false);
	}
	lpC->raise();
	if( !lpC->hasFocus() ) lpC->setFocus();
}

void KviMdiManager::minimizeTopChild()
{
	KviMdiChild *lpC = m_pZ->last();
	if( !lpC ) return;
	if( lpC->state() != KviMdiChild::Minimized ) lpC->minimizePressed();
}

void KviMdiManager::restoreTopChild()
{
	KviMdiChild *lpC = m_pZ->last();
	if( !lpC ) return;
	if( lpC->state() == KviMdiChild::Maximized) lpC->maximizePressed();
}

void KviMdiManager::closeTopChild()
{
	KviMdiChild *lpC = m_pZ->last();
	if( !lpC ) return;
	lpC->closePressed();
}

// ============= cascadeWindows ===============
void KviMdiManager::cascadeWindows()
{
	int idx = 0;
	QPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while( !list.isEmpty() ) {
		KviMdiChild *lpC = list.first();
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			lpC->move(getCascadePoint(idx));
			lpC->resize(lpC->minimumSize());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

// ============= cascadeMaximized ===============
void KviMdiManager::cascadeMaximized()
{
	int idx = 0;
	QPtrList<KviMdiChild> list(*m_pZ);

	list.setAutoDelete(false);
	while( !list.isEmpty() ) {
		KviMdiChild *lpC = list.first();
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			QPoint pnt(getCascadePoint(idx));
			lpC->move(pnt);
			QSize curSize(width() - pnt.x(), height() - pnt.y());
			if( (lpC->minimumSize().width()  > curSize.width()) ||
				(lpC->minimumSize().height() > curSize.height()) )
				lpC->resize(lpC->minimumSize());
			else
				lpC->resize(curSize);
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandVertical()
{
	int idx = 0;
	QPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while( !list.isEmpty() ) {
		KviMdiChild *lpC = list.first();
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			lpC->setGeometry(lpC->x(), 0, lpC->width(), height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expandHorizontal()
{
	int idx = 0;
	QPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while( !list.isEmpty() ) {
		KviMdiChild *lpC = list.first();
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			lpC->setGeometry(0, lpC->y(), width(), lpC->height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}

void KviMdiManager::expand()
{
	int idx = 0;
	QPtrList<KviMdiChild> list(*m_pZ);
	list.setAutoDelete(false);
	while( !list.isEmpty() ) {
		KviMdiChild *lpC = list.first();
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			lpC->setGeometry(0, 0, width(), height());
			idx++;
		}
		list.removeFirst();
	}
	focusTopChild();
}
// ============= getVisibleChildCount =============
int KviMdiManager::getVisibleChildCount()
{
	int cnt = 0;
	for( KviMdiChild *lpC = m_pZ->first(); lpC; lpC = m_pZ->next() ) {
		if( lpC->m_state != KviMdiChild::Minimized ) cnt++;
	}
	return cnt;
}

// ============ tilePragma ============
void KviMdiManager::tilePragmaHorizontal()
{
	tileAllInternal(9, true);
}

void KviMdiManager::tilePragmaVertical()
{
	tileAllInternal(9, false);
}

// ============ tileAllInternal ============
void KviMdiManager::tileAllInternal(int maxWnds, bool bHorizontal)
{
	// NUM WINDOWS =            1, 2, 3, 4, 5, 6, 7, 8, 9
	static int colstable[9] = { 1, 1, 1, 2, 2, 2, 3, 3, 3 }; // Num columns
	static int rowstable[9] = { 1, 2, 3, 2, 3, 3, 3, 3, 3 }; // Num rows
	static int lastwindw[9] = { 1, 1, 1, 1, 2, 1, 3, 2, 1 }; // Last window multiplier
	static int colrecall[9] = { 0, 0, 0, 3, 3, 3, 6, 6, 6 }; // Adjust self
	static int rowrecall[9] = { 0, 0, 0, 0, 4, 4, 4, 4, 4 }; // Adjust self

	int *pColstable = bHorizontal ? colstable : rowstable;
	int *pRowstable = bHorizontal ? rowstable : colstable;
	int *pColrecall = bHorizontal ? colrecall : rowrecall;
	int *pRowrecall = bHorizontal ? rowrecall : colrecall;

	KviMdiChild *lpTop = topChild();
	int numVisible = getVisibleChildCount();
	if( numVisible < 1 ) return;
	int numToHandle = ((numVisible > maxWnds) ? maxWnds : numVisible);
	int xQuantum = width() / pColstable[numToHandle - 1];
	if( xQuantum < ((lpTop->minimumSize().width() > KVI_MDI_CHILD_MIN_WIDTH) ? lpTop->minimumSize().width() : KVI_MDI_CHILD_MIN_WIDTH) ) {
		if( pColrecall[numToHandle - 1] == 0 )
			debug(__tr("Tile: not enough space"));
		else
			tileAllInternal(pColrecall[numToHandle - 1], bHorizontal);
		return;
	}
	int yQuantum = height() / pRowstable[numToHandle - 1];
	if( yQuantum < ((lpTop->minimumSize().height() > KVI_MDI_CHILD_MIN_HEIGHT) ? lpTop->minimumSize().height() : KVI_MDI_CHILD_MIN_HEIGHT) ) {
		if( pRowrecall[numToHandle - 1] == 0 )
			debug(__tr("Tile: not enough space"));
		else
			tileAllInternal(pRowrecall[numToHandle - 1], bHorizontal);
		return;
	}
	int curX   = 0;
	int curY   = 0;
	int curRow = 1;
	int curCol = 1;
	int curWin = 1;
	for( KviMdiChild *lpC = m_pZ->first(); lpC; lpC = m_pZ->next() ) {
		if( lpC->m_state != KviMdiChild::Minimized ) {
			// Restore the window
			if( lpC->m_state == KviMdiChild::Maximized ) lpC->setState(KviMdiChild::Normal, false);
			if( (curWin % numToHandle) == 0 )
				lpC->setGeometry(curX, curY, xQuantum * lastwindw[numToHandle - 1], yQuantum);
			else
				lpC->setGeometry(curX, curY, xQuantum, yQuantum);
			// Example: 12 windows : 3 cols 3 rows
			if( curCol<pColstable[numToHandle - 1] ) { // curCol < 3
				curX += xQuantum; // Add a column in the same row
				curCol++;         // Increase current column
			} else {
				curX   = 0;       // New row
				curCol = 1;       // Column 1
				if( curRow < pRowstable[numToHandle - 1] ) { // curRow < 3
					curY += yQuantum; // Add a row
					curRow++;         //
				} else {
					curY   = 0;       // Restart from beginning
					curRow = 1;       //
				}
			}
			curWin++;
		}
	}
	if( lpTop ) lpTop->setFocus();
}

// ============ tileAnodine ============
void KviMdiManager::tileAnodine()
{
	KviMdiChild *lpTop = topChild();
	int numVisible = getVisibleChildCount(); // Count visible windows
	if( numVisible < 1 ) return;
	int numCols = int(sqrt(numVisible)); // Set columns to square root of visible count
	// Create an array to form grid layout
	int *numRows = new int[numCols];
	__range_valid(numRows);
	int numCurCol = 0;
	while( numCurCol < numCols ) {
		numRows[numCurCol] = numCols; // Create primary grid values
		numCurCol++;
	}
	int numDiff = numVisible - (numCols*numCols); // Count extra rows
	int numCurDiffCol = numCols; // Set column limiting for grid updates
	while( numDiff > 0 ) {
		numCurDiffCol--;
		numRows[numCurDiffCol]++; // Add extra rows to column grid
		if( numCurDiffCol < 1 ) numCurDiffCol = numCols; // Rotate through the grid
		numDiff--;
	}
	numCurCol = 0;
	int numCurRow = 0;
	int curX = 0;
	int curY = 0;
	// The following code will size everything based on my grid above.
	// There is no limit to the number of windows it will handle.
	// It is great when a kick-ass theory works!!!
	// Pragma :)
	int xQuantum = width()  / numCols;
	int yQuantum = height() / numRows[numCurCol];
	for( KviMdiChild *lpC = m_pZ->first(); lpC; lpC = m_pZ->next() ) {
		if( lpC->m_state != KviMdiChild::Minimized ) {
			if( lpC->m_state == KviMdiChild::Maximized) lpC->setState(KviMdiChild::Normal, false);
			lpC->setGeometry(curX, curY, xQuantum, yQuantum);
			numCurRow++;
			curY += yQuantum;
			if( numCurRow == numRows[numCurCol] ) {
				numCurRow = 0;
				numCurCol++;
				curY = 0;
				curX += xQuantum;
				if( numCurCol != numCols )
					yQuantum = height() / numRows[numCurCol];
			}
		}
	}
	if( numRows ) delete[] numRows;
	if( lpTop ) lpTop->setFocus();
}

#include "m_kvi_mdi_manager.moc"
