## This file is part of the KVIrc distribution
## (C) 2000-2001 Szymon Stefanek & Triskelios
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; if not, write to the Free Software
## Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
##

AC_DEFUN(AC_SS_CXXFLAGS,
	[
		AC_MSG_CHECKING([if debugging symbols are requested])
		AC_ARG_ENABLE(debug,
			[[  --enable-debug          enable debugging symbols [default=no]]],
			[SS_ADD_DEBUG_SYMBOLS="$enableval"],
			[SS_ADD_DEBUG_SYMBOLS="no"]
		)
		AC_MSG_RESULT($SS_ADD_DEBUG_SYMBOLS)
		ac_save_CXXFLAGS="$CXXFLAGS"
		if test "x$SS_ADD_DEBUG_SYMBOLS" = "xyes"; then
			CXXFLAGS="$CXXFLAGS -g"
		else
			CXXFLAGS="$ac_save_CXXFLAGS"
		fi

		AC_MSG_CHECKING([if O3 optimisation is requested])
		AC_ARG_WITH(o3-optimisation,
			[[  --with-o3-optimisation  use the -O3 flag when compiling [default=no]]],
			[SS_O3_OPTIMISATION="$withval"],
			[SS_O3_OPTIMISATION="no"]
		)
		AC_MSG_RESULT($SS_O3_OPTIMISATION)
		if test "$SS_O3_OPTIMISATION" = "yes"; then
			CXXFLAGS="-O3 $CXXFLAGS"
		else
			CXXFLAGS="-O2 $CXXFLAGS"
		fi

		AC_ARG_WITH(i386-asm,
			[[  --with-i386-asm         build with x86 assembly optimisations [default=no]]],
			[SS_COMPILE_i386_ASM="$withval"],
			[SS_COMPILE_i386_ASM="no"]
		)

		AC_MSG_CHECKING([whether to add extra include search paths])
		AC_ARG_WITH(extra-includes,
			[  --with-extra-includes=DIR
                          add the specified include paths],
			[SS_DO_IT="$withval"],
			[SS_DO_IT="no"]
		)
		if test "$SS_DO_IT" != "no"; then
			AC_MSG_RESULT(yes)
			SS_OTHER_INCDIRS="$SS_OTHER_INCDIRS $SS_DO_IT"
		else
			AC_MSG_RESULT(no)
		fi

		AC_MSG_CHECKING([whether to add extra library search paths])
		AC_ARG_WITH(extra-libs,
			[  --with-extra-libs=DIR   add the specified library paths],
			[SS_DO_IT="$withval"],
			[SS_DO_IT="no"]
		)
		if test "$SS_DO_IT" != "no"; then
			AC_MSG_RESULT(yes)
			SS_OTHER_LIBDIRS="$SS_DO_IT"
		else
			AC_MSG_RESULT(no)
		fi

		AC_ARG_WITH(pedantic-gcc,
			[[  --with-pedantic-gcc     compile with all warnings enabled [default=no]]],
			[SS_COMPILE_USE_PEDANTIC_GCC="$withval"],
			[SS_COMPILE_USE_PEDANTIC_GCC="no"]
		)

		AC_ARG_WITH(verbose-compilation,
			[[  --with-verbose-compilation
                          compile with verbose output [default=no]]],
			[SS_VERBOSE_COMPILATION="$withval"],
			[SS_VERBOSE_COMPILATION="no"]
		)

		AC_ARG_WITH(pipes,
			[[  --with-pipes            use pipes for compilation [default=no]]],
			[SS_USE_PIPES="$withval"],
			[SS_USE_PIPES="no"]
		)
		if test "$SS_USE_PIPES" = "yes"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -pipe"
		fi
	]
)

AC_DEFUN(AC_SS_FUNCTION_CHECKS,
	[
		AC_CHECK_FUNCS(vsnprintf strerror inet_aton inet_ntoa usleep getenv uname)
	]
)

AC_DEFUN(AC_SS_HEADER_CHECKS,
	[
		AC_CHECK_HEADERS(unistd.h)
		AC_CHECK_HEADERS(linux/soundcard.h sys/soundcard.h soundcard.h)
		AC_CHECK_HEADERS(math.h strings.h jpeglib.h setjmp.h limits.h sys/utsname.h)
	]
)

AC_DEFUN(AC_SS_LIBRARY_CHECKS,
	[
		AC_CHECK_LIB(compat, main, [LIBCOMPAT="-lcompat"])
		HAVE_LIBCRYPT=
		AC_CHECK_LIB(crypt, crypt, [LIBCRYPT="-lcrypt"; HAVE_LIBCRYPT="yes"],
			AC_CHECK_LIB(c, crypt, [LIBCRYPT=""; HAVE_LIBCRYPT="yes"],
			[AC_MSG_WARN([you have no crypt in either libcrypt or libc.
You should install libcrypt from another source])
		]))
		AC_SUBST(LIBCRYPT)
		if test "x$HAVE_LIBCRYPT" = "xyes"; then
			AC_DEFINE_UNQUOTED(HAVE_CRYPT, 1, [Define if your system has the crypt function])
		fi
		AC_CHECK_LIB(socket, socket, [LIBSOCKET="-lsocket -lnsl"])
	]
)

AC_DEFUN(AC_SS_FINAL_CHECK_FOR_PLUGINS,
	[
		AC_MSG_CHECKING([if we should compile plugin support])
		AC_MSG_RESULT($SS_PLUGIN_REQUESTED)
		if test "$SS_PLUGIN_REQUESTED" = "yes"; then
			if test "$SS_LINK_TO_LIBDL" = "yes"; then
				SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -export-dynamic -ldl"
			else
				SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -export-dynamic"
			fi
			SS_PLUGINS_DIRECTORY_COMPILE="plugins"
		else
			SS_PLUGINS_DIRECTORY_COMPILE=""
			if test "$SS_KDE_REQUESTED" = "yes"; then
				dnl Cannot compile with KDE and without libdl
				AC_MSG_RESULT(["Warning: Cannot compile KDE support without dlopen support"])
				AC_MSG_RESULT(["Warning: Disabling KDE support"])
				SS_KDE_REQUESTED="no"
			fi
		fi
		AC_SUBST(SS_PLUGINS_DIRECTORY_COMPILE)
	]
)

AC_DEFUN(AC_SS_FINAL_CHECK_FOR_KDE,
	[
		AC_MSG_CHECKING([if we should compile KDE support])
		AC_MSG_RESULT($SS_KDE_REQUESTED)
		if test "$SS_KDE_REQUESTED" = "yes"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -I$SS_KDE_INCLUDE_DIR"
			SS_FLAGS_LIBDIRS="$SS_FLAGS_LIBDIRS -L$SS_KDE_LIBRARY_DIR"
			SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -lkdecore -lkdeui -lkio"
			AC_DEFINE(COMPILE_NEED_KDE)
			AC_PREFIX_DEFAULT(${KDEDIR:-/usr/local})
		fi
	]
)

AC_DEFUN(AC_SS_FINAL_CONFIG,
	[
		AC_ARG_WITH(local-8bit-encoding,
			[[  --with-local-8bit-encoding
                          preserve local 8-bit charset encoding [default=no]]],
			[SS_LOCAL_8BIT_ENCODING="$withval"],
			[SS_LOCAL_8BIT_ENCODING="no"]
		)

		AC_ARG_WITH(ignore-sigalarm,
			[[  --with-ignore-sigalarm  ignore SIGALARM in all threads [default=no]]],
			[SS_COMPILE_IGNORE_SIGALARM="$withval"],
			[SS_COMPILE_IGNORE_SIGALARM="no"]
		)

		AC_ARG_WITH(solaris-libs,
			[[  --with-solaris-libs     link to lsocket, lresolv, and lnsl [default=no]]],
			[SS_SOLARIS_LIBS="$withval"],
			[SS_SOLARIS_LIBS="no"]
		)
		if test "SS_SOLARIS_LIBS" = "yes"; then
			SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -lnsl -lsocket -lresolv"
		fi

		AC_ARG_WITH(own-inet-aton,
			[[  --with-own-inet-aton    use own inet aton implementation [default=no]]],
			[SS_COMPILE_USE_OWN_INET_ATON="$withval"],
			[SS_COMPILE_USE_OWN_INET_ATON="no"]
		)

		AC_ARG_WITH(own-inet-ntoa,
			[[  --with-own-inet-ntoa    use own inet ntoa implementation [default=no]]],
			[SS_COMPILE_USE_OWN_INET_NTOA="$withval"],
			[SS_COMPILE_USE_OWN_INET_NTOA="no"]
		)

		AC_ARG_WITH(ipv6-support,
			[[  --with-ipv6-support     enable IPv6 support [default=no]]],
			[SS_COMPILE_USE_IPV6="$withval"],
			[SS_COMPILE_USE_IPV6="no"]
		)

		AC_ARG_WITH(charset-translation,
			[[  --without-charset-translation
                          disable character set translation support [default=yes]]],
			[SS_CHARSET_TRANSLATION="$withval"],
			[SS_CHARSET_TRANSLATION="yes"]
		)

		NEW_PICS_DIRECTORY=""
		AC_MSG_CHECKING([whether to use new pics])
		AC_ARG_WITH(new-pics,
			[[  --with-new-pics         use new KDE-style icons [default=no]]],
			[WITH_NEW_PICS="$withval"],
			[WITH_NEW_PICS="no"]
		)
		AC_MSG_RESULT($WITH_NEW_PICS)
		if test "x$WITH_NEW_PICS" = "xyes"; then
			NEW_PICS_DIRECTORY="newpics"
		fi
		AC_SUBST(NEW_PICS_DIRECTORY)

		SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -l$SS_QTLIB_NAME $SS_X_EXTRA_LIBS -lXext -lX11"

		SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -I$SS_QT_INCLUDE_DIR -D_REENTRANT"
		SS_FLAGS_LIBDIRS="$SS_FLAGS_LIBDIRS -L$SS_QT_LIBRARY_DIR"

		if test -n "$SS_OTHER_INCDIRS"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -I$SS_OTHER_INCDIRS"
		fi

		if test -n "$SS_OTHER_LIBDIRS"; then
			SS_FLAGS_LIBDIRS="-L$SS_OTHER_LIBDIRS $SS_FLAGS_LIBDIRS"
		fi

		AC_SUBST(SS_QT_LIBRARY_DIR)
		AC_SUBST(SS_X_LIBRARY_DIR)
		AC_SUBST(SS_QT_INCLUDE_DIR)
		AC_SUBST(SS_X_INCLUDE_DIR)

		if test -n "$SS_X_EXTRA_GCC_FLAGS"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS $SS_X_EXTRA_GCC_FLAGS"
		fi

		if test -n "$SS_X_EXTRA_LDD_FLAGS"; then
			SS_FLAGS_L="$SS_X_EXTRA_LDD_FLAGS $SS_FLAGS_LIBDIRS"
		fi

		AC_SS_FINAL_CHECK_FOR_PLUGINS
		AC_SS_FINAL_CHECK_FOR_KDE

		if test "x$SS_QT_JPEG" = "xno"; then
			if test "$SS_HAVE_LIBJPEG" = "yes"; then
				SS_FLAGS_LIBLINK="$SS_FLAGS_LIBLINK -ljpeg"
				AC_DEFINE_UNQUOTED(COMPILE_JPEG_SUPPORT, 1, [Define if you wish to use the internal JPEG support])
			fi
		fi

		if test "$SS_CHARSET_TRANSLATION" = "yes"; then
			AC_DEFINE(COMPILE_NEED_CHARSET_TRANSLATION)
		fi

		if test "$SS_VERBOSE_COMPILATION" = "yes"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -v"
		fi

		if test "$SS_COMPILE_USE_PEDANTIC_GCC" = "yes"; then
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -Wall -pedantic"
		fi

		if test "$SS_COMPILE_USE_IPV6" = "yes"; then
			AC_DEFINE(COMPILE_NEED_IPV6)
		fi

		if test "$SS_COMPILE_IGNORE_SIGALARM" = "yes"; then
			AC_DEFINE(COMPILE_NEED_IGNORE_SIGALARM)
		fi

		if test "$SS_LOCAL_8BIT_ENCODING" = "yes"; then
			AC_DEFINE(COMPILE_USE_LOCAL_8BIT_ENCODING)
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -DQT_NO_ASCII_CAST -DQT_NO_COMPAT"
		fi

		if test "$SS_COMPILE_USE_OWN_INET_ATON" = "yes"; then
			AC_DEFINE(COMPILE_NEED_OWN_INET_ATON)
		fi

		if test "$SS_COMPILE_USE_OWN_INET_NTOA" = "yes"; then
			AC_DEFINE(COMPILE_NEED_OWN_INET_NTOA)
		fi

		if test "$SS_COMPILE_i386_ASM" = "yes"; then
			AC_MSG_CHECKING(target system CPU)
			if test "$target_cpu" = "i386"; then
				AC_MSG_RESULT([i386, compiling asm code])
				AC_DEFINE(HAVE_i386_BASED_CPU)
			else
				if test "$target_cpu" = "i486"; then
					AC_MSG_RESULT([i486, compiling asm code])
					AC_DEFINE(HAVE_i386_BASED_CPU)
				else
					if test "$target_cpu" = "i586"; then
						AC_MSG_RESULT([i586, compiling asm code])
						AC_DEFINE(HAVE_i386_BASED_CPU)
					else
						if test "$target_cpu" = "i686"; then
							AC_MSG_RESULT([i686, compiling asm code])
							AC_DEFINE(HAVE_i386_BASED_CPU)
						else
							AC_MSG_RESULT([$target_cpu\, not using i386 optimisations])
							unset SS_COMPILE_i386_ASM
						fi
					fi
				fi
			fi
		fi

		build_date=`date`
		AC_DEFINE_UNQUOTED(BUILD_DATE,"$build_date")

		charmapsdir="\${exec_prefix}/share/kvirc/charmaps"
		configdir="\${exec_prefix}/share/kvirc/config"
		globalkvircdir="\${exec_prefix}/share/kvirc"
		helpdir="\${exec_prefix}/share/kvirc/help/en"
		icondir="\${exec_prefix}/share/apps/kvirc/pics"
		imagedir="\${exec_prefix}/share/kvirc/pics"
		includedir="\${exec_prefix}/include/kvirc"
		localedir="\${exec_prefix}/share/kvirc/locale"
		mandir="\${exec_prefix}/share/man/man1"
		msgcolordir="\${exec_prefix}/share/kvirc/msgcolors"
		pluglibdir="\${exec_prefix}/share/kvirc/plugins"
		if test -n "$KDEDIR"; then
			sharedir="$KDEDIR/share"
		else
			sharedir="\${exec_prefix}/share"
		fi
		topdir=`pwd`

		AC_SUBST(charmapsdir)
		AC_SUBST(configdir)
		AC_SUBST(globalkvircdir)
		AC_SUBST(helpdir)
		AC_SUBST(icondir)
		AC_SUBST(imagedir)
		AC_SUBST(includedir)
		AC_SUBST(localedir)
		AC_SUBST(msgcolordir)
		AC_SUBST(pluglibdir)
		AC_SUBST(sharedir)
		AC_SUBST(SS_FLAGS_INCDIRS)
		AC_SUBST(SS_FLAGS_LIBDIRS)
		AC_SUBST(SS_FLAGS_LIBLINK)
		AC_SUBST(topdir)
	]
)

dnl ------------------------------------------------------------------------
dnl Old stuff
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_CHECK_BOOL,
	[
		AC_MSG_CHECKING([for bool])
		AC_CACHE_VAL(ac_cv_have_bool,
			[
			AC_LANG_CPLUSPLUS
				AC_TRY_COMPILE([],
					[bool aBool = true;],
					[ac_cv_have_bool="yes"],
					[ac_cv_have_bool="no"]
				)
			]
		)
		AC_MSG_RESULT($ac_cv_have_bool)
		if test "$ac_cv_have_bool" = "yes"; then
			AC_DEFINE(HAVE_BOOL)
		fi
	]
)
dnl ------------------------------------------------------------------------
dnl Find the KDE library path and includes
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_FIND_KDE,
	[
		SS_KDE_GENERAL_DIR="none"
		SS_KDE_INCLUDE_DIR="none"
		SS_KDE_LIBRARY_DIR="none"

		AC_SS_CHECK_IF_KDE_IS_REQUESTED
		dnl if test "$SS_KDE_REQUESTED" = "yes"; then
			dnl AC_SS_FIND_KDE_GENERAL_DIR
			if test "$SS_KDE_REQUESTED" = "yes"; then
				AC_SS_FIND_KDE_INCLUDE_DIR
				if test "$SS_KDE_REQUESTED" = "yes"; then
					AC_SS_FIND_KDE_LIBRARY_DIR
					if test "$SS_KDE_REQUESTED" = "yes"; then
						AC_SS_ENSURE_CAN_COMPILE_X_QT_AND_KDE
					fi
				fi
			fi
		dnl fi
		AM_CONDITIONAL([USE_KDE_TOOLKIT], [test "$SS_KDE_REQUESTED" = "yes"])
		dnl AC_SUBST(SS_KDE_REQUESTED)
		dnl AC_SUBST(SS_KDE_GENERAL_DIR)
		dnl AC_SUBST(SS_KDE_INCLUDE_DIR)
		dnl AC_SUBST(SS_KDE_LIBRARY_DIR)
	]
)

AC_DEFUN(AC_SS_CHECK_IF_KDE_IS_REQUESTED,
	[
		AC_MSG_CHECKING(if KDE support is requested)
		AC_ARG_WITH(kde-support,
			[  --without-kde-support   build without KDE support],
			[SS_KDE_REQUESTED="$withval"],
			[SS_KDE_REQUESTED="yes"]
		)
		AC_MSG_RESULT($SS_KDE_REQUESTED)
	]
)

AC_DEFUN(AC_SS_FIND_KDE_GENERAL_DIR,
	[
		AC_MSG_CHECKING(for the KDE base directory)
		ss_kde_general_test_dirs="/usr /usr/local /usr /usr/local /usr/build /usr/X11R6 /opt /usr/opt /usr/lib /usr/lib /usr/src /usr/lib /usr/local/lib /"
		if test -n "$HOME"; then
			ss_kde_general_test_dirs="$ss_kde_general_test_dirs $HOME $HOME/lib"
		fi
		if test -n "$KDEDIR"; then
			ss_kde_general_test_dirs="$KDEDIR $ss_kde_general_test_dirs"
		fi
		ss_kde_general_test_path_suffix=". kde kde2.3 kde2.2 kde2.1 kde2.0 kde2 KDE Kde KDE2"

		AC_SS_FIND_FILE_PATH_EXT(include/kwin.h,$ss_kde_general_test_dirs,$ss_kde_general_test_path_suffix,SS_KDE_GENERAL_DIR)

		if test "$SS_KDE_GENERAL_DIR" = "FAILED"; then
			AC_MSG_RESULT(failed... continuing with normal checks)
			unset SS_KDE_GENERAL_DIR
		else
			AC_MSG_RESULT(seems to be $SS_KDE_GENERAL_DIR)
		fi
	]
)

AC_DEFUN(AC_SS_FIND_KDE_INCLUDE_DIR,
	[
		AC_ARG_WITH(kde-includes,
			[  --with-kde-includes=DIR directory the KDE headers are in],
			[SS_KDE_INCLUDE_DIR="$withval"],
			[SS_KDE_INCLUDE_DIR="NOTGIVEN"]
		)
		if test "$SS_KDE_INCLUDE_DIR" = "NOTGIVEN"; then
			AC_MSG_CHECKING(for the KDE header files)

			ss_kde_include_test_dirs="/include /usr/include /usr/local/include /usr/kde/include /usr/local/kde/include /usr/X11R6/include /opt/kde/include /usr/lib/kde/include /usr/lib/kde /usr/include/X11"
			test -n "$KDEDIR" && ss_kde_include_test_dirs="$KDEDIR/include $KDEDIR $ss_kde_include_test_dirs"
			ss_kde_include_test_dirs="$ss_kde_include_test_dirs /usr/include/kde /usr/local/include/kde /include/kde /usr/X11R6/include/kde /usr/build/kde/include"
			ss_kde_include_test_dirs="$ss_kde_include_test_dirs /usr/include/kde2 /usr/local/include/kde2 /include/kde2 /usr/X11R6/include/kde2 /usr/include/X11/kde2"
			test -n "$SS_KDE_GENERAL_DIR" && ss_kde_include_test_dirs="$SS_KDE_GENERAL_DIR/include $ss_kde_include_test_dirs"

			AC_SS_FIND_FILE_PATH(kwin.h,$ss_kde_include_test_dirs,SS_KDE_INCLUDE_DIR)

			if test "$SS_KDE_INCLUDE_DIR" = "FAILED"; then
				AC_MSG_RESULT(FAILED)
				SS_KDE_REQUESTED="no"
			else
				AC_MSG_RESULT(found in $SS_KDE_INCLUDE_DIR)
			fi
		else
			AC_MSG_RESULT(using user supplied path for the KDE include files $SS_KDE_INCLUDE_DIR)
			AC_SUBST(SS_KDE_INCLUDE_DIR)
		fi
	]
)

AC_DEFUN(AC_SS_FIND_KDE_LIBRARY_DIR,
	[
		AC_ARG_WITH(kde-libraries,
			[  --with-kde-libraries=DIR
                          directory the KDE libraries are in],
			[SS_KDE_LIBRARY_DIR="$withval"],
			[SS_KDE_LIBRARY_DIR="NOTGIVEN"]
		)
		if test "$SS_KDE_LIBRARY_DIR" = "NOTGIVEN"; then
			AC_MSG_CHECKING(for the KDE libraries)

			ss_kde_library_test_dirs="/lib /usr/lib /usr/local/lib /usr/kde/lib /opt/kde/lib /opt/lib /usr/opt/kde/lib /usr/local/kde/lib /usr/X11R6/lib /usr/lib/kde/lib /usr/lib/kde /usr/local/lib/kde/lib"
			test -n "$KDEDIR" && ss_kde_library_test_dirs="$KDEDIR/lib $KDEDIR $ss_kde_library_test_dirs"
			ss_kde_library_test_dirs="$ss_kde_library_test_dirs /usr/local/lib/kde /usr/X11R6/lib/kde /usr/build/kde/lib"
			if test -n "$SS_KDE_GENERAL_DIR"; then
				ss_kde_library_test_dirs="$SS_KDE_GENERAL_DIR/lib $ss_kde_library_test_dirs"
			fi

			AC_SS_FIND_FILE_PATH(libkdecore.so.3.0.0 libkdecore.so.3.0 libkdecore.a libkdecore.so,$ss_kde_library_test_dirs,SS_KDE_LIBRARY_DIR)

			if test "$SS_KDE_LIBRARY_DIR" = "FAILED"; then
				AC_MSG_RESULT(FAILED)
				SS_KDE_REQUESTED="no"
			else
				AC_MSG_RESULT(found in $SS_KDE_LIBRARY_DIR)
			fi
		else
			AC_MSG_RESULT(using user supplied path for the KDE libraries $SS_KDE_LIBRARY_DIR)
			AC_SUBST(SS_KDE_LIBRARY_DIR)
		fi
	]
)

dnl ------------------------------------------------------------------------
dnl Make sure we can compile Qt and X stuff
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_ENSURE_CAN_COMPILE_X_QT_AND_KDE,
	[
		AC_ARG_WITH(kde-check,
			[  --without-kde-check     do not check whether KDE compiles],
			[SS_CHECK_KQTX_COMPILE="$withval"],
			[SS_CHECK_KQTX_COMPILE="yes"]
		)
		if test "$SS_CHECK_KQTX_COMPILE" = "yes"; then
			AC_MSG_CHECKING(if we can compile an X-Qt-KDE application)
			AC_LANG_CPLUSPLUS
			ss_save_CPPFLAGS="$CPPFLAGS"
			ss_save_CXXFLAGS="$CXXFLAGS"
			ss_save_LDFLAGS="$LDFLAGS"
			ss_save_LIBS="$LIBS"
			CPPFLAGS="-I$SS_QT_INCLUDE_DIR -I$SS_KDE_INCLUDE_DIR $CPPFLAGS $X_CFLAGS"
			if test -n "$SS_OTHER_INCDIRS"; then
				CPPFLAGS="-I$SS_OTHER_INCDIRS $CPPFLAGS"
			fi
			CXXFLAGS="-O -Wall -g $CXXFLAGS"
			LIBS="-L$SS_KDE_LIBRARY_DIR -L$SS_QT_LIBRARY_DIR $X_ $X_EXTRA_LIBS $X_LIBS $LIBS -lkdecore -l$SS_QTLIB_NAME -lXext -lX11 -ldl"
			if test -n "$SS_OTHER_LIBDIRS"; then
				LIBS="-L$SS_OTHER_LIBDIRS $LIBS"
			fi
			LDFLAGS="-s $LDFLAGS"
			AC_TRY_LINK(
				[
					#include "kapp.h"
					#include "kwin.h"
				],[
					int a = KWin::currentDesktop();
					KApplication app(a,0,"kvirc");
				],
				[SS_KQTX_LINKED_OK="TRUE"],
				[SS_KQTX_LINKED_OK="FALSE"]
			)
			if test "$SS_KQTX_LINKED_OK" = "FALSE"; then
				SS_KDE_REQUESTED="no"
				AC_MSG_RESULT(FAILED)
			else
				AC_MSG_RESULT(success)
			fi
			LIBS="$ss_save_LIBS"
			LDFLAGS="$ss_save_LDFLAGS"
			CXXFLAGS="$ss_save_CXXFLAGS"
			CPPFLAGS="$ss_save_CPPFLAGS"
		fi
	]
)
dnl ------------------------------------------------------------------------
dnl Find the dl library
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_CHECK_IF_PLUGIN_SUPPORT_IS_REQUESTED,
	[
		AC_MSG_CHECKING(if plugin support is requested)
		AC_ARG_WITH(plugin-support,
			[  --without-plugin-support
                          do not build with plugin support],
			[SS_PLUGIN_REQUESTED="$withval"],
			[SS_PLUGIN_REQUESTED="yes"]
		)
		AC_MSG_RESULT($SS_PLUGIN_REQUESTED)
	]
)

AC_DEFUN(AC_SS_FIND_DL,
	[
		AC_SS_CHECK_IF_PLUGIN_SUPPORT_IS_REQUESTED
		if test "$SS_PLUGIN_REQUESTED" = "yes"; then
			AC_MSG_CHECKING([for dlopen capabilities])

			SS_LINK_TO_LIBDL="no"

			SS_SAVE_LIBS="$LIBS"
			LIBS="-rdynamic"
			SS_SAVE_CFLAGS="$CFLAGS";
			if test -n "$SS_OTHER_INCDIRS"; then
				CFLAGS="-I$SS_OTHER_INCDIRS $CFLAGS"
			fi
			AC_TRY_LINK(
				[#include <dlfcn.h>],
				[dlopen("/lib/libc.so.6",RTLD_NOW);],
				[eval "SS_DLOPEN_OK=yes"],
				[eval "SS_DLOPEN_OK=no"]
			)
			LIBS="$SS_SAVE_LIBS"
			CFLAGS="$SS_SAVE_CFLAGS"
			if test "$SS_DLOPEN_OK" = "yes"; then
				AC_MSG_RESULT(built into libc)
				AC_DEFINE_UNQUOTED(HAVE_LIBDL)
			else
				SS_SAVE_LIBS="$LIBS"
				SS_SAVE_CFLAGS="$CFLAGS";
				if test -n "$SS_OTHER_LIBDIRS"; then
					LIBS="-L$SS_OTHER_LIBDIRS -rdynamic -ldl"
				else
					LIBS="-rdynamic -ldl"
				fi
				if test -n "$SS_OTHER_INCDIRS"; then
					CFLAGS="-I$SS_OTHER_INCDIRS $CFLAGS"
				fi
				AC_TRY_LINK(
					[#include <dlfcn.h>],
					[dlopen("/usr/lib/libdl.so",RTLD_NOW);],
					[eval "SS_DLOPEN_OK=yes"],
					[eval "SS_DLOPEN_OK=no"]
				)
				LIBS="$SS_SAVE_LIBS"
				CFLAGS="$SS_SAVE_CFLAGS"
				if test "$SS_DLOPEN_OK" = "yes"; then
					SS_LINK_TO_LIBDL="yes"
					AC_MSG_RESULT(external in libdl)
					AC_DEFINE_UNQUOTED(HAVE_LIBDL)
				else
					AC_MSG_RESULT([none available])
					SS_PLUGIN_REQUESTED="no";
				fi
			fi
		fi
	]
)
dnl ------------------------------------------------------------------------
dnl Find a file path (list of files, list of dirs, variable)
dnl Searches for one of the files in the directories specified
dnl Sets the veriable to "FAILED" or to the path where the file was found
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_FIND_FILE_PATH,
	[
		$3="FAILED"
		for a_dir in $2; do
			for a_file in $1; do
				if test -r "$a_dir/$a_file"; then
					$3=$a_dir
					break 2
				fi
			done
		done
	]
)

AC_DEFUN(AC_SS_FIND_FILE_PATH_EXT,
	[
		$4="FAILED"
		for a_dir in $2; do
			for a_semidir in $3; do
				for a_file in $1; do
					if test -r "$a_dir/$a_semidir/$a_file"; then
						$4="$a_dir/$a_semidir"
						break 3
					fi
				done
			done
		done
	]
)
dnl ------------------------------------------------------------------------
dnl Find the Qt library and includes.
dnl Sets SS_QT_INCLUDE_DIR, SS_QT_LIBRARY_DIR, SS_QT_MOC
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_FIND_QT,
	[
		AC_SS_CHECK_QT_MT
		AC_SS_CHECK_QT_NAME
		dnl AC_SS_FIND_QT_GENERAL_DIR
		AC_SS_FIND_QT_INCLUDE_DIR
		AC_SS_FIND_QT_LIBRARY_DIR
		AC_SS_FIND_QT_MOC
		AC_SS_ENSURE_CAN_COMPILE_X_AND_QT
	]
)

AC_DEFUN(AC_SS_CHECK_QT_MT,
	[
		AC_ARG_ENABLE(mt,
			[  --disable-mt            link to single-threaded Qt],
			[SS_USE_QT_MT=$enableval],
			[SS_USE_QT_MT=yes]
		)
		USING_QT_MT=""
		if test "x$SS_USE_QT_MT" = "xyes"; then
			case $host in
			*-*-linux-*)
			;;
			*)
			AC_MSG_WARN([mt may not be supported on $host])
			AC_MSG_WARN([configure with --disable-mt if you have problems])
			;;
			esac
			USING_QT_MT=", using -mt"
		fi
	]
)

AC_DEFUN(AC_SS_CHECK_QT_NAME,
	[
		SS_QTLIB_NAME="qt"
		AC_MSG_CHECKING([for the Qt library name])
		AC_ARG_WITH(qt-lib-name,
			[  --with-qt-lib-name=NAME use NAME instead of 'qt' for the Qt library],
			[SS_QTLIB_NAME="$withval"],
			[SS_QTLIB_NAME="qt"]
		)
		if test "x$SS_USE_QT_MT" = "xyes"; then
			SS_QTLIB_NAME="$SS_QTLIB_NAME-mt"
		fi
		AC_MSG_RESULT($SS_QTLIB_NAME)
	]
)

AC_DEFUN(AC_SS_FIND_QT_GENERAL_DIR,
	[
		SS_QTDIR="$QTDIR"
		AC_MSG_CHECKING([for the Qt base directory])
		AC_ARG_WITH(qt-dir,
			[  --with-qt-dir=DIR       base directory of the Qt installation],
			[SS_QTDIR="$withval"],
			[SS_QTDIR="$QTDIR"]
		)
		ss_qt_general_test_dirs="/usr/local /usr /usr/build /usr/X11R6 /usr/lib /usr/src /usr/local/lib /"
		ss_qt_general_test_dirs="$ss_qt_general_test_dirs /opt/kde /kde /usr/kde /usr/local/kde /usr/opt/kde"
		if test -n "$HOME"; then
			ss_qt_general_test_dirs="$ss_qt_general_test_dirs $HOME $HOME/lib"
		fi
		if test -n "$QTDIR"; then
			ss_qt_general_test_dirs="$QTDIR $ss_qt_general_test_dirs"
		fi
		if test -n "$SS_QTDIR"; then
			ss_qt_general_test_dirs="$SS_QTDIR $ss_qt_general_test_dirs"
		fi
		ss_qt_general_test_path_suffix=". qt3 qt.3 qt-3 qt-3.1 qt-3.0 qt-3.0.1 qt-3.0.0 qt2 qt.2 qt-2 qt-2.3 qt-2.3.2 qt-2.3.1 qt-2.3.0 qt-2.2 qt-2.2.4 qt-2.2.3 qt-2.2.2 qt-2.2.1 qt-2.2.0"

		AC_SS_FIND_FILE_PATH_EXT(include/qinputdialog.h,$ss_qt_general_test_dirs,$ss_qt_general_test_path_suffix,SS_QT_GENERAL_DIR)

		if test "$SS_QT_GENERAL_DIR" = "FAILED"; then
			AC_MSG_RESULT(failed, continuing with normal checks)
			unset SS_QT_GENERAL_DIR
		else
			AC_MSG_RESULT(seems to be $SS_QT_GENERAL_DIR)
		fi
	]
)

AC_DEFUN(AC_SS_FIND_QT_INCLUDE_DIR,
	[
		AC_ARG_WITH(qt-includes,
			[  --with-qt-includes=DIR  directory the Qt headers are in],
			[SS_QT_INCLUDE_DIR="$withval"],
			[SS_QT_INCLUDE_DIR="NOTGIVEN"]
		)
		if test "$SS_QT_INCLUDE_DIR" = "NOTGIVEN"; then
			AC_MSG_CHECKING(for the Qt header files)

			ss_qt_include_test_dirs="$SS_QT_GENERAL_DIR/include /usr/include/qt3 /include /usr/include /usr/local/include /usr/qt/include /usr/local/qt/include /usr/X11R6/include /usr/lib/qt/include /usr/lib/qt /usr/include/X11"
			test -n "$QTDIR" && ss_qt_include_test_dirs="$QTDIR/include $QTDIR $ss_qt_include_test_dirs"
			ss_qt_include_test_dirs="$ss_qt_include_test_dirs /usr/include/qt /usr/local/include/qt /include/qt /usr/X11R6/include/qt /usr/build/qt/include"
			ss_qt_include_test_dirs="$ss_qt_include_test_dirs /usr/include/qt2 /usr/local/include/qt2 /include/qt2 /usr/X11R6/include/qt2 /usr/include/X11/qt2"
			test -n "$SS_QT_GENERAL_DIR" && ss_qt_include_test_dirs="$SS_QT_GENERAL_DIR/include $ss_qt_include_test_dirs"

			AC_SS_FIND_FILE_PATH(qiconview.h,$ss_qt_include_test_dirs,SS_QT_INCLUDE_DIR)

			if test "$SS_QT_INCLUDE_DIR" = "FAILED"; then
				AC_MSG_RESULT(FAILED)
				echo "################################################################################"
				echo "### CONFIGURE ERROR:"
				echo "###"
				echo "### Cannot find Qt >= 3.0.0 header files."
				echo "### Make sure that Qt is correctly installed on your system,"
				echo "### and that the Qt version is the one requested by this version of KVIrc."
				echo "### Try to run configure again, this time passing the --with-qt-includes=DIR"
				echo "### option (see ./configure --help)."
				echo "### You may also take a look at the config.log file in this directory,"
				echo "### that will tell you which check has failed and maybe more about the reason"
				echo "### of the failure."
				echo "### If you're feeling that this may be a bug in this configure script"
				echo "### and want to report this, please include your configure script,"
				echo "### the config.log file, as well as the complete configure output."
				echo "################################################################################"
				AC_MSG_ERROR(["This was a fatal one... aborting"])
			else
				AC_MSG_RESULT(found in $SS_QT_INCLUDE_DIR)
				dnl AC_SUBST(SS_QT_INCLUDE_DIR)
			fi

		else
			AC_MSG_RESULT(using user-supplied Qt header directory $SS_QT_INCLUDE_DIR)
			dnl AC_SUBST(SS_QT_INCLUDE_DIR)
		fi
		if test "x$SS_USE_QT_MT" = "xyes"; then
			SS_QT_INCLUDE_DIR="$SS_QT_INCLUDE_DIR"
			SS_FLAGS_INCDIRS="$SS_FLAGS_INCDIRS -DQT_THREAD_SUPPORT"
		fi
	]
)

AC_DEFUN(AC_SS_FIND_QT_LIBRARY_DIR,
	[
		AC_ARG_WITH(qt-libraries,
			[  --with-qt-libraries=DIR directory the Qt library is in],
			[SS_QT_LIBRARY_DIR="$withval"],
			[SS_QT_LIBRARY_DIR="NOTGIVEN"]
		)
		if test "$SS_QT_LIBRARY_DIR" = "NOTGIVEN"; then
			AC_MSG_CHECKING(for the Qt library)

			ss_qt_library_test_dirs="$SS_QT_GENERAL_DIR/lib /lib /usr/lib /usr/local/lib /usr/qt/lib /usr/local/qt/lib /usr/X11R6/lib /usr/lib/qt/lib /usr/lib/qt /usr/local/lib/qt/lib"
			test -n "$QTDIR" && ss_qt_library_test_dirs="$QTDIR/lib $QTDIR $ss_qt_library_test_dirs"
			ss_qt_library_test_dirs="$ss_qt_library_test_dirs /usr/local/lib/qt /usr/X11R6/lib/qt /usr/build/qt/lib"
			if test -n "$SS_QT_GENERAL_DIR"; then
				ss_qt_library_test_dirs="$SS_QT_GENERAL_DIR/lib $ss_qt_library_test_dirs"
			fi

			AC_SS_FIND_FILE_PATH(libqt.so.2.3 libqt.so.2.2 libqt.so.2 libqt.a libqt-mt.so libqt.so libqt.so.3 libqt.so.3.0 libqt.so.3.1 lib$SS_QTLIB_NAME.so lib$SS_QTLIB_NAME.a,$ss_qt_library_test_dirs,SS_QT_LIBRARY_DIR)

			if test "$SS_QT_LIBRARY_DIR" = "FAILED"; then
				AC_MSG_RESULT(FAILED)
				echo "################################################################################"
				echo "### CONFIGURE ERROR:"
				echo "###"
				echo "### Cannot find a Qt >= 3.0.0 library file."
				echo "### Make sure that Qt is correctly installed on your system,"
				echo "### and that the Qt version is the one requested by this version of KVIrc."
				echo "### Try to run configure again, this time passing the --with-qt-libraries=DIR"
				echo "### option (see ./configure --help)."
				echo "### You may also take a look at the config.log file in this directory,"
				echo "### that will tell you which check has failed and maybe more about the reason"
				echo "### of the failure."
				echo "### If you feel this may be a bug in this configure script"
				echo "### and want to report this, please include your configure script,"
				echo "### and config.log file, as well as the complete configure output."
				echo "################################################################################"
				AC_MSG_ERROR(["This was a fatal one... aborting"])
			else
				AC_MSG_RESULT(found in $SS_QT_LIBRARY_DIR$USING_QT_MT)
				dnl AC_SUBST(SS_QT_LIBRARY_DIR)
			fi
		else
			AC_MSG_RESULT(using user supplied path for the Qt library $SS_QT_LIBRARY_DIR)
			dnl AC_SUBST(SS_QT_LIBRARY_DIR)
		fi
	]
)

dnl ------------------------------------------------------------------------
dnl Find the Qt meta object preprocessor.
dnl Sets SS_MOC to the path/filename of moc
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_FIND_QT_MOC,
	[
		AC_ARG_WITH(qt-moc,
			[  --with-qt-moc=PATH      filename of the Qt meta object compiler],
			[SS_QT_MOC="$withval"],
			[SS_QT_MOC="NOTGIVEN"]
		)
		if test "$SS_QT_MOC" = "NOTGIVEN"; then
			unset SS_QT_MOC
			if test -n "$SS_QT_GENERAL_DIR"; then
				AC_PATH_PROG(SS_QT_MOC,moc,"FAILED",$SS_QT_GENERAL_DIR/bin:$QTDIR/bin:$PATH:/usr/bin:/bin:/usr/local/bin:/usr/X11R6/bin:/usr/lib/qt/bin:/usr/local/qt/bin:/usr/X11R6/qt/bin:/usr/qt/bin:/usr/build/qt/bin)
			else
				AC_PATH_PROG(SS_QT_MOC,moc,"FAILED",$QTDIR/bin:$PATH:/usr/bin:/bin:/usr/local/bin:/usr/X11R6/bin:/usr/lib/qt/bin:/usr/local/qt/bin:/usr/X11R6/qt/bin:/usr/qt/bin:/usr/build/qt/bin)
			fi

			if test "$SS_QT_MOC" = "FAILED"; then
				AC_MSG_RESULT(FAILED)
				echo "################################################################################"
				echo "### CONFIGURE ERROR:"
				echo "###"
				echo "### The Qt meta object compiler cannot be found."
				echo "### Make sure that qt is correctly installed on your system,"
				echo "### and the qt version is the one requested by this version of kvirc."
				echo "### Try to run configure again, this time passing the --with-qt-moc"
				echo "### option (see ./configure --help)."
				echo "### You may also take a look at the config.log file in this directory,"
				echo "### that will tell you which check has failed and maybe more about the reason"
				echo "### of the failure."
				echo "### If you feel that this may be a bug in this configure script"
				echo "### and want to report this, please include your configure script,"
				echo "### and config.log file, as well as the complete configure output."
				echo "################################################################################"
				AC_MSG_ERROR(["This was a fatal one... aborting"])
			fi
		else
			AC_MSG_RESULT(using user supplied path for moc $SS_QT_MOC)
			dnl AC_SUBST(SS_QT_MOC)
		fi
	]
)

dnl ------------------------------------------------------------------------
dnl Make sure we can compile Qt and X stuff
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_ENSURE_CAN_COMPILE_X_AND_QT,
	[
		AC_ARG_WITH(qt-check,
			[  --without-qt-check      do not check whether Qt compiles],
			[SS_CHECK_QTX_COMPILE="$withval"],
			[SS_CHECK_QTX_COMPILE="yes"]
		)
		if test "$SS_CHECK_QTX_COMPILE" = "yes"; then
			AC_MSG_CHECKING(if we can compile an X-Qt application)
			AC_LANG_CPLUSPLUS
			ss_save_CPPFLAGS="$CPPFLAGS"
			ss_save_CXXFLAGS="$CXXFLAGS"
			ss_save_LDFLAGS="$LDFLAGS"
			ss_save_LIBS="$LIBS"
			CPPFLAGS="-I$SS_QT_INCLUDE_DIR $CPPFLAGS $X_CFLAGS"
			if test -n "$SS_OTHER_INCDIRS"; then
				CPPFLAGS="-I$SS_OTHER_INCDIRS $CPPFLAGS"
			fi
			CXXFLAGS="-O -Wall -g $CXXFLAGS"
			LIBS="-L$SS_QT_LIBRARY_DIR $X_EXTRA_LIBS $X_LIBS $LIBS -l$SS_QTLIB_NAME -lXext -lX11"
			if test -n "$SS_OTHER_LIBDIRS"; then
				LIBS="-L$SS_OTHER_LIBDIRS $LIBS"
			fi
			LDFLAGS="-s $LDFLAGS"
			AC_TRY_LINK(
				[
					#include <qglobal.h>
					#if (QT_VERSION < 300)
					#error "Bad Qt version: need at least 3.0.0"
					#endif
				],
				[return 0;],
				[SS_QTX_LINKED_OK="TRUE"],
				[SS_QTX_LINKED_OK="FALSE"]
			)
			if test "$SS_QTX_LINKED_OK" = "FALSE"; then
				AC_MSG_RESULT(FAILED);
				echo "################################################################################"
				echo "### CONFIGURE ERROR:"
				echo "###"
				echo "### Failed to compile the Qt library test program."
				echo "### This may be a bad sign :)"
				echo "### First of all, make sure that Qt is correctly installed on your system,"
				echo "### and that the Qt version is the one requested by this version of KVIrc."
				echo "### Ensure that you have only one copy of Qt visible at a time."
				echo "### You may also take a look at the config.log file in this directory,"
				echo "### which will tell you which check has failed and maybe more about the reason"
				echo "### of the failure."
				echo "### The CPPFLAGS used were:"
				for a_flag in $CPPFLAGS ; do
					echo "###   $a_flag"
				done
				echo "### The CXXFLAGS used were:"
				for a_flag in $CXXFLAGS ; do
					echo "###   $a_flag"
				done
				echo "### The LIBS used were:"
				for a_flag in $LIBS ; do
					echo "###   $a_flag"
				done
				echo "### The LDFLAGS used were:"
				for a_flag in $LDFLAGS ; do
					echo "###   $a_flag"
				done
				echo "### If you're sure that Qt is correctly installed , you may force configure"
				echo "### to skip this check and try to compile KVIrc anyway."
				echo "### Try using the --without-qt-check option."
				echo "### If you're feeling that this may be a bug in this configure script"
				echo "### and want to report this, please include your configure script,"
				echo "### and config.log file, as well as the complete configure output."
				echo "################################################################################"
				AC_MSG_ERROR(["This was a fatal one... aborting"])
			fi
			AC_MSG_RESULT(success$USING_QT_MT)

			AC_MSG_CHECKING([Qt version])
			AC_TRY_RUN(dnl
				[
					#include <qglobal.h>
					int main()
					{
						if (QT_VERSION < 300)
							return -1;
						else
							return 0;
					}
				],
				[SS_QT_VERSION="3"],
				[SS_QT_VERSION="2"],
				[AC_MSG_RESULT(cross compiling. We hope so)]
			)
			if test "x$SS_QT_VERSION" = "x3"; then
				AC_MSG_RESULT(using Qt version 3.x)
			else
				AC_MSG_RESULT(using Qt version 2.x)
			fi

			AC_MSG_CHECKING([for JPEG support in Qt])
			if test "x$SS_QT_VERSION" = "x3"; then
				SS_QT_JPEG="yes"
			else
				AC_TRY_LINK(
					[#include <qjpegio.h>],
					[qInitJpegIO();],
					[SS_QT_JPEG="yes"],
					[SS_QT_JPEG="no"]
				)
			fi
			if test "x$SS_QT_JPEG" = "xyes"; then
				AC_MSG_RESULT(yes, disabling internal JPEG support)
			else
				AC_MSG_RESULT(no, using internal JPEG support instead)
			fi

			SS_FREETYPE_LIBS=
			SS_FREETYPE_CFLAGS=
			AC_PATH_PROG(FREETYPE_CONFIG, freetype-config, no)
			if test "x$FREETYPE_CONFIG" != "xno" ; then
				SS_FREETYPE_CFLAGS=`freetype-config --cflags`
				SS_FREETYPE_LIBS=`freetype-config --libs`
			fi

			LIBS="$ss_save_LIBS"
			LDFLAGS="$ss_save_LDFLAGS"
			CXXFLAGS="$ss_save_CXXFLAGS"
			CPPFLAGS="$ss_save_CPPFLAGS"
		fi
	]
)
dnl -------------------------------------------------------------------------
dnl Find the X libraries and includes
dnl Sets SS_X_INCLUDE_DIR, SS_X_LIBRARY_DIR,
dnl      SS_X_EXTRA_GCC_FLAGS, SS_X_EXTRA_LDD_FLAGS
dnl -------------------------------------------------------------------------

AC_DEFUN(AC_SS_FIND_X,
	[
		AC_PATH_XTRA
		if test -n "$no_x"; then
			echo "################################################################################"
			echo "### CONFIGURE ERROR:"
			echo "###"
			echo "### Cannot find the X libraries."
			echo "### Make sure that X is installed on your system, and try to run configure again,"
			echo "### this time passing the --x-includes and --x-libraries options."
			echo "### You may also take a look at the config.log file in this directory,"
			echo "### That will tell you which checks have failed and maybe more about the"
			echo "### reason of the failure."
			echo "################################################################################"
			AC_MSG_ERROR(["This was a fatal one... aborting"])
		else
			dnl Set the X include and lib dir
			SS_X_INCLUDE_DIR="$x_includes"
			SS_X_LIBRARY_DIR="$x_libraries"

			dnl ac_cpp_safe=$ac_cpp
			dnl ac_cpp='$CPPFLAGS $x_includes'
			dnl AC_CHECK_HEADERS(X11/extensions/XShm.h,have_mitshm=yes)
			dnl ac_cpp=$ac_cpp_safe

			dnl AC_MSG_CHECKING(if MIT-SHM support is requested)
			dnl AC_ARG_ENABLE(mitshm,
			dnl 	[  --enable-mitshm         use MIT-SHM for pixmap loading/saving [default=no]],
			dnl 	[AC_MSG_RESULT($enableval)
			dnl 	if test "$enableval" = yes; then
			dnl 		want_mitshm="yes"
			dnl 	fi],
			dnl 	[AC_MSG_RESULT(yes); want_mitshm="yes"]
			dnl )

			dnl if test "$want_mitshm" = "yes" -a "$have_mitshm" = "yes"; then
			dnl 	AC_SUBST(HAVE_MITSHM)
			dnl 	AC_DEFINE(HAVE_MITSHM,1,[Define if you want MIT-SHM support])
			dnl fi

			dnl Add any extra libs needed
			SS_X_EXTRA_LDD_FLAGS="$X_EXTRA_LIBS"
			for a_flag in "$X_LIBS"; do
				if test "$a_flag" != "-L$x_libraries"; then
					SS_X_EXTRA_LDD_FLAGS="$SS_X_EXTRA_LDD_FLAGS $a_flag"
				fi
			done

			dnl Add any extra gcc flags
			SS_X_EXTRA_GCC_FLAGS=""
			for a_flag in "$X_CFLAGS"; do
				if test "$a_flag" != "-I$x_includes"; then
					SS_X_EXTRA_GCC_FLAGS="$SS_X_EXTRA_GCC_FLAGS $a_flag"
				fi
			done

			dnl AC_SUBST(SS_X_INCLUDE_DIR)
			dnl AC_SUBST(SS_X_LIBRARY_DIR)
			dnl AC_SUBST(SS_X_EXTRA_GCC_FLAGS)
			dnl AC_SUBST(SS_X_EXTRA_LDD_FLAGS)
		fi
	]
)
dnl ------------------------------------------------------------------------
dnl Find the JPEG library
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_CHECK_IF_JPEG_SUPPORT_IS_REQUESTED,
	[
		AC_MSG_CHECKING(if JPEG support is requested)
		AC_ARG_WITH(jpeg-support,
			[  --without-jpeg-support  do not use internal JPEG support],
			[SS_JPEG_REQUESTED="$withval"],
			[SS_JPEG_REQUESTED="yes"]
		)
		AC_MSG_RESULT($SS_JPEG_REQUESTED)
	]
)

AC_DEFUN(AC_SS_FIND_JPEG,
	[
		AC_SS_CHECK_IF_JPEG_SUPPORT_IS_REQUESTED
		if test "$SS_JPEG_REQUESTED" = "yes"; then
			AC_MSG_CHECKING([for jpeglib])
			AC_CACHE_VAL(ac_cv_lib_jpeg,[
				ac_save_LIBS="$LIBS"
				ac_save_CFLAGS="$CFLAGS"
				if test -n "$SS_OTHER_INCDIRS"; then
					CFLAGS="-I$SS_OTHER_INCDIRS $CFLAGS"
				fi
				if test -n "$SS_OTHER_LIBDIRS"; then
					LIBS="-L$SS_OTHER_LIBDIRS -ljpeg"
				else
					LIBS="-ljpeg"
				fi

				AC_TRY_LINK(
					[ /* Override internal prototypes */
					struct jpeg_decompress_struct;
					typedef struct jpeg_decompress_struct * j_decompress_ptr;
					typedef int size_t;
					#ifdef __cplusplus
					extern "C" {
					#endif
						void jpeg_CreateDecompress(j_decompress_ptr cinfo,int version,size_t struct_size);
					#ifdef __cplusplus
					}
					#endif
					],
					[jpeg_CreateDecompress(0,0,0);],
					[eval "ac_cv_lib_jpeg=-ljpeg"],
					[eval "ac_cv_lib_jpeg=no"]
				)
				LIBS="$ac_save_LIBS"
				CFLAGS="$ac_save_CFLAGS"
			])
			if eval "test ! \"`echo $ac_cv_lib_jpeg`\" = no"; then
				SS_HAVE_LIBJPEG="yes"
				dnl AC_SUBST(SS_HAVE_LIBJPEG)
				AC_MSG_RESULT([OK, seems to work])
			else
				AC_MSG_RESULT([not found])
			fi
		fi
	]
)
dnl ------------------------------------------------------------------------
dnl Check if localisation support has to be compiled and installed
dnl ------------------------------------------------------------------------

AC_DEFUN(AC_SS_CHECK_LOCALIZE,
	[
		AC_MSG_CHECKING(if locale support is requested)
		AC_ARG_WITH(locale,
			[  --without-locale        do not enable localisation support],
			[SS_LOCALE_REQUESTED="$withval"],
			[SS_LOCALE_REQUESTED="yes"]
		)
		AC_MSG_RESULT($SS_LOCALE_REQUESTED)

		if test "$SS_LOCALE_REQUESTED" = "yes"; then
			AC_PATH_PROG(SS_MSGFMT,msgfmt,":",$PATH:/bin:/usr/bin:/bin:/usr/local/bin:/usr/X11R6/bin:/usr/lib/qt/bin:/usr/local/qt/bin:/usr/X11R6/qt/bin:/usr/qt/bin:/usr/build/qt/bin)
			if test "SS_MSGFMT" != ":"; then
				AC_DEFINE(COMPILE_NEED_LOCALE)
			fi
		else
			SS_MSGFMT=":"
		fi

		AC_SUBST(SS_MSGFMT)
	]
)
