using System;
using System.Collections;
using Gtk;
using GtkExtra;

namespace Kurush
{

public class ReportFrame : Frame
{
    public override void UpdateActions (Actions actions)
    {
        actions.EditMove.Enabled = false;
        actions.ViewExpandCollapse.Enabled = false;

        actions.FilePrint.Enabled = true;
        actions.EditProperties.Enabled = true;
        actions.EditDetails.Enabled = false;
        actions.EditAdd.Enabled = false;
        actions.EditRemove.Enabled = false;
    }

    [Glade.Widget] private VBox reportFrame;
    [Glade.Widget] private Label totalPagesLabel;
    [Glade.Widget] private SpinButton pageSpinButton;

    protected DateTime beginningDate;
    protected DateTime endingDate;
    protected bool showBeginningDate = false;

    protected PrintPreviewEx printPreview;
    private EventHandler spinButtonChangedHandler;

    public ReportFrame (FrameManager owner) : base (owner, "reportFrame")
    {
        beginningDate = Owner.DataBase.ReportStart;
        endingDate = DateTime.Today;

        printPreview = new PrintPreviewEx ();
        printPreview.Show ();
        reportFrame.PackStart (printPreview, true, true, 0);

        printPreview.Assigned += new EventHandler (PreviewAssigned);
        printPreview.Changed += new EventHandler (PreviewChanged);
        spinButtonChangedHandler =  new EventHandler (PageSpinButtonChanged);

        pageSpinButton.ValueChanged += spinButtonChangedHandler;
    }

    private void PreviewAssigned (object obj, EventArgs args)
    {
        pageSpinButton.ValueChanged -= spinButtonChangedHandler;
        pageSpinButton.SetRange (1, printPreview.Pages);
        totalPagesLabel.Text = printPreview.Pages.ToString ();
        pageSpinButton.ValueChanged += spinButtonChangedHandler;
    }

    private void PreviewChanged (object obj, EventArgs args)
    {
        pageSpinButton.ValueChanged -= spinButtonChangedHandler;
        pageSpinButton.Value = printPreview.Page;
        pageSpinButton.ValueChanged += spinButtonChangedHandler;
    }

    private void PageSpinButtonChanged (object obj, EventArgs args)
    {
        printPreview.RenderPage (pageSpinButton.ValueAsInt - 1);
    }

    private void on_firstButton_clicked (object obj, EventArgs args)
    {
        printPreview.RenderFirstPage ();
    }

    private void on_prevButton_clicked (object obj, EventArgs args)
    {
        printPreview.RenderPrevPage ();
    }

    private void on_nextButton_clicked (object obj, EventArgs args)
    {
        printPreview.RenderNextPage ();
    }

    private void on_lastButton_clicked (object obj, EventArgs args)
    {
        printPreview.RenderLastPage ();
    }

    private void on_zoomFitButton_clicked (object obj, EventArgs args)
    {
        printPreview.ZoomFit = true;
    }

    private void on_zoomAllButton_clicked (object obj, EventArgs args)
    {
        printPreview.ZoomAll ();
    }

    private void on_zoomInButton_clicked (object obj, EventArgs args)
    {
        printPreview.ZoomIn ();
    }

    private void on_zoomOutButton_clicked (object obj, EventArgs args)
    {
        printPreview.ZoomOut ();
    }

    private void on_refreshButton_clicked (object obj, EventArgs args)
    {
        Refresh ();
    }

    public virtual void Refresh ()
    {
    }

    public override void PrintReport ()
    {
        printPreview.Print ();
    }

    private void DateDialogResponse (object obj, ResponseArgs args)
    {
        DateDialog dialog = obj as DateDialog;

        if (args.ResponseId == ResponseType.Ok)
        {
            beginningDate = dialog.BeginningDate;
            endingDate =  dialog.EndingDate;
            Refresh ();
        }

        dialog.Self.Destroy ();
        RemoveDialog (dialog.Self);
    }

    public override void Properties ()
    {
        DateDialog dialog = new DateDialog (Owner.Toplevel as Window, showBeginningDate);
        dialog.BeginningDate = beginningDate;
        dialog.EndingDate = endingDate;
        AddDialog (dialog.Self);
        dialog.Response += new ResponseHandler (DateDialogResponse);
    }
}

}
